#!/usr/bin/env python3
from bluezero import adapter
from bluezero import central
from time import sleep
import sys
import zlib
import os
import datetime

BLE_OTA_SERVICE_UUID = "dac890c2-35a1-11ef-aba0-9b95565f4ffb"
BLE_OTA_CHARACTERISTIC_UUID_RX = "dac89194-35a1-11ef-aba1-b37714ad9a54"
BLE_OTA_CHARACTERISTIC_UUID_TX = "dac89266-35a1-11ef-aba2-0f0127bce478"
BLE_OTA_CHARACTERISTIC_UUID_MF_NAME = "dac89338-35a1-11ef-aba3-8746a2fdea8c"
BLE_OTA_CHARACTERISTIC_UUID_HW_NAME = "dac89414-35a1-11ef-aba4-7fa301ad5c49"
BLE_OTA_CHARACTERISTIC_UUID_HW_VER = "dac894e6-35a1-11ef-aba5-0fcd13588409"
BLE_OTA_CHARACTERISTIC_UUID_SW_NAME = "dac895b8-35a1-11ef-aba6-63ebb073a878"
BLE_OTA_CHARACTERISTIC_UUID_SW_VER = "dac89694-35a1-11ef-aba7-bf64db99d724"

OK = 0x00
NOK = 0x01
INCORRECT_FORMAT = 0x02
INCORRECT_FIRMWARE_SIZE = 0x03
CHECKSUM_ERROR = 0x04
INTERNAL_STORAGE_ERROR = 0x05
UPLOAD_DISABLED = 0x06

BEGIN = 0x10
PACKAGE = 0x11
END = 0x12

respToStr = {
    NOK: "Not ok",
    INCORRECT_FORMAT: "Incorrect format",
    INCORRECT_FIRMWARE_SIZE: "Incorrect firmware size",
    CHECKSUM_ERROR: "Checksum error",
    INTERNAL_STORAGE_ERROR: "Internal storage error",
    UPLOAD_DISABLED: "Upload disabled"
}

U8_BYTES_NUM = 1
U32_BYTES_NUM = 4
HEAD_BYTES_NUM = U8_BYTES_NUM
ATTR_SIZE_BYTES_NUM = U32_BYTES_NUM
BUFFER_SIZE_BYTES_NUM = U32_BYTES_NUM
BEGIN_RESP_BYTES_NUM = HEAD_BYTES_NUM + ATTR_SIZE_BYTES_NUM + BUFFER_SIZE_BYTES_NUM
HEAD_POS = 0
ATTR_SIZE_POS = HEAD_POS + HEAD_BYTES_NUM
BUFFER_SIZE_POS = ATTR_SIZE_POS + ATTR_SIZE_BYTES_NUM


def file_size(path):
    if os.path.isfile(path):
        file_info = os.stat(path)
        return file_info.st_size


def bytes_to_int(value):
    return int.from_bytes(value, 'little', signed=False)


def int_to_u8_bytes(value):
    return list(int.to_bytes(value, U8_BYTES_NUM, 'little', signed=False))


def int_to_u32_bytes(value):
    return list(int.to_bytes(value, U32_BYTES_NUM, 'little', signed=False))


def scan_ota_devices(adapter_address=None, timeout=5.0):
    for dongle in adapter.Adapter.available():
        if adapter_address and adapter_address.upper() != dongle.address():
            continue

        dongle.nearby_discovery(timeout=timeout)

        for dev in central.Central.available(dongle.address):
            if BLE_OTA_SERVICE_UUID.lower() in dev.uuids:
                yield dev


def handle_response(resp):
    resp = bytes_to_int(resp)
    if resp == OK:
        return True

    print(respToStr[resp])
    return False


def handle_begin_response(resp):
    respList = list(bytearray(resp))
    head = respList[HEAD_POS]
    if head != OK:
        print(respToStr[head])
        return

    if len(respList) != BEGIN_RESP_BYTES_NUM:
        print("Incorrect begin responce")
        return
    return bytes_to_int(respList[ATTR_SIZE_POS:BUFFER_SIZE_POS]), bytes_to_int(respList[BUFFER_SIZE_POS:])


def connect(dev):
    device = central.Central(adapter_addr=dev.adapter, device_addr=dev.address)
    rx_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_RX)
    tx_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_TX)
    mf_name_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_MF_NAME)
    hw_name_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_HW_NAME)
    hw_ver_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_HW_VER)
    sw_name_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_SW_NAME)
    sw_ver_char = device.add_characteristic(
        BLE_OTA_SERVICE_UUID, BLE_OTA_CHARACTERISTIC_UUID_SW_VER)

    print(f"Connecting to {dev.alias}")
    device.connect()
    if not device.connected:
        print("Didn't connect to device!")
        return

    try:
        print(", ".join([f"MF: {str(bytearray(mf_name_char.value), 'utf-8')}",
                         f"HW: {str(bytearray(hw_name_char.value), 'utf-8')}",
                         f"VER: {list(bytearray(hw_ver_char.value))}",
                         f"SW: {str(bytearray(sw_name_char.value), 'utf-8')}",
                         f"VER: {list(bytearray(sw_ver_char.value))}"]))
    except Exception as e:
        print(e)
        return

    return device, rx_char, tx_char


def upload(rx_char, tx_char, path):
    crc = 0
    uploaded_len = 0
    firmware_len = file_size(path)
    current_buffer_len = 0

    if not firmware_len:
        print(f"File not exist: {path}")
        return False

    rx_char.value = int_to_u8_bytes(BEGIN) + int_to_u32_bytes(firmware_len)
    begin_resp = handle_begin_response(tx_char.value)
    if not begin_resp:
        return False
    attr_size, buffer_size = begin_resp
    print(f"Begin upload: attr size: {attr_size}, buffer size: {buffer_size}")

    with open(path, 'rb') as f:
        while True:
            data = f.read(attr_size - HEAD_BYTES_NUM)
            if not len(data):
                break

            rx_char.value = int_to_u8_bytes(PACKAGE) + list(data)
            if current_buffer_len + len(data) > buffer_size:
                if not handle_response(tx_char.value):
                    return False
                current_buffer_len = 0
            current_buffer_len += len(data)

            uploaded_len += len(data)
            crc = zlib.crc32(data, crc)
            print(f"Uploaded: {uploaded_len}/{firmware_len}")

    rx_char.value = int_to_u8_bytes(END) + int_to_u32_bytes(crc)
    if not handle_response(tx_char.value):
        return False

    return True


def try_upload(rx_char, tx_char, path):
    time = datetime.datetime.now()

    try:
        if not upload(rx_char, tx_char, path):
            return False
    except Exception as e:
        print(e)
        return False

    upload_time = datetime.datetime.now() - time
    print(f"Installing. Upload time: {upload_time}")
    return True


def connect_and_upload(dev, path):
    res = connect(dev)
    if not res:
        return
    device, rx_char, tx_char = res

    if not try_upload(rx_char, tx_char, path):
        device.disconnect()
        return
    device.disconnect()
    sleep(1)

    res = connect(dev)
    if not res:
        return
    device, rx_char, tx_char = res

    device.disconnect()
    print("Success!")


def scan_and_upload(path):
    devices = list()

    print("Devices:")
    for device in scan_ota_devices():
        print(f"{len(devices)}. [{device.address}] {device.alias}")
        devices.append(device)

    if not len(devices):
        print("Device not found")
        exit()

    user_input = input("Chose device [0]: ")

    try:
        device_num = int(user_input)
        if device_num >= len(devices) or device_num < 0:
            print("Incorrect device number")
            exit()
    except ValueError:
        device_num = 0
        if len(user_input):
            print("Incorrect input")
            exit()

    connect_and_upload(devices[device_num], path)


if __name__ == '__main__':
    scan_and_upload(sys.argv[1])
