/*
 *
 * Copyright (c) [2020] by InvenSense, Inc.
 * 
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */
 
#include "Arduino.h"
#include "TAD214x.h"

static int i2c_write(void *serif, uint8_t reg, const uint16_t * wbuffer, uint32_t wlen);
static int i2c_read(void *serif, uint8_t reg, uint16_t * rbuffer, uint32_t rlen);
static int spi_write(void *serif, uint8_t reg, const uint16_t * wbuffer, uint32_t wlen);
static int spi_read(void *serif, uint8_t reg, uint16_t * rbuffer, uint32_t rlen);
static void sleep_us(uint32_t us);
static void int_encoder_cb(void);

// i2c and SPI interfaces are used from C driver callbacks, without any knowledge of the object
// As they are declared as static, they will be overriden each time a new TAD214x object is created
// i2c

/* SPI/I2C configuration for TAD I/O */
/* tad214x_I2C_ADDR = 0x17 with SA1=0 and SA0=0*/
/* tad214x_I2C_ADDR = 0x23 with SA1=0 and SA0=1*/
/* tad214x_I2C_ADDR = 0x56 with SA1=1 and SA0=0*/
/* tad214x_I2C_ADDR = 0x65 with SA1=1 and SA0=1*/
uint8_t i2c_address = 0;
static TwoWire *i2c = NULL;
#define I2C_DEFAULT_CLOCK 100000
#define I2C_MAX_CLOCK 4000000
#define TAD214x_I2C_ADDRESS 0x65
#define ARDUINO_I2C_BUFFER_LENGTH 32

// spi
static SPIClass *spi = NULL;
static uint8_t chip_select_id = 0;
#define SPI_READ 0x80
#define SPI_DEFAULT_CLOCK 10000000
#define SPI_MAX_CLOCK 10000000

#define MOSI 11 // a
#define SCK  13 // z
#define SS   10 
#define MISO 12 // b
#define CS   8

// ENC
static bool ENC_MODE = false;
static uint32_t encoder_position = 0;

// i2c/spi clock frequency
static uint32_t clk_freq = 0;

// TAD214x constructor for I2c interface
TAD214x::TAD214x(TwoWire &i2c_ref,bool lsb, uint32_t freq) {
  i2c = &i2c_ref; 
  i2c_address = TAD214x_I2C_ADDRESS | (lsb ? 0x1 : 0);
  if ((freq <= I2C_MAX_CLOCK) && (freq >= 100000))
  {
    clk_freq = freq;
  } else {
    clk_freq = I2C_DEFAULT_CLOCK;
  }
}

// TAD214x constructor for I2c interface, default frequency
TAD214x::TAD214x(TwoWire &i2c_ref,bool lsb) {
  i2c = &i2c_ref; 
  i2c_address = TAD214x_I2C_ADDRESS | (lsb ? 0x1 : 0);
  clk_freq = I2C_DEFAULT_CLOCK;
}

// TAD214x constructor for spi interface
TAD214x::TAD214x(SPIClass &spi_ref,uint8_t cs_id, uint32_t freq) {
  spi = &spi_ref;
  chip_select_id = cs_id; 
  if ((freq <= SPI_MAX_CLOCK) && (freq >= 100000))
  {
    clk_freq = freq;
  } else {
    clk_freq = SPI_DEFAULT_CLOCK;
  }
}

// TAD214x constructor for spi interface, default frequency
TAD214x::TAD214x(SPIClass &spi_ref,uint8_t cs_id) {
  spi = &spi_ref;
  chip_select_id = cs_id; 
  clk_freq = SPI_DEFAULT_CLOCK;
}

// TAD214x constructor for encoder interface
TAD214x::TAD214x(void) {
	ENC_MODE = true;
}


/* starts communication with the TAD214x */
int TAD214x::begin() {
  if (i2c != NULL) {
    pinMode(CS, OUTPUT);
    digitalWrite(CS, HIGH);
    pinMode(MISO, OUTPUT);
    digitalWrite(MISO, HIGH);

    i2c->begin();	
    i2c->setClock(clk_freq);
    tad214x_driver.serif.if_tad = IF_I2C;
    tad214x_driver.serif.read_reg  = i2c_read;
    tad214x_driver.serif.write_reg = i2c_write;
    tad214x_driver.serif.max_read  = 8;
    tad214x_driver.serif.max_write = 6;
  } else if(ENC_MODE == false) {
    pinMode(chip_select_id,OUTPUT);
    digitalWrite(chip_select_id,HIGH);

    spi->begin();
    tad214x_driver.serif.if_tad = IF_SPI;
    tad214x_driver.serif.read_reg  = spi_read;
    tad214x_driver.serif.write_reg = spi_write;
    tad214x_driver.serif.max_read  = 8;
    tad214x_driver.serif.max_write = 6;
  } else { // Encoder mode
    pinMode(CS, OUTPUT);
    digitalWrite(CS, HIGH);
    pinMode(MOSI, INPUT);
	attachInterrupt(MOSI,int_encoder_cb,CHANGE);
    pinMode(MISO, INPUT);
    attachInterrupt(MISO,int_encoder_cb,CHANGE);
    pinMode(SCK, INPUT);
	attachInterrupt(SCK,int_encoder_cb,CHANGE);
    tad214x_driver.serif.if_tad = IF_ENC;
  }

  if(ENC_MODE == false)
  {
    TAD214x_Init(&tad214x_driver);
    sleep_us(30000);
    TAD214x_SetMode(&tad214x_driver, TAD214X_MODE_CONT);
    sleep_us(30000);	
    TAD214x_Unlock(&tad214x_driver);
    TAD214x_ProgramModeEnable(&tad214x_driver);
    TAD214x_PredictionDisable(&tad214x_driver);
    TAD214x_ProgramModeDisable(&tad214x_driver);
    TAD214x_Lock(&tad214x_driver);
  }

  sleep_us(3000);
 
  return 0;
}

int TAD214x::getDataFromEncoder(float& angle)
{
  angle = (encoder_position*360.0)/16384.0 - 180.0;
  return INV_ERROR_SUCCESS;
}

int TAD214x::getDataFromRegisters(float& angle, float& temp)
{
  int rc = INV_ERROR_SUCCESS;
  uint16_t  TMRData = 0;
  int16_t  TempData = 0;
  uint64_t irq_timestamp = 0;

  /* Read tad214x data */
  rc = TAD214x_GetData(&tad214x_driver, &TMRData, &TempData);
  if (rc == 1) {
    return rc;
  }
  angle = fmod((TMRData*360.0)/65536.0 + 180.0,360.0) - 180;
  temp = 25.0 + TempData* 0.00625;

  return rc;
}

int TAD214x::getODR(void)
{
  int rc = INV_ERROR_SUCCESS;
  TAD214X_ODR_t odr=TAD214X_ODR_10;
  
  /* TAD214x Set Odr at 100Hz and LPM Mode */
  rc = TAD214x_GetODR(&tad214x_driver, &odr);
  return (int)odr;
}

int TAD214x::getMode(void)
{
  int rc = INV_ERROR_SUCCESS;
  TAD214X_PowerMode_t mode =TAD214X_MODE_SBY;
  
  /* TAD214x Set Odr at 100Hz and LPM Mode */
  rc = TAD214x_GetMode(&tad214x_driver, &mode);
  return (int)mode;
}

int TAD214x::setODR(TAD214X_ODR_t odr)
{
  int rc = INV_ERROR_SUCCESS;
  /* TAD214x Set Odr at 100Hz and LPM Mode */
  rc = TAD214x_SetODR(&tad214x_driver, odr);
  return rc;
}

int TAD214x::setMode(TAD214X_PowerMode_t mode)
{
  int rc = INV_ERROR_SUCCESS;
  /* TAD214x Set Mode */
  rc = TAD214x_SetMode(&tad214x_driver, mode);
  return rc;
}

int TAD214x::enableInterrupt(uint8_t intpin, TAD214x_irq_handler handler)
{
  int rc = INV_ERROR_SUCCESS;
  
  if(handler == NULL) {
    return INV_ERROR;
  }

  rc |= setup_irq(intpin, handler);

  /* clear the register to get the interrupt */
  float angle, temp;
  getDataFromRegisters(angle, temp);
  return rc;
}

int TAD214x::setup_irq(uint8_t intpin, TAD214x_irq_handler handler)
{
  pinMode(intpin,INPUT);
  attachInterrupt(intpin,handler,HIGH);
  return INV_ERROR_SUCCESS;
}

int TAD214x::write_reg(uint8_t addr, int *value)
{
  int rc = INV_ERROR_SUCCESS;
  rc = TAD214x_WriteReg(&tad214x_driver.serif, addr, (uint16_t *)value);
  return rc;
}

int TAD214x::read_reg(uint8_t addr, int *value)
{
  int rc = INV_ERROR_SUCCESS;
  rc = TAD214x_ReadReg(&tad214x_driver.serif, addr, (uint16_t *)value);
  return rc;
}

static void memswap16( void* ptr1, unsigned int bytes )
{
  unsigned char* s1 = (unsigned char*)ptr1;
  unsigned char tmp;
  for( unsigned int i = 0; i < bytes; i=i+2 )
  {
    tmp = s1[i];
    s1[i] = s1[i+1];
    s1[i+1] = tmp;
  }
}

// Polynomial for CRC-8-SAE J1850: x^8 + x^4 + x^3 + x^2 + 1 (0x1D)
#define POLYNOMIAL 0x1D
#define INITIAL_VALUE 0xFF

static unsigned char crc8_sae_j1850(const unsigned char *data, unsigned int length) {
    unsigned char crc = INITIAL_VALUE;
    for (unsigned int i = 0; i < length; i++) {
        crc ^= data[i];
        for (unsigned char bit = 0; bit < 8; bit++) {
            if (crc & 0x80) {
                crc = (crc << 1) ^ POLYNOMIAL;
            } else {
                crc <<= 1;
            }
        }
    }
    return crc ^ 0xFF;
}

static int i2c_write(void *serif, uint8_t reg, const uint16_t * wbuffer, uint32_t wlen) {
  (void)serif;
  uint8_t buf[257];

  buf[0] = reg;
  memcpy(&buf[1],(uint8_t *)wbuffer, wlen*2);
  memswap16(&buf[1],wlen*2);
  buf[wlen*2+1] = crc8_sae_j1850(buf, wlen*2+1);

  i2c->beginTransmission(i2c_address);
  for(uint8_t i = 0; i < wlen*2+2; i++) {
    i2c->write(buf[i]);
  }
  i2c->endTransmission();
  return 0;
}

static int i2c_read(void *serif, uint8_t reg, uint16_t * rbuffer, uint32_t rlen) {
  uint16_t offset = 0;
  uint8_t buf[257];
  int ret = 0;
  int rx_length = rlen*2 +1;

  (void)serif;

  buf[0] = reg;
  buf[1] = crc8_sae_j1850(&reg,1);

  i2c->beginTransmission(i2c_address);
  ret = i2c->write(buf, 2);
  ret = i2c->endTransmission(false);

  while(offset < rx_length)
  {
    uint16_t rx_bytes = 0;
    if(offset != 0)
      i2c->beginTransmission(i2c_address);
    uint16_t length = ((rx_length - offset) > ARDUINO_I2C_BUFFER_LENGTH) ? ARDUINO_I2C_BUFFER_LENGTH : (rx_length - offset) ;
    rx_bytes = i2c->requestFrom(i2c_address, length);
    if (rx_bytes == length) {
      for(uint8_t i = 0; i < length; i++) {
        buf[offset+i] = i2c->read();
      }
      offset += length;
      i2c->endTransmission((offset == rx_length));
    } else {
      i2c->endTransmission((offset == rx_length));
    }
  }

  if(offset == rx_length)
  {
    memswap16(&buf[0],rlen*2);
    memcpy((uint8_t *) rbuffer, buf, rlen*2);  
    return 0;
  } else {
    return -1;
  }
}

static int spi_write(void *serif, uint8_t reg, const uint16_t * wbuffer, uint32_t wlen) {
  (void)serif;
  uint8_t addr[3];
  uint8_t buf[257];

  memset(buf, 0x00, 257); 

  buf[0] = 0x33;
  buf[1] = reg;

  memcpy(&buf[2], (uint8_t *)wbuffer, wlen*2);
  memswap16(&buf[2], wlen*2);
  buf[wlen*2+2] = crc8_sae_j1850(&buf[0], wlen*2+2); // CRC Compute on command, Adr and data

  digitalWrite(chip_select_id,LOW);
  for(uint8_t i = 0; i < wlen*2+3; i++) {
    spi->transfer(buf[i]);
  }
  digitalWrite(chip_select_id,HIGH);
  
  return 0;
}

static int spi_read(void *serif, uint8_t reg, uint16_t * rbuffer, uint32_t rlen) {
  (void)serif;
  uint8_t addr[3];
  uint8_t buf[257];

  memset(buf, 0x00, 257);

  addr[0] = 0x3C;
  addr[1] = reg;
  addr[2] = crc8_sae_j1850(&addr[0], 2);

  digitalWrite(chip_select_id,LOW);
  spi->transfer(addr, 3);
  spi->transfer(buf, rlen*2+1);
  digitalWrite(chip_select_id,HIGH);

  if (crc8_sae_j1850(&buf[0], rlen*2) != buf[rlen*2])
    return 1;

  memswap16(&buf[0], rlen*2);
  memcpy((uint8_t*) rbuffer, buf, rlen*2); 

  return 0;
}

static void sleep_us(uint32_t us)
{
  delayMicroseconds(us);
}

/* Encoder interrupt handler. */
static void int_encoder_cb(void)
{
  static volatile uint8_t last_encoder_state = 0;

  uint8_t a = digitalRead(MOSI);
  uint8_t b = digitalRead(MISO);
  uint8_t z = digitalRead(SCK);

  uint8_t current_state = (a << 1) | b;
  // Check for index pulse (Z channel)
  if (z)
    encoder_position = 0; // Reset position on index pulse
  else if (current_state != last_encoder_state) {
    if (((last_encoder_state == 0x00) && (current_state == 0x02)) ||
        ((last_encoder_state == 0x01) && (current_state == 0x00)) ||
        ((last_encoder_state == 0x02) && (current_state == 0x03)) ||
        ((last_encoder_state == 0x03) && (current_state == 0x01))) {
      encoder_position++;
    } else {
      encoder_position--;
    }
  }

  encoder_position = encoder_position%16384;
  last_encoder_state = current_state;
}


