/*
 *
 * Copyright (c) [2018] by InvenSense, Inc.
 * 
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef _INV_IMU_DEFS_H_
#define _INV_IMU_DEFS_H_

#ifdef __cplusplus
extern "C" {
#endif

/** @file inv_imu_defs.h
 * File exposing the device register map
 */

#include <stdint.h>

#define INV_IMU_REV_A 0xA
#define INV_IMU_REV_B 0XB

/* Include device definition */
#include "imu/inv_imu.h"

#if INV_IMU_REV == INV_IMU_REV_A
#include "imu/inv_imu_regmap_rev_a.h"
#elif INV_IMU_REV == INV_IMU_REV_B
#include "imu/inv_imu_regmap_rev_b.h"
#endif

#define ACCEL_DATA_SIZE 6
#define GYRO_DATA_SIZE  6
#define TEMP_DATA_SIZE  2

#define FIFO_HEADER_SIZE              1
#define FIFO_ACCEL_DATA_SIZE          ACCEL_DATA_SIZE
#define FIFO_GYRO_DATA_SIZE           GYRO_DATA_SIZE
#define FIFO_TEMP_DATA_SIZE           1
#define FIFO_TS_FSYNC_SIZE            2
#define FIFO_TEMP_HIGH_RES_SIZE       1
#define FIFO_ACCEL_GYRO_HIGH_RES_SIZE 3

#define FIFO_16BYTES_PACKET_SIZE                                                                   \
	(FIFO_HEADER_SIZE + FIFO_ACCEL_DATA_SIZE + FIFO_GYRO_DATA_SIZE + FIFO_TEMP_DATA_SIZE +         \
	 FIFO_TS_FSYNC_SIZE)
#define FIFO_20BYTES_PACKET_SIZE                                                                   \
	(FIFO_HEADER_SIZE + FIFO_ACCEL_DATA_SIZE + FIFO_GYRO_DATA_SIZE + FIFO_TEMP_DATA_SIZE +         \
	 FIFO_TS_FSYNC_SIZE + FIFO_TEMP_HIGH_RES_SIZE + FIFO_ACCEL_GYRO_HIGH_RES_SIZE)

#define FIFO_HEADER_ODR_ACCEL 0x01
#define FIFO_HEADER_ODR_GYRO  0x02
#define FIFO_HEADER_FSYNC     0x04
#define FIFO_HEADER_TMST      0x08
#define FIFO_HEADER_HEADER_20 0x10
#define FIFO_HEADER_GYRO      0x20
#define FIFO_HEADER_ACC       0x40
#define FIFO_HEADER_MSG       0x80

#define INVALID_VALUE_FIFO                   ((int16_t)0x8000)
#define INVALID_VALUE_FIFO_1B                ((int8_t)0x80)
#define OUT_OF_BOUND_TEMPERATURE_NEG_FIFO_1B ((int8_t)0x81)
#define OUT_OF_BOUND_TEMPERATURE_POS_FIFO_1B ((int8_t)0x7F)

/** Describe the content of the FIFO header */
typedef union {
	unsigned char Byte;
	struct {
#if INV_IMU_IS_GYRO_SUPPORTED
		unsigned char gyro_odr_different : 1;
#else
		unsigned char reserved1 : 1;
#endif
		unsigned char accel_odr_different : 1;
#if INV_IMU_IS_GYRO_SUPPORTED
		unsigned char fsync_bit : 1;
#else
		unsigned char reserved2 : 1;
#endif
		unsigned char timestamp_bit : 1;
		unsigned char twentybits_bit : 1;
#if INV_IMU_IS_GYRO_SUPPORTED
		unsigned char gyro_bit : 1;
#else
		unsigned char reserved3 : 1;
#endif
		unsigned char accel_bit : 1;
		unsigned char msg_bit : 1;
	} bits;
} fifo_header_t;

/* ----------------------------------------------------------------------------
 * Device registers description
 *
 * Next section defines some of the registers bitfield and declare corresponding
 * accessors.
 * Note that descriptors and accessors are not provided for all the registers 
 * but only for the most useful ones.
 * For all details on registers and bitfields functionalities please refer to 
 * the device datasheet.
 * ---------------------------------------------------------------------------- */

/* ---------------------------------------------------------------------------
 * register bank 0 
 * ---------------------------------------------------------------------------- */

/*
 * DEVICE_CONFIG
 * Register Name : DEVICE_CONFIG
 */

/* SPI_MODE */
typedef enum {
	DEVICE_CONFIG_SPI_MODE_1_2 = (0x1 << DEVICE_CONFIG_SPI_MODE_POS),
	DEVICE_CONFIG_SPI_MODE_0_3 = (0x0 << DEVICE_CONFIG_SPI_MODE_POS),
} DEVICE_CONFIG_SPI_MODE_t;

/* SPI_AP_4WIRE */
typedef enum {
	DEVICE_CONFIG_SPI_4WIRE = (0x1 << DEVICE_CONFIG_SPI_AP_4WIRE_POS),
	DEVICE_CONFIG_SPI_3WIRE = (0x0 << DEVICE_CONFIG_SPI_AP_4WIRE_POS),
} DEVICE_CONFIG_SPI_AP_4WIRE_t;

/*
 * SIGNAL_PATH_RESET
 * Register Name: SIGNAL_PATH_RESET
 */

/* SOFT_RESET_DEVICE_CONFIG */
typedef enum {
	SIGNAL_PATH_RESET_SOFT_RESET_DEVICE_CONFIG_EN =
	    (0x01 << SIGNAL_PATH_RESET_SOFT_RESET_DEVICE_CONFIG_POS),
	SIGNAL_PATH_RESET_SOFT_RESET_DEVICE_CONFIG_DIS =
	    (0x00 << SIGNAL_PATH_RESET_SOFT_RESET_DEVICE_CONFIG_POS),
} SIGNAL_PATH_RESET_SOFT_RESET_DEVICE_CONFIG_t;

/* FIFO_FLUSH */
typedef enum {
	SIGNAL_PATH_RESET_FIFO_FLUSH_EN  = (0x01 << SIGNAL_PATH_RESET_FIFO_FLUSH_POS),
	SIGNAL_PATH_RESET_FIFO_FLUSH_DIS = (0x00 << SIGNAL_PATH_RESET_FIFO_FLUSH_POS),
} SIGNAL_PATH_RESET_FIFO_FLUSH_t;

/*
 * DRIVE_CONFIG3
 * Register Name: DRIVE_CONFIG3
 */

/* SPI_SLEW_RATE */
typedef enum {
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_2_NS  = (0x05 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_8_NS  = (0x04 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_14_NS = (0x03 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_19_NS = (0x02 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_36_NS = (0x01 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
	DRIVE_CONFIG3_SPI_SLEW_RATE_MAX_60_NS = (0x00 << DRIVE_CONFIG3_SPI_SLEW_RATE_POS),
} DRIVE_CONFIG3_SPI_SLEW_RATE_t;

/*
 * INT_CONFIG
 * Register Name: INT_CONFIG
 */

/* INT2_MODE */
typedef enum {
	INT_CONFIG_INT2_MODE_LATCHED = (0x01 << INT_CONFIG_INT2_MODE_POS),
	INT_CONFIG_INT2_MODE_PULSED  = (0x00 << INT_CONFIG_INT2_MODE_POS),
} INT_CONFIG_INT2_MODE_t;

/* INT2_DRIVE_CIRCUIT */
typedef enum {
	INT_CONFIG_INT2_DRIVE_CIRCUIT_PP = (0x01 << INT_CONFIG_INT2_DRIVE_CIRCUIT_POS),
	INT_CONFIG_INT2_DRIVE_CIRCUIT_OD = (0x00 << INT_CONFIG_INT2_DRIVE_CIRCUIT_POS),
} INT_CONFIG_INT2_DRIVE_CIRCUIT_t;

/* INT2_POLARITY */
typedef enum {
	INT_CONFIG_INT2_POLARITY_HIGH = (0x01 << INT_CONFIG_INT2_POLARITY_POS),
	INT_CONFIG_INT2_POLARITY_LOW  = (0x00 << INT_CONFIG_INT2_POLARITY_POS),
} INT_CONFIG_INT2_POLARITY_t;

/* INT1_MODE */
typedef enum {
	INT_CONFIG_INT1_MODE_LATCHED = (0x01 << INT_CONFIG_INT1_MODE_POS),
	INT_CONFIG_INT1_MODE_PULSED  = (0x00 << INT_CONFIG_INT1_MODE_POS),
} INT_CONFIG_INT1_MODE_t;

/* INT1_DRIVE_CIRCUIT */
typedef enum {
	INT_CONFIG_INT1_DRIVE_CIRCUIT_PP = (0x01 << INT_CONFIG_INT1_DRIVE_CIRCUIT_POS),
	INT_CONFIG_INT1_DRIVE_CIRCUIT_OD = (0x00 << INT_CONFIG_INT1_DRIVE_CIRCUIT_POS),
} INT_CONFIG_INT1_DRIVE_CIRCUIT_t;

/* INT1_POLARITY */
typedef enum {
	INT_CONFIG_INT1_POLARITY_HIGH = 0x01,
	INT_CONFIG_INT1_POLARITY_LOW  = 0x00,
} INT_CONFIG_INT1_POLARITY_t;

/*
 * PWR_MGMT0
 * Register Name: PWR_MGMT0
 */

/* ACCEL_LP_CLK_SEL */
typedef enum {
	PWR_MGMT0_ACCEL_LP_CLK_WUOSC = (0x00 << PWR_MGMT0_ACCEL_LP_CLK_SEL_POS),
	PWR_MGMT0_ACCEL_LP_CLK_RCOSC = (0x01 << PWR_MGMT0_ACCEL_LP_CLK_SEL_POS),
} PWR_MGMT0_ACCEL_LP_CLK_t;

/* IDLE */
typedef enum {
	PWR_MGMT0_IDLE_DIS = (0x01 << PWR_MGMT0_IDLE_POS),
	PWR_MGMT0_IDLE_EN  = (0x00 << PWR_MGMT0_IDLE_POS),
} PWR_MGMT0_IDLE_t;

#if INV_IMU_IS_GYRO_SUPPORTED
/* GYRO_MODE */
typedef enum {
	PWR_MGMT0_GYRO_MODE_LN      = (0x03 << PWR_MGMT0_GYRO_MODE_POS),
	PWR_MGMT0_GYRO_MODE_LP      = (0x02 << PWR_MGMT0_GYRO_MODE_POS),
	PWR_MGMT0_GYRO_MODE_STANDBY = (0x01 << PWR_MGMT0_GYRO_MODE_POS),
	PWR_MGMT0_GYRO_MODE_OFF     = (0x00 << PWR_MGMT0_GYRO_MODE_POS),
} PWR_MGMT0_GYRO_MODE_t;
#endif

/* ACCEL_MODE */
typedef enum {
	PWR_MGMT0_ACCEL_MODE_LN  = 0x03,
	PWR_MGMT0_ACCEL_MODE_LP  = 0x02,
	PWR_MGMT0_ACCEL_MODE_OFF = 0x00,
} PWR_MGMT0_ACCEL_MODE_t;

#if INV_IMU_IS_GYRO_SUPPORTED
/*
 * GYRO_CONFIG0
 * Register Name: GYRO_CONFIG0
 */

/* GYRO_FS_SEL*/
typedef enum {
#if INV_IMU_HFSR_SUPPORTED
	GYRO_CONFIG0_FS_SEL_500dps  = (3 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_1000dps = (2 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_2000dps = (1 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_4000dps = (0 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
#else
	GYRO_CONFIG0_FS_SEL_250dps  = (3 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_500dps  = (2 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_1000dps = (1 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
	GYRO_CONFIG0_FS_SEL_2000dps = (0 << GYRO_CONFIG0_GYRO_UI_FS_SEL_POS),
#endif
} GYRO_CONFIG0_FS_SEL_t;

/* GYRO_ODR */
typedef enum {
	GYRO_CONFIG0_ODR_1_5625_HZ = 0xF,
	GYRO_CONFIG0_ODR_3_125_HZ  = 0xE,
	GYRO_CONFIG0_ODR_6_25_HZ   = 0xD,
	GYRO_CONFIG0_ODR_12_5_HZ   = 0xC,
	GYRO_CONFIG0_ODR_25_HZ     = 0xB,
	GYRO_CONFIG0_ODR_50_HZ     = 0xA,
	GYRO_CONFIG0_ODR_100_HZ    = 0x9,
	GYRO_CONFIG0_ODR_200_HZ    = 0x8,
	GYRO_CONFIG0_ODR_400_HZ    = 0x7,
	GYRO_CONFIG0_ODR_800_HZ    = 0x6,
	GYRO_CONFIG0_ODR_1600_HZ   = 0x5,
} GYRO_CONFIG0_ODR_t;
#endif

/*
 * ACCEL_CONFIG0
 * Register Name: ACCEL_CONFIG0
 */

/* ACCEL_FS_SEL */
typedef enum {
#if INV_IMU_HFSR_SUPPORTED
	ACCEL_CONFIG0_FS_SEL_4g  = (0x3 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_8g  = (0x2 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_16g = (0x1 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_32g = (0x0 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
#else
	ACCEL_CONFIG0_FS_SEL_2g  = (0x3 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_4g  = (0x2 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_8g  = (0x1 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
	ACCEL_CONFIG0_FS_SEL_16g = (0x0 << ACCEL_CONFIG0_ACCEL_UI_FS_SEL_POS),
#endif
} ACCEL_CONFIG0_FS_SEL_t;

/* ACCEL_ODR */
typedef enum {
	ACCEL_CONFIG0_ODR_1_5625_HZ = 0xF,
	ACCEL_CONFIG0_ODR_3_125_HZ  = 0xE,
	ACCEL_CONFIG0_ODR_6_25_HZ   = 0xD,
	ACCEL_CONFIG0_ODR_12_5_HZ   = 0xC,
	ACCEL_CONFIG0_ODR_25_HZ     = 0xB,
	ACCEL_CONFIG0_ODR_50_HZ     = 0xA,
	ACCEL_CONFIG0_ODR_100_HZ    = 0x9,
	ACCEL_CONFIG0_ODR_200_HZ    = 0x8,
	ACCEL_CONFIG0_ODR_400_HZ    = 0x7,
	ACCEL_CONFIG0_ODR_800_HZ    = 0x6,
	ACCEL_CONFIG0_ODR_1600_HZ   = 0x5,
} ACCEL_CONFIG0_ODR_t;

#if INV_IMU_IS_GYRO_SUPPORTED
/*
 * GYRO_CONFIG1
 * Register Name: GYRO_CONFIG1
 */

/* GYRO_UI_FILT_BW_IND */
typedef enum {
	GYRO_CONFIG1_GYRO_FILT_BW_16        = (0x07 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_25        = (0x06 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_34        = (0x05 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_53        = (0x04 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_73        = (0x03 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_121       = (0x02 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_180       = (0x01 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
	GYRO_CONFIG1_GYRO_FILT_BW_NO_FILTER = (0x00 << GYRO_CONFIG1_GYRO_UI_FILT_BW_POS),
} GYRO_CONFIG1_GYRO_FILT_BW_t;
#endif

/*
 * ACCEL_CONFIG1
 * Register Name: ACCEL_CONFIG1
 */

/* ACCEL_UI_AVG_IND */
typedef enum {
	ACCEL_CONFIG1_ACCEL_FILT_AVG_64 = (0x5 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
	ACCEL_CONFIG1_ACCEL_FILT_AVG_32 = (0x4 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
	ACCEL_CONFIG1_ACCEL_FILT_AVG_16 = (0x3 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
	ACCEL_CONFIG1_ACCEL_FILT_AVG_8  = (0x2 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
	ACCEL_CONFIG1_ACCEL_FILT_AVG_4  = (0x1 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
	ACCEL_CONFIG1_ACCEL_FILT_AVG_2  = (0x0 << ACCEL_CONFIG1_ACCEL_UI_AVG_POS),
} ACCEL_CONFIG1_ACCEL_FILT_AVG_t;

/* ACCEL_UI_FILT_BW_IND */
typedef enum {
	ACCEL_CONFIG1_ACCEL_FILT_BW_16        = (0x7 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_25        = (0x6 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_34        = (0x5 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_53        = (0x4 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_73        = (0x3 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_121       = (0x2 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_180       = (0x1 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
	ACCEL_CONFIG1_ACCEL_FILT_BW_NO_FILTER = (0x0 << ACCEL_CONFIG1_ACCEL_UI_FILT_BW_POS),
} ACCEL_CONFIG1_ACCEL_FILT_BW_t;

/*
 * APEX_CONFIG0
 * Register Name: APEX_CONFIG0
 */

/* DMP_POWER_SAVE_EN */
typedef enum {
	APEX_CONFIG0_DMP_POWER_SAVE_EN  = (0x1 << APEX_CONFIG0_DMP_POWER_SAVE_EN_POS),
	APEX_CONFIG0_DMP_POWER_SAVE_DIS = (0x0 << APEX_CONFIG0_DMP_POWER_SAVE_EN_POS),
} APEX_CONFIG0_DMP_POWER_SAVE_t;

/* DMP_INIT_EN */
typedef enum {
	APEX_CONFIG0_DMP_INIT_EN  = (0x01 << APEX_CONFIG0_DMP_INIT_EN_POS),
	APEX_CONFIG0_DMP_INIT_DIS = (0x00 << APEX_CONFIG0_DMP_INIT_EN_POS),
} APEX_CONFIG0_DMP_INIT_t;

/* DMP_MEM_RESET */
typedef enum {
	APEX_CONFIG0_DMP_MEM_RESET_APEX_ST_EN = (0x01 << APEX_CONFIG0_DMP_MEM_RESET_EN_POS),
	APEX_CONFIG0_DMP_MEM_RESET_DIS        = (0x00 << APEX_CONFIG0_DMP_MEM_RESET_EN_POS),
} APEX_CONFIG0_DMP_MEM_RESET_t;

/*
 * APEX_CONFIG1
 * Register Name: APEX_CONFIG1
 */

/* SMD_ENABLE */
typedef enum {
	APEX_CONFIG1_SMD_ENABLE_DIS = (0x00 << APEX_CONFIG1_SMD_ENABLE_POS),
	APEX_CONFIG1_SMD_ENABLE_EN  = (0x01 << APEX_CONFIG1_SMD_ENABLE_POS),
} APEX_CONFIG1_SMD_ENABLE_t;

/* FF_ENABLE */
typedef enum {
	APEX_CONFIG1_FF_ENABLE_DIS = (0x00 << APEX_CONFIG1_FF_ENABLE_POS),
	APEX_CONFIG1_FF_ENABLE_EN  = (0x01 << APEX_CONFIG1_FF_ENABLE_POS),
} APEX_CONFIG1_FF_ENABLE_t;

/* TILT_ENABLE */
typedef enum {
	APEX_CONFIG1_TILT_ENABLE_DIS = (0x0 << APEX_CONFIG1_TILT_ENABLE_POS),
	APEX_CONFIG1_TILT_ENABLE_EN  = (0x1 << APEX_CONFIG1_TILT_ENABLE_POS),
} APEX_CONFIG1_TILT_ENABLE_t;

/* PED_ENABLE */
typedef enum {
	APEX_CONFIG1_PED_ENABLE_DIS = (0x0 << APEX_CONFIG1_PED_ENABLE_POS),
	APEX_CONFIG1_PED_ENABLE_EN  = (0x1 << APEX_CONFIG1_PED_ENABLE_POS),
} APEX_CONFIG1_PED_ENABLE_t;

/* DMP_ODR */
typedef enum {
	APEX_CONFIG1_DMP_ODR_25Hz  = (0x0 << APEX_CONFIG1_DMP_ODR_POS),
	APEX_CONFIG1_DMP_ODR_50Hz  = (0x2 << APEX_CONFIG1_DMP_ODR_POS),
	APEX_CONFIG1_DMP_ODR_100Hz = (0x3 << APEX_CONFIG1_DMP_ODR_POS),
	APEX_CONFIG1_DMP_ODR_400Hz = (0x1 << APEX_CONFIG1_DMP_ODR_POS),
} APEX_CONFIG1_DMP_ODR_t;

/*
 * WOM_CONFIG
 * Register Name: WOM_CONFIG
 */

/* WOM_INT_DUR */
typedef enum {
	WOM_CONFIG_WOM_INT_DUR_1_SMPL = (0x00 << WOM_CONFIG_WOM_INT_DUR_POS),
	WOM_CONFIG_WOM_INT_DUR_2_SMPL = (0x01 << WOM_CONFIG_WOM_INT_DUR_POS),
	WOM_CONFIG_WOM_INT_DUR_3_SMPL = (0x02 << WOM_CONFIG_WOM_INT_DUR_POS),
	WOM_CONFIG_WOM_INT_DUR_4_SMPL = (0x03 << WOM_CONFIG_WOM_INT_DUR_POS),
} WOM_CONFIG_WOM_INT_DUR_t;

/* WOM_INT_MODE */
typedef enum {
	WOM_CONFIG_WOM_INT_MODE_ANDED = (0x01 << WOM_CONFIG_WOM_INT_MODE_POS),
	WOM_CONFIG_WOM_INT_MODE_ORED  = (0x00 << WOM_CONFIG_WOM_INT_MODE_POS),
} WOM_CONFIG_WOM_INT_MODE_t;

/* WOM_MODE */
typedef enum {
	WOM_CONFIG_WOM_MODE_CMP_PREV = (0x01 << WOM_CONFIG_WOM_MODE_POS),
	WOM_CONFIG_WOM_MODE_CMP_INIT = (0x00 << WOM_CONFIG_WOM_MODE_POS),
} WOM_CONFIG_WOM_MODE_t;

/* WOM_ENABLE */
typedef enum {
	WOM_CONFIG_WOM_EN_ENABLE  = (0x01 << WOM_CONFIG_WOM_EN_POS),
	WOM_CONFIG_WOM_EN_DISABLE = (0x00 << WOM_CONFIG_WOM_EN_POS),
} WOM_CONFIG_WOM_EN_t;

/*
 * FIFO_CONFIG1
 * Register Name: FIFO_CONFIG
 */

/* FIFO_MODE */
typedef enum {
	FIFO_CONFIG1_FIFO_MODE_SNAPSHOT = (0x01 << FIFO_CONFIG1_FIFO_MODE_POS),
	FIFO_CONFIG1_FIFO_MODE_STREAM   = (0x00 << FIFO_CONFIG1_FIFO_MODE_POS)
} FIFO_CONFIG1_FIFO_MODE_t;

/* FIFO_BYPASS */
typedef enum {
	FIFO_CONFIG1_FIFO_BYPASS_ON  = (0x01 << FIFO_CONFIG1_FIFO_BYPASS_POS),
	FIFO_CONFIG1_FIFO_BYPASS_OFF = (0x00 << FIFO_CONFIG1_FIFO_BYPASS_POS),
} FIFO_CONFIG1_FIFO_BYPASS_t;

/*
 * APEX_DATA3
 * Register Name: APEX_DATA3
 */

/* DMP_IDLE */
typedef enum {
	APEX_DATA3_DMP_IDLE_ON  = (0x01 << APEX_DATA3_DMP_IDLE_POS),
	APEX_DATA3_DMP_IDLE_OFF = (0x00 << APEX_DATA3_DMP_IDLE_POS),
} APEX_DATA3_DMP_IDLE_OFF_t;

/* ACTIVITY_CLASS */
typedef enum {
	APEX_DATA3_ACTIVITY_CLASS_OTHER = 0x0,
	APEX_DATA3_ACTIVITY_CLASS_WALK  = 0x1,
	APEX_DATA3_ACTIVITY_CLASS_RUN   = 0x2,
} APEX_DATA3_ACTIVITY_CLASS_t;

/*
 * INTF_CONFIG0
 * Register Name: INTF_CONFIG0
 */

/* FIFO_COUNT_REC */
typedef enum {
	INTF_CONFIG0_FIFO_COUNT_REC_RECORD = (0x01 << INTF_CONFIG0_FIFO_COUNT_FORMAT_POS),
	INTF_CONFIG0_FIFO_COUNT_REC_BYTE   = (0x00 << INTF_CONFIG0_FIFO_COUNT_FORMAT_POS),
} INTF_CONFIG0_FIFO_COUNT_REC_t;

/* FIFO_COUNT_ENDIAN */
typedef enum {
	INTF_CONFIG0_FIFO_COUNT_BIG_ENDIAN    = (0x01 << INTF_CONFIG0_FIFO_COUNT_ENDIAN_POS),
	INTF_CONFIG0_FIFO_COUNT_LITTLE_ENDIAN = (0x00 << INTF_CONFIG0_FIFO_COUNT_ENDIAN_POS),
} INTF_CONFIG0_FIFO_COUNT_ENDIAN_t;

/* DATA_ENDIAN */
typedef enum {
	INTF_CONFIG0_DATA_BIG_ENDIAN    = (0x01 << INTF_CONFIG0_SENSOR_DATA_ENDIAN_POS),
	INTF_CONFIG0_DATA_LITTLE_ENDIAN = (0x00 << INTF_CONFIG0_SENSOR_DATA_ENDIAN_POS),
} INTF_CONFIG0_DATA_ENDIAN_t;

/* ---------------------------------------------------------------------------
 * register bank MREG1 
 * ---------------------------------------------------------------------------- */

/*
 * TMST_CONFIG1_MREG1
 * Register Name: TMST_CONFIG1
 */

/* TMST_RES */
typedef enum {
	TMST_CONFIG1_RESOL_16us = (0x01 << TMST_CONFIG1_TMST_RES_POS),
	TMST_CONFIG1_RESOL_1us  = (0x00 << TMST_CONFIG1_TMST_RES_POS),
} TMST_CONFIG1_RESOL_t;

/* TMST_FSYNC */
typedef enum {
	TMST_CONFIG1_TMST_FSYNC_EN  = (0x01 << TMST_CONFIG1_TMST_FSYNC_EN_POS),
	TMST_CONFIG1_TMST_FSYNC_DIS = (0x00 << TMST_CONFIG1_TMST_FSYNC_EN_POS),
} TMST_CONFIG1_TMST_FSYNC_EN_t;

/* TMST_EN */
typedef enum {
	TMST_CONFIG1_TMST_EN  = 0x01,
	TMST_CONFIG1_TMST_DIS = 0x00,
} TMST_CONFIG1_TMST_EN_t;

/*
 * FIFO_CONFIG5_MREG1
 * Register Name: FIFO_CONFIG5
 */
/* FIFO_WM_GT_TH */
typedef enum {
	FIFO_CONFIG5_WM_GT_TH_EN  = (0x1 << FIFO_CONFIG5_FIFO_WM_GT_TH_POS),
	FIFO_CONFIG5_WM_GT_TH_DIS = (0x0 << FIFO_CONFIG5_FIFO_WM_GT_TH_POS),
} FIFO_CONFIG5_WM_GT_t;

/* FIFO_HIRES_EN */
typedef enum {
	FIFO_CONFIG5_HIRES_EN  = (0x1 << FIFO_CONFIG5_FIFO_HIRES_EN_POS),
	FIFO_CONFIG5_HIRES_DIS = (0x0 << FIFO_CONFIG5_FIFO_HIRES_EN_POS),
} FIFO_CONFIG5_HIRES_t;

/* FIFO_TMST_FSYNC_EN */
typedef enum {
	FIFO_CONFIG5_TMST_FSYNC_EN  = (0x1 << FIFO_CONFIG5_FIFO_TMST_FSYNC_EN_POS),
	FIFO_CONFIG5_TMST_FSYNC_DIS = (0x0 << FIFO_CONFIG5_FIFO_TMST_FSYNC_EN_POS),
} FIFO_CONFIG5_TMST_FSYNC_t;

#if INV_IMU_IS_GYRO_SUPPORTED
/* FIFO_GYRO_EN */
typedef enum {
	FIFO_CONFIG5_GYRO_EN  = (0x1 << FIFO_CONFIG5_FIFO_GYRO_EN_POS),
	FIFO_CONFIG5_GYRO_DIS = (0x0 << FIFO_CONFIG5_FIFO_GYRO_EN_POS),
} FIFO_CONFIG5_GYRO_t;
#endif

/* FIFO_ACCEL_EN*/
typedef enum {
	FIFO_CONFIG5_ACCEL_EN  = 0x01,
	FIFO_CONFIG5_ACCEL_DIS = 0x00,
} FIFO_CONFIG5_ACCEL_t;

#if INV_IMU_IS_GYRO_SUPPORTED
/*
 * FSYNC_CONFIG_MREG1
 * Register Name: FSYNC_CONFIG
 */

/* FSYNC_UI_SEL */
typedef enum {
	FSYNC_CONFIG_UI_SEL_NO      = (0x0 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_TEMP    = (0x1 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_GYRO_X  = (0x2 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_GYRO_Y  = (0x3 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_GYRO_Z  = (0x4 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_ACCEL_X = (0x5 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_ACCEL_Y = (0x6 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
	FSYNC_CONFIG_UI_SEL_ACCEL_Z = (0x7 << FSYNC_CONFIG_FSYNC_UI_SEL_POS),
} FSYNC_CONFIG_UI_SEL_t;
#endif

/*
 * ST_CONFIG_MREG1
 * Register Name: ST_CONFIG
 */
typedef enum {
	ST_CONFIG_16_SAMPLES  = (0 << ST_CONFIG_ST_NUMBER_SAMPLE_POS),
	ST_CONFIG_200_SAMPLES = (1 << ST_CONFIG_ST_NUMBER_SAMPLE_POS),
} ST_CONFIG_NUM_SAMPLES_t;

typedef enum {
	ST_CONFIG_ACCEL_ST_LIM_50 = (7 << ST_CONFIG_ACCEL_ST_LIM_POS),
} ST_CONFIG_ACCEL_ST_LIM_t;

#if INV_IMU_IS_GYRO_SUPPORTED
typedef enum {
	ST_CONFIG_GYRO_ST_LIM_50 = (7 << ST_CONFIG_GYRO_ST_LIM_POS),
} ST_CONFIG_GYRO_ST_LIM_t;
#endif

/*
 * SELFTEST_MREG1
 * Register Name: SELFTEST
 */

/* GYRO_ST_EN and ACCEL_ST_EN */
typedef enum {
	SELFTEST_DIS      = 0,
	SELFTEST_ACCEL_EN = SELFTEST_ACCEL_ST_EN_MASK,
#if INV_IMU_IS_GYRO_SUPPORTED
	SELFTEST_GYRO_EN = SELFTEST_GYRO_ST_EN_MASK,
	SELFTEST_EN      = (SELFTEST_ACCEL_ST_EN_MASK | SELFTEST_GYRO_ST_EN_MASK)
#endif
} SELFTEST_ACCEL_GYRO_ST_EN_t;

/*
 * OTP_CONFIG_MREG1
 * Register Name: OTP_CONFIG
 */

/* OTP_CONFIG */
typedef enum {
	OTP_CONFIG_OTP_COPY_TRIM    = (1 << OTP_CONFIG_OTP_COPY_MODE_POS),
	OTP_CONFIG_OTP_COPY_ST_DATA = (3 << OTP_CONFIG_OTP_COPY_MODE_POS),
} OTP_CONFIG_COPY_MODE_t;

/*
 * APEX_CONFIG2_MREG1
 * Register Name: APEX_CONFIG2
*/

/* LOW_ENERGY_AMP_TH_SEL */
typedef enum {
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_30_MG  = (0 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_35_MG  = (1 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_40_MG  = (2 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_45_MG  = (3 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_50_MG  = (4 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_55_MG  = (5 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_60_MG  = (6 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_65_MG  = (7 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_70_MG  = (8 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_75_MG  = (9 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_80_MG  = (10 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_85_MG  = (11 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_90_MG  = (12 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_95_MG  = (13 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_100_MG = (14 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
	APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_105_MG = (15 << APEX_CONFIG2_LOW_ENERGY_AMP_TH_SEL_POS),
} APEX_CONFIG2_LOW_ENERGY_AMP_TH_t;

/* DMP_POWER_SAVE_TIME_SEL */
typedef enum {
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_0_S  = 0x0,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_4_S  = 0x1,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_8_S  = 0x2,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_12_S = 0x3,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_16_S = 0x4,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_20_S = 0x5,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_24_S = 0x6,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_28_S = 0x7,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_32_S = 0x8,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_36_S = 0x9,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_40_S = 0xA,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_44_S = 0xB,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_48_S = 0xC,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_52_S = 0xD,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_56_S = 0xE,
	APEX_CONFIG2_DMP_POWER_SAVE_TIME_SEL_60_S = 0xF,
} APEX_CONFIG2_DMP_POWER_SAVE_TIME_t;

/*
 * APEX_CONFIG3_MREG1
 * Register Name: APEX_CONFIG3
*/

/* PEDO_AMP_TH_SEL */
typedef enum {
	APEX_CONFIG3_PEDO_AMP_TH_30_MG = (0 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_34_MG = (1 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_38_MG = (2 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_42_MG = (3 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_46_MG = (4 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_50_MG = (5 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_54_MG = (6 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_58_MG = (7 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_62_MG = (8 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_66_MG = (9 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_70_MG = (10 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_74_MG = (11 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_78_MG = (12 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_82_MG = (13 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_86_MG = (14 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
	APEX_CONFIG3_PEDO_AMP_TH_90_MG = (15 << APEX_CONFIG3_PED_AMP_TH_SEL_POS),
} APEX_CONFIG3_PEDO_AMP_TH_t;

/*
 * APEX_CONFIG4_MREG1
 * Register Name: APEX_CONFIG4
*/

/* PEDO_SB_TIMER_TH_SEL */
typedef enum {
	APEX_CONFIG4_PEDO_SB_TIMER_TH_50_SAMPLES  = (0 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_75_SAMPLES  = (1 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_100_SAMPLES = (2 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_125_SAMPLES = (3 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_150_SAMPLES = (4 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_175_SAMPLES = (5 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_200_SAMPLES = (6 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
	APEX_CONFIG4_PEDO_SB_TIMER_TH_225_SAMPLES = (7 << APEX_CONFIG4_PED_SB_TIMER_TH_SEL_POS),
} APEX_CONFIG4_PEDO_SB_TIMER_TH_t;

/* PEDO_HI_ENRGY_TH_SEL */
typedef enum {
	APEX_CONFIG4_PEDO_HI_ENRGY_TH_88_MG  = (0 << APEX_CONFIG4_PED_HI_EN_TH_SEL_POS),
	APEX_CONFIG4_PEDO_HI_ENRGY_TH_104_MG = (1 << APEX_CONFIG4_PED_HI_EN_TH_SEL_POS),
	APEX_CONFIG4_PEDO_HI_ENRGY_TH_133_MG = (2 << APEX_CONFIG4_PED_HI_EN_TH_SEL_POS),
	APEX_CONFIG4_PEDO_HI_ENRGY_TH_155_MG = (3 << APEX_CONFIG4_PED_HI_EN_TH_SEL_POS),
} APEX_CONFIG4_PEDO_HI_ENRGY_TH_t;

/*
 * APEX_CONFIG5_MREG1
 * Register Name: APEX_CONFIG5
*/

/* TILT_WAIT_TIME_SEL */
typedef enum {
	APEX_CONFIG5_TILT_WAIT_TIME_0_S = (0 << APEX_CONFIG5_TILT_WAIT_TIME_SEL_POS),
	APEX_CONFIG5_TILT_WAIT_TIME_2_S = (1 << APEX_CONFIG5_TILT_WAIT_TIME_SEL_POS),
	APEX_CONFIG5_TILT_WAIT_TIME_4_S = (2 << APEX_CONFIG5_TILT_WAIT_TIME_SEL_POS),
	APEX_CONFIG5_TILT_WAIT_TIME_6_S = (3 << APEX_CONFIG5_TILT_WAIT_TIME_SEL_POS),
} APEX_CONFIG5_TILT_WAIT_TIME_t;

/* LOWG_PEAK_TH_HYST_SEL */
typedef enum {
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_31_MG  = (0 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_63_MG  = (1 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_94_MG  = (2 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_125_MG = (3 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_156_MG = (4 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_188_MG = (5 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_219_MG = (6 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_LOWG_PEAK_TH_HYST_250_MG = (7 << APEX_CONFIG5_LOWG_PEAK_TH_HYST_SEL_POS),
} APEX_CONFIG5_LOWG_PEAK_TH_HYST_t;

/* HIGHG_PEAK_TH_HYST_SEL */
typedef enum {
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_31_MG  = (0 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_63_MG  = (1 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_94_MG  = (2 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_125_MG = (3 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_156_MG = (4 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_188_MG = (5 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_219_MG = (6 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
	APEX_CONFIG5_HIGHG_PEAK_TH_HYST_250_MG = (7 << APEX_CONFIG5_HIGHG_PEAK_TH_HYST_SEL_POS),
} APEX_CONFIG5_HIGHG_PEAK_TH_HYST_t;

/*
 * APEX_CONFIG9_MREG1
 * Register Name: APEX_CONFIG9
*/

/* FF_DEBOUNCE_DURATION_SEL */
typedef enum {
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_0_MS    = (0 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1250_MS = (1 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1375_MS = (2 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1500_MS = (3 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1625_MS = (4 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1750_MS = (5 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_1875_MS = (6 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2000_MS = (7 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2125_MS = (8 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2250_MS = (9 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2375_MS = (10 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2500_MS = (11 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2625_MS = (12 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2750_MS = (13 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_2875_MS = (14 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
	APEX_CONFIG9_FF_DEBOUNCE_DURATION_3000_MS = (15 << APEX_CONFIG9_FF_DEBOUNCE_DURATION_SEL_POS),
} APEX_CONFIG9_FF_DEBOUNCE_DURATION_t;

/* SMD_SENSITIVITY_SEL */
typedef enum {
	APEX_CONFIG9_SMD_SENSITIVITY_0 = (0 << APEX_CONFIG9_SMD_SENSITIVITY_SEL_POS),
	APEX_CONFIG9_SMD_SENSITIVITY_1 = (1 << APEX_CONFIG9_SMD_SENSITIVITY_SEL_POS),
	APEX_CONFIG9_SMD_SENSITIVITY_2 = (2 << APEX_CONFIG9_SMD_SENSITIVITY_SEL_POS),
	APEX_CONFIG9_SMD_SENSITIVITY_3 = (3 << APEX_CONFIG9_SMD_SENSITIVITY_SEL_POS),
	APEX_CONFIG9_SMD_SENSITIVITY_4 = (4 << APEX_CONFIG9_SMD_SENSITIVITY_SEL_POS),
} APEX_CONFIG9_SMD_SENSITIVITY_t;

/* SMD_SENSITIVITY_MODE */
typedef enum {
	APEX_CONFIG9_SENSITIVITY_MODE_NORMAL    = (0 << APEX_CONFIG9_SENSITIVITY_MODE_POS),
	APEX_CONFIG9_SENSITIVITY_MODE_SLOW_WALK = (1 << APEX_CONFIG9_SENSITIVITY_MODE_POS),
} APEX_CONFIG9_SENSITIVITY_MODE_t;

/*
 * APEX_CONFIG10_MREG1
 * Register Name: APEX_CONFIG10
*/

/* LOWG_PEAK_TH_SEL */
typedef enum {
	APEX_CONFIG10_LOWG_PEAK_TH_31_MG   = (0x00 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_63_MG   = (0x01 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_94_MG   = (0x02 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_125_MG  = (0x03 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_156_MG  = (0x04 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_188_MG  = (0x05 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_219_MG  = (0x06 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_250_MG  = (0x07 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_281_MG  = (0x08 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_313_MG  = (0x09 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_344_MG  = (0x0A << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_375_MG  = (0x0B << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_406_MG  = (0x0C << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_438_MG  = (0x0D << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_469_MG  = (0x0E << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_500_MG  = (0x0F << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_531_MG  = (0x10 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_563_MG  = (0x11 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_594_MG  = (0x12 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_625_MG  = (0x13 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_656_MG  = (0x14 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_688_MG  = (0x15 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_719_MG  = (0x16 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_750_MG  = (0x17 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_781_MG  = (0x18 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_813_MG  = (0x19 << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_844_MG  = (0x1A << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_875_MG  = (0x1B << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_906_MG  = (0x1C << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_938_MG  = (0x1D << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_969_MG  = (0x1E << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
	APEX_CONFIG10_LOWG_PEAK_TH_1000_MG = (0x1F << APEX_CONFIG10_LOWG_PEAK_TH_SEL_POS),
} APEX_CONFIG10_LOWG_PEAK_TH_t;

/* LOWG_TIME_TH_SEL */
typedef enum {
	APEX_CONFIG10_LOWG_TIME_TH_1_SAMPLE  = (0x00 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_2_SAMPLES = (0x01 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_3_SAMPLES = (0x02 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_4_SAMPLES = (0x03 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_5_SAMPLES = (0x04 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_6_SAMPLES = (0x05 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_7_SAMPLES = (0x06 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
	APEX_CONFIG10_LOWG_TIME_TH_8_SAMPLES = (0x07 << APEX_CONFIG10_LOWG_TIME_TH_SEL_POS),
} APEX_CONFIG10_LOWG_TIME_TH_SAMPLES_t;

/*
 * APEX_CONFIG11_MREG1
 * Register Name: APEX_CONFIG11
*/

/* HIGHG_PEAK_TH_SEL */
typedef enum {
	APEX_CONFIG11_HIGHG_PEAK_TH_250_MG  = (0x00 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_500_MG  = (0x01 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_750_MG  = (0x02 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_1000MG  = (0x03 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_1250_MG = (0x04 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_1500_MG = (0x05 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_1750_MG = (0x06 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_2000_MG = (0x07 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_2250_MG = (0x08 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_2500_MG = (0x09 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_2750_MG = (0x0A << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_3000_MG = (0x0B << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_3250_MG = (0x0C << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_3500_MG = (0x0D << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_3750_MG = (0x0E << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_4000_MG = (0x0F << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_4250_MG = (0x10 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_4500_MG = (0x11 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_4750_MG = (0x12 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_5000_MG = (0x13 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_5250_MG = (0x14 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_5500_MG = (0x15 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_5750_MG = (0x16 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_6000_MG = (0x17 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_6250_MG = (0x18 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_6500_MG = (0x19 << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_6750_MG = (0x1A << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_7000_MG = (0x1B << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_7250_MG = (0x1C << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_7500_MG = (0x1D << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_PEAK_TH_7750_MG = (0x1E << APEX_CONFIG11_HIGHG_PEAK_TH_SEL_POS),
} APEX_CONFIG11_HIGHG_PEAK_TH_t;

/* HIGHG_TIME_TH_SEL */
typedef enum {
	APEX_CONFIG11_HIGHG_TIME_TH_1_SAMPLE  = (0x00 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_2_SAMPLES = (0x01 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_3_SAMPLES = (0x02 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_4_SAMPLES = (0x03 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_5_SAMPLES = (0x04 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_6_SAMPLES = (0x05 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_7_SAMPLES = (0x06 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
	APEX_CONFIG11_HIGHG_TIME_TH_8_SAMPLES = (0x07 << APEX_CONFIG11_HIGHG_TIME_TH_SEL_POS),
} APEX_CONFIG11_HIGHG_TIME_TH_SAMPLES_t;

/*
 * FDR_CONFIG_MREG1
 * Register Name: FDR_CONFIG
 */

/* FDR_SEL */
typedef enum {
	FDR_CONFIG_FDR_SEL_DIS        = (0x0 << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_2   = (0x8 << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_4   = (0x9 << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_8   = (0xA << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_16  = (0xB << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_32  = (0xC << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_64  = (0xD << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_128 = (0xE << FDR_CONFIG_FDR_SEL_POS),
	FDR_CONFIG_FDR_SEL_FACTOR_256 = (0xF << FDR_CONFIG_FDR_SEL_POS),
} FDR_CONFIG_FDR_SEL_t;

/*
 * APEX_CONFIG12_MREG1
 * Register Name: APEX_CONFIG12
*/
/* FF_MAX_DURATION_SEL */
typedef enum {
	APEX_CONFIG12_FF_MAX_DURATION_102_CM = (0 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_120_CM = (1 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_139_CM = (2 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_159_CM = (3 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_181_CM = (4 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_204_CM = (5 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_228_CM = (6 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_254_CM = (7 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_281_CM = (8 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_310_CM = (9 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_339_CM = (10 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_371_CM = (11 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_403_CM = (12 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_438_CM = (13 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_473_CM = (14 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MAX_DURATION_510_CM = (15 << APEX_CONFIG12_FF_MAX_DURATION_SEL_POS),
} APEX_CONFIG12_FF_MAX_DURATION_t;

/* FF_MIN_DURATION_SEL */
typedef enum {
	APEX_CONFIG12_FF_MIN_DURATION_10_CM = (0 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_12_CM = (1 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_13_CM = (2 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_16_CM = (3 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_18_CM = (4 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_20_CM = (5 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_23_CM = (6 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_25_CM = (7 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_28_CM = (8 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_31_CM = (9 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_34_CM = (10 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_38_CM = (11 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_41_CM = (12 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_45_CM = (13 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_48_CM = (14 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
	APEX_CONFIG12_FF_MIN_DURATION_52_CM = (15 << APEX_CONFIG12_FF_MIN_DURATION_SEL_POS),
} APEX_CONFIG12_FF_MIN_DURATION_t;

/* ---------------------------------------------------------------------------
 * register bank MREG2 
 * ---------------------------------------------------------------------------- */

/*
 * OTP_CTRL7_MREG2
 * Register Name: OTP_CTRL7
*/

/* OTP_CTRL7 */
typedef enum {
	OTP_CTRL7_OTP_RELOAD_EN  = (1 << OTP_CTRL7_OTP_RELOAD_POS),
	OTP_CTRL7_OTP_RELOAD_DIS = (0 << OTP_CTRL7_OTP_RELOAD_POS),
} OTP_CTRL7_OTP_RELOAD_t;

/* OTP_PWR_DOWN */
typedef enum {
	OTP_CTRL7_PWR_DOWN_EN  = (1 << OTP_CTRL7_OTP_PWR_DOWN_POS),
	OTP_CTRL7_PWR_DOWN_DIS = (0 << OTP_CTRL7_OTP_PWR_DOWN_POS),
} OTP_CTRL7_PWR_DOWN_t;

#ifdef __cplusplus
}
#endif

#endif /* #ifndef _INV_IMU_DEFS_H_ */
