#!/usr/bin/env python3
"""Fetch and convert the ESP-IDF X.509 bundle into an Arduino header."""

import argparse
import datetime as _dt
import os
import pathlib
import subprocess
import sys
import urllib.request
from email.utils import parsedate_to_datetime


GEN_SCRIPT_URL = "https://raw.githubusercontent.com/espressif/esp-idf/master/components/mbedtls/esp_crt_bundle/gen_crt_bundle.py"
CACERT_URL = "https://curl.se/ca/cacert.pem"


def _file_timestamp(path: pathlib.Path) -> _dt.datetime:
    return _dt.datetime.fromtimestamp(path.stat().st_mtime, tz=_dt.timezone.utc)


def _normalize_dt(value: _dt.datetime) -> _dt.datetime:
    if value.tzinfo is None:
        return value.replace(tzinfo=_dt.timezone.utc)
    return value.astimezone(_dt.timezone.utc)


def download(url: str, dest: pathlib.Path, force: bool = False) -> _dt.datetime:
    if dest.exists() and not force:
        return _file_timestamp(dest)
    last_modified = None
    with urllib.request.urlopen(url) as response:
        data = response.read()
        last_modified_header = response.info().get("Last-Modified")
        if last_modified_header:
            last_modified = parsedate_to_datetime(last_modified_header)
    dest.write_bytes(data)
    if last_modified:
        normalized = _normalize_dt(last_modified)
        ts = normalized.timestamp()
        os.utime(dest, (ts, ts))
    return _file_timestamp(dest)


def run_gen_script(gen_script: pathlib.Path, cacert: pathlib.Path, workdir: pathlib.Path) -> pathlib.Path:
    args = [sys.executable, str(gen_script), "-i", str(cacert)]
    subprocess.run(args, check=True, cwd=workdir)
    candidates = [
        workdir / "x509_crt_bundle.bin",
        workdir / "x509_crt_bundle",
    ]
    for candidate in candidates:
        if candidate.exists():
            if candidate.suffix != ".bin":
                target = candidate.with_suffix(".bin")
                candidate.replace(target)
                return target
            return candidate
    raise FileNotFoundError("x509_crt_bundle(.bin) was not created by gen_crt_bundle.py")


def format_as_c_array(data: bytes) -> str:
    hex_bytes = [f"0x{b:02x}" for b in data]
    lines = []
    for i in range(0, len(hex_bytes), 16):
        chunk = ", ".join(hex_bytes[i : i + 16])
        lines.append(f"    {chunk}")
    return ",\n".join(lines)


def _format_timestamp(dt: _dt.datetime) -> str:
    return _normalize_dt(dt).isoformat(timespec="seconds")


def write_header(
    bin_path: pathlib.Path, header_path: pathlib.Path, source_timestamp: _dt.datetime
) -> None:
    data = bin_path.read_bytes()
    header_path.parent.mkdir(parents=True, exist_ok=True)
    c_array = format_as_c_array(data)
    header_lines = [
        f"// This file is auto-generated by tools/update_bundle.py on {_format_timestamp(source_timestamp)}",
        f"// Source bundle: {CACERT_URL}",
        "",
        "#pragma once",
        "",
        "#include <pgmspace.h>",
        "#include <stddef.h>",
        "",
        "static const unsigned char x509_crt_bundle[] PROGMEM = {",
        f"{c_array}",
        "};",
        "",
        "static const unsigned int x509_crt_bundle_len = sizeof(x509_crt_bundle);",
        "",
    ]
    header_path.write_text("\n".join(header_lines), encoding="utf-8")


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument(
        "--force-download",
        action="store_true",
        help="Re-download helper scripts even if cached versions exist.",
    )
    parser.add_argument(
        "--cacert-url",
        default=CACERT_URL,
        help="Override the source CA bundle URL.",
    )
    return parser.parse_args()


def main() -> None:
    args = parse_args()
    repo_root = pathlib.Path(__file__).resolve().parents[1]
    cache_dir = repo_root / "tools" / "cache"
    cache_dir.mkdir(parents=True, exist_ok=True)

    gen_script = cache_dir / "gen_crt_bundle.py"
    cacert = cache_dir / "cacert.pem"

    download(GEN_SCRIPT_URL, gen_script, force=args.force_download)
    cacert_timestamp = download(args.cacert_url, cacert, force=args.force_download)

    bundle_bin = run_gen_script(gen_script, cacert, cache_dir)
    header_path = repo_root / "src" / "esp32_cert_bundle.h"
    write_header(bundle_bin, header_path, cacert_timestamp)


if __name__ == "__main__":
    main()
