# ESP32AutoTask

[English README](README.md) is available.

ESP32 の Arduino 環境で FreeRTOS タスクを手軽に使うためのヘルパーです。`begin()` を呼ぶだけで各コア（0/1）の Low / Normal / High 優先度タスクを用意し、決められたフック関数をスケッチ側に書けば定期的に実行されます。フックを定義しなければタスクは即終了するため、余計なリソースは使いません。

デフォルト値:
- 実行周期 `periodMs = 1`。
- スタックサイズ `stackSize = ARDUINO_LOOP_STACK_SIZE`（ESP32 Arduino の標準は 8192 バイト）。足りないときは `begin(stackBytes)` で増やす。
- `periodMs` を 0 にすると最速で実行されますが、低優先度タスクが走りにくくなるため 1 以上を推奨します。
- 優先度は FreeRTOS の範囲（0〜24; 数字が大きいほど高い）に収めてください。初心者は 1〜4 程度にとどめるのが安全です。
  - デフォルト優先度: Low=1, Normal=2, High=3。Core1 Low は `loop()` と同じ優先度 (~1) なので、`delay` なしで長時間処理すると `loop()` が遅くなります。

## コアと優先度のイメージ

- Arduino の `loop()` は ESP32 Arduino ではコア1にピン留めされたタスク（優先度 ~1、スタック 8192B）として動いています。Wi-Fi/BT などのシステムタスクは主にコア0の高優先度で動きます。
- 本ライブラリはコア0/1それぞれに Low / Normal / High フックを用意し、デフォルト優先度を `1 / 2 / 3` に設定する想定です。`loop()` と同じ優先度で回したい処理は Core1 Low（長く走らせない）、少し上なら Normal、より重い処理は High、バックグラウンドの軽作業は Low を使う、と覚えると良いです。
- シングルコアの ESP32シリーズ (例: ESP32-SOLO / ESP32-C3 / C2 / C6 / S2) では、Core1 向けフックも Core0 で実行されます（実行順だけ分けているイメージ）。

## 使い方（最短）

1. `ESP32AutoTask.h` をインクルード。
2. `setup()` で `AutoTask.begin()` を呼ぶ。
3. 呼ばせたいフック関数（例: `LoopCore0_Low` や `LoopCore1_Normal`）を定義する。

```cpp
#include <ESP32AutoTask.h>

void setup() {
    ESP32AutoTask::AutoTask.begin();  // デフォルト設定でタスクを用意
}

// 必要なフックだけ定義すれば動く
void LoopCore0_Low() {
    // コア0・低優先度のループ
}

void LoopCore1_Normal() {
    // コア1・通常優先度のループ
}

void LoopCore1_High() {
    // コア1・高優先度のループ
}
```

- 各コアに Low / Normal / High のフックが用意されています。
- フックを定義しない場合、内部で `vTaskDelete(NULL)` されるため常駐負荷はありません。

## スタックサイズだけ変えたいとき

「デフォルトより足りないかも」と思ったら全タスク共通のスタックサイズを引数で渡します（例: 16384）。

```cpp
void setup() {
    ESP32AutoTask::AutoTask.begin(/*stackBytes=*/16384);  // 既定値からスタックだけ上書き
}
```

- 初心者は基本 `begin()` のみで OK。スタック不足が疑わしいときだけ数字を足す運用を想定。

## さらに細かく設定したいとき（上級者向け）

優先度やスタックサイズ、実行周期を変えたい場合だけ `Config` を渡します。

```cpp
void setup() {
  ESP32AutoTask::Config cfg;
  cfg.core0.low = {1, ARDUINO_LOOP_STACK_SIZE, 1};
  cfg.core0.normal = {2, ARDUINO_LOOP_STACK_SIZE, 1};
  cfg.core0.high = {3, ARDUINO_LOOP_STACK_SIZE, 1};
  cfg.core1.low = {1, ARDUINO_LOOP_STACK_SIZE, 1};
  cfg.core1.normal = {2, ARDUINO_LOOP_STACK_SIZE, 1};
  cfg.core1.high = {3, ARDUINO_LOOP_STACK_SIZE, 1};

  ESP32AutoTask::AutoTask.begin(cfg);  // 全パラメータを指定
}
```

よくある使い方（例）: デフォルト値をベースに一部だけ変える。

```cpp
ESP32AutoTask::Config cfg;  // デフォルト値で初期化される想定
cfg.core1.high.priority = 4;  // コア1・高優先度だけ強めに
ESP32AutoTask::AutoTask.begin(cfg);
```

## マルチコアと優先度の基本（初心者→中級者へのヒント）

- マルチコアとは: ESP32 は通常 2 コア（Core0/Core1）を持ち、OS がタスクを割り当てて並列実行します。  
- Core0 と Wi‑Fi/Bluetooth: 無線を使うと Core0 に無線関連の高優先度タスクが常駐し、ユーザーの Core0 タスクは周期が乱れやすくなります。無線を使わないときは Core0 に余裕があるので重めの処理も載せやすいです。  
- Core1 と `loop()`: Arduino の `loop()` は Core1 で優先度 ~1 で動いています。Core1 でユーザータスクを動かすときは、`loop()` との優先度関係を意識して決めてください。  
- 同じ優先度だと: 同一コア・同一優先度のタスクは協調的にスケジューリングされ、実質ラウンドロビンになります。周期性や応答性を分けたいなら優先度をずらす方が安全です。  
- 割り込みとの関係: 割り込みはタスクより高い優先度ですが、できることは限定されます。重い処理は ISR で完結させず、FreeRTOS の通知やキューで高優先度タスクに渡して処理させるのが定石です。  
- 高優先度タスクの作法: 実行時間を短くし、重い処理ほど優先度を下げるのが基本。`delay()`（このライブラリのフックは `delay` を使う）で他タスクに CPU を譲り、処理が終わったら必ず待ちを入れてください。入れないとそれ以下の優先度が動けずデッドロックや WDT の原因になります。  
- I2C などの通信: 複数タスクから同時にアクセスするとパニックでリセットする場合があります。通信・共有リソースへのアクセスは特定のタスクに集約し、他のタスクからはキュー/通知で依頼する形にしてください。  
- タスク間同期の例: 状態を共有するときは FreeRTOS のキュー／通知／ミューテックスを使うと安全です。通知は「1回のイベントを伝える」用途、キューはデータを順番に渡す用途、ミューテックスは排他制御に使います。I2C や SPI などの共有バスは「専任タスクがキューを受けて処理する」形にすると衝突を避けられます。  
- 時間精度について: タスクは厳密な時間精度を保証しません。より高精度が必要ならハードウェアタイマー（割り込み）を使い、割り込み内で通知を送ってタスク側で処理するのがおすすめです。  
- 通知を使う理由: タスクは通知を待機している間はブロックできます。優先度の高いタスクが通知待ちで眠っていれば、割り込み発生→通知送信→直後に高優先度タスクが即座に再開、という流れでリアルタイムに近い応答ができます。  
- WDT（ウォッチドッグタイマ）: 一定時間 CPU を占有すると WDT が発火し、パニック→リセットに至ります。重い処理は分割して `delay()` を挟むか、優先度を下げて他タスクに譲ってください。  
- パニック時に確認する項目: (1) スタック不足（`begin(stackBytes)` で増やす）、(2) 無限ループで `delay` を入れ忘れていないか、(3) 同じ優先度で重い処理を回していないか、(4) 共有リソース（I2C/SPI/Serial）への同時アクセスがないか。  
- Serial/ログの影響: 高優先度タスクで大量の `Serial.print` を行うとブロックして他タスクが動けなくなることがあります。ログは低優先度でまとめて出すか、リングバッファに貯めて専用タスクで送る運用が安全です。  
- 細かい制御が必要になったら: 本ライブラリで足りなければ、直接 FreeRTOS のタスク API（`xTaskCreatePinnedToCore` など）や通知・キューを使う方が望ましいです。このライブラリは「最初に学ぶための足場」として位置づけています。  
- 追加の注意: 長時間ブロックする I/O を高優先度で回すのは避ける、スタック不足が疑わしければ `begin(stackBytes)` で増やす、ウォッチドッグ（WDT）を意識して `delay` を挟む、といった基本も守ってください。  
- FreeRTOS の TICK について: ESP32 Arduino のデフォルトでは 1ms ごとに TICK が入り、各コアで優先度の高いタスクから順に実行されます。`delay` を呼ぶとそのタスクは CPU を解放し、同じコア上の次のタスクに切り替わります。CPU を解放しないタスクがあると、そのコアで同等以下の優先度タスクは動けません。1ms 経過して次の TICK が来ると再び優先度順で実行が回ります。ESP-IDF のデフォルト TICK は 10ms なので、`vTaskDelay(1)` でも 10ms 待つ点が Arduino とは異なります。  
- FreeRTOS API を直接使う場合: Arduino の `delay` は便利ですが、ESP-IDF では `vTaskDelay` / `vTaskDelayUntil` と `pdMS_TO_TICKS` を使います（例: `vTaskDelay(pdMS_TO_TICKS(1000))` は 100 TICK で 1000ms 待機）。  

## もっと詳しく

設計の考え方や優先度の扱いなど、詳しい技術メモは `SPEC.ja.md` を参照してください。

## ライセンス

MIT License (`LICENSE` を参照)。
