# DCF77FreeRTOS

A lightweight and reliable **DCF77 time signal receiver** library for **ESP32**, powered by **FreeRTOS**.  
The decoder runs in a dedicated background task, providing precise time synchronization events without blocking other system operations.

---

## ⚙️ Features

- Runs DCF77 decoding in its own FreeRTOS task  
- Non-blocking, thread-safe event handling  
- Robust signal classification and noise filtering  
- Automatic error detection and frame validation  
- Real-time receiver status reporting (no signal / signal / synced)  
- Works seamlessly alongside GPS or NTP synchronization  
- Designed for long-term high-stability clock systems  

---

## 🧩 Requirements

- **ESP32** board (tested on ESP32 DevKit and D1 Mini ESP32)  
- Arduino Core for ESP32 **v2.0+**  
- FreeRTOS (built into the ESP32 Arduino core)

---

## 🕹️ Usage Example

```cpp
#include <DCF77FreeRTOS.h>

DCF77FreeRTOS dcf(4);   // DCF input pin = 4

void setup() {
  Serial.begin(115200);
  dcf.begin(2);         // optional LED pin = 2
}

void loop() {
  DCFtime t;
  if (dcf.getTime(&t)) {
    Serial.printf("DCF time: %02d-%02d-%02d %02d:%02d (dow=%d)\n",
                  t.year, t.month, t.day, t.hour, t.minute, t.dow);
  }
  delay(1000);
}
```

---

## 📶 Status Codes

The current receiver status can be retrieved using the `getStatus()` method.  
Each state automatically times out to a lower level if no valid signal is received.

| Code | Status          | Meaning |
|------|------------------|---------|
| 0 | **Disconnected** | No impulses detected on the input pin |
| 1 | **No signal** | Impulses detected, but not valid |
| 2 | **Signal present** | Valid impulses received |
| 3 | **Synced** | A full valid time frame has been decoded |

---

## 🧠 Design Overview

The receiver logic is fully **interrupt-driven**. Pulse lengths are measured from edge to edge with **dead-time based noise suppression**. Each impulse is classified by its duration and interpreted as a received bit. Valid DCF77 frames are decoded **once per minute** at the end of the 59-bit frame. Synchronization events are broadcast via **FreeRTOS event groups**. Several integrity checks ensure that only valid time data is reported.

This design allows for precise and efficient time decoding without blocking other tasks on the ESP32.

---

🪪 License

MIT License
Copyright © 2025
Zoltán Szilvásy
