/* Arduino library for the EA DOGM128 (128x64), DOGM132 (132x32), DOGL128 (128x64) pixel LCD
 * using ST7565R display controller
 * Copyright (c) 2025 Stefan Staub
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef ST7565DOG_H
#define ST7565DOG_H

#include <Arduino.h>
#include <SPI.h>
#include <Print.h>

#define DOG132_LCD_WIDTH          132
#define DOG132_LCD_HEIGHT         32

#define DOG128_LCD_WIDTH          128
#define DOG128_LCD_HEIGHT         64
#define DOG128_SHIFT_ADDR_TOPVIEW 4

#define LCD_PIXEL_PER_BYTE        8

#define COMMAND_COLUMN_ADDRESS_LSB            0x00
#define COMMAND_COLUMN_ADDRESS_MSB            0x10
#define COMMAND_V0_VRS_BASE                   0x20
#define COMMAND_V0_VRS_VALUE_DOGM132          0x23 // DOGM132
#define COMMAND_V0_VRS_VALUE_DOGx128          0x27 // DOGM128, DOGL128
#define COMMAND_POWER_CONTROL_SET_BASE        0x28
#define COMMAND_POWER_CONTROL_SET_LOW_VOLTAGE 0x2B // Booster Off, Regulator On, Follower On
#define COMMAND_POWER_CONTROL_SET_LOW_POWER   0x2F // Booster On, Regulator On, Follower On 
#define COMMAND_POWER_CONTROL_SET_WIDE_RANGE  0x2F // Booster On, Regulator On, Follower On
#define COMMAND_START_LINE_0                  0x40
#define COMMAND_VOLUME_MODE_SET               0x81
#define COMMAND_CONTRAST_VALUE_DOGM132        0x1F
#define COMMAND_CONTRAST_VALUE_DOGM128        0x16
#define COMMAND_CONTRAST_VALUE_DOGL128        0x10
#define COMMAND_ADC_NORMAL                    0xA0
#define COMMAND_ADC_REVERSE                   0xA1
#define COMMAND_BIAS_1_9                      0xA2
#define COMMAND_BIAS_1_7                      0xA3
#define COMMAND_ALL_POINTS_NORMAL             0xA4
#define COMMAND_ALL_POINTS_ON                 0xA5
#define COMMAND_DISPLAY_NORMAL                0xA6
#define COMMAND_DISPLAY_INVERT                0xA7
#define COMMAND_INICATOR_OFF                  0xAC
#define COMMAND_INICATOR_ON                   0xAD
#define COMMAND_INICATOR_MODE_FLASH_OFF       0x00
#define COMMAND_INICATOR_MODE_FLASH_ON        0x01
#define COMMAND_DISPLAY_OFF                   0xAE
#define COMMAND_DISPLAY_ON                    0xAF
#define COMMAND_PAGE_ADDRESS                  0xB0
#define COMMAND_COM_NORMAL                    0xC0
#define COMMAND_COM_REVERSE                   0xC8
#define COMMAND_READ_MODIFY_WRITE             0xE0
#define COMMAND_RESET                         0xE2
#define COMMAND_NOP                           0xE3
#define COMMAND_END                           0xEE
#define COMMAND_BOOSTER_RATIO                 0xF8
#define COMMAND_BOOSTER_VALUE_4x              0x00
#define COMMAND_BOOSTER_VALUE_5x              0x01
#define COMMAND_BOOSTER_VALUE_6x              0x03

/** display type
	* 
	* @param DOGM128 128x64 2.3''
	* @param DOGM132 132x32 2.1''
	* @param DOGL128 128x64 2.8''
	* 
	*/
typedef enum {
	DOGM128,
	DOGM132,
	DOGL128
	} display_t;

/** update modes
	* 
	* @param AUTO default
	* @param MANUELL
	* 
	*/
typedef enum {
	MANUAL,
	AUTO
	} update_t;

/** display settings
	* 
	* @param ON
	* @param OFF
	* @param SLEEP
	* @param DEFAULT
	* @param INVERT
	* @param VIEW_TOP
	* @param VIEW_BOTTOM
	* @param CONTRAST
	* 
	*/
typedef enum {
	ON,
	OFF,
	SLEEP,
	DEFAULT,
	INVERT,
	VIEW_TOP,
	VIEW_BOTTOM,
	CONTRAST
	} dispmode_t;

/** bitmap
	* 
	*/
struct Bitmap {
	int32_t xSize;
	int32_t ySize;
	int32_t byteInLine;
	char *data;
	};

class ST7565DOG : public Print {

	public:

	/** Construct a new ST7565DOG object
	 * 
	 * @param reset 
	 * @param a0 
	 * @param cs 
	 * 
	 */
	ST7565DOG(uint8_t reset, uint8_t a0, uint8_t cs);

	/** init the DOGM132 LCD controller
		* 
		* @param id DOGM128, DOGM132, DOGL128
		* @param spi spi class
		* 
		*/
	void begin(display_t id, SPIClass &spi = SPI);

	/** display functions
		* 
		* @param mode ON switch display on, or wake up from sleep
		* @param mode OFF set display off
		* @param mode SLEEP set display off and to sleep mode
		* @param mode VIEW_BOTTOM (default) set display orientation 0°
		* @param mode VIEW_TOP draw display oriention to 180°
		* @param mode INVERT invert the pixels
		* @param mode DEFAULT normal pixel display
		* @param mode CONTRAST set display contrast to default
		* 
		*/
	void display(dispmode_t mode);

	/** display functions
	 * 
	 * @param mode CONTRAST set display contrast with value,
	 * @param value sets display contrast 0 - 63, default 31
	 * 
	 */
	void display(dispmode_t mode, uint8_t value);

	/** set update mode
		* 
		* @param AUTO set update mode to auto, default
		* @param MANUAL the update function must manually set
		* 
		*/
	void update(update_t mode);

	/** update copy display buffer to lcd
		* 
		*/
	void update();

	/** clear the screen
		* 
		*/
	void cls();

	/** clear an area
		* 
		* @param x0,y0 top left corner
		* @param x1,y1 down right corner
		* @param color 1 set pixel, 0 erase pixel
		* 
		*/
	void cla(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 0);

	/** draw a single point
		* 
		* @param x horizontal position
		* @param y vertical position
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void point(int32_t x, int32_t y, uint8_t color = 1);

	/** draw a 1 pixel line
		* 
		* @param x0,y0 start point
		* @param x1,y1 end point
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void line(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);

	/** draw a rect
		* 
		* @param x0,y0 top left corner
		* @param x1,y1 down right corner
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void rectangle(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);

	/** draw a filled rect
		* 
		* @param x0,y0 top left corner
		* @param x1,y1 down right corner
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void rectangleFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);

	/** draw a rounded rect
		* 
		* @param x0,y0 top left corner
		* @param x1,y1 down right corner
		* @param rnd radius of the rounding
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void rectangleRound(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color = 1);

	/** draw a filled rounded rect
		* 
		* @param x0, y0 top left corner
		* @param x1, y1 down right corner
		* @param rnd radius of the rounding
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void rectangleRoundFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color = 1);


	/** draw a circle
		* 
		* @param x, y center
		* @param r radius
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void circle(int32_t x, int32_t y, int32_t r, uint8_t color = 1);

	/** draw a filled circle
		* 
		* @param x, y center
		* @param r radius
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		* use circle with different radius,
		* can miss some pixel
		* 
		*/
	void circleFill(int32_t x, int32_t y, int32_t r, uint8_t color = 1);

	/** draw an ellipse
		* 
		* @param x, y center
		* @param a, b radius
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void ellipse(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color = 1);

	/** draw an ellipse
		* 
		* @param x, y center
		* @param a, b radius
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void ellipseFill(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color = 1);

	/** set top left position of char/printf
		* 
		* @param x x-position
		* @param y y-position
		* 
		*/
	void locate(uint8_t x, uint8_t y);

	/** select the font to use
		* 
		* @param f pointer to font array
		*
		*   font array can created with GLCD Font Creator from https://www.mikroe.com/glcd-font-creator
		*   you have to add 4 parameter at the beginning of the font array to use:
		*   - the number of byte / char
		*   - the vertial size in pixel
		*   - the horizontal size in pixel
		*   - the number of byte per vertical line
		*   you also have to change the array to char[]
		* @code
		* lcd.font((unsigned char*)Small_7);
		* @endcode
		*/
	void font(uint8_t *fnt);

	/** print bitmap to buffer
		* 
		* @param bm Bitmap in flash
		* @param x  x start
		* @param y  y start
		* 
		*/
	void bitmap(Bitmap bm, int32_t x, int32_t y);

	/** function used for print functionality
		* 
		* @param value 
		* @return size_t 
		* 
		*/
	virtual size_t write(uint8_t value);

	private:

	/** draw a pixel in the graphic buffer
		* 
		* @param x horizontal position
		* @param y vertical position
		* @param color optional, 1 set pixel, 0 erase pixel, default 1
		* 
		*/
	void pixel(int32_t x, int32_t y, uint8_t color = 1);

	/** draw a character on given position out of the active font to the LCD buffer
		* 
		* @param x x-position of char (top left)
		* @param y y-position
		* @param c char to print
		* 
		*/
	void character(uint8_t x, uint8_t y, uint8_t c);

	/** Write a page of data to the LCD controller
		* 
		* @param page page number from 0-7
		* 
		*/
	void writePage(uint8_t page);

	/** write data to the LCD controller
		* 
		* @param data data written to LCD controller
		* 
		*/
	void writeData(uint8_t data); // Write data to the LCD controller

	/** Write a command the LCD controller
		* 
		* @param command command to be written
		* 
		*/
	void writeCommand(uint8_t command); // Write a command the LCD controller

	SPIClass *spi;
	display_t id;
	uint8_t reset;
	uint8_t a0;
	uint8_t cs;
	uint8_t charX;
	uint8_t charY;
	uint8_t offset;
	bool autoUpdate;
	uint8_t width;
	uint8_t height;
	uint32_t graphic_buffer_size;
	uint8_t *font_buffer;
	uint8_t *graphic_buffer;
	};

#endif
