#include "ST7565DOG.h"
#include "fonts/Small_7.h"

static void inline swap(int32_t &a, int32_t &b) {
	int32_t c = a;
	a = b;
	b = c;
	}

ST7565DOG::ST7565DOG(uint8_t reset, uint8_t a0, uint8_t cs) {
	this->reset = reset;
	this->a0 = a0;
	this->cs = cs;
	pinMode(reset, OUTPUT);
	digitalWrite(reset, HIGH);
	pinMode(a0, OUTPUT);
	pinMode(cs, OUTPUT);
	digitalWrite(cs, HIGH);
	}

void ST7565DOG::begin(display_t id, SPIClass &spi) {
	this->spi = &spi;
	this->id = id;
	if (id == DOGM132) {
		width = DOG132_LCD_WIDTH;
		height = DOG132_LCD_HEIGHT;
		graphic_buffer_size = (DOG132_LCD_WIDTH * DOG132_LCD_HEIGHT) / LCD_PIXEL_PER_BYTE;
		graphic_buffer = new uint8_t [graphic_buffer_size];
		}

	if (id == DOGM128 || id == DOGL128) {
		width = DOG128_LCD_WIDTH;
		height = DOG128_LCD_HEIGHT;
		graphic_buffer_size = (DOG128_LCD_WIDTH * DOG128_LCD_HEIGHT) / LCD_PIXEL_PER_BYTE;
		graphic_buffer = new uint8_t [graphic_buffer_size];
		}

	this->spi->begin();
	this->spi->beginTransaction(SPISettings(10000000, MSBFIRST, SPI_MODE3));

	digitalWrite(a0, LOW);
	digitalWrite(cs, HIGH);
	digitalWrite(reset, LOW);
	delayMicroseconds(50);
	digitalWrite(reset, HIGH);
	delay(5);
	offset = 0;

	// Start Initial Sequence
	writeCommand(COMMAND_START_LINE_0); // display start line 0 0x40
	writeCommand(COMMAND_ADC_REVERSE); // ADC reverse 0xA1
	writeCommand(COMMAND_COM_NORMAL); // normal com0-com31 0xC0
	writeCommand(COMMAND_DISPLAY_NORMAL); // display normal 0xA6
	writeCommand(COMMAND_BIAS_1_9); // set bias 1/9 (duty 1/33) 0xA2
	writeCommand(COMMAND_POWER_CONTROL_SET_WIDE_RANGE); // booster, regulator and follower 0x2F
	writeCommand(COMMAND_BOOSTER_RATIO); // set internal booster to 3x/4x 0xF3
	writeCommand(COMMAND_BOOSTER_VALUE_4x);
	if (id == DOGM132) {
		writeCommand(COMMAND_V0_VRS_VALUE_DOGM132); //  set contrast
		writeCommand(COMMAND_VOLUME_MODE_SET);
		writeCommand(COMMAND_CONTRAST_VALUE_DOGM132);
		}
	if (id == DOGM128) {
		writeCommand(COMMAND_V0_VRS_VALUE_DOGx128); //  set contrast
		writeCommand(COMMAND_VOLUME_MODE_SET);
		writeCommand(COMMAND_CONTRAST_VALUE_DOGM128);
		}
	if (id == DOGL128) {
		writeCommand(COMMAND_V0_VRS_VALUE_DOGx128); //  set contrast
		writeCommand(COMMAND_VOLUME_MODE_SET);
		writeCommand(COMMAND_CONTRAST_VALUE_DOGL128);
		}
	writeCommand(COMMAND_INICATOR_OFF); // no indicator
	writeCommand(COMMAND_INICATOR_MODE_FLASH_OFF);
	writeCommand(COMMAND_DISPLAY_ON); // display on

	// clear and update LCD
	cls();
	autoUpdate = true; // switch on auto update
	locate(0, 0);
	font((uint8_t*)Small_7); // default font
	}

void ST7565DOG::display(dispmode_t mode) {
	switch (mode) {
		case ON:
			writeCommand(COMMAND_ALL_POINTS_NORMAL);
			writeCommand(COMMAND_DISPLAY_ON);
			break;
		case OFF:
			writeCommand(COMMAND_DISPLAY_OFF);
			break;
		case SLEEP:
			writeCommand(COMMAND_ALL_POINTS_ON);
			writeCommand(COMMAND_DISPLAY_OFF);
			break;
		case INVERT:
			writeCommand(COMMAND_DISPLAY_INVERT);
			break;
		case DEFAULT:
			writeCommand(COMMAND_DISPLAY_NORMAL);
			break;
		case VIEW_TOP:
			if(id == DOGM128 || id == DOGL128) {
				offset = DOG128_SHIFT_ADDR_TOPVIEW;
				}
			else offset = 0;
			writeCommand(COMMAND_ADC_NORMAL); // ADC normal
			writeCommand(COMMAND_COM_REVERSE); // reversed com31-com0
			update(); // update necessary
			break;
		case VIEW_BOTTOM:
			offset = 0;
			writeCommand(COMMAND_ADC_REVERSE); // ADC reverse
			writeCommand(COMMAND_COM_NORMAL); // normal com0-com31
			update(); // update necessary
			break;
		case CONTRAST:
			if (id == DOGM132) {
				writeCommand(COMMAND_VOLUME_MODE_SET); // set contrast to default for dogm 128
				writeCommand(COMMAND_CONTRAST_VALUE_DOGM132);
				}
			if (id == DOGM128) {
				writeCommand(COMMAND_VOLUME_MODE_SET); // set contrast to default for dogm132
				writeCommand(COMMAND_CONTRAST_VALUE_DOGM128);
				}
			if (id == DOGL128) {
				writeCommand(COMMAND_VOLUME_MODE_SET); // set contrast to default for dogl132
				writeCommand(COMMAND_CONTRAST_VALUE_DOGL128);
				}
			break;
		}
	}

void ST7565DOG::display(dispmode_t mode, uint8_t value) {
	if (mode == CONTRAST) {
		if (value < 64) {
			writeCommand(COMMAND_VOLUME_MODE_SET); // set contrast
			writeCommand(value & 0x3F);
			}
		}
	}

void ST7565DOG::update() {
	for (int32_t i = 0; i < 4; i++)
		writePage(i);
	if (id == DOGM128 || id == DOGL128) {
		for (int32_t i = 4; i < 8; i++) 
			writePage(i);
			}
	digitalWrite(cs, LOW);
	}

void ST7565DOG::update(update_t mode) {
	if (mode == MANUAL) autoUpdate = false;
	if (mode == AUTO) autoUpdate = true;
	}

void ST7565DOG::cls() {
	memset(graphic_buffer, 0x00, graphic_buffer_size); // clear display graphic_buffer
	update();
	}

void ST7565DOG::cla(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color) {
	autoUpdate = false;
	if (x0 > x1) swap(x0, x1);
	if (y0 > y1) swap(y0, y1);
	for (int32_t i = x0; i <= x1; i++) {
		for (int32_t j = y0; j <= y1; j++) {
			pixel(i, j, color);
			}
		}
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::pixel(int32_t x, int32_t y, uint8_t color) {
	if (x > width - 1 || y > height - 1 || x < 0 || y < 0) return;
	if (color == 0) graphic_buffer[x + ((y / 8) * width)] &= ~(1 << (y % 8)); // erase pixel
	else graphic_buffer[x + ((y / 8) * width)] |= (1 << (y % 8));   // set pixel
	}

void ST7565DOG::point(int32_t x, int32_t y, uint8_t color) {
	autoUpdate = false;
	pixel(x, y, color);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::line(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color) {
	autoUpdate = false;
	int32_t dx =  abs(x1 - x0), sx = x0 < x1 ? 1 : -1;
	int32_t dy = -abs(y1 - y0), sy = y0 < y1 ? 1 : -1;
	int32_t err = dx + dy, e2; /* error value e_xy */

	while(true) {
		pixel(x0, y0, color);
		if (x0 == x1 && y0 == y1) break;
		e2 = 2 * err;
		if (e2 > dy) { err += dy; x0 += sx; } /* e_xy+e_x > 0 */
		if (e2 < dx) { err += dx; y0 += sy; } /* e_xy+e_y < 0 */
		}
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::rectangle(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color) {
	autoUpdate = false;
	line(x0, y0, x1, y0, color);
	line(x0, y1, x1, y1, color);
	line(x0, y0, x0, y1, color);
	line(x1, y0, x1, y1, color);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::rectangleFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color) {
	autoUpdate = false;
	if (x0 > x1) swap(x0, x1);
	if (y0 > y1) swap(y0, y1);
	for (int32_t i = x0; i <= x1; i++) {
		for (int32_t j = y0; j <= y1; j++) {
			pixel(i, j, color);
			}
		}
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::rectangleRound(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color) {
	if (x0 > x1) swap(x0, x1);
	if (y0 > y1) swap(y0, y1);
	autoUpdate = false;
	int32_t r = rnd;
	int32_t x = -r, y = 0, err = 2 - 2 * r;
	line(x0 + rnd, y0, x1 - rnd, y0, color);
	line(x0 + rnd, y1, x1 - rnd, y1, color);
	line(x0, y0 + rnd, x0, y1 - rnd, color);
	line(x1, y0 + rnd, x1, y1 - rnd, color);
	do {
		pixel(x1 - rnd + y, y0 + x + rnd, color); // 1 I. quadrant
		pixel(x1 - rnd - x, y1 + y - rnd, color); // 2 IV. quadrant
		pixel(x0 + rnd - y, y1 - rnd - x, color); // 3 III. quadrant
		pixel(x0 + rnd + x, y0 + rnd - y, color); // 4 II. quadrant
		r = err;
		if (r <= y) err += ++y * 2 + 1;
		if (r > x || err > y) err += ++x * 2 + 1;
		} while (x < 0);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::rectangleRoundFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color) {
	if (x0 > x1) swap(x0, x1);
	if (y0 > y1) swap(y0, y1);
	autoUpdate = false;
	int32_t r = rnd;
	int32_t x = -r, y = 0, err = 2 - 2 * r;
	for (int32_t i = x0; i <= x1; i++) {
		for (int32_t j = y0+rnd; j <= y1-rnd; j++) {
			pixel(i, j, color);
			}
		}
	do {
		line(x0 + rnd - y, y0 + rnd + x, x1 - rnd + y, y0 + rnd + x, color);
		line(x0 + rnd + x, y1 - rnd + y, x1 - rnd - x, y1 - rnd + y, color);
		r = err;
		if (r <= y) err += ++y * 2 + 1;
		if (r > x || err > y) err += ++x * 2 + 1;
		} while (x < 0);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::circle(int32_t x, int32_t y, int32_t r, uint8_t color) {
	autoUpdate = false;
	int32_t dx = -r, dy = 0, err = 2 - 2 * r;
	do {
		pixel(x + dy, y + dx, color); // I. quadrant
		pixel(x + dx, y - dy, color); // II. quadrant
		pixel(x - dy, y - dx, color); // III. quadrant
		pixel(x - dx, y + dy, color); // IV. quadrant
		
		r = err;
		if (r <= dy) err += ++dy * 2 + 1;
		if (r > dx || err > dy) err += ++dx * 2 + 1;
		} while (dx < 0);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::circleFill(int32_t x, int32_t y, int32_t r, uint8_t color) {
	autoUpdate = false;
	int32_t dx = -r, dy = 0, err = 2 - 2 * r;
	do {
		line(x - dy, y + dx, x + dy, y + dx, color);
		line(x + dx, y + dy, x - dx, y + dy, color);
		r = err;
		if (r <= dy) err += ++dy * 2 + 1;
		if (r > dx || err > dy) err += ++dx * 2 + 1;
		} while (dx < 0);
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::ellipse(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color) {
	autoUpdate = false;
	int32_t dx = 0, dy = b; // I. Quadrant from left to right bottom
	int32_t a2 = a * a, b2 = b * b;
	int32_t err = b2 - (2 * b - 1) * a2, e2; // error in 1. step
	do {
		pixel(x + dx, y + dy, color); // I. quadrant
		pixel(x - dx, y + dy, color); // II. quadrant
		pixel(x - dx, y - dy, color); // III. quadrant
		pixel(x + dx, y - dy, color); // IV. quadrant
		e2 = 2 * err;
		if (e2 <  (2 * dx + 1) * b2) { ++dx; err += (2 * dx + 1) * b2; }
		if (e2 > -(2 * dy - 1) * a2) { --dy; err -= (2 * dy - 1) * a2; }
		} while (dy >= 0);
	while (dx++ < a) {
		pixel(x + dx, y, color); // -> finish the top of the ellipse
		pixel(x - dx, y, color);
		}
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::ellipseFill(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color) {
	autoUpdate = false;
	int32_t dx = 0, dy = b; // I. quadrant from left to right bottom
	int32_t a2 = a * a, b2 = b * b;
	int32_t err = b2 - (2 * b - 1) * a2, e2; // error in 1. step
	do {
		line(x - dx, y - dy, x + dx, y - dy, color);
		line(x + dx, y + dy, x - dx, y + dy, color); // I. II.
		e2 = 2 * err;
		if (e2 <  (2 * dx + 1) * b2) { ++dx; err += (2 * dx + 1) * b2; }
		if (e2 > -(2 * dy - 1) * a2) { --dy; err -= (2 * dy - 1) * a2; }
		} while (dy >= 0);
	while (dx++ < a) {
		line(x + dx, y, x - dx, y + y, color);
		}
	autoUpdate = true;
	if (autoUpdate) update();
	}

void ST7565DOG::locate(uint8_t x, uint8_t y) {
	charX = x;
	charY = y;
	}

void ST7565DOG::character(uint8_t x, uint8_t y, uint8_t c) {
	int32_t hor, vert, offset, bpl, b;
	uint8_t *sign;
	uint8_t z, w;

	if ((c < 31) || (c > 127)) return;   // test char range

	// read font parameter from start of array
	offset = font_buffer[0]; // bytes / char
	hor = font_buffer[1];    // get hor size of font
	vert = font_buffer[2];   // get vert size of font
	bpl = font_buffer[3];    // bytes per line

	if (charX + hor > width) {
		charX = 0;
		charY = charY + vert;
		if (charY >= height - font_buffer[2]) {
			charY = 0;
			}
		}

	sign = &font_buffer[((c - 32) * offset) + 4]; // start of char bitmap
	w = sign[0]; // width of actual char
	// construct the char into the font_graphic_buffer
	for (int32_t j = 0; j < vert; j++) {  // vert line
		for (int32_t i = 0; i < hor; i++) { // horz line
			z =  sign[bpl * i + ((j & 0xF8) >> 3) + 1];
			b = 1 << (j & 0x07);
			if (( z & b ) == 0x00) {
				pixel(x + i, y + j, 0);
				}
			else {
				pixel(x + i, y + j, 1);
				}
			}
		}
	charX += w;
	}

void ST7565DOG::font(uint8_t *fnt) {
	font_buffer = fnt;
	}

void ST7565DOG::bitmap(Bitmap bm, int32_t x, int32_t y) {
	int32_t b;
	char d;

	for (int32_t v = 0; v < bm.ySize; v++) {   // lines
		for (int32_t h = 0; h < bm.xSize; h++) { // pixel
			if (h + x >= width) break;
			if (v + y >= height) break;
			d = bm.data[bm.byteInLine * v + ((h & 0xF8) >> 3)];
			b = 0x80 >> (h & 0x07);
			if ((d & b) == 0) {
				pixel(x + h, y + v, 0);
				}
			else {
				pixel(x + h, y + v, 1);
				}
			}
		}
		if (autoUpdate) update();
	}

size_t ST7565DOG::write(uint8_t value) {
	if (value == '\n') { // new line
		charX = 0;
		charY = charY + font_buffer[2];
		if (charY >= height - font_buffer[2]) {
			charY = 0;
			}
		}
	else {
		character(charX, charY, value);
		if (autoUpdate) update();
		}
	return 1;
	}

void ST7565DOG::writePage(uint8_t page) {
	writeCommand(offset); // set column low nibble
	writeCommand(COMMAND_COLUMN_ADDRESS_MSB); // set column high nibble
	writeCommand(COMMAND_PAGE_ADDRESS + page); // set page address
	for (int32_t i = width * page; i < width * (page+1); i++) {
		writeData(graphic_buffer[i]);
		}
	}

void ST7565DOG::writeCommand(uint8_t command) {
	digitalWrite(a0, LOW);
	digitalWrite(cs, LOW);
	spi->transfer(command);
	digitalWrite(cs, HIGH);
	}

void ST7565DOG::writeData(uint8_t data) {
	digitalWrite(a0, HIGH);
	digitalWrite(cs, LOW);
	spi->transfer(data);
	digitalWrite(cs, HIGH);
	}