# Arduino LCD SPI library for EA DOG graphic LCDs from Electronic Assembly with ST7565R controller

This library is for the LCDs from Electronic Assembly
http://www.lcd-module.de/produkte/dog.html <br>
The library support following types:
- EA DOGL128 (128 x 64 pixel)
- EA DOGM128 (128 x 64 pixel)
- EA DOGM132 (132 x 32 pixel)

Because of RAM consumption the library is intended for modern Microcontrollers like RaspberryPI Pico(2).

## Installation

1. Download from https://github.com/sstaub/ST7652DOG/master.zip or use the the Arduino Library installer.
2. For PIO extract the file and move the folder to your Library folder.

## Example

Here is an example for an EA DOGM132 display with a RaspberryPi Pi(2).

```cpp
#include <Arduino.h>
#include <ST7565DOG.h>
#include <fonts/Small_7.h>
#include <fonts/ArialR12x14.h>

// DOGM132x-5, DOGM128x-6, DOGL128x6 Single Supply
// Pins for RaspberryPi Pico(2), SPI0

// DOG Pin   Purpose   Pico Pin
// 40        CS        17 (SPI0 CSn)
// 39        RST       21
// 38        A0        20
// 37        SCL       18 (SPI0 SCK)
// 36        SI        19 (SPI0 Tx)
// 35        VDD       3.3V
// 34        VDD2      3.3V
// 33        VSS       GND
// 32        VOUT      1uF -> GND
// 31        CAP3P     1uF -> CAP1N
// 30        CAP1N     1uF -> CAP1P
// 29        CAP1P
// 28        CAP2P     1uF -> CAP2N
// 27        CAP2N
// 26        VSS       GND
// 25        V4        1uF -> GND
// 24        V3        1uF -> GND
// 23        V2        1uF -> GND
// 22        V1        1uF -> GND
// 21        V0        1uF -> GND

#define CS 17
#define A_0 20
#define RESET 21

ST7565DOG lcd(RESET, A_0, CS);

void exampleDOGM132();
void exampleDOGM128();

void setup() {
  lcd.begin(DOGM132);
	lcd.rectangle(0, 0, 131, 31);
	lcd.rectangleFill(2, 2, 4, 4);
	lcd.rectangleFill(2, 27, 4, 29);
	lcd.rectangleFill(61, 2, 63, 4);
	lcd.rectangleFill(61, 27, 63, 29);
	lcd.line(65, 0, 65, 31);
	lcd.ellipse(33, 15, 20, 10);
	lcd.circle(33, 15, 6);
	lcd.circleFill(33, 15, 3);
	lcd.locate(70, 1);
	lcd.font((uint8_t*)ArialR12x14);
	lcd.printf("Arduino");
	lcd.locate(70, 13);
	lcd.font((uint8_t*)Small_7);
	lcd.printf("EA");
	lcd.locate(70, 22);
 	lcd.printf("DOGM132");
	}

void loop() {
	lcd.display(INVERT); // display inverted
	delay(1000);
	lcd.display(DEFAULT); // display normal
	delay(1000);
	lcd.display(VIEW_TOP); // display rotated
	delay(1000);
	lcd.display(VIEW_BOTTOM); // display normal orientation
	delay(1000);
	}

```

# Documentation

## Constructor
```cpp
ST7565DOG(uint8_t reset, uint8_t a0, uint8_t cs);
```
- **reset** pin for reset
- **a0** pin for a0
- **cs** pin for cs (chip select)

Create a LCD object.

**Example**
```cpp
#define CS 17
#define A_0 20
#define RESET 21

ST7565DOG lcd(RESET, A_0, CS);
```
	
## Methods

### **begin()**
```cpp
void begin(display_t id, SPIClass &spi = SPI);
```
- **id** ID of the display DOGM132, DOGM128, DOGL128
- **spi** optional SPI port, this is only needed if you use another interface than the standard SPI like SPI1 ...

Initialize the display, this must done in `setup()`.

**Example**
```cpp
lcd.begin(DOGM132);
```


### **display()**
```cpp
void display(dispmode_t mode);
void display(dispmode_t mode, uint8_t value);
```

- **mode** display function you want to use
- **value** set additional value, e.g. for contrast

Set the modes and functions of the display.

- DISPLAY_ON witch display on, or wake up from sleep
- DISPLAY_OFF set display off
- VIEW_TOP set display orientation to 180°
- VIEW_BOTTOM (default) set display orientation 0°
- SLEEP set display off and to sleep mode
- INVERT invert the pixels
- DEFAULT normal pixel display
- CONTRAST change the contrast from 0...63 

**Example**
```cpp
lcd.display(DISPLAY_OFF); // turn the display OFF
lcd.display(DISPLAY_ON); // turn the display ON
```


### **update()**
```cpp
void update(update_t mode);
void update();
```

Set the update modes and function of the display.
By default the graphic buffer is send to the controller immediately. But you can also collect the methods and send them once at all.
Call `update()` to send the graphic buffer to the display manually.

- AUTO set update mode to auto, default
- MANUAL the update function must manually set

**Example**
```cpp
lcd.update(MANUAL); // set update mode to MANUAL
```


## Graphic Methods

The coordinate system start on the top left corner 0, 0  and end at the bottom right corner (131, 31 for DOGM132 and 127, 63 for DOGx128). 

### **cls()**
```cpp
void cls();
```

Clears the display.

**Example**
```cpp
lcd.cls();
```


### **cla()**
```cpp
void cla(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 0);
```
- **x0** upper left start position x
- **y0** upper left start position y
- **x1** lower right end position x
- **y1** lower right end position y
- **color** color of the area to clear, 0 (default) to erase the pixel, 1 for set the pixel

Clears a defined area of the display.

**Example**
```cpp
lcd.cla(10, 10, 40, 20); // clears the area
```


### **point()**
void point(int32_t x, int32_t y, uint8_t color = 1);
- **x** position x
- **y** position y
- **color** color of the point, 1 (default) to set the pixel, 0 for erase the pixel

Draw a single point.

**Example**
```cpp
lcd.point(1, 1); // draw a point at position 1, 1
```


### **line()**
void line(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);
- **x0** start position x
- **y0** endposition y
- **x1** end position x
- **y1** end position y
- **color** color of the line, 1 (default) to set the pixels, 0 for erase the pixels

Draw a line.

**Example**
```cpp
lcd.line(0, 0, 20, 20); // draw a line
```


### **rectangle()**
```cpp
void rectangle(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);
```
- **x0** upper left start position x
- **y0** upper left start position y
- **x1** lower right end position x
- **y1** lower right end position y
- **color** color of the rectangle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a rectangle.

**Example**
```cpp
lcd.rectangle(10, 10, 40, 20);
```


### **rectangleFill()**
```cpp
void rectangleFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, uint8_t color = 1);
```
- **x0** upper left start position x
- **y0** upper left start position y
- **x1** lower right end position x
- **y1** lower right end position y
- **color** color of the rectangle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a filled rectangle.

**Example**
```cpp
lcd.rectangleFill(10, 10, 40, 20);
```


### **rectangleRound()**
```cpp
void rectangleRound(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color = 1);
```
- **x0** upper left start position x
- **y0** upper left start position y
- **x1** lower right end position x
- **y1** lower right end position y
- **rnd** radius of the curve
- **color** color of the rectangle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a rectangle with curved corners.

**Example**
```cpp
lcd.rectangleRound(10, 10, 40, 20, 3);
```


### **rectangleRoundFill()**
```cpp
rectangleRoundFill(int32_t x0, int32_t y0, int32_t x1, int32_t y1, int32_t rnd, uint8_t color = 1);
```
- **x0** upper left start position x
- **y0** upper left start position y
- **x1** lower right end position x
- **y1** lower right end position y
- **rnd** radius of the curve
- **color** color of the rectangle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a filled rectangle with curved cornes.

**Example**
```cpp
lcd.rectangleRoundFill(10, 10, 40, 20, 3);
```


### **circle()**
void circle(int32_t x, int32_t y, int32_t r, uint8_t color = 1);
- **x** position x
- **y** position y
- **r** radius y
- **color** color of the circle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a circle.

**Example**
```cpp
lcd.circle(10, 10, 4); // draw a circle at 10, 10 with radius 4
```


### **circleFill()**
void circleFill(int32_t x, int32_t y, int32_t r, uint8_t color = 1);
- **x** position x
- **y** position y
- **r** radius y
- **color** color of the circle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a filled circle.

**Example**
```cpp
lcd.circleFill(10, 10, 4); // draw a filled circle at 10, 10 with radius 4
```


### **ellipse()**
void ellipse(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color = 1);
- **x** position x
- **y** position y
- **a** radius size x axis
- **b** radius size y axis
- **color** color of the circle, 1 (default) to set the pixels, 0 for erase the pixels

Draw an ellipse.

**Example**
```cpp
lcd.ellipse(33, 31, 20, 10); // draw a circle at 33, 31 with with x size 2x20 and y size 2x10
```


### **ellipseFill()**
void ellipseFill(int32_t x, int32_t y, int32_t a, int32_t b, uint8_t color = 1);
- **x** position x
- **y** position y
- **a** radius size x axis
- **b** radius size y axis
- **color** color of the circle, 1 (default) to set the pixels, 0 for erase the pixels

Draw a filled ellipse.

**Example**
```cpp
lcd.ellipseFill(33, 31, 20, 10); // draw a circle at 33, 31 with with x size 2x20 and y size 2x10
```


## Text Methods

In general you can use the build in functions `print()` and `printf()` for text output.

### **locate()**
```cpp
void locate(uint8_t x, uint8_t y);
```
- **x** position x
- **y** position y

Set the x, y position of the text for printing.

**Example**
```cpp
lcd.locate(10, 15);
```


### **font()**
```cpp
void font(uint8_t *fnt);
```
- **fnt** pointer to font array

Setting the current font you want to use for test output. `Small_7` is the default font.
You find some fonts in the fonts folder.

The font must included in the main `.ino` or `.cpp` (PIO) file, e.g. `#include <fonts/ArialR12x14.h>`<br>
This is needed for every font you use even for the default font if you change the font to a new one and back. 

The font array can also created with GLCD Font Creator from https://www.mikroe.com/glcd-font-creator

You have to add 4 parameter at the beginning of the font array to use:
- the number of byte / char
- the vertical size in pixel
- the horizontal size in pixel
- the number of byte per vertical line

**Example**
```cpp
lcd.font((uint8_t*)ArialR12x14); // cast is necessary
```


## Bitmaps

### Structure
```cpp
struct Bitmap {
	int32_t xSize;
	int32_t ySize;
	int32_t byteInLine;
	char *data;
	};
```

### **bitmap()**
```cpp
void bitmap(Bitmap bm, int32_t x, int32_t y);
```

- **bm** bitmap in flash<br> 
- **x** start point x<br> 
- **y** start point x<br>

Draw a bitmap.


