/**
 * @file sfDevFPC2534.h
 * @brief header file for the SparkFun FPC2534 core implementation. The core implements all logic
 * and functionality of the FPC2534 sensor, independent of the communication protocol used.
 *
 * @copyright  (c) 2025, SparkFun Electronics Inc.
 *
 * SPDX-License-Identifier: MIT
 *
 *---------------------------------------------------------------------------------
 */

// Header file for the main class of the library.

#pragma once

// from the FPC SDK
#include "fpc_api.h"

// The interface definition for communication classes
#include "sfDevFPC2534IComm.h"

// This library is dependent on Arduino framework
#include <Arduino.h>

// Define the LED pin on the FPC2534 board
const uint8_t SPARKFUN_FPC2534_LED_PIN = 1;

// The design pattern that the library implements follows the standard implementation
// pattern of the FPC SDK - response from the sensor is delivered via callback functions.
//
// Define a struct that can hold all the callback functions
// that the library can call on events from the sensor

/// @struct sfDevFPC2534Callbacks_t
/// @brief Struct holding all callback function pointers for sfDevFPC2534 events
typedef struct
{
    void (*on_error)(uint16_t error);
    void (*on_status)(uint16_t event, uint16_t state);
    void (*on_version)(char *version);
    void (*on_enroll)(uint8_t feedback, uint8_t samples_remaining);
    void (*on_identify)(bool is_match, uint16_t id);
    void (*on_list_templates)(uint16_t num_templates, uint16_t *template_ids);
    void (*on_navigation)(uint16_t gesture);
    void (*on_gpio_control)(uint8_t state);
    void (*on_system_config_get)(fpc_system_config_t *cfg);
    void (*on_bist_done)(uint16_t test_verdict);
    void (*on_data_transfer_done)(uint8_t *data, size_t size);
    void (*on_mode_change)(uint16_t new_mode);
    void (*on_finger_change)(bool present);
    void (*on_is_ready_change)(bool isReady);

} sfDevFPC2534Callbacks_t;

/// @class sfDevFPC2534
/// @brief Core class implementing FPC2534 functionality independent of communication protocol
class sfDevFPC2534
{
  public:
    sfDevFPC2534();

    /**
     * @brief Request the status from the device.
     *
     * @return Result Code
     */
    fpc_result_t requestStatus(void);

    /**
     * @brief Request the version from the device.
     *
     * @return Result Code
     */
    fpc_result_t requestVersion(void);

    /**
     * @brief Request to start an enrollment operation.
     *
     * id type can be ID_TYPE_SPECIFIED or ID_TYPE_GENERATE_NEW
     *
     * @param id The User ID to be used for the new template.
     * @return Result Code
     */
    fpc_result_t requestEnroll(fpc_id_type_t &id);

    /**
     * @brief Request to start an identification operation.
     *
     * id type can be ID_TYPE_SPECIFIED or ID_TYPE_ALL
     *
     * @param id The User ID to be used for the new template.
     * @param tag Operation tag. Will be returned in the response.
     *
     * @return Result Code
     */
    fpc_result_t requestIdentify(fpc_id_type_t &id, uint16_t tag);

    /**
     * @brief Send an abort command to the device.
     *
     * @return Result Code
     */
    fpc_result_t requestAbort(void);

    /**
     * @brief Send a list templates command to the device.
     *
     * @return Result Code
     */
    fpc_result_t requestListTemplates(void);

    /**
     * @brief Request to delete a template.
     *
     * id type can be ID_TYPE_SPECIFIED or ID_TYPE_ALL
     *
     * @param id The User ID to be deleted.
     *
     * @return Result Code
     */
    fpc_result_t requestDeleteTemplate(fpc_id_type_t &id);

    /**
     * @brief Request to send a reset command to the device.
     *
     * @return Result Code
     */
    fpc_result_t sendReset(void);

    /**
     * @brief Populate and transfer a CMD_SET_CRYPTO_KEY request.
     *
     * @return Result Code
     */
    // fpc_result_t requestCryptoKey(void);

    /**
     * @brief Start navigation mode on the device.
     *
     * Starts the navigation mode.
     *
     * @param orientation Orientation in 90 degrees per step (0-3).
     *
     * @return Result Code
     */
    fpc_result_t startNavigationMode(uint8_t orientation);

    /**
     * @brief Start the BuiltIn Self Test. (BIST )
     *
     * Runs the BuiltIn Self Test.
     *
     * @return Result Code
     */
    fpc_result_t startBuiltInSelfTest(void);

    /**
     * @brief Send GPIO control command to the device.
     *
     * Configure gpio pins.
     *
     * @param pin   Pin to configure (see product specification).
     * @param mode  Mode selection (GPIO_CONTROL_MODE_*).
     * @param state State of pin if output (GPIO_CONTROL_STATE_*).
     *
     * @return Result Code
     */
    fpc_result_t requestSetGPIO(uint8_t pin, uint8_t mode, uint8_t state);

    /**
     * @brief Respond to a GPIO get state command to the device.
     *
     * Configure gpio pins.
     *
     * @param pin Pin to get state of (see product specification).
     *
     * @return Result Code
     */
    fpc_result_t requestGetGPIO(uint8_t pin);

    /**
     * @brief Set the system configuration on the device
     *
     * Configure various system settings.
     *
     * @param cfg Pointer to ::fpc_system_config_t.
     *
     * @return Result Code
     */
    fpc_result_t setSystemConfig(fpc_system_config_t *cfg);

    /**
     * @brief Request the system configuration from the device
     *
     * Configure various system settings.
     *
     * @param type One of FPC_SYS_CFG_TYPE_*.
     *
     * @return Result Code
     */
    fpc_result_t requestGetSystemConfig(uint8_t type);

    // /**
    //  * @brief Populate and transfer a CMD_PUT_TEMPLATE_DATA request
    //  *
    //  * Send template to device.
    //  *
    //  * @param id   Template id.
    //  * @param data Pointer to template data.
    //  * @param size Size of template data.
    //  *
    //  * @return Result Code
    //  */
    // fpc_result_t requestPutTemplateData(uint16_t id, uint8_t *data, size_t size);

    // /**
    //  * @brief Populate and transfer a CMD_GET_TEMPLATE_DATA request.
    //  *
    //  * Read template from device. This function will allocate memory for the template and
    //  * return data and size in the callback function (on_data_transfer_done).
    //  *
    //  * @param id Template id.
    //  *
    //  * @return Result Code
    //  */
    // fpc_result_t requestGetTemplateData(uint16_t id);

    /**
     * @brief Send a factory reset command to the device.
     *
     * Perform factory reset (system config flags must be set to support this).
     *
     * @return Result Code
     */
    fpc_result_t factoryReset(void);

    /**
     * @brief Get the current operating mode of the device.
     *
     * @return Current mode (STATE_ENROLL, STATE_IDENTIFY, STATE_NAVIGATION)
     */

    uint16_t currentMode(void) const
    {
        return _current_state & (STATE_ENROLL | STATE_IDENTIFY | STATE_NAVIGATION);
    }

    /**
     * @brief Check if a finger is currently present on the sensor.
     *
     * @return true - if finger is present
     * @return false - if no finger is present
     */
    bool isFingerPresent(void) const
    {
        return _finger_present;
    }

    /**
     * @brief Set the callback functions for the library to call on events. This is required.
     *
     * @param callbacks Struct holding all callback function pointers.
     */
    // for the library to actually work, user provided callbacks are needed ...
    void setCallbacks(const sfDevFPC2534Callbacks_t &callbacks)
    {
        _callbacks = callbacks;
    }

    /**
     * @brief initialize the library with a communication interface.
     *
     * @param comm The communication interface to use.
     * @return true - if initialization was successful
     */
    bool initialize(sfDevFPC2534IComm &comm)
    {
        _comm = &comm;
        return true;
    }

    /**
     * @brief Check if the device firmware is ready.
     *
     * @return true - if device firmware is ready
     */
    bool isReady(void) const
    {
        return (_current_state & STATE_APP_FW_READY) == STATE_APP_FW_READY;
    }

    /**
     * @brief Check if data is available to read from the device.
     *
     * @return true - if data is available
     */
    bool isDataAvailable(void) const
    {
        if (_comm == nullptr)
            return false;
        return _comm->dataAvailable();
    }

    /**
     * @brief Clear any available data from the device.
     *
     */
    void clearData(void)
    {
        if (_comm != nullptr)
            _comm->clearData();
    }
    /**
     * @brief Set the state of the on-board LED.
     *
     * @param on true to turn LED on, false to turn it off
     * @return Result Code
     */
    fpc_result_t setLED(bool on = true);

    // Process the next response from the device
    // If flushNone is true, it will skip over any EVENT_NONE events

    /**
     * @brief Process the next response message from the device. This should be called regularly (in loop)
     *
     * @param flushNone  - if true, EVENT_NONE events will be skipped
     * @return fpc_result_t
     */
    fpc_result_t processNextResponse(bool flushNone);
    fpc_result_t processNextResponse(void)
    {
        return processNextResponse(false);
    };

  private:
    // NOTE:
    // In general, messages are received from the device, identified and sent to the
    // appropriate parser function.

    fpc_result_t sendCommand(fpc_cmd_hdr_t &cmd, size_t size);
    fpc_result_t parseStatusCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseVersionCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseEnrollStatusCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseIdentifyCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseListTemplatesCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseNavigationEventCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseGPIOControlCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseGetSystemConfigCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseBISTCommand(fpc_cmd_hdr_t *, size_t);
    fpc_result_t parseCommand(uint8_t *frame_payload, size_t payload_size);

    bool checkForNoneEvent(uint8_t *payload, size_t size);
    fpc_result_t flushNoneEvent(void);

    // internal pointer to the communication interface. The comm device being used (I2C, Serial) is abstract
    // to the library via this interface.
    sfDevFPC2534IComm *_comm = nullptr;

    // Internal copy of the callback functions
    sfDevFPC2534Callbacks_t _callbacks;

    // current state of the sensor
    uint16_t _current_state = 0;

    // Is a finger present?
    bool _finger_present = false;
};