/*
  WebApp.ino - Web application to control, monitor and configure Iono Uno Ethernet

    Copyright (C) 2014-2022 Sfera Labs S.r.l. - All rights reserved.

    For information, see:
    https://www.sferalabs.cc/

  This code is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  See file LICENSE.txt for further informations on licensing terms.
*/

#include <SPI.h>
#include <EEPROM.h>
#ifdef ARDUINO_AVR_LEONARDO_ETH
  #include <Ethernet2.h>
#else
  #include <Ethernet.h>
#endif
#include <Iono.h>
#include <IonoWeb.h>

byte mac[6];
byte ip[4];
byte dnsa[4];
byte gateway[4];
byte subnet[4];

char pwd[9];

void configCmd(WebServer &webServer, WebServer::ConnectionType type, char* urlTail, bool tailComplete) {

  webServer.httpSuccess();
  webServer.print("{");

  webServer.print("\"i\":\"");
  for (int i = 0; i < 4; i++) {
    if (i != 0) {
      webServer.print(".");
    }
    webServer.print(ip[i]);
  }

  webServer.print("\",\"d\":\"");
  for (int i = 0; i < 4; i++) {
    if (i != 0) {
      webServer.print(".");
    }
    webServer.print(dnsa[i]);
  }

  webServer.print("\",\"g\":\"");
  for (int i = 0; i < 4; i++) {
    if (i != 0) {
      webServer.print(".");
    }
    webServer.print(gateway[i]);
  }

  webServer.print("\",\"s\":\"");
  for (int i = 0; i < 4; i++) {
    if (i != 0) {
      webServer.print(".");
    }
    webServer.print(subnet[i]);
  }

  webServer.print("\",\"m\":\"");
  for (int i = 0; i < 6; i++) {
    if (i != 0) {
      webServer.print(":");
    }
    char hex[3];
    hex[0] = (mac[i] >> 4) + 0x30;
    if (hex[0] > 0x39) hex[0] +=7;
    hex[1] = (mac[i] & 0x0f) + 0x30;
    if (hex[1] > 0x39) hex[1] +=7;
    hex[2] = '\0';
    webServer.print(hex);
  }

  webServer.print("\",\"p\":\"");
  webServer.print(pwd);

  webServer.print("\"}");
}

void webPageCmd(WebServer &webServer, WebServer::ConnectionType type, char* urlTail, bool tailComplete) {
  if (strlen(urlTail) == 0) { // serve page
    webServer.httpSuccess("text/html", "Content-Encoding: gzip\r\n");

    // These are the bytes of the zipped web page.
    // HOW-TO:
    // minify iono.html (e.g. use https://kangax.github.io/html-minifier/)
    // gzip -n -k -9 iono-min.html
    // hexdump -ve '1/1 "0x%.2x,"' iono-min.html.gz > bytes.txt
    P(page) = {0x1f,0x8b,0x08,0x00,0x00,0x00,0x00,0x00,0x02,0x03,0x95,0x58,0x7f,0x73,0xe2,0x46,0x12,0xfd,0xff,0x3e,0x85,0x76,0x36,0x87,0xa5,0x42,0x08,0x84,0xc1,0xf6,0x0a,0x04,0x95,0xcb,0xa6,0x12,0x5f,0x25,0xd9,0xd4,0xed,0x56,0xea,0xaa,0xbc,0x4e,0x95,0x7e,0x0c,0x30,0xb1,0x90,0x14,0x69,0x10,0x78,0x31,0xdf,0xfd,0x5e,0xcf,0x48,0x58,0x06,0x67,0x53,0xe7,0xb2,0x61,0xd4,0xd3,0x33,0xd3,0xfd,0xba,0xfb,0x4d,0xcb,0xd3,0x35,0x97,0x81,0x11,0x65,0xa9,0xe4,0xa9,0xf4,0xd9,0x56,0xc4,0x72,0xe5,0xc7,0xbc,0x12,0x11,0xef,0xa9,0x07,0x5b,0xa4,0x42,0x8a,0x20,0xe9,0x95,0x51,0x90,0x70,0xdf,0x65,0x69,0xb0,0xe6,0x7e,0x25,0xf8,0x36,0xcf,0x0a,0x39,0x9b,0x4a,0x21,0x13,0x3e,0x13,0x59,0x9a,0x4d,0xfb,0x7a,0x3c,0x2d,0xe5,0x23,0xbe,0xc2,0x2c,0x7e,0xdc,0x4b,0xbe,0x93,0xbd,0x20,0x11,0xcb,0xd4,0x8b,0x70,0x00,0x2f,0x26,0x0b,0x1c,0xe5,0xb9,0x57,0xf9,0xce,0x58,0xf1,0xa4,0xe2,0x52,0x44,0xc1,0x64,0x1d,0x14,0x4b,0x91,0x7a,0x83,0x49,0x1e,0xc4,0xb1,0x48,0x97,0x18,0x85,0x41,0xf4,0xb0,0x2c,0xb2,0x4d,0x1a,0x7b,0x6f,0x2f,0xaf,0xde,0x1d,0x9c,0x64,0xbf,0x48,0xb2,0x40,0x7a,0x09,0x5f,0xc8,0x83,0x53,0xd4,0x4f,0x85,0x58,0xae,0xf0,0x18,0xec,0xf3,0xac,0x84,0x99,0x59,0xea,0x05,0x61,0x99,0x25,0x1b,0xc9,0x0f,0x22,0xcd,0x37,0x72,0xdf,0xdb,0xf2,0xf0,0x41,0xc0,0x86,0x3c,0xe7,0x41,0x11,0xa4,0x11,0xf7,0xd2,0x2c,0xe5,0x93,0x46,0xae,0x0c,0x2c,0xc5,0x17,0xde,0x0b,0xe2,0x3f,0x36,0xa5,0xd4,0xb3,0x2b,0x4e,0xfb,0x7a,0xc3,0xcb,0x7c,0x77,0x70,0x08,0x9c,0xbd,0xc2,0xc2,0x73,0x07,0x83,0x7f,0x1e,0x1c,0xe5,0x99,0x16,0x0c,0x6f,0x06,0xf9,0xee,0x68,0xbf,0x11,0x6c,0x64,0x76,0x70,0xca,0xad,0xd8,0xd7,0xa2,0x31,0x6d,0x80,0x67,0x67,0xdd,0x92,0x18,0xc3,0x51,0x2d,0x36,0xb4,0x91,0x67,0xc6,0xd7,0x3b,0xf6,0xc8,0x59,0xaf,0xf7,0x0e,0x3f,0x38,0xa5,0x12,0xa5,0x08,0x45,0x22,0xe4,0xa3,0xb7,0x12,0x71,0xcc,0x53,0xbd,0x45,0x12,0x84,0x3c,0xd9,0xc7,0xa2,0xcc,0x93,0xe0,0xd1,0x0b,0x93,0x2c,0x7a,0x98,0x1c,0x37,0x2c,0x78,0x12,0x48,0x51,0xf1,0x49,0xb6,0x91,0x89,0x48,0x79,0x0b,0xe3,0x21,0xb6,0xd4,0x4e,0x5c,0x91,0x0f,0xb5,0xcb,0x97,0x34,0x6e,0x83,0x7f,0x73,0xf3,0x6e,0x12,0x66,0x45,0xcc,0x8b,0x5e,0x11,0xc4,0x62,0x53,0x7a,0x6e,0xe3,0x94,0x3e,0xda,0x0b,0x16,0x88,0xab,0xdd,0x12,0x84,0x7c,0x91,0x15,0xfc,0x15,0xaf,0xea,0x34,0xf3,0x2e,0x2e,0x0e,0xe7,0xea,0x32,0xcb,0x95,0x4d,0xca,0x67,0x1a,0x84,0x99,0x94,0xd9,0x5a,0x0d,0x0b,0x1d,0x8e,0x13,0xd3,0x16,0x8b,0xc5,0xa9,0x69,0xa3,0x57,0x4c,0xdb,0x4b,0xc4,0xbd,0x31,0x25,0x49,0x0c,0x67,0x58,0x4e,0xe8,0xb0,0x51,0x73,0xd8,0xe8,0xf9,0xb0,0xd1,0x11,0x94,0xe1,0xd5,0xc9,0x69,0x61,0x18,0x46,0xd1,0xe8,0x2f,0x0f,0x54,0x91,0xf4,0xa2,0x15,0x8f,0x1e,0x78,0xdc,0x6d,0x1f,0xdf,0x8e,0xe5,0x19,0xbc,0x57,0x83,0xc8,0xbd,0xb9,0x3a,0x84,0x85,0xbd,0x2a,0xf6,0x51,0x82,0x1c,0xf5,0x60,0xca,0xea,0xe0,0xf0,0x02,0xcf,0x59,0x92,0x15,0xda,0xcf,0x3a,0x7b,0x28,0x0f,0x9a,0x50,0x53,0xa6,0xe2,0x6c,0x59,0x1a,0x8e,0x6c,0x54,0xaf,0xaf,0x6f,0x9a,0x00,0xf7,0x6a,0x97,0x74,0xb4,0x48,0x8b,0xd7,0x49,0x7b,0x3d,0x3a,0x8a,0x44,0x2d,0x1a,0x91,0x5d,0x67,0xc5,0xaa,0x95,0x62,0x51,0xb5,0x6a,0xaf,0xc9,0x99,0x9b,0x97,0x0b,0xea,0x42,0x54,0x0c,0x70,0x6e,0x77,0x8f,0xf0,0x56,0xbe,0x53,0xf9,0xab,0x7a,0x43,0xe9,0x90,0x15,0x79,0x53,0x17,0x43,0xcc,0xaa,0xfa,0x31,0x74,0x19,0x1d,0xf3,0xd4,0x3d,0x81,0xac,0x58,0x86,0x81,0x39,0x1c,0x0d,0x6c,0xf5,0x37,0x1e,0xd8,0xce,0x8d,0x75,0x12,0x95,0xf1,0x19,0x4a,0x79,0x5d,0x69,0xcf,0xc7,0xbb,0x3a,0xea,0xb4,0x8c,0x12,0xcb,0x40,0x8a,0x8a,0xd8,0x78,0x25,0xdd,0x15,0x07,0x50,0xed,0x34,0x81,0x6c,0xf2,0x72,0xdc,0x4c,0xd4,0x25,0xd8,0x62,0xa4,0x33,0x64,0x9a,0xf8,0x61,0x33,0x43,0x31,0x00,0x79,0x3b,0xa8,0x97,0xd7,0xa6,0xb5,0x96,0xd7,0x6c,0x33,0x3e,0x1e,0xe1,0x94,0x79,0x70,0x72,0xfe,0xe5,0x90,0x26,0xc3,0x86,0x9a,0xae,0x4f,0x33,0x8b,0xe3,0x67,0x71,0xdd,0x16,0xd1,0x46,0x41,0xd1,0x5b,0x92,0x63,0x88,0xaf,0x49,0x01,0xb2,0xdf,0x82,0x49,0x1a,0xfc,0x54,0xb5,0x1b,0x6e,0x83,0xc6,0x09,0x10,0x04,0x58,0x13,0x94,0x6b,0xed,0xc7,0x24,0xda,0x14,0x25,0x42,0x1d,0xf3,0x45,0xb0,0x49,0x24,0x99,0xe3,0xad,0xb2,0x0a,0x49,0x7f,0x52,0xa7,0x6a,0x26,0x88,0x88,0x8a,0x5e,0x4c,0xe1,0x70,0x9a,0x72,0xa2,0xda,0x8b,0x31,0xbc,0x38,0xbc,0x8d,0xc2,0x7d,0x3b,0x7c,0x4d,0x45,0x0e,0x5a,0x06,0x0c,0xb5,0x01,0xa4,0x3b,0xa3,0x04,0x6d,0xa9,0x34,0x4c,0x76,0x82,0x07,0x95,0x46,0x1d,0x85,0x11,0x81,0x7f,0x5a,0xc9,0x84,0x66,0x90,0xed,0x5b,0x09,0x7b,0x7d,0xa4,0xf6,0xde,0x33,0x3d,0x1c,0xde,0x7e,0xfb,0xc1,0x6d,0xae,0x03,0x17,0x2a,0x87,0x69,0x5f,0x5f,0x78,0xd3,0x32,0x2a,0x44,0x2e,0x67,0x8b,0x4d,0x1a,0x11,0xd9,0x18,0x3b,0xd3,0xda,0x57,0x41,0x61,0x70,0x5f,0xae,0x44,0x39,0xe1,0x4e,0xe1,0xa7,0x7c,0x6b,0xfc,0xf7,0xe7,0x9f,0x7e,0x94,0x32,0xff,0x0f,0xff,0x73,0xc3,0x4b,0x69,0x43,0xec,0x64,0x29,0x82,0x1f,0xfb,0xcd,0x4a,0xac,0x1b,0xf9,0xbe,0x4f,0x33,0x05,0x0f,0xe2,0xc7,0x8f,0x32,0x90,0xbc,0xd3,0x31,0x87,0x83,0x41,0x2d,0x2e,0x21,0xd9,0x94,0x73,0xee,0x64,0xa6,0xe5,0x71,0x87,0x9b,0x96,0x75,0xa8,0x77,0x02,0x71,0x64,0x45,0x7b,0x2b,0x35,0x7d,0x38,0x1c,0xcd,0xfa,0xc6,0xe4,0xd6,0xbe,0xe0,0x72,0x53,0xa4,0x46,0x9c,0x45,0x9b,0x35,0x72,0xc1,0x59,0x72,0xf9,0x7d,0xc2,0x69,0xf8,0xaf,0xc7,0xdb,0x18,0x1a,0xcf,0xfa,0x65,0xed,0x46,0xea,0x0f,0x6c,0xb4,0x08,0x53,0xc0,0x6d,0x44,0x49,0x50,0x96,0xfe,0x05,0xb3,0x85,0xcf,0x2e,0x0c,0x11,0xfb,0x17,0xb7,0x0c,0xb5,0x5d,0x98,0x6b,0x3c,0xcf,0x98,0xcd,0xa1,0xd7,0x87,0x22,0x86,0x5f,0x30,0x0c,0x0b,0x0c,0x52,0xdf,0x9d,0xa4,0xd3,0xeb,0x49,0xda,0xed,0x5a,0xdf,0x98,0xac,0xdc,0x32,0xeb,0xae,0xba,0xef,0xfa,0xb2,0xcb,0x14,0x63,0x1b,0xac,0xcb,0x0c,0x63,0xad,0x7e,0xd9,0x5d,0x7a,0xdf,0xc5,0x4e,0x53,0x55,0x1b,0xea,0x80,0xf7,0x1f,0x58,0x37,0x85,0xcc,0x90,0x8f,0x39,0xf7,0x2f,0x14,0xd3,0x86,0xd9,0xee,0xc2,0xc8,0xd2,0x28,0x11,0xd1,0x83,0x7f,0x51,0x4a,0xd3,0xc2,0x12,0x55,0x8d,0x06,0x8c,0x79,0x5e,0x33,0x9b,0xf6,0x95,0x74,0xa6,0x05,0x5c,0x99,0x0a,0x7f,0x8e,0xf6,0x28,0x07,0x11,0x9e,0x39,0x63,0x1e,0x93,0xcc,0xce,0x60,0x55,0xd1,0x15,0xb4,0xd8,0x61,0x13,0xb2,0x56,0x96,0xcf,0xe6,0x0a,0x98,0x5a,0x74,0xd7,0x5d,0x33,0x7d,0x7a,0x62,0xca,0x42,0x66,0x75,0x79,0x37,0xeb,0xb2,0xf7,0x4c,0x89,0xe7,0x6c,0x8e,0x7d,0xde,0x2b,0xa9,0x99,0x4e,0xc7,0x73,0x4c,0xfd,0xd6,0x9e,0xfa,0x4d,0x4d,0xd1,0x19,0xa6,0x3e,0xd3,0xe0,0x90,0xe8,0xf3,0x6e,0xdb,0x8a,0xb7,0x86,0xb9,0xfe,0xd6,0x22,0x6d,0x8f,0xe1,0xf3,0xcb,0xe1,0xd9,0xf2,0x2b,0x65,0x39,0x05,0x00,0xc0,0x89,0x78,0x59,0xae,0xf3,0x1a,0x36,0x66,0xc3,0xe2,0x68,0xf1,0x6c,0x30,0x55,0x3e,0xc3,0x9e,0xb3,0xd1,0x9c,0x19,0xa0,0x12,0xa6,0x36,0x5b,0x77,0xd9,0x33,0xbe,0x0c,0xa7,0x33,0x43,0x35,0x77,0x34,0xac,0x75,0x35,0xd6,0x39,0x82,0xbd,0x45,0xb5,0x5c,0xe8,0x75,0xac,0x8d,0xb1,0x5a,0x37,0x63,0xdd,0x3b,0x76,0xfb,0x2b,0xb3,0xd9,0xfb,0x5f,0x3e,0xe2,0xf3,0x07,0xe4,0xeb,0x36,0x78,0xc4,0xe8,0xe3,0x26,0x4c,0x39,0x00,0x65,0x3f,0x7f,0xfb,0x1d,0x3e,0x7f,0xad,0x77,0x62,0xf7,0xca,0x52,0xc3,0x33,0x9a,0xc8,0x30,0xe0,0xf1,0x65,0xd2,0xb6,0xfb,0x0b,0xf0,0x61,0xa1,0x34,0x92,0x56,0x88,0xa3,0x05,0x85,0x38,0xa2,0x3e,0x2e,0x61,0x0a,0x41,0xd2,0x28,0xda,0x49,0x50,0x91,0x46,0x19,0x54,0x5c,0xed,0x68,0xe7,0x89,0xe9,0xb6,0x92,0x19,0xd5,0x61,0x72,0x3b,0xb5,0xa5,0xb5,0x97,0x02,0x35,0xa5,0xae,0xdc,0x4f,0x62,0xcd,0xd1,0x20,0x99,0x90,0x58,0x13,0xca,0x04,0xe1,0xc3,0x10,0xa8,0x32,0x6b,0x42,0x75,0x37,0x78,0xe3,0xfb,0x12,0xdf,0x98,0xf7,0x4b,0x2e,0x1b,0x75,0x28,0xd8,0xf2,0xe9,0x69,0xcc,0x2f,0x2d,0xcb,0x16,0xb0,0x19,0xf9,0x83,0x30,0xa9,0x02,0x34,0x4c,0x3a,0x9e,0x21,0x72,0xb6,0x70,0x14,0x3f,0x38,0x35,0xa5,0xf9,0x7c,0xce,0x54,0x97,0x06,0x30,0x89,0xdd,0xd8,0xb3,0x71,0xc5,0x9f,0x64,0x9b,0xb5,0x17,0x0b,0xf3,0x4d,0x98,0xe8,0xa4,0x94,0xfe,0x56,0xa4,0x71,0xb6,0xbd,0x63,0x3b,0xec,0x78,0xef,0x14,0x13,0xe9,0x64,0x39,0x4f,0x4d,0xf6,0xc3,0xf7,0x9f,0x50,0x54,0x5d,0xd6,0xa1,0xc8,0x82,0x57,0xde,0x03,0x74,0x8b,0x0a,0x99,0xec,0x33,0x11,0x26,0x9f,0xd9,0x6f,0x06,0x96,0x2d,0x9d,0x92,0xa7,0xb1,0x99,0x6e,0x92,0xa4,0x4d,0x03,0x65,0x6e,0x12,0x06,0xf9,0x19,0x06,0xb9,0x65,0xef,0xdc,0x4e,0x67,0xe7,0x82,0x4e,0x02,0x90,0x24,0xaa,0xea,0x79,0x15,0xe0,0x04,0x7b,0xd0,0x5a,0x1b,0x9a,0x6d,0x34,0x5a,0x8c,0x03,0x3f,0x5c,0x3b,0x5a,0xce,0x59,0x3f,0xc8,0x45,0x9f,0xb8,0x8a,0x53,0x2e,0xf7,0xd1,0xff,0x2d,0x04,0xc4,0xc4,0x55,0x73,0x77,0xe0,0xb9,0x40,0xae,0xb5,0x35,0x11,0x0d,0x70,0xcf,0x41,0x0c,0x4e,0x15,0x24,0x1b,0xee,0xfb,0xd1,0xd2,0xc9,0xe7,0xa6,0xa8,0xfc,0x37,0x03,0x7b,0x83,0xc8,0xda,0x14,0x3f,0x4b,0x07,0x15,0xc4,0x87,0x07,0xb6,0x2d,0xb2,0x74,0x69,0x34,0x29,0xaa,0x5c,0x6e,0x39,0x59,0x35,0x1c,0x8c,0x98,0xa3,0x54,0xfa,0x73,0x4d,0x53,0x12,0x15,0x24,0x51,0x41,0x92,0x6a,0x1f,0x70,0x23,0xe9,0x9b,0x0a,0x92,0xf7,0x60,0x2a,0xe2,0x48,0x5b,0xa2,0x70,0xdf,0xa4,0xda,0x94,0xa7,0xa7,0x66,0xe4,0xac,0x03,0x19,0xad,0x4c,0x39,0xbd,0x9c,0xf7,0x3f,0xc7,0x9f,0xe3,0x39,0xfd,0x3a,0x5f,0x1b,0xf5,0x3d,0x39,0x1d,0xcd,0xfb,0xbf,0x9b,0xa6,0x3b,0xbc,0x79,0x72,0xdf,0x0d,0x9f,0x86,0xc3,0xd1,0x13,0xda,0x17,0xfc,0xdd,0x3c,0x0d,0xc7,0x78,0x1e,0x8f,0xac,0xcf,0xce,0x40,0xff,0x5a,0x4f,0xe6,0x70,0x3c,0xfe,0xec,0x98,0xa6,0x39,0x78,0xfa,0xfb,0x15,0xff,0x8f,0xba,0xf5,0xa4,0x55,0xbf,0xae,0x48,0x3f,0xdf,0xf4,0x3d,0x18,0x7c,0x37,0xe8,0xbd,0x0b,0x7a,0x8b,0xfb,0xbd,0x6b,0x0f,0x0f,0x77,0x9f,0x1d,0xaf,0x77,0x6f,0xed,0xc7,0x87,0x97,0x72,0xe8,0x0a,0xcb,0xab,0xd1,0x41,0x95,0x34,0x38,0x25,0x3c,0x5d,0xca,0xd5,0xf4,0xf2,0x0c,0xba,0xfe,0xef,0x7a,0xfd,0x97,0xfb,0x2e,0x2d,0xb5,0xac,0xfa,0x32,0xc2,0xfb,0x25,0x92,0x8d,0xfd,0x8a,0x74,0x2c,0xb9,0xa1,0x3a,0x50,0x23,0x30,0xb0,0x12,0x8d,0xd9,0x09,0xc1,0x2c,0x8f,0x04,0x53,0xb6,0x08,0xc6,0x40,0x73,0x50,0xf0,0xb2,0xc4,0x53,0x93,0x0f,0xa0,0x4f,0xbc,0xc0,0xae,0x37,0x6b,0xe3,0xd2,0x88,0x56,0x78,0x03,0x8c,0xb0,0x6b,0x69,0x1b,0x09,0x97,0x34,0x30,0x82,0x34,0x46,0x83,0xbb,0x14,0xe8,0x73,0x71,0x09,0x3f,0x5a,0x20,0x26,0x79,0x6f,0xd9,0xa9,0xb3,0xc0,0xcd,0x88,0x84,0xb4,0xdf,0xb8,0x13,0xd1,0xf5,0x39,0x55,0x53,0xb7,0x76,0x83,0x4a,0xee,0x90,0x20,0xfb,0x91,0xac,0xa2,0xc9,0x55,0x1b,0x49,0x3f,0xb4,0x85,0x65,0x87,0x09,0x92,0xf5,0x39,0x03,0x29,0x6b,0xf7,0x94,0x72,0xba,0x13,0x18,0x4c,0xf8,0x74,0x34,0xe1,0xea,0x0a,0xbc,0x63,0x39,0x99,0x1a,0x85,0xf8,0xa8,0xb6,0x34,0x5a,0xe0,0x78,0x7e,0x6f,0x9d,0x10,0xc6,0x9d,0xa8,0xe6,0x9a,0x2a,0xbc,0x9a,0x39,0x6c,0x92,0xbc,0x20,0x11,0x3b,0xaa,0x80,0xf3,0xb9,0x62,0x54,0x75,0x3a,0x67,0xca,0x74,0x48,0xab,0x46,0x24,0x95,0x34,0x45,0x2e,0x28,0xc9,0x25,0xf4,0x37,0x47,0xa7,0x5e,0x3e,0xfa,0x60,0x7f,0xaa,0x3d,0x4d,0x92,0x29,0x75,0x03,0xba,0xc2,0xb9,0xac,0xeb,0x4a,0xdf,0xf1,0xbe,0xbe,0x9a,0x64,0xd7,0x67,0xf5,0x0d,0x0c,0xf0,0x4c,0xec,0xa5,0x9e,0x2c,0xa7,0x7e,0x4b,0x9a,0x33,0x17,0x16,0x0d,0xe8,0x46,0xe9,0xe8,0xe5,0xc0,0xf0,0xd2,0x96,0x5d,0x18,0x32,0xa7,0x73,0xb1,0x2a,0x28,0xe9,0xca,0xb1,0xe8,0xa8,0xa6,0x83,0x38,0xdf,0xdd,0xfb,0x9b,0xdd,0x8d,0x16,0xc7,0xd2,0x0d,0xb2,0x8f,0x40,0x28,0x51,0xa5,0x18,0xe5,0xa0,0x48,0x56,0xe4,0xa0,0x33,0x0e,0xce,0x02,0xb2,0xc0,0xcc,0x0e,0x4a,0x5b,0xd8,0x8b,0x08,0x1d,0x0f,0x85,0xd3,0xb5,0x29,0xda,0x30,0xe0,0x01,0x82,0x9d,0xab,0xba,0xb8,0x9d,0xbd,0x1b,0x36,0x83,0xcb,0x7a,0x50,0x81,0x45,0x52,0x34,0x60,0x3f,0x7e,0xfa,0xf9,0x27,0x36,0x01,0x81,0x66,0xed,0x3e,0x8c,0xb6,0x9b,0x34,0x5c,0x24,0xfd,0x7f,0x7f,0xfc,0xf0,0x8b,0x93,0x07,0x45,0xc9,0x4d,0x45,0xb5,0x48,0xdc,0x3c,0x4b,0x4b,0xfe,0x09,0xaf,0x0e,0xd6,0x84,0xae,0x80,0x68,0x09,0x4b,0x97,0xbe,0x54,0xd0,0xd0,0x6b,0x82,0x21,0x2d,0x22,0xa6,0x3a,0x18,0x12,0x51,0x9c,0xd4,0xa5,0x43,0x0c,0x29,0x9d,0x5c,0x73,0x64,0x95,0x89,0x18,0x9c,0xd9,0xe9,0x68,0x9a,0x3c,0xb4,0x96,0x63,0x5b,0xb4,0x25,0xe8,0x21,0xef,0x06,0xf7,0x7a,0xaf,0x1a,0x2d,0x1f,0x68,0xf9,0x7a,0x4b,0x9e,0xa0,0xfa,0x54,0x30,0xd5,0x1a,0x4a,0x48,0x68,0x52,0x23,0x04,0x74,0xe9,0x96,0x23,0x11,0xae,0xef,0x09,0x65,0xc2,0x01,0x78,0x38,0xbc,0xed,0x66,0xb7,0xbb,0x88,0x66,0xc3,0x4e,0x47,0x91,0xf3,0xce,0xad,0x33,0x06,0x7a,0xc3,0x36,0x1c,0xcd,0x25,0x52,0x5f,0xf8,0x27,0x19,0x5f,0xe7,0x34,0x26,0xab,0xed,0x57,0x26,0xcb,0xea,0x7c,0xb2,0xce,0x7b,0xa4,0xe4,0x2d,0x11,0x08,0xa0,0x6a,0xdf,0x4c,0xfa,0x2e,0x75,0xa0,0x13,0x90,0xc4,0x67,0x2b,0x74,0xe2,0x5e,0xbf,0xcf,0xba,0x88,0xf0,0xc1,0x1e,0xd2,0x85,0x44,0x96,0xbe,0xf0,0x48,0x7b,0x32,0x64,0x56,0x9d,0x0c,0xba,0xa5,0x40,0xd8,0x5f,0xc6,0x17,0xd5,0xc3,0x98,0x12,0xa3,0x4b,0x90,0x78,0x2b,0xd0,0xaf,0x03,0x53,0xfa,0x77,0x91,0x51,0x77,0xf8,0x60,0x94,0x59,0xab,0x81,0x66,0xfa,0x5f,0x6f,0xac,0x2d,0x53,0xea,0xea,0xd5,0xd1,0xd7,0x6f,0xe1,0xed,0x49,0xfd,0x1f,0xb6,0xdb,0x0f,0xbf,0x7c,0x68,0xef,0x52,0x18,0xa1,0xc4,0xfd,0xe5,0x47,0xe1,0xb1,0x0f,0xa2,0x34,0x57,0x0b,0x67,0xba,0x0f,0xff,0x8b,0x61,0xff,0xa5,0xb8,0xde,0x30,0xa7,0x86,0xb0,0xac,0x66,0x1f,0x83,0x0a,0xef,0x5a,0x86,0xbe,0xb6,0x37,0x85,0x82,0xcc,0x46,0x0b,0xa0,0xf8,0x79,0x1b,0x08,0xe9,0x38,0xce,0xeb,0x8b,0xf3,0xb2,0x2d,0xa2,0xf6,0xb3,0xd5,0xca,0xe7,0x5b,0xdd,0x67,0xe2,0x3b,0x4b,0x1f,0xf8,0x63,0x4e,0x9c,0x8d,0x04,0xbc,0x44,0x5e,0x56,0xf4,0x32,0x02,0xd9,0x77,0x59,0x0c,0x46,0xa2,0x96,0x80,0xad,0x83,0x9d,0xbe,0x4a,0xfc,0x1b,0xdd,0x8f,0x36,0xdc,0xde,0x6e,0x43,0xf3,0xed,0xac,0xe9,0x2d,0x0d,0xaf,0xe9,0x29,0x6b,0xdb,0xc2,0x17,0x10,0x6a,0xb4,0x1a,0x9c,0xe8,0x84,0x59,0xf6,0xd0,0xd2,0x7c,0xd5,0x9f,0x68,0xf1,0x17,0x13,0xd5,0xf6,0x95,0x58,0x11,0x7a,0xdb,0x66,0xc1,0xea,0xc5,0xe9,0xf4,0x0f,0x15,0x9a,0x96,0xe5,0xeb,0xf3,0x2c,0xc8,0x8c,0xe4,0x45,0x83,0x0d,0x1e,0x9c,0xe1,0xcf,0xf0,0x3a,0x69,0x58,0xe6,0x93,0xa3,0x6f,0x47,0x38,0x69,0x52,0xe1,0x49,0x83,0xbf,0x01,0x14,0x5c,0xef,0x5a,0x6c,0x66,0xfc,0x76,0xe6,0x8d,0x82,0xc5,0x88,0x8e,0xc0,0x28,0xcd,0x19,0xca,0xe8,0x2b,0xd0,0x60,0x0d,0x35,0xc9,0xb0,0x01,0x5f,0xaf,0xa4,0xd4,0x3f,0xfe,0x07,0x56,0xaf,0x35,0xf1,0x5e,0x16,0x00,0x00};
    webServer.writeP(page, sizeof(page));

  } else {
    if (tailComplete) {
      char name[8];
      char value[30];
      URLPARAM_RESULT rc;

      while (strlen(urlTail)) {
        rc = webServer.nextURLparam(&urlTail, name, 8, value, 30);
        if (rc == URLPARAM_EOS) {
          webServer.httpFail();
          return;
        }

        if (strcmp(name, "i") == 0) {
          setField(ip, value, '.', 10);
        } else if (strcmp(name, "d") == 0) {
          setField(dnsa, value, '.', 10);
        } else if (strcmp(name, "g") == 0) {
          setField(gateway, value, '.', 10);
        } else if (strcmp(name, "s") == 0) {
          setField(subnet, value, '.', 10);
        } else if (strcmp(name, "m") == 0) {
          setField(mac, value, ':', 16);
        } else if (strcmp(name, "p") == 0) {
          for (int i = 0; i < 9; i++) {
            pwd[i] = '\0';
          }
          strncpy(pwd, value, 8);
        }
      }

      byte checksum = 7;
      int a = 0;
      for (; a < 4; a++) {
        EEPROM.write(a, ip[a]);
        checksum ^= ip[a];
      }
      for (; a < 8; a++) {
        EEPROM.write(a, dnsa[a - 4]);
        checksum ^= dnsa[a - 4];
      }
      for (; a < 12; a++) {
        EEPROM.write(a, gateway[a - 8]);
        checksum ^= gateway[a - 8];
      }
      for (; a < 16; a++) {
        EEPROM.write(a, subnet[a - 12]);
        checksum ^= subnet[a - 12];
      }
      for (; a < 22; a++) {
        EEPROM.write(a, mac[a - 16]);
        checksum ^= mac[a - 16];
      }
      for (; a < 38; a+=2) {
        char c = pwd[(a - 22) / 2];
        byte l = c & 0xff;
        byte h = (c >> 8) & 0xff;
        EEPROM.write(a, l);
        EEPROM.write(a + 1, h);
        checksum ^= l;
        checksum ^= h;
      }
      EEPROM.write(a, checksum);

      webServer.httpSuccess();
      webServer.flushBuf();
      webServer.reset();

      delay(500);

      Ethernet.begin(mac, ip, dnsa, gateway, subnet);

    } else {
      webServer.httpFail();
    }
  }
}

void setField(byte *ip, char *value, char sep, int base) {
  int p = 0;
  int g = 0;
  for (int c = 0; c < 30; c++) {
    if (value[c] == sep || value[c] == '\0') {
      char n[4];
      int i = p;
      for (; i < c; i++) {
        n[i - p] = value[i];
      }
      n[i - p] = '\0';
      ip[g] = strtoul(n, NULL, base);
      if (value[c] == '\0') {
        break;
      }
      p = c + 1;
      g++;
    }
  }
}

void setup() {
  Iono.setup();
  pinMode(2, INPUT);
  pinMode(3, OUTPUT);
  boolean reset = true;
  for (int i = 0; i < 5; i++) {
    digitalWrite(3, i % 2);
    if (digitalRead(2) != i % 2) {
      reset = false;
    }
  }
  pinMode(3, INPUT);

  if (!reset) {
    byte checksum = 7;
    int a = 0;
    for (; a < 4; a++) {
      ip[a] = EEPROM.read(a);
      checksum ^= ip[a];
    }
    for (; a < 8; a++) {
      dnsa[a - 4] = EEPROM.read(a);
      checksum ^= dnsa[a - 4];
    }
    for (; a < 12; a++) {
      gateway[a - 8] = EEPROM.read(a);
      checksum ^= gateway[a - 8];
    }
    for (; a < 16; a++) {
      subnet[a - 12] = EEPROM.read(a);
      checksum ^= subnet[a - 12];
    }
    for (; a < 22; a++) {
      mac[a - 16] = EEPROM.read(a);
      checksum ^= mac[a - 16];
    }
    for (; a < 38; a+=2) {
      byte l = EEPROM.read(a);
      byte h = EEPROM.read(a + 1);
      pwd[(a - 22) / 2] = (h << 8) + l;
      checksum ^= l;
      checksum ^= h;
    }

    if (EEPROM.read(a) != checksum) {
      reset = true;
    }
  }

  if (reset) {
    ip[0] = 192;
    ip[1] = 168;
    ip[2] = 0;
    ip[3] = 100;

    dnsa[0] = 192;
    dnsa[1] = 168;
    dnsa[2] = 0;
    dnsa[3] = 1;

    gateway[0] = 192;
    gateway[1] = 168;
    gateway[2] = 0;
    gateway[3] = 1;

    subnet[0] = 255;
    subnet[1] = 255;
    subnet[2] = 255;
    subnet[3] = 0;

    mac[0] = 0xDE;
    mac[1] = 0xAD;
    mac[2] = 0xBE;
    mac[3] = 0xEF;
    mac[4] = 0xFE;
    mac[5] = 0xED;

    pwd[0] = '\0';
  }

  Ethernet.begin(mac, ip, dnsa, gateway, subnet);

  IonoWeb.begin(80);
  WebServer &webServer = IonoWeb.getWebServer();
  webServer.setDefaultCommand(&webPageCmd);
  webServer.addCommand("config", &configCmd);
}

void loop() {
  // Process incoming requests
  IonoWeb.processRequest();
  // Check all the inputs - needed for subscribe
  Iono.process();
}
