/*
  ModbusRtuApp.ino - A configurable Modbus RTU Slave for Iono Uno, Iono MKR and Iono RP

    Copyright (C) 2016-2022 Sfera Labs S.r.l. - All rights reserved.

    For information, see:
    https://www.sferalabs.cc/

  This code is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  See file LICENSE.txt for further informations on licensing terms.
*/

#include <IonoModbusRtuSlave.h>

#ifdef ARDUINO_ARCH_SAMD
#include <FlashAsEEPROM.h>
#else
#include <EEPROM.h>
#endif

#ifdef IONO_RP
#include "hardware/watchdog.h"
#endif

#define DELAY  25                           // the debounce delay in milliseconds
#define BOOT_CONSOLE_TIMEOUT_MILLIS 15000   // if 5 consecutive spaces are received within this time interval after boot, enter console mode

const PROGMEM char CONSOLE_MENU_HEADER[]  = {"=== Sfera Labs - Modbus RTU Slave configuration menu - v6.1 ==="};
const PROGMEM char CONSOLE_MENU_CURRENT_CONFIG[]  = {"Print current configuration"};
const PROGMEM char CONSOLE_MENU_SPEED[]  = {"Speed (baud)"};
const PROGMEM char CONSOLE_MENU_PARITY[]  = {"Parity"};
const PROGMEM char CONSOLE_MENU_MIRROR[]  = {"Input/Output rules"};
const PROGMEM char CONSOLE_MENU_ADDRESS[]  = {"Modbus device address"};
const PROGMEM char CONSOLE_MENU_SAVE[]  = {"Save configuration and restart"};
const PROGMEM char CONSOLE_MENU_TYPE[]  = {"Type a menu number (0, 1, 2, 3, 4, 5): "};
const PROGMEM char CONSOLE_TYPE_SPEED[]  = {"Type serial port speed (1: 1200, 2: 2400, 3: 4800, 4: 9600; 5: 19200; 6: 38400, 7: 57600, 8: 115200): "};
const PROGMEM char CONSOLE_TYPE_PARITY[]  = {"Type serial port parity (1: Even, 2: Odd, 3: None): "};
const PROGMEM char CONSOLE_TYPE_ADDRESS[]  = {"Type Modbus device address (1-247): "};
#ifdef IONO_ARDUINO
#define IORULES_LEN 6
const PROGMEM char CONSOLE_TYPE_MIRROR[]  = {"Type Input/Output rules (xxxxxx, F: follow, I: invert, H: flip on L>H transition, L: flip on H>L transition, T: flip on any transition, -: no rule): "};
#else
#define IORULES_LEN 4
const PROGMEM char CONSOLE_TYPE_MIRROR[]  = {"Type Input/Output rules (xxxx, F: follow, I: invert, H: flip on L>H transition, L: flip on H>L transition, T: flip on any transition, -: no rule): "};
#endif
const PROGMEM char CONSOLE_TYPE_SAVE[]  = {"Confirm? (Y/N): "};
const PROGMEM char CONSOLE_CURRENT_CONFIG[]  = {"Current configuration:"};
const PROGMEM char CONSOLE_NEW_CONFIG[]  = {"New configuration:"};
const PROGMEM char CONSOLE_ERROR[]  = {"Error"};
const PROGMEM char CONSOLE_SAVED[]  = {"Saved"};

#ifndef SERIAL_PORT_MONITOR
#define SERIAL_PORT_MONITOR SERIAL_PORT_HARDWARE
#endif

const long SPEED_VALUE[] = {0, 1200, 2400, 4800, 9600, 19200, 38400, 57600, 115200};

byte consoleState = 0; // 0: wait for menu selection number; 1: speed; 2: parity; 3: address; 4: i/o rules; 5 confirm to save
byte opMode = 0; // 0: boot sequence, wait to enter console mode; 1: console mode; 2: Modbus slave mode
boolean validConfiguration;
unsigned long bootTimeMillis;
short spacesCounter = 0;
char consoleInputBuffer[7];

byte speedCurrent;
byte parityCurrent;
byte addressCurrent;
char rulesCurrent[7];
byte speedNew;
byte parityNew;
byte addressNew;
char rulesNew[IORULES_LEN + 1];

Stream *consolePort = NULL;

bool configReset = false;

void setup() {
  bootTimeMillis = millis();

  Iono.setup();

  SERIAL_PORT_HARDWARE.begin(9600);
  if ((int) &SERIAL_PORT_HARDWARE != (int) &SERIAL_PORT_MONITOR) {
    SERIAL_PORT_MONITOR.begin(9600);
  }

  // retrieve settings from EEPROM
  validConfiguration = getEEPROMConfig();
}

void loop() {
  if (opMode == 2) {
    IonoModbusRtuSlave.process();

    if (configReset) {
      delay(1000);
      softReset();
    }

  } else {
    if (consolePort == NULL) {
      if (SERIAL_PORT_HARDWARE.available()) {
        consolePort = &SERIAL_PORT_HARDWARE;
      } else if (SERIAL_PORT_MONITOR.available()) {
        consolePort = &SERIAL_PORT_MONITOR;
      }
    } else if (consolePort->available()) {
      int b = consolePort->read();
      if (opMode == 0) {
        if (b == ' ') {
          if (spacesCounter >= 4) {
            delay(50);
            Iono.serialTxEn(true);
            printConsoleMenu();
            consolePort->flush();
            Iono.serialTxEn(false);
            opMode = 1;
          } else {
            spacesCounter++;
          }
        } else if (validConfiguration) {
          startModbus();
        } else {
          consolePort = NULL;
        }
      } else { // opMode == 1
        serialConsole(b);
      }
    }

    if (validConfiguration && opMode == 0 && bootTimeMillis + BOOT_CONSOLE_TIMEOUT_MILLIS < millis()) {
      startModbus();
    }
  }
}

void startModbus() {
  SERIAL_PORT_HARDWARE.end();
  SERIAL_PORT_MONITOR.end();

  switch (parityCurrent) {
    case 2:
      IonoModbusRtuSlave.begin(addressCurrent, SPEED_VALUE[speedCurrent], SERIAL_8O1, DELAY);
      break;
    case 3:
      IonoModbusRtuSlave.begin(addressCurrent, SPEED_VALUE[speedCurrent], SERIAL_8N2, DELAY);
      break;
    default:
      IonoModbusRtuSlave.begin(addressCurrent, SPEED_VALUE[speedCurrent], SERIAL_8E1, DELAY);
  }

  IonoModbusRtuSlave.setCustomHandler(&modbusConfigHandler);

  if (rulesCurrent[0] != 0) {
    setLink(rulesCurrent[0], DI1, DO1);
    setLink(rulesCurrent[1], DI2, DO2);
    setLink(rulesCurrent[2], DI3, DO3);
    setLink(rulesCurrent[3], DI4, DO4);
#ifdef IONO_ARDUINO
    setLink(rulesCurrent[4], DI5, DO5);
    setLink(rulesCurrent[5], DI6, DO6);
#endif
  }

  opMode = 2;
}

byte modbusConfigHandler(byte unitAddr, byte function, word regAddr, word qty, byte *data) {
  switch (function) {
    case MB_FC_WRITE_MULTIPLE_REGISTERS:
    case MB_FC_WRITE_SINGLE_REGISTER:
      if (regAddr == 3000) {
        word val = ModbusRtuSlave.getDataRegister(function, data, 0);
        if (qty != 1 || val != 0xABCD) {
          return MB_EX_ILLEGAL_DATA_VALUE;
        }
        if (saveConfig()) {
          configReset = true;
          return MB_RESP_OK;
        } else {
          return MB_EX_SERVER_DEVICE_FAILURE;
        }
      }
      if (modbusCheckAddrRange(regAddr, qty, 3001, 3004 + IORULES_LEN - 1)) {
        for (word i = regAddr; i < regAddr + qty; i++) {
          word val = ModbusRtuSlave.getDataRegister(function, data, i - regAddr);
          if (i == 3001) {
            if (val < 1 || val > 247) {
              return MB_EX_ILLEGAL_DATA_VALUE;
            }
            addressNew = val;
          } else if (i == 3002) {
            if (val < 1 || val > 8) {
              return MB_EX_ILLEGAL_DATA_VALUE;
            }
            speedNew = val;
          } else if (i == 3003) {
            if (val < 1 || val > 3) {
              return MB_EX_ILLEGAL_DATA_VALUE;
            }
            parityNew = val;
          } else if (i >= 3004) {
            if (val != '-' && val != 'F' && val != 'I' && val != 'H' && val != 'L' && val != 'T') {
              return MB_EX_ILLEGAL_DATA_VALUE;
            }
            rulesNew[i - 3004] = val;
          }
        }
        return MB_RESP_OK;
      }
      break;

    case MB_FC_READ_HOLDING_REGISTERS:
      if (modbusCheckAddrRange(regAddr, qty, 3001, 3004 + IORULES_LEN - 1)) {
        for (word i = regAddr; i < regAddr + qty; i++) {
          if (i == 3001) {
            ModbusRtuSlave.responseAddRegister(((addressNew == 0) ? addressCurrent : addressNew) & 0xff);
          } else if (i == 3002) {
            ModbusRtuSlave.responseAddRegister(((speedNew == 0) ? speedCurrent : speedNew) & 0xff);
          } else if (i == 3003) {
            ModbusRtuSlave.responseAddRegister(((parityNew == 0) ? parityCurrent : parityNew) & 0xff);
          } else if (i >= 3004) {
            ModbusRtuSlave.responseAddRegister(((rulesNew[i - 3004] == 0) ? rulesCurrent[i - 3004] : rulesNew[i - 3004]) & 0xff);
          }
        }
        return MB_RESP_OK;
      }
      break;

    default:
      break;
  }

  return MB_RESP_PASS;
}

bool modbusCheckAddrRange(word regAddr, word qty, word min, word max) {
  return regAddr >= min && regAddr <= max && regAddr + qty <= max + 1;
}

void setLink(char rule, uint8_t dix, uint8_t dox) {
  switch (rule) {
    case 'F':
      Iono.linkDiDo(dix, dox, LINK_FOLLOW, DELAY);
      break;
    case 'I':
      Iono.linkDiDo(dix, dox, LINK_INVERT, DELAY);
      break;
    case 'T':
      Iono.linkDiDo(dix, dox, LINK_FLIP_T, DELAY);
      break;
    case 'H':
      Iono.linkDiDo(dix, dox, LINK_FLIP_H, DELAY);
      break;
    case 'L':
      Iono.linkDiDo(dix, dox, LINK_FLIP_L, DELAY);
      break;
    default:
      break;
  }
}

void serialConsole(int b) {
  Iono.serialTxEn(true);
  delayMicroseconds(4000); // this is to let the console also work over the RS485 interface
  switch (consoleState) {
    case 0: // waiting for menu selection number
      switch (b) {
        case '0':
          consolePort->println((char)b);
          consolePort->println();
          printlnProgMemString(CONSOLE_CURRENT_CONFIG);
          printConfiguration(speedCurrent, parityCurrent, addressCurrent, rulesCurrent);
          printConsoleMenu();
          break;
        case '1':
          consoleState = 1;
          consoleInputBuffer[0] = 0;
          consolePort->println((char)b);
          consolePort->println();
          printProgMemString(CONSOLE_TYPE_SPEED);
          break;
        case '2':
          consoleState = 2;
          consoleInputBuffer[0] = 0;
          consolePort->println((char)b);
          consolePort->println();
          printProgMemString(CONSOLE_TYPE_PARITY);
          break;
        case '3':
          consoleState = 3;
          consoleInputBuffer[0] = 0;
          consolePort->println((char)b);
          consolePort->println();
          printProgMemString(CONSOLE_TYPE_ADDRESS);
          break;
        case '4':
          consoleState = 4;
          consoleInputBuffer[0] = 0;
          consolePort->println((char)b);
          consolePort->println();
          printProgMemString(CONSOLE_TYPE_MIRROR);
          break;
        case '5':
          consoleState = 5;
          consolePort->println((char)b);
          consolePort->println();
          printlnProgMemString(CONSOLE_NEW_CONFIG);
          printConfiguration((speedNew == 0) ? speedCurrent : speedNew, (parityNew == 0) ? parityCurrent : parityNew, (addressNew == 0) ? addressCurrent : addressNew, (rulesNew[0] == 0) ? rulesCurrent : rulesNew);
          printProgMemString(CONSOLE_TYPE_SAVE);
          break;
      }
      break;
    case 1: // speed
      if (numberEdit(consoleInputBuffer, &speedNew, b, 1, 1, 8)) {
        consoleState = 0;
        consolePort->println();
        printConsoleMenu();
      }
      break;
    case 2: // parity
      if (numberEdit(consoleInputBuffer, &parityNew, b, 1, 1, 3)) {
        consoleState = 0;
        consolePort->println();
        printConsoleMenu();
      }
      break;
    case 3: // address
      if (numberEdit(consoleInputBuffer, &addressNew, b, 3, 1, 247)) {
        consoleState = 0;
        consolePort->println();
        printConsoleMenu();
      }
      break;
    case 4: // rules
      if (rulesEdit(consoleInputBuffer, rulesNew, b, IORULES_LEN)) {
        consoleState = 0;
        consolePort->println();
        printConsoleMenu();
      }
      break;
    case 5: // confirm to save
      switch (b) {
        case 'Y':
        case 'y':
          consoleState = 0;
          consolePort->println('Y');
          if (saveConfig()) {
            printlnProgMemString(CONSOLE_SAVED);
            delay(1000);
            softReset();
          } else {
            printlnProgMemString(CONSOLE_ERROR);
          }
          printConsoleMenu();
          break;
        case 'N':
        case 'n':
          consoleState = 0;
          consolePort->println('N');
          consolePort->println();
          printConsoleMenu();
          break;
      }
      break;
    default:
      break;
  }
  consolePort->flush();
  Iono.serialTxEn(false);
}

boolean saveConfig() {
  if (speedNew == 0) {
    speedNew = speedCurrent;
  }
  if (parityNew == 0) {
    parityNew = parityCurrent;
  }
  if (addressNew == 0) {
    addressNew = addressCurrent;
  }
  for (int i = 0; i < IORULES_LEN; i++) {
    if (rulesNew[i] == 0) {
      rulesNew[i] = rulesCurrent[i];
    }
  }
  return writeEepromConfig(speedNew, parityNew, addressNew, rulesNew);
}

boolean writeEepromConfig(byte speed, byte parity, byte address, char *rules) {
  byte checksum = 7;
  if (speed != 0 && parity != 0 && address != 0) {
    EEPROM.write(0, speed);
    checksum ^= speed;
    EEPROM.write(1, parity);
    checksum ^= parity;
    EEPROM.write(2, address);
    checksum ^= address;
    for (int a = 0; a < IORULES_LEN; a++) {
      EEPROM.write(a + 3, rules[a]);
      checksum ^= rules[a];
    }
    EEPROM.write(9, checksum);
#if defined(ARDUINO_ARCH_SAMD) || defined(IONO_RP)
    EEPROM.commit();
#endif
    return true;
  } else {
    return false;
  }
}

boolean readEepromConfig(byte *speedp, byte *parityp, byte *addressp, char *rulesp) {
  byte checksum = 7;

#ifdef ARDUINO_ARCH_SAMD
  if (!EEPROM.isValid()) {
    return false;
  }
#endif
  *speedp = EEPROM.read(0);
  checksum ^= *speedp;
  *parityp = EEPROM.read(1);
  checksum ^= *parityp;
  *addressp = EEPROM.read(2);
  checksum ^= *addressp;
  for (int a = 0; a < IORULES_LEN; a++) {
    rulesp[a] = EEPROM.read(a + 3);
    checksum ^= rulesp[a];
  }
  return (EEPROM.read(9) == checksum);
}

boolean getEEPROMConfig() {
#ifdef IONO_RP
  EEPROM.begin(256);
#endif
  if (!readEepromConfig(&speedCurrent, &parityCurrent, &addressCurrent, rulesCurrent)) {
    speedCurrent = 0;
    parityCurrent = 0;
    addressCurrent = 0;
    for (int i = 0; i < IORULES_LEN; i++) {
      rulesCurrent[i] = '-';
    }
  }
  return (speedCurrent != 0 && parityCurrent != 0 && addressCurrent != 0);
}

void softReset() {
#ifdef IONO_RP
  watchdog_enable(10, 1);
  while (1);
#elif defined(ARDUINO_ARCH_SAMD)
  NVIC_SystemReset();
#else
  asm volatile ("  jmp 0");
#endif
}

void printlnProgMemString(const char* s) {
  printProgMemString(s);
  consolePort->println();
}

void printProgMemString(const char* s) {
  int len = strlen_P(s);
  for (int k = 0; k < len; k++) {
    consolePort->print((char)pgm_read_byte_near(s + k));
  }
}

void printConsoleMenu() {
  consolePort->println();
  printlnProgMemString(CONSOLE_MENU_HEADER);
  for (int i = 0; i <= 5; i++) {
    consolePort->print(i);
    consolePort->print(". ");
    switch (i) {
      case 0:
        printlnProgMemString(CONSOLE_MENU_CURRENT_CONFIG);
        break;
      case 1:
        printlnProgMemString(CONSOLE_MENU_SPEED);
        break;
      case 2:
        printlnProgMemString(CONSOLE_MENU_PARITY);
        break;
      case 3:
        printlnProgMemString(CONSOLE_MENU_ADDRESS);
        break;
      case 4:
        printlnProgMemString(CONSOLE_MENU_MIRROR);
        break;
      case 5:
        printlnProgMemString(CONSOLE_MENU_SAVE);
        break;
    }
  }
  printProgMemString(CONSOLE_MENU_TYPE);
}

void printConfiguration(byte speed, byte parity, byte address, char *rules) {
  char s[] = ": ";
  printProgMemString(CONSOLE_MENU_SPEED);
  consolePort->print(s);
  if (speed == 0) {
    consolePort->println();
  } else {
    consolePort->println(SPEED_VALUE[speed]);
  }
  printProgMemString(CONSOLE_MENU_PARITY);
  consolePort->print(s);
  switch (parity) {
    case 1:
      consolePort->print("Even");
      break;
    case 2:
      consolePort->print("Odd");
      break;
    case 3:
      consolePort->print("None");
      break;
  }
  consolePort->println();
  printProgMemString(CONSOLE_MENU_ADDRESS);
  consolePort->print(s);
  if (address == 0) {
    consolePort->println();
  } else {
    consolePort->println(address);
  }
  printProgMemString(CONSOLE_MENU_MIRROR);
  consolePort->print(s);
  if (rules[0] == 0) {
    consolePort->println();
  } else {
    consolePort->println(rules);
  }
}

boolean rulesEdit(char *buffer, char *value, int c, int size) {
  int i;
  switch (c) {
    case 8: case 127: // backspace
      i = strlen(buffer);
      if (i > 0) {
        consolePort->print('\b');
        consolePort->print(' ');
        consolePort->print('\b');
        buffer[i - 1] = 0;
      }
      break;
    case 10: // newline
    case 13: // enter
      if (strlen(buffer) == size) {
        strcpy(value, buffer);
        return true;
      } else {
        return false;
      }
      break;
    default:
      if (strlen(buffer) < size) {
        if (c >= 'a') {
          c -= 32;
        }
        if (c == 'F' || c == 'I' || c == 'H' || c == 'L' || c == 'T' || c == '-') {
          consolePort->print(' ');
          consolePort->print('\b');
          consolePort->print((char)c);
          strcat_c(buffer, c);
        }
      }
      break;
  }
  return false;
}

boolean numberEdit(char *buffer, byte *value, int c, int length, long min, long max) {
  int i;
  long v;
  switch (c) {
    case 8: case 127: // backspace
      i = strlen(buffer);
      if (i > 0) {
        consolePort->print('\b');
        consolePort->print(' ');
        consolePort->print('\b');
        buffer[i - 1] = 0;
      }
      break;
    case 10: // newline
    case 13: // enter
      v = strtol(buffer, NULL, 10);
      if (v >= min && v <= max) {
        *value = (byte)v;
        return true;
      } else {
        return false;
      }
      break;
    default:
      if (strlen(buffer) < length) {
        if (c >= '0' && c <= '9') {
          consolePort->print(' ');
          consolePort->print('\b');
          consolePort->print((char)c);
          strcat_c(buffer, c);
        }
      }
      break;
  }
  return false;
}

void strcat_c(char *s, char c) {
  for (; *s; s++);
  *s++ = c;
  *s++ = 0;
}
