#include "ufm02.h"
#include <string.h>

static const char* ufm02_ErrorNames[UFM02_AMOUNT_ERROR_FLAGS] =
{
    "", "", "", "", "", "",
    "Measurement not ok, e.g. low or no signal",
    "", "", "", "", "", "", "",
    "Bubbles detected",
    "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""
};

UFM02::UFM02() {}

UFM02::~UFM02() {}

void UFM02::begin(SPIClass* spi, uint8_t chipSelect, const SPISettings settings)
{
    spiConfig                   = { 0 };
    spiConfig.spi               = spi;
    spiConfig.csPin             = chipSelect;
    spiConfig.settings          = settings;
    spiConfig.useSpiSettings    = true;

    io.transfer                 = ScioSense_Spi_Transfer;
    io.write                    = ScioSense_Spi_Write_Data;
    io.wait                     = ScioSense_Arduino_Spi_Wait;
    io.config                   = &spiConfig;
    io.protocol                 = UFM02_PROTOCOL_SPI;

    volumeIntRaw                = 0;
    volumeFracRaw               = 0;
    flowRaw                     = 0;
    tempRaw                     = 0;
    dateCode                    = 0;
    materialId                  = 0;
    errors                      = 0;

    pinMode(spiConfig.csPin, OUTPUT);
    digitalWrite(spiConfig.csPin, LOW);
    delay(1);
    digitalWrite(spiConfig.csPin, HIGH);
}


void UFM02::begin(Stream* serial)
{
    serialConfig                = { 0 };
    serialConfig.serial         = serial;

    io.transfer                 = ScioSense_Arduino_Serial_Transfer;
    io.write                    = ScioSense_Arduino_Serial_Write_Data;
    io.wait                     = ScioSense_Arduino_Serial_Wait;
    io.clear                    = ScioSense_Arduino_Serial_Clear;
    io.protocol                 = UFM02_PROTOCOL_UART;
    io.config                   = &serialConfig;
    io.serialMeasurementMode    = UFM02_SERIAL_MODE_PASSIVE;

    volumeIntRaw                = 0;
    volumeFracRaw               = 0;
    flowRaw                     = 0;
    tempRaw                     = 0;
    dateCode                    = 0;
    materialId                  = 0;
    errors                      = 0;
}

void UFM02::begin(TwoWire* wire, const uint8_t address)
{
    i2cConfig                   = { 0 };
    i2cConfig.wire              = wire;
    i2cConfig.address           = address;
    
    io.transfer                 = ScioSense_Arduino_I2c_Transfer;
    io.write                    = ScioSense_Arduino_I2c_Write_Data;
    io.wait                     = ScioSense_Arduino_I2c_Wait;
    io.protocol                 = UFM02_PROTOCOL_I2C;
    io.config                   = &i2cConfig;

    volumeIntRaw                = 0;
    volumeFracRaw               = 0;
    flowRaw                     = 0;
    tempRaw                     = 0;
    dateCode                    = 0;
    materialId                  = 0;
    errors                      = 0;
}

Result UFM02::reset()
{
    return Ufm02_Reset(this);
}

bool UFM02::init()
{
    Ufm02_Init(this);

    return isConnected();
}

bool UFM02::isConnected()
{
    return (bool)Ufm02_isConnected(this);
}

Result UFM02::update()
{
    return Ufm02_Update(this);
}

uint32_t UFM02::getVolumeIntegerPartRaw()
{
    return Ufm02_Parse_Flow_Volume_Int(this);
}

uint32_t UFM02::getVolumeFractionalPartRaw()
{
    return Ufm02_Parse_Flow_Volume_Frac(this);
}

uint32_t UFM02::getInstantFlowRaw()
{
    return Ufm02_Parse_Flow_Rate_Raw(this);
}

uint32_t UFM02::getInstantFilteredFlowRaw()
{
    return Ufm02_Parse_Flow_Rate_Filtered_Raw(this);
}

uint32_t UFM02::getTemperatureRaw()
{
    return Ufm02_Parse_Temperature_Raw(this);
}

uint32_t UFM02::getErrorFlags()
{
    return Ufm02_Parse_Error_Flags(this);
}

float UFM02::getVolumeM3()
{
    return Ufm02_Parse_Flow_Volume_M3(this);
}

float UFM02::getInstantFlowLPerHr()
{
    return Ufm02_Parse_Flow_Rate_L_Per_Hr(this);
}

float UFM02::getInstantFilteredFlowLPerHr()
{
    return Ufm02_Parse_Flow_Rate_Filtered_L_Per_Hr(this);
}

float UFM02::getTemperatureDegC()
{
    return Ufm02_Parse_Temperature_Deg_C(this);
}

uint32_t UFM02::getMaterialId()
{
    return Ufm02_ParseSensorProductGroupId(this);
}

uint32_t UFM02::getThread()
{
    return Ufm02_ParseSensorThreadId(this);
}

uint32_t UFM02::getFamilyNumber()
{
    return Ufm02_ParseSensorFamilyId(this);
}

uint32_t UFM02::getApplicationFamily()
{
    return Ufm02_ParseSensorApplicationId(this);
}

Result UFM02::setSerialPassiveMode()
{
    Result result = RESULT_INVALID;
    if( this->io.protocol == UFM02_PROTOCOL_UART )
    {
        result = (bool)Ufm02_SetPassiveMeasurement(this);
    }
    return result;
}

Result UFM02::setSerialActiveMode()
{
    Result result = RESULT_INVALID;
    if( this->io.protocol == UFM02_PROTOCOL_UART )
    {
        result = (bool)Ufm02_SetActiveMeasurement(this);
    }
    return result;
}

bool UFM02::hasErrorFlag(uint8_t errorFlag)
{
    return (bool)Ufm02_HasErrorFlag(this, errorFlag);
}

uint8_t UFM02::hasErrors()
{
    uint8_t errorsFound = 0;

    uint8_t errorsToCheck[] = {UFM02_ERROR_FLAG_MEAS_NOT_OK, UFM02_ERROR_FLAG_BUBBLES};
    uint8_t amountErrorsToCheck = (uint8_t)( sizeof(errorsToCheck) / sizeof(errorsToCheck[0]) );

    for( uint8_t error = 0; error < amountErrorsToCheck; error++ )
    {
        errorsFound += Ufm02_HasErrorFlag(this, errorsToCheck[error]);
    }

    return errorsFound;
}

const char* UFM02::errorToString(uint8_t errorFlag)
{
    return ufm02_ErrorNames[errorFlag];
}
