#ifndef SCIOSENSE_UFM02_DEFINES_H
#define SCIOSENSE_UFM02_DEFINES_H

#include <stdint.h>
#include <stddef.h>

/*******************  Array sizes for the UFM02 structs  **********************/
#define UFM02_AMOUNT_OUTPUT_REGISTERS           (40)                                            // Amount of output registers of the UFM02 outputs
#define UFM02_AMOUNT_ERROR_FLAGS                (32)                                            // Amount of error flags of the UFM02
#define UFM02_REGISTER_LENGTH_BYTES             (4)                                             // Size of the registers of the UFM02 in bytes

/************************  UFM02 timing definitions  **************************/
#define UFM02_BOOTUP_TIME_MS                    (100)                                           // Time needed for the sensor to boot up

/****************  Conversion constants for floating points  ******************/
#define UFM02_FD8_DIVIDER_FLOAT                 (256.0)                                         // Floating point value of 2^16 
#define UFM02_FD16_DIVIDER_FLOAT                (65536.0)                                       // Floating point value of 2^16 
#define UFM02_FD32_DIVIDER_FLOAT                (4294967296.0)                                  // Floating point value of 2^32 

/************************  Remote commands for UFM02  *************************/
#define UFM02_REMOCE_COMMAND_SYS_INIT           (0x9A)                                          // Resets main part of digital core without register part and triggers bootloading process
#define UFM02_REMOCE_COMMAND_RD_RAM             (0x7A)                                          // Read from RAM or register area
#define UFM02_REMOCE_COMMAND_RD_FWD             (0x7B)                                          // Read from firmware data area (NVRAM)

/***********************  Byte definition for UFM02  **************************/
#define UFM02_VOLUME_INT_ADDRESS                (0x00)                                          // Address of the register of the integer part of the accumulated flow volume data
#define UFM02_VOLUME_FRAC_ADDRESS               (0x01)                                          // Address of the register of the fractional part of the accumulated flow volume data
#define UFM02_UNFILTERED_FLOW_LPH_ADDRESS       (0x02)                                          // Address of the register of the unfiltered instant flow data
#define UFM02_FILTERED_FLOW_LPH_ADDRESS         (0x03)                                          // Address of the register of the filtered instant flow data
#define UFM02_TEMPERATURE_ADDRESS               (0x04)                                          // Address of the register of the temperature data
#define UFM02_DEVICE_ID_ADDRESS                 (0x09)                                          // Address of the register of the serial number and datecode
#define UFM02_PART_ID_ADDRESS                   (0x0A)                                          // Address of the register of the part number
#define UFM02_ERROR_FLAGS_ADDRESS               (0x27)                                          // Address of the register of the error flags data
#define UFM02_INTERRUPT_CLEAR_ADDRESS           (0xDD)                                          // Address of the register of the interrupt clear instructions

/***********************  Byte index for UFM02 data  *************************/
#define UFM02_VOLUME_INT_INDEX                  (0)                                             // Index of the integer part of the accumulated flow volume data on the output data array
#define UFM02_VOLUME_FRAC_INDEX                 (1)                                             // Index of the fractional part of the accumulated flow volume data on the output data array
#define UFM02_UNFILTERED_FLOW_LPH_INDEX         (2)                                             // Index of the unfiltered instant flow data on the output data array
#define UFM02_FILTERED_FLOW_LPH_INDEX           (3)                                             // Index of the filtered instant flow data on the output data array
#define UFM02_TEMPERATURE_INDEX                 (4)                                             // Index of the temperature data on the output data array
#define UFM02_DEVICE_ID_INDEX                   (9)                                             // Index of the serial number and datecode on the output data array
#define UFM02_PART_ID_INDEX                     (10)                                            // Index of the part number on the output data array
#define UFM02_ERROR_FLAGS_INDEX                 (39)                                            // Index of the error flags data on the output data array

/****************  Bit definition for Error flags register  *******************/
#define UFM02_ERROR_FLAG_MEAS_NOT_OK_Pos        (6U)                                            // Position of the Measurement not OK flag on the Error flags register
#define UFM02_ERROR_FLAG_MEAS_NOT_OK_Msk        (0x1UL << UFM02_ERROR_FLAG_MEAS_NOT_OK_Pos)     // 0x00000040
#define UFM02_ERROR_FLAG_MEAS_NOT_OK            (UFM02_ERROR_FLAG_MEAS_NOT_OK_Pos)              // Error flag indicating Measurement not OK
#define UFM02_ERROR_FLAG_BUBBLES_Pos            (14U)                                           // Position of the Bubbles detected flag on the Error flags register
#define UFM02_ERROR_FLAG_BUBBLES_Msk            (0x1UL << UFM02_ERROR_FLAG_BUBBLES_Pos)         // 0x00004000
#define UFM02_ERROR_FLAG_BUBBLES                (UFM02_ERROR_FLAG_BUBBLES_Pos)                  // Error flag indicating Bubbles detected

/**************  Bit definition for Interrupt clear register  *****************/
#define UFM02_INTERRUPT_BOOT_Pos                (0U)                                            // Bit to clear interrupts related to boot completed
#define UFM02_INTERRUPT_BOOT_Msk                (0x1UL << UFM02_INTERRUPT_BOOT_Pos)             // 0x00000001
#define UFM02_INTERRUPT_BOOT_CLEAR              (UFM02_INTERRUPT_BOOT_Msk)                      // Clear boot finished interrupt flag 
#define UFM02_INTERRUPT_ERROR_Pos               (1U)                                            // Bit to clear interrupts related to error detected
#define UFM02_INTERRUPT_ERROR_Msk               (0x1UL << UFM02_INTERRUPT_ERROR_Pos)            // 0x00000002
#define UFM02_INTERRUPT_ERROR_CLEAR             (UFM02_INTERRUPT_ERROR_Msk)                     // Clear error detected flag
#define UFM02_INTERRUPT_MEASUREMENT_Pos         (2U)                                            // Bit to clear interrupts related to measurement completed
#define UFM02_INTERRUPT_MEASUREMENT_Msk         (0x1UL << UFM02_INTERRUPT_MEASUREMENT_Pos)      // 0x00000004
#define UFM02_INTERRUPT_MEASUREMENT_CLEAR       (UFM02_INTERRUPT_MEASUREMENT_Msk)               // Clear measurement ready flag

/**************************  Device ID decoding  ******************************/
#define UFM02_DEVICE_ID_SERIAL_NUMBER_DIVIDER   (1)                                             // Value to divide the DEVICE ID register to have the serial number at the LSB
#define UFM02_DEVICE_ID_SERIAL_NUMBER_MODULUS   (10000)                                         // Value to use as modulus to obtain the serial number when the value is at the LSB
#define UFM02_DEVICE_ID_DATE_DAY_DIVIDER        (10000)                                         // Value to divide the DEVICE ID register to have the production day at the LSB
#define UFM02_DEVICE_ID_DATE_DAY_MODULUS        (100)                                           // Value to use as modulus to obtain the production day when the value is at the LSB
#define UFM02_DEVICE_ID_DATE_MONTH_DIVIDER      (1000000)                                       // Value to divide the DEVICE ID register to have the production month at the LSB
#define UFM02_DEVICE_ID_DATE_MONTH_MODULUS      (100)                                           // Value to use as modulus to obtain the production month when the value is at the LSB
#define UFM02_DEVICE_ID_DATE_YEAR_DIVIDER       (100000000)                                     // Value to divide the DEVICE ID register to have the production year at the LSB
#define UFM02_DEVICE_ID_DATE_YEAR_MODULUS       (100)                                           // Value to use as modulus to obtain the production year when the value is at the LSB

/***************************  Part ID decoding  *******************************/
#define UFM02_PRODUCT_GROUP_ID                  (501)                                           // Code of the Part ID shared by all devices on the product family

#define UFM02_PART_ID_THREAD_DIVIDER            (1)                                             // Value to divide the PART ID register to have the thread size at the LSB
#define UFM02_PART_ID_THREAD_MODULUS            (100)                                           // Value to use as modulus to obtain the thread size when the value is at the LSB
#define UFM02_PART_ID_FAMILY_DIVIDER            (100)                                           // Value to divide the PART ID register to have the family number at the LSB
#define UFM02_PART_ID_FAMILY_MODULUS            (10)                                            // Value to use as modulus to obtain the family number when the value is at the LSB
#define UFM02_PART_ID_APPLICATION_DIVIDER       (1000)                                          // Value to divide the PART ID register to have the Application family at the LSB
#define UFM02_PART_ID_APPLICATION_MODULUS       (10)                                            // Value to use as modulus to obtain the Application family when the value is at the LSB
#define UFM02_PART_ID_PRODUCT_LINE_DIVIDER      (10000)                                         // Value to divide the PART ID register to have the product line at the LSB
#define UFM02_PART_ID_PRODUCT_LINE_MODULUS      (1000)                                          // Value to use as modulus to obtain the product line when the value is at the LSB

#define UFM02_THREAD_ID_03_IN                   (03)                                            // Register content on part ID indicating a transducer pipe of 3/8 inch diameter
#define UFM02_THREAD_ID_05_IN                   (05)                                            // Register content on part ID indicating a transducer pipe of 1/2 inch diameter
#define UFM02_THREAD_ID_07_IN                   (07)                                            // Register content on part ID indicating a transducer pipe of 3/4 inch diameter
#define UFM02_THREAD_ID_10_IN                   (10)                                            // Register content on part ID indicating a transducer pipe of 1 inch diameter
#define UFM02_THREAD_ID_15_IN                   (15)                                            // Register content on part ID indicating a transducer pipe of 1.5 inch diameter

#define UFM02_NOMINAL_FLOW_UFM02_03             (800.0)                                         // Nominal flow in liters for the UFM-02-03 (with 3/8 inch pipe)
#define UFM02_NOMINAL_FLOW_UFM02_05             (2000.0)                                        // Nominal flow in liters for the UFM-02-05 (with 1/2 inch pipe)
#define UFM02_NOMINAL_FLOW_UFM02_07             (8000.0)                                        // Nominal flow in liters for the UFM-02-07 (with 3/4 inch pipe)
#define UFM02_NOMINAL_FLOW_UFM02_10             (8000.0)                                        // Nominal flow in liters for the UFM-02-10 (with 1 inch pipe)
#define UFM02_NOMINAL_FLOW_UFM02_15             (18000.0)                                       // Nominal flow in liters for the UFM-02-15 (with 1.5 inch pipe)

/***************  SPI and I2C command protocol structure values  **************/
#define UFM02_COMMAND_OPCODE_LENGTH             (1)                                             // Length of the operational codes to be written through SPI or I2C
#define UFM02_COMMAND_READ_REGISTER_LENGTH      (2)                                             // Length of the command to read a specified register through SPI or I2C
#define UFM02_COMMAND_WRITE_REGISTER_LENGTH     (6)                                             // Length of the command to write a specified register through SPI or I2C
#define UFM02_COMMAND_OPCODE_INDEX              (0)                                             // Index of the opcode for communications through SPI or I2C
#define UFM02_COMMAND_ADDRESS_INDEX             (1)                                             // Index of the address of the register to read/write  for communications through SPI or I2C
#define UFM02_COMMAND_WRITE_MSB_BYTE_INDEX      (2)                                             // Index of the MSB of the register to write for communications through SPI or I2C
#define UFM02_COMMAND_WRITE_SECOND_BYTE_INDEX   (3)                                             // Index of the second MSB of the register to write for communications through SPI or I2C
#define UFM02_COMMAND_WRITE_THIRD_BYTE_INDEX    (4)                                             // Index of the third MSB of the register to write for communications through SPI or I2C
#define UFM02_COMMAND_WRITE_LSB_BYTE_INDEX      (5)                                             // Index of the LSB of the register to write for communications through SPI or I2C
#define UFM02_COMMAND_READ_MSB_BYTE_INDEX       (0)                                             // Index of the MSB of the register to read for communications through SPI or I2C
#define UFM02_COMMAND_READ_SECOND_BYTE_INDEX    (1)                                             // Index of the second MSB of the register to read for communications through SPI or I2C
#define UFM02_COMMAND_READ_THIRD_BYTE_INDEX     (2)                                             // Index of the third MSB of the register to read for communications through SPI or I2C
#define UFM02_COMMAND_READ_LSB_BYTE_INDEX       (3)                                             // Index of the LSB of the register to read for communications through SPI or I2C

/*******************  Commands definition for SPI and I2C  ********************/
#define UFM02_COMMAND_SYS_INIT                  (0x9AU)                                         // Resets main part of digital core without register part and triggers bootloading process                   */
#define UFM02_COMMAND_RAA_RD                    (0x7AU)                                         // Read from RAM or register area                                                                             */
#define UFM02_COMMAND_RAA_WR                    (0x5AU)                                         // Read from RAM or register area                                                                             */
#define UFM02_COMMAND_RAA_FWD                   (0x7BU)                                         // Read from firmware data area (NVRAM)                                                                             */

/*****************  Serial command protocol structure values  *****************/
#define UFM02_SERIAL_COMMAND_LENGTH                                 (7)

/******************************  Serial commands  *****************************/
typedef const uint8_t Ufm02_SerialCommand[UFM02_SERIAL_COMMAND_LENGTH];
#define UFM02_SERIAL_COMMAND_CLEAR_ACCUMULATED_FLOW                 {0xFE, 0xFE, 0x11, 0x5A, 0xFD, 0x57, 0x16}
#define UFM02_SERIAL_COMMAND_SET_PASSIVE_MODE                       {0xFE, 0xFE, 0x11, 0x5C, 0x01, 0x5D, 0x16}
#define UFM02_SERIAL_COMMAND_SET_ACTIVE_MODE                        {0xFE, 0xFE, 0x11, 0x5C, 0x00, 0x5C, 0x16}
#define UFM02_SERIAL_COMMAND_READ_SENSOR_DATA_NO_ID                 {0xFE, 0xFE, 0x11, 0x5B, 0x0F, 0x6A, 0x16}
#define UFM02_SERIAL_COMMAND_READ_SENSOR_DATA_WITH_ID               {0xFE, 0xFE, 0x11, 0x5B, 0xCB, 0x26, 0x16}
#define UFM02_SERIAL_COMMAND_RESET_MODULE                           {0xFE, 0xFE, 0x11, 0x5D, 0xCB, 0x28, 0x16}
#define UFM02_SERIAL_COMMAND_GET_SOFTWARE_VERSION                   {0xFE, 0xFE, 0x11, 0x5E, 0x62, 0xC0, 0x16}
#define UFM02_SERIAL_COMMAND_GET_MATERIAL_ID                        {0xFE, 0xFE, 0x11, 0x5E, 0x70, 0xCE, 0x16}
#define UFM02_SERIAL_COMMAND_GET_RAW_DATA                           {0xFE, 0xFE, 0x11, 0x5F, 0x00, 0x5F, 0x16}

#define UFM02_SERIAL_COMMAND_ADDRESS_START_BYTE_1                   (0xFE)
#define UFM02_SERIAL_COMMAND_ADDRESS_START_BYTE_2                   (0xFE)
#define UFM02_SERIAL_COMMAND_ADDRESS_START_BYTE_3                   (0x11)
#define UFM02_SERIAL_COMMAND_ADDRESS_STOP_BYTE                      (0x16)

#define UFM02_SERIAL_COMMAND_STARTUP_FLOW_RATE_LENGTH               (3)                         // Length in bytes of the Startup Flow Rate parameter

#define UFM02_SERIAL_COMMAND_CONFIGURATION_START_BYTE               (3)                         // First byte of the payload in the configuration command
#define UFM02_SERIAL_COMMAND_A0_START_BYTE                          (5)                         // First byte of the A0 configuration in the configuration command
#define UFM02_SERIAL_COMMAND_A1_START_BYTE                          (7)                         // First byte of the A1 configuration in the configuration command
#define UFM02_SERIAL_COMMAND_A2_START_BYTE                          (14)                        // First byte of the A2 configuration in the configuration command
#define UFM02_SERIAL_COMMAND_A3_START_BYTE                          (16)                        // First byte of the A3 configuration in the configuration command
#define UFM02_SERIAL_COMMAND_A4_START_BYTE                          (20)                        // First byte of the A4 configuration in the configuration command
#define UFM02_SERIAL_COMMAND_CHECKSUM_START_BYTE                    (23)                        // Checksum byte in the configuration command

#define UFM02_SERIAL_COMMAND_A0_LENGTH                              (1)                         // Length in bytes of the A0 parameter
#define UFM02_SERIAL_COMMAND_A1_LENGTH                              (6)                         // Length in bytes of the A1 parameter
#define UFM02_SERIAL_COMMAND_A2_LENGTH                              (1)                         // Length in bytes of the A2 parameter
#define UFM02_SERIAL_COMMAND_A3_LENGTH                              (3)                         // Length in bytes of the A3 parameter
#define UFM02_SERIAL_COMMAND_A4_LENGTH                              (3)                         // Length in bytes of the A4 parameter

/******************************  Serial replies  ******************************/
typedef size_t Ufm02_SerialCommandResponse;
#define UFM02_SERIAL_RESPONSE_ACKNOWLEDGE                           (0xE5)                      // Response of the UFM02 module to commands where no data must be sent

#define UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH                    (1)                         // Command response length to a command that requests no data
#define UFM02_SERIAL_RESPONSE_ACTIVE_MODE_LENGTH                    (32)                        // Command response length of the data that gets printed periodically in active mode
#define UFM02_SERIAL_RESPONSE_PASSIVE_MODE_WITH_ID_LENGTH           (39)                        // Command response length for the "ReadDataWithId" command
#define UFM02_SERIAL_RESPONSE_PASSIVE_MODE_NO_ID_LENGTH             (23)                        // Command response length for the "ReadDataNoId" command
#define UFM02_SERIAL_RESPONSE_GET_SOFTWARE_VERSION_LENGTH           (7)                         // Command response length for the "GetSoftwareVersion" command
#define UFM02_SERIAL_RESPONSE_GET_MATERIAL_ID_LENGTH                (7)                         // Command response length for the "GetMaterialId" command
#define UFM02_SERIAL_RESPONSE_GET_RAW_DATA_LENGTH                   (28)                        // Command response length for the "GetRawData" command

#define UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_1                  (0)
#define UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_2                  (1)
#define UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_BYTE_ADDRESS              (0)

#define UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_INTEGER_FLAG_ADDRESS     (2)                         // Byte address of the flag of the integer part of the accumulated flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_INTEGER_ADDRESS          (3)                         // Byte address of the start of the integer part of the accumulated flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_FRACTIONAL_FLAG_ADDRESS  (7)                         // Byte address of the flag of the fractional part of the accumulated flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_FRACTIONAL_ADDRESS       (8)                         // Byte address of the start of the fractional part of the accumulated flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_INSTANT_FLOW_FLAG_ADDRESS         (12)                        // Byte address of the flag of the instant flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_INSTANT_FLOW_ADDRESS              (13)                        // Byte address of the start of the instant flow data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_TEMP_FLAG_ADDRESS                 (17)                        // Byte address of the flag of the temperature data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_TEMP_ADDRESS                      (18)                        // Byte address of the start of the temperature data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_ERRORS_ADDRESS                    (22)                        // Byte address of the start of the error data in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_CHECKSUM_ADDRESS                  (26)                        // Byte address of the checksum byte in raw mode
#define UFM02_SERIAL_RESPONSE_RAW_STOP_ADDRESS                      (27)                        // Byte address of the stop byte in raw mode

#define UFM02_SERIAL_RESPONSE_SOFTWARE_VERSION_START_BYTE_ADDRESS   (1)                         // Address of the first byte of the software version response
#define UFM02_SERIAL_RESPONSE_SOFTWARE_CHECKSUM_BYTE_ADDRESS        (5)                         // Address of the checksum byte of the software version response
#define UFM02_SERIAL_RESPONSE_SOFTWARE_STOP_BYTE_ADDRESS            (6)                         // Address of the stop byte of the software version response

#define UFM02_SERIAL_RESPONSE_MATERIAL_ID_START_BYTE_ADDRESS        (1)                         // Address of the first byte of the material id response
#define UFM02_SERIAL_RESPONSE_MATERIAL_ID_CHECKSUM_BYTE_ADDRESS     (5)                         // Address of the checksum byte of the material id response
#define UFM02_SERIAL_RESPONSE_MATERIAL_ID_STOP_BYTE_ADDRESS         (6)                         // Address of the stop byte of the material id response

#define UFM02_SERIAL_RESPONSE_DATA_OUT_START_BYTE_1                 (0x3C)                      // First byte of the data output response
#define UFM02_SERIAL_RESPONSE_RAW_DATA_OUT_START_BYTE_2             (0x70)                      // Second byte of the data output response for the raw data
#define UFM02_SERIAL_RESPONSE_FLAG_A0                               (0xA0)                      // Flag for parameter A0
#define UFM02_SERIAL_RESPONSE_FLAG_A1                               (0xA1)                      // Flag for parameter A1
#define UFM02_SERIAL_RESPONSE_FLAG_A2                               (0xA2)                      // Flag for parameter A2
#define UFM02_SERIAL_RESPONSE_FLAG_A3                               (0xA3)                      // Flag for parameter A3
#define UFM02_SERIAL_RESPONSE_FLAG_A4                               (0xA4)                      // Flag for parameter A4
#define UFM02_SERIAL_RESPONSE_STOP_BYTE                             (0x16)                      // Stop byte of the data output responsenses

/*******************************  IO Protocols  *******************************/
typedef uint8_t Ufm02_Protocol;
#define UFM02_PROTOCOL_SPI                      (0)
#define UFM02_PROTOCOL_I2C                      (1)
#define UFM02_PROTOCOL_UART                     (2)

typedef uint8_t Ufm02_SerialOperatingMode;
#define UFM02_SERIAL_MODE_PASSIVE               (0)                                             // Device sends data on request
#define UFM02_SERIAL_MODE_ACTIVE                (1)                                             // Device sends data every second (Default)

/*******************  Error code definitions of ScioSense  ********************/
#ifndef SCIOSENSE_RESULT_CODES
#define SCIOSENSE_RESULT_CODES
typedef int8_t Result;
#define RESULT_TIMEOUT                          (5)                                             // A timeout was triggered before the result was produced
#define RESULT_NOT_ALLOWED                      (4)                                             // The requested command is not allowed
#define RESULT_CHECKSUM_ERROR                   (3)                                             // The value was read, but the checksum over the payload (valid and data) does not match
#define RESULT_INVALID                          (2)                                             // The value was read, but the data is invalid
#define RESULT_IO_ERROR                         (1)                                             // There was an IO communication error, read/write the stream failed
#define RESULT_OK                               (0)                                             // All OK; The value was read, the checksum matches, and data is valid
#endif  

#endif /* SCIOSENSE_UFM02_DEFINES_H */