#ifndef SCIOSENSE_UFM02_INL_H
#define SCIOSENSE_UFM02_INL_H
#include <Arduino.h>

#define transfer(dataToWrite, lenToWrite, dataToRead, lenToRead)    (Result)ufm02->io.transfer(ufm02->io.config, (uint8_t*)(dataToWrite), (lenToWrite), (uint8_t*)(dataToRead), (lenToRead))
#define write(data, len)                                            (Result)ufm02->io.write(ufm02->io.config, (uint8_t*)(data), (len))
#define wait(ms)                                                    ufm02->io.wait(ms)

/***************************  SPI and I2C commands  ***************************/

static inline uint32_t Ufm02_ComposeDWordMsbFirst(uint8_t* array)
{
    uint32_t outputValue    =   ( ((uint32_t)array[0]) << 24    );
    outputValue             +=  ( ((uint32_t)array[1]) << 16    );
    outputValue             +=  ( ((uint32_t)array[2]) << 8     );
    outputValue             +=  ( ((uint32_t)array[3])          );

    return outputValue;
}

static inline Result Ufm02_WriteOpcode(ScioSense_Ufm02* ufm02, uint8_t opcode)
{
    Result result = RESULT_INVALID;
    
    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) || (ufm02->io.protocol == UFM02_PROTOCOL_I2C) )
    {
        uint8_t valuesToWrite[UFM02_COMMAND_OPCODE_LENGTH];
        valuesToWrite[UFM02_COMMAND_OPCODE_INDEX] = opcode;
        
        result = write(valuesToWrite, UFM02_COMMAND_OPCODE_LENGTH);
    }

    return result;
}

static inline uint32_t Ufm02_ReadRegister1Dword(ScioSense_Ufm02* ufm02, uint16_t address)
{
    uint32_t readValue = 0;
    
    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) || (ufm02->io.protocol == UFM02_PROTOCOL_I2C) )
    {
        uint8_t dataToWrite[    UFM02_COMMAND_READ_REGISTER_LENGTH  ];
        uint8_t dataRead[       UFM02_REGISTER_LENGTH_BYTES         ];

        uint8_t opcode = UFM02_COMMAND_RAA_RD;
        if( address >= 0x100 )
        {
            // The request is to read from the NVRAM
            opcode  = UFM02_COMMAND_RAA_FWD;
            address = address & 0x00FF;
        }

        uint8_t amountBytesToWrite = 0;

        if( ufm02->io.protocol == UFM02_PROTOCOL_SPI )
        {
            dataToWrite[UFM02_COMMAND_OPCODE_INDEX]     = opcode;
            dataToWrite[UFM02_COMMAND_ADDRESS_INDEX]    = (uint8_t)address;
            amountBytesToWrite = UFM02_COMMAND_READ_REGISTER_LENGTH;
        }
        else if( ufm02->io.protocol == UFM02_PROTOCOL_I2C )
        {
            dataToWrite[0]     = (uint8_t)address;
            amountBytesToWrite = 1;
        }

        Result result = transfer(dataToWrite, amountBytesToWrite, dataRead, UFM02_REGISTER_LENGTH_BYTES);

        if ( result == RESULT_OK )
        {
            readValue = Ufm02_ComposeDWordMsbFirst(dataRead + UFM02_COMMAND_READ_MSB_BYTE_INDEX);
        }
    }
    
    return readValue;
}

static inline void Ufm02_ReadRegisterNDword(ScioSense_Ufm02* ufm02, uint16_t address, uint32_t* outputData, uint16_t amountRegistersToRead)
{
    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) || (ufm02->io.protocol == UFM02_PROTOCOL_I2C) )
    {
        for( uint16_t i = 0; i < amountRegistersToRead; i++ )
        {
            outputData[i] = Ufm02_ReadRegister1Dword(ufm02, address+i);
        }
    }
}

static inline Result Ufm02_Write8BitAddressRegister(ScioSense_Ufm02* ufm02, uint8_t address, uint32_t dataToWrite)
{
    Result result = RESULT_INVALID;

    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) )
    {
        if( ufm02->io.protocol == UFM02_PROTOCOL_SPI )
        {
            uint8_t opcode = UFM02_COMMAND_RAA_WR;
            uint8_t valuesToWrite[UFM02_COMMAND_WRITE_REGISTER_LENGTH];

            valuesToWrite[UFM02_COMMAND_OPCODE_INDEX            ] = opcode;
            valuesToWrite[UFM02_COMMAND_ADDRESS_INDEX           ] = address;
            valuesToWrite[UFM02_COMMAND_WRITE_MSB_BYTE_INDEX    ] = (uint8_t)(dataToWrite >> 24 );
            valuesToWrite[UFM02_COMMAND_WRITE_SECOND_BYTE_INDEX ] = (uint8_t)(dataToWrite >> 16 );
            valuesToWrite[UFM02_COMMAND_WRITE_THIRD_BYTE_INDEX  ] = (uint8_t)(dataToWrite >> 8  );
            valuesToWrite[UFM02_COMMAND_WRITE_LSB_BYTE_INDEX    ] = (uint8_t)(dataToWrite       );

            result = write(valuesToWrite, UFM02_COMMAND_WRITE_REGISTER_LENGTH);
        }
    }

    return result;
}

static inline Result Ufm02_ClearInterrupts(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_INVALID;

    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) )
    {
        uint32_t interruptsToClear = UFM02_INTERRUPT_BOOT_CLEAR | UFM02_INTERRUPT_ERROR_CLEAR | UFM02_INTERRUPT_MEASUREMENT_CLEAR;
        result = Ufm02_Write8BitAddressRegister(ufm02, UFM02_INTERRUPT_CLEAR_ADDRESS, interruptsToClear );
    }

    return result;
}

static inline Result Ufm02_Reset(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_IO_ERROR;
    
    switch( ufm02->io.protocol )
    {
        case UFM02_PROTOCOL_SPI:

            result = Ufm02_WriteOpcode(ufm02, UFM02_COMMAND_SYS_INIT);
            result |= Ufm02_ClearInterrupts(ufm02);
            break;

        case UFM02_PROTOCOL_I2C:

            result = Ufm02_WriteOpcode(ufm02, UFM02_COMMAND_SYS_INIT);
            break;

        case UFM02_PROTOCOL_UART:

            result = Ufm02_SerialReset(ufm02);
            break;

        default:

            result = RESULT_INVALID;
    }

    return result;
}

static inline Result Ufm02_Init(ScioSense_Ufm02* ufm02)
{
    Result result = Ufm02_Reset(ufm02);

    if( result == RESULT_OK )
    {
        wait(200);
    }

    return result;
}

static inline uint8_t Ufm02_isConnected(ScioSense_Ufm02* ufm02)
{
    uint8_t isConnected = 0;
    Ufm02_Update(ufm02);
    
    if( ufm02->io.protocol == UFM02_PROTOCOL_UART )
    {
        Ufm02_SerialGetMaterialId(ufm02);
    }
    
    if( Ufm02_ParseSensorProductGroupId(ufm02) == UFM02_PRODUCT_GROUP_ID )
    {
        isConnected = 1;
    }
    return isConnected;
}

static inline uint32_t Ufm02_GetRawErrorFlags(ScioSense_Ufm02* ufm02)
{
    return ufm02->errors;
}

static inline uint8_t Ufm02_HasErrorFlag(ScioSense_Ufm02* ufm02, uint8_t errorFlag)
{
    return (ufm02->errors  >> errorFlag ) & 1;
}

static inline Result Ufm02_ReadOutputData(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_INVALID;

    if( (ufm02->io.protocol == UFM02_PROTOCOL_SPI) || (ufm02->io.protocol == UFM02_PROTOCOL_I2C) )
    {
        ufm02->volumeIntRaw     = Ufm02_ReadRegister1Dword(ufm02, UFM02_VOLUME_INT_ADDRESS          );
        ufm02->volumeFracRaw    = Ufm02_ReadRegister1Dword(ufm02, UFM02_VOLUME_FRAC_ADDRESS         );
        ufm02->flowRaw          = Ufm02_ReadRegister1Dword(ufm02, UFM02_UNFILTERED_FLOW_LPH_ADDRESS );
        ufm02->flowFilteredRaw  = Ufm02_ReadRegister1Dword(ufm02, UFM02_FILTERED_FLOW_LPH_ADDRESS   );
        ufm02->tempRaw          = Ufm02_ReadRegister1Dword(ufm02, UFM02_TEMPERATURE_ADDRESS         );
        ufm02->dateCode         = Ufm02_ReadRegister1Dword(ufm02, UFM02_DEVICE_ID_ADDRESS           );
        ufm02->materialId       = Ufm02_ReadRegister1Dword(ufm02, UFM02_PART_ID_ADDRESS             );
        ufm02->errors           = Ufm02_ReadRegister1Dword(ufm02, UFM02_ERROR_FLAGS_ADDRESS         );
    }

    return result;
}

static inline Result Ufm02_Update(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_IO_ERROR;

    switch( ufm02->io.protocol )
    {
        case UFM02_PROTOCOL_SPI:

            Ufm02_ReadOutputData(ufm02);
            result = Ufm02_ClearInterrupts(ufm02);
            break;
            
        case UFM02_PROTOCOL_I2C:
            Ufm02_ReadOutputData(ufm02);
            result = RESULT_OK;
            break;
        
        case UFM02_PROTOCOL_UART:
            result = Ufm02_SerialGetSensorRawData(ufm02);
            break;
        
        default:
            result = RESULT_INVALID;
    }

    
    return result;
}

/***************************  Serial communication  ***************************/

static inline uint8_t Ufm02_CompareArrays(const uint8_t* a, const uint8_t* b, size_t s)
{
    for (size_t i = 0; i < s; i++)
    {
        if (a[i] != b[i]) return 0;
    }
    return 1;
}

static inline uint8_t Ufm02_SerialCalculateChecksum(uint8_t *val, uint8_t startCrcByte, uint8_t stopCrcByte)
{
    uint8_t crc = 0x00;

    for( uint8_t i=startCrcByte; i<stopCrcByte; i++ )
    {
        crc += val[i];
    }
    return crc;
}

static inline Result Ufm02_SerialCheckAcknowledgeResponse(uint8_t* data, const Ufm02_SerialCommandResponse size)
{
    Result result = RESULT_INVALID;

    if( size != UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH )
    {
        return result;
    }

    if( data[UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_BYTE_ADDRESS] == UFM02_SERIAL_RESPONSE_ACKNOWLEDGE ) 
    {
        result = RESULT_OK;
    }

    return result;
}

static inline Result Ufm02_SetPassiveMeasurement(ScioSense_Ufm02* ufm02)
{
    static Ufm02_SerialCommand commandSetPassiveMode = UFM02_SERIAL_COMMAND_SET_PASSIVE_MODE;

    size_t responseLength = UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH;
    uint8_t buff[UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH];
    
    Result result = transfer(commandSetPassiveMode, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);

    if ( result == RESULT_OK )
    {
        result = Ufm02_SerialCheckAcknowledgeResponse(buff, responseLength);
        if (result == RESULT_OK)
        {
            ufm02->serialMeasurementMode = UFM02_SERIAL_MODE_PASSIVE;
        }
    }

    return result;
}

static inline Result Ufm02_SetActiveMeasurement(ScioSense_Ufm02* ufm02)
{
    static Ufm02_SerialCommand commandSetActiveMode = UFM02_SERIAL_COMMAND_SET_ACTIVE_MODE;

    size_t responseLength = UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH;
    uint8_t buff[UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH];
    
    Result result = transfer(commandSetActiveMode, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);

    if ( result == RESULT_OK )
    {
        result = Ufm02_SerialCheckAcknowledgeResponse(buff, responseLength);
        if (result == RESULT_OK)
        {
            ufm02->serialMeasurementMode = UFM02_SERIAL_MODE_ACTIVE;
        }
    }

    return result;
}

static inline Result Ufm02_SerialCheckSensorRawDataResponse(uint8_t* data, const Ufm02_SerialCommandResponse size)
{
    Result result = RESULT_INVALID;
    
    if( size != UFM02_SERIAL_RESPONSE_GET_RAW_DATA_LENGTH )
    {
        return result;
    }

    if(     (data[UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_1                    ] != UFM02_SERIAL_RESPONSE_DATA_OUT_START_BYTE_1        )
        ||  (data[UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_2                    ] != UFM02_SERIAL_RESPONSE_RAW_DATA_OUT_START_BYTE_2    )
        ||  (data[UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_INTEGER_FLAG_ADDRESS       ] != UFM02_SERIAL_RESPONSE_FLAG_A0                      )
        ||  (data[UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_FRACTIONAL_FLAG_ADDRESS    ] != UFM02_SERIAL_RESPONSE_FLAG_A1                      )
        ||  (data[UFM02_SERIAL_RESPONSE_RAW_INSTANT_FLOW_FLAG_ADDRESS           ] != UFM02_SERIAL_RESPONSE_FLAG_A2                      )
        ||  (data[UFM02_SERIAL_RESPONSE_RAW_TEMP_FLAG_ADDRESS                   ] != UFM02_SERIAL_RESPONSE_FLAG_A3                      )
        ||  (data[UFM02_SERIAL_RESPONSE_RAW_STOP_ADDRESS                        ] != UFM02_SERIAL_RESPONSE_STOP_BYTE                    ) )
    {
        return result;
    }

    if( data[UFM02_SERIAL_RESPONSE_RAW_CHECKSUM_ADDRESS] == Ufm02_SerialCalculateChecksum( data, 0, UFM02_SERIAL_RESPONSE_RAW_CHECKSUM_ADDRESS ) )
    {
        result = RESULT_OK;
    }
    else
    {
        result = RESULT_CHECKSUM_ERROR;
    }
    
    return result;
}

static inline Result Ufm02_SerialCheckSWVersionResponse(uint8_t* data, const Ufm02_SerialCommandResponse size)
{
    Result result = RESULT_INVALID;
    
    if( size != UFM02_SERIAL_RESPONSE_GET_SOFTWARE_VERSION_LENGTH )
    {
        return result;
    }

    if(     (data[UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_1       ]   != UFM02_SERIAL_RESPONSE_ACKNOWLEDGE   )
        ||  (data[UFM02_SERIAL_RESPONSE_SOFTWARE_STOP_BYTE_ADDRESS ]   != UFM02_SERIAL_RESPONSE_STOP_BYTE     ) )
    {
        return result;
    }

    if( data[UFM02_SERIAL_RESPONSE_SOFTWARE_CHECKSUM_BYTE_ADDRESS] == Ufm02_SerialCalculateChecksum( data, UFM02_SERIAL_RESPONSE_SOFTWARE_VERSION_START_BYTE_ADDRESS, UFM02_SERIAL_RESPONSE_SOFTWARE_CHECKSUM_BYTE_ADDRESS ) )
    {
        result = RESULT_OK;
    }
    else
    {
        result = RESULT_CHECKSUM_ERROR;
    }
    
    return result;
}

static inline Result Ufm02_SerialCheckMaterialIdResponse(uint8_t* data, const Ufm02_SerialCommandResponse size)
{
    Result result = RESULT_INVALID;
    
    if( size != UFM02_SERIAL_RESPONSE_GET_MATERIAL_ID_LENGTH )
    {
        return result;
    }

    if(     (data[UFM02_SERIAL_RESPONSE_START_BYTE_ADDRESS_1            ] != UFM02_SERIAL_RESPONSE_ACKNOWLEDGE  )
        ||  (data[UFM02_SERIAL_RESPONSE_MATERIAL_ID_STOP_BYTE_ADDRESS   ] != UFM02_SERIAL_RESPONSE_STOP_BYTE    ) )
    {
        return result;
    }

    if( data[UFM02_SERIAL_RESPONSE_MATERIAL_ID_CHECKSUM_BYTE_ADDRESS] == Ufm02_SerialCalculateChecksum( data, UFM02_SERIAL_RESPONSE_MATERIAL_ID_START_BYTE_ADDRESS, UFM02_SERIAL_RESPONSE_MATERIAL_ID_CHECKSUM_BYTE_ADDRESS ) )
    {
        result = RESULT_OK;
    }
    else
    {
        result = RESULT_CHECKSUM_ERROR;
    }
    
    return result;
}

static inline Result Ufm02_SerialReset(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_INVALID;

    if (ufm02->io.protocol == UFM02_PROTOCOL_UART)
    {
        static Ufm02_SerialCommand commandReset = UFM02_SERIAL_COMMAND_RESET_MODULE;

        size_t responseLength = UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH;
        uint8_t buff[UFM02_SERIAL_RESPONSE_ACKNOWLEDGE_LENGTH];
        
        result = transfer(commandReset, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);
        
        if (result == RESULT_OK)
        {
            result = Ufm02_SerialCheckAcknowledgeResponse(buff, responseLength);
        }
    }

    return result;
}

static inline Result Ufm02_SerialGetSensorRawData(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_IO_ERROR;

    if (ufm02->io.protocol == UFM02_PROTOCOL_UART)
    {
        static Ufm02_SerialCommand commandGetSensorRawData = UFM02_SERIAL_COMMAND_GET_RAW_DATA;

        size_t responseLength = UFM02_SERIAL_RESPONSE_GET_RAW_DATA_LENGTH;
        uint8_t buff[UFM02_SERIAL_RESPONSE_GET_RAW_DATA_LENGTH];
        
        result = transfer(commandGetSensorRawData, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);
        
        if (result == RESULT_OK)
        {
            result = Ufm02_SerialCheckSensorRawDataResponse(buff, responseLength);

            if (result == RESULT_OK)
            {
                ufm02->volumeIntRaw     = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_INTEGER_ADDRESS      );
                ufm02->volumeFracRaw    = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_RAW_ACC_FLOW_FRACTIONAL_ADDRESS   );
                ufm02->flowRaw          = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_RAW_INSTANT_FLOW_ADDRESS          );
                ufm02->tempRaw          = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_RAW_TEMP_ADDRESS                  );
                ufm02->errors           = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_RAW_ERRORS_ADDRESS                );
            }
        }
    }

    return result;
}

static inline Result Ufm02_SerialGetSoftwareVersion(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_INVALID;

    if (ufm02->io.protocol == UFM02_PROTOCOL_UART)
    {
        static Ufm02_SerialCommand commandGetSoftwareVersion = UFM02_SERIAL_COMMAND_GET_SOFTWARE_VERSION;

        size_t responseLength = UFM02_SERIAL_RESPONSE_GET_SOFTWARE_VERSION_LENGTH;
        uint8_t buff[UFM02_SERIAL_RESPONSE_GET_SOFTWARE_VERSION_LENGTH];
        
        result = transfer(commandGetSoftwareVersion, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);

        if (result == RESULT_OK)
        {
            result = Ufm02_SerialCheckSWVersionResponse(buff, responseLength);

            if (result == RESULT_OK)
            {
                ufm02->materialId = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_SOFTWARE_VERSION_START_BYTE_ADDRESS);
            }
        }
    }

    return result;
}

static inline Result Ufm02_SerialGetMaterialId(ScioSense_Ufm02* ufm02)
{
    Result result = RESULT_INVALID;

    if (ufm02->io.protocol == UFM02_PROTOCOL_UART)
    {
        static Ufm02_SerialCommand commandGetMaterialId = UFM02_SERIAL_COMMAND_GET_MATERIAL_ID;

        size_t responseLength = UFM02_SERIAL_RESPONSE_GET_MATERIAL_ID_LENGTH;
        uint8_t buff[UFM02_SERIAL_RESPONSE_GET_MATERIAL_ID_LENGTH];
        
        result = transfer(commandGetMaterialId, UFM02_SERIAL_COMMAND_LENGTH, buff, responseLength);

        if (result == RESULT_OK)
        {
            result = Ufm02_SerialCheckMaterialIdResponse(buff, responseLength);

            if (result == RESULT_OK)
            {
                ufm02->materialId = Ufm02_ComposeDWordMsbFirst(buff + UFM02_SERIAL_RESPONSE_MATERIAL_ID_START_BYTE_ADDRESS);
            }
        }
    }

    return result;
}

/*******************************  Data parsing  *******************************/


static inline int32_t Ufm02_Parse_Flow_Volume_Int(ScioSense_Ufm02* ufm02)
{
    return (int32_t)(ufm02->volumeIntRaw);
}

static inline uint32_t Ufm02_Parse_Flow_Volume_Frac(ScioSense_Ufm02* ufm02)
{
    return ufm02->volumeFracRaw;
}

static inline float Ufm02_Parse_Flow_Volume_M3(ScioSense_Ufm02* ufm02)
{
    float flowIntegerPartM3 = (float)Ufm02_Parse_Flow_Volume_Int(ufm02);
    float flowFracPartM3    = (float)Ufm02_Parse_Flow_Volume_Frac(ufm02);
    
    float flowVolumeM3 = flowIntegerPartM3 + flowFracPartM3 / UFM02_FD32_DIVIDER_FLOAT;
    
    if( flowVolumeM3 < 0 )
    {
        flowVolumeM3 = 0;
    }
    return flowVolumeM3;
}

static inline int32_t Ufm02_Parse_Flow_Rate_Raw(ScioSense_Ufm02* ufm02)
{
    return (int32_t)(ufm02->flowRaw);
}

static inline float Ufm02_Parse_Flow_Rate_L_Per_Hr(ScioSense_Ufm02* ufm02)
{
    float flowRateLPerH = ((float)Ufm02_Parse_Flow_Rate_Raw(ufm02)) / UFM02_FD16_DIVIDER_FLOAT;
    if( flowRateLPerH < 0 )
    {
        flowRateLPerH = 0;
    }
    return flowRateLPerH;
}

static inline int32_t Ufm02_Parse_Flow_Rate_Filtered_Raw(ScioSense_Ufm02* ufm02)
{
    return (int32_t)(ufm02->flowFilteredRaw);
}

static inline float Ufm02_Parse_Flow_Rate_Filtered_L_Per_Hr(ScioSense_Ufm02* ufm02)
{
    float flowRateLPerH = ((float)Ufm02_Parse_Flow_Rate_Filtered_Raw(ufm02)) / UFM02_FD16_DIVIDER_FLOAT;
    if( flowRateLPerH < 0 )
    {
        flowRateLPerH = 0;
    }
    return flowRateLPerH;
}

static inline uint32_t Ufm02_Parse_Temperature_Raw(ScioSense_Ufm02* ufm02)
{
    return ufm02->tempRaw;
}

static inline float Ufm02_Parse_Temperature_Deg_C(ScioSense_Ufm02* ufm02)
{
    return ( ((float)Ufm02_Parse_Temperature_Raw(ufm02)) / UFM02_FD16_DIVIDER_FLOAT );
}

static inline uint32_t Ufm02_Parse_Error_Flags(ScioSense_Ufm02* ufm02)
{
    return ufm02->errors;
}

static inline uint16_t Ufm02_ParseSerialNo(ScioSense_Ufm02* ufm02)
{
    return (ufm02->dateCode / UFM02_DEVICE_ID_SERIAL_NUMBER_DIVIDER) % UFM02_DEVICE_ID_SERIAL_NUMBER_MODULUS;
}

static inline uint8_t Ufm02_ParseDateProductionDay(ScioSense_Ufm02* ufm02)
{
    return (ufm02->dateCode / UFM02_DEVICE_ID_DATE_DAY_DIVIDER) % UFM02_DEVICE_ID_DATE_DAY_MODULUS;
}

static inline uint8_t Ufm02_ParseDateProductionMonth(ScioSense_Ufm02* ufm02)
{
    return (ufm02->dateCode / UFM02_DEVICE_ID_DATE_MONTH_DIVIDER) % UFM02_DEVICE_ID_DATE_MONTH_MODULUS;
}

static inline uint8_t Ufm02_ParseDateProductionYear(ScioSense_Ufm02* ufm02)
{
    return (ufm02->dateCode / UFM02_DEVICE_ID_DATE_YEAR_DIVIDER) % UFM02_DEVICE_ID_DATE_YEAR_MODULUS;
}

static inline uint32_t Ufm02_ParsePartId(ScioSense_Ufm02* ufm02)
{
    return ufm02->materialId;
}

static inline uint8_t Ufm02_ParseSensorThreadId(ScioSense_Ufm02* ufm02)
{
    return (uint8_t)( (ufm02->materialId / UFM02_PART_ID_THREAD_DIVIDER) % UFM02_PART_ID_THREAD_MODULUS );
}

static inline uint8_t Ufm02_ParseSensorFamilyId(ScioSense_Ufm02* ufm02)
{
    return (uint8_t)( (ufm02->materialId / UFM02_PART_ID_FAMILY_DIVIDER) % UFM02_PART_ID_FAMILY_MODULUS );
}

static inline uint8_t Ufm02_ParseSensorApplicationId(ScioSense_Ufm02* ufm02)
{
    return (uint8_t)( (ufm02->materialId / UFM02_PART_ID_APPLICATION_DIVIDER) % UFM02_PART_ID_APPLICATION_MODULUS );
}

static inline uint16_t Ufm02_ParseSensorProductGroupId(ScioSense_Ufm02* ufm02)
{
    return (uint16_t)( (ufm02->materialId / UFM02_PART_ID_PRODUCT_LINE_DIVIDER) % UFM02_PART_ID_PRODUCT_LINE_MODULUS );
}

static inline float Ufm02_GetNominalFlow(ScioSense_Ufm02* ufm02)
{
    float nominalFlow = 0;
    switch( Ufm02_ParseSensorThreadId(ufm02) )
    {
        case UFM02_THREAD_ID_03_IN:
            nominalFlow = UFM02_NOMINAL_FLOW_UFM02_03;
            break;
        case UFM02_THREAD_ID_05_IN:
            nominalFlow = UFM02_NOMINAL_FLOW_UFM02_05;
            break;
        case UFM02_THREAD_ID_07_IN:
            nominalFlow = UFM02_NOMINAL_FLOW_UFM02_07;
            break;
        case UFM02_THREAD_ID_10_IN:
            nominalFlow = UFM02_NOMINAL_FLOW_UFM02_10;
            break;
        case UFM02_THREAD_ID_15_IN:
            nominalFlow = UFM02_NOMINAL_FLOW_UFM02_15;
            break;
        default:
            nominalFlow = 0;
    }
    return nominalFlow;
}

#undef read

#undef transfer
#undef write
#undef wait

#endif // SCIOSENSE_UFM02_INL_H
