#ifndef SCIOSENSE_UFM02_H
#define SCIOSENSE_UFM02_H

#include <stddef.h>

#include "ScioSense_Ufm02_defines.h"

typedef struct _ScioSense_Ufm02_IO
{
    Result  (*transfer)     (void* config, uint8_t* dataToWrite, const size_t sizeToWrite, uint8_t* dataToRead, const size_t sizeToRead);
    Result  (*write)        (void* config, uint8_t* data, const size_t size);
    Result  (*clear)        (void* config);
    void    (*wait)         (const uint32_t ms);
    
    Ufm02_Protocol              protocol;
    Ufm02_SerialOperatingMode   serialMeasurementMode;
    void*   config;
} ScioSense_Ufm02_IO;

typedef struct _ScioSense_Ufm02
{
    ScioSense_Ufm02_IO  io;
    Ufm02_SerialOperatingMode serialMeasurementMode;

    uint32_t volumeIntRaw;
    uint32_t volumeFracRaw;
    uint32_t flowRaw;
    uint32_t flowFilteredRaw;
    uint32_t tempRaw;
    uint32_t dateCode;
    uint32_t materialId;
    uint32_t errors;    
} ScioSense_Ufm02;

static inline Result    Ufm02_Reset                             (ScioSense_Ufm02* ufm02);                                                                           // Performs a software reset. Returns OK if the state indicates RESET afterwards
static inline Result    Ufm02_Init                              (ScioSense_Ufm02* ufm02);                                                                           // Initializes the device

static inline Result    Ufm02_WriteOpcode                       (ScioSense_Ufm02* ufm02, uint8_t opcode);                                                           // Writes 8 bit opcode
static inline uint32_t  Ufm02_ReadRegister1Dword                (ScioSense_Ufm02* ufm02, uint16_t address);                                                         // Writes 8 bit opcode,  8 bit address, and then reads 4 bytes and returns them as a number
static inline void      Ufm02_ReadRegisterNDword                (ScioSense_Ufm02* ufm02, uint16_t address, uint32_t* outputData, uint16_t amountRegistersToRead);   // Writes 8 bit opcode,  8 bit address, and then reads N DWORDs (4 bytes)
static inline Result    Ufm02_ClearInterrupts                   (ScioSense_Ufm02* ufm02);                                                                           // Clears the interrupt on the device, which will release the INTN output
static inline uint8_t   Ufm02_isConnected                       (ScioSense_Ufm02* ufm02);                                                                           // Returns whether a successful communication was achieved with the device
static inline uint32_t  Ufm02_GetRawErrorFlags                  (ScioSense_Ufm02* ufm02);                                                                           // Returns the error register content
static inline uint8_t   Ufm02_HasErrorFlag                      (ScioSense_Ufm02* ufm02, uint8_t errorFlag);                                                        // Returns whether the designated flag is active
static inline Result    Ufm02_Update                            (ScioSense_Ufm02* ufm02);                                                                           // Checks reads the output values of the UFM02

static inline uint8_t   Ufm02_CompareArrays                     (const uint8_t* a, const uint8_t* b, size_t s);                                                     // Returns true if the contents of the arrays is the same
static inline uint8_t   Ufm02_SerialCalculateChecksum           (uint8_t *val, uint8_t startCrcByte, uint8_t stopCrcByte);                                          // Calculate the checksum in a serial messge
static inline Result    Ufm02_SerialCheckAcknowledgeResponse    (uint8_t* data, const Ufm02_SerialCommandResponse size);                                            // Check if the serial ACK reply is correct
static inline Result    Ufm02_SerialCheckSensorRawDataResponse  (uint8_t* data, const Ufm02_SerialCommandResponse size);                                            // Check if the Raw data message is correct
static inline Result    Ufm02_SerialCheckSWVersionResponse      (uint8_t* data, const Ufm02_SerialCommandResponse size);                                            // Check if the software version message is correct
static inline Result    Ufm02_SerialReset                       (ScioSense_Ufm02* ufm02);                                                                           // Send the reset command through serial
static inline Result    Ufm02_SerialGetSoftwareVersion          (ScioSense_Ufm02* ufm02);                                                                           // Gets the software version from the sensor
static inline Result    Ufm02_SerialGetMaterialId               (ScioSense_Ufm02* ufm02);                                                                           // Gets the material ID from the sensor
static inline Result    Ufm02_SerialGetSensorRawData            (ScioSense_Ufm02* ufm02);                                                                           // Gets the raw data from the sensor

static inline int32_t   Ufm02_Parse_Flow_Volume_Int             (ScioSense_Ufm02* ufm02);                                                                           // Read the signed integer part of total volume of water flow in cubic meters calculated by the UFM-02
static inline uint32_t  Ufm02_Parse_Flow_Volume_Frac            (ScioSense_Ufm02* ufm02);                                                                           // Read unsigned factional part of total volume of water flow in cubic meters calculated by the UFM-02
static inline int32_t   Ufm02_Parse_Flow_Rate_Filtered_Raw      (ScioSense_Ufm02* ufm02);                                                                           // Read raw register content of the presently filtered calculated flow volume calculated by the UFM-02
static inline uint32_t  Ufm02_Parse_Temperature_Raw             (ScioSense_Ufm02* ufm02);                                                                           // Read raw register content of the temperature calculated by the UFM-02
static inline uint32_t  Ufm02_Parse_Error_Flags                 (ScioSense_Ufm02* ufm02);                                                                           // Read raw register content of the error flags

static inline float     Ufm02_Parse_Flow_Volume_M3              (ScioSense_Ufm02* ufm02);                                                                           // Read total volume of water flow in cubic meters calculated by the UFM-02
static inline float     Ufm02_Parse_Flow_Rate_Filtered_L_Per_Hr (ScioSense_Ufm02* ufm02);                                                                           // Read filtered flow rate in litres per hour calculated by the UFM-02
static inline float     Ufm02_Parse_Temperature_Deg_C           (ScioSense_Ufm02* ufm02);                                                                           // Read temperature in degrees Celsius calculated by the UFM-02
static inline float     Ufm02_GetNominalFlow                    (ScioSense_Ufm02* ufm02);                                                                           // Returns the nominal flow of the UFM-02

static inline uint16_t  Ufm02_ParseSerialNo                     (ScioSense_Ufm02* ufm02);                                                                           // Returns the serial number of the UFM-02
static inline uint8_t   Ufm02_ParseDateProductionDay            (ScioSense_Ufm02* ufm02);                                                                           // Returns the day of production
static inline uint8_t   Ufm02_ParseDateProductionMonth          (ScioSense_Ufm02* ufm02);                                                                           // Returns the month of production
static inline uint8_t   Ufm02_ParseDateProductionYear           (ScioSense_Ufm02* ufm02);                                                                           // Returns the year of production
static inline uint32_t  Ufm02_ParsePartId                       (ScioSense_Ufm02* ufm02);                                                                           // Returns the Part ID register content
static inline uint8_t   Ufm02_ParseSensorThreadId               (ScioSense_Ufm02* ufm02);                                                                           // Returns the identifyier of the thread
static inline uint8_t   Ufm02_ParseSensorFamilyId               (ScioSense_Ufm02* ufm02);                                                                           // Returns the product family
static inline uint8_t   Ufm02_ParseSensorApplicationId          (ScioSense_Ufm02* ufm02);                                                                           // Returns the product application
static inline uint16_t  Ufm02_ParseSensorProductGroupId         (ScioSense_Ufm02* ufm02);                                                                           // Returns the product group

#include "ScioSense_Ufm02.inl.h"
#endif // SCIOSENSE_UFM02_H
