#include <Arduino.h>
#include <ScioSense_UFM02.h>

#define UFM02_BAUDRATE              115200
#define UPDATE_PERIOD_MS            1000
#define AMOUNT_DECIMALS_VOLUME_M3   6

#define rxPin 16
#define txPin 17

// If your board only has one hardware serial bus, use SoftwareSerial
#ifndef ESP32
  #include <SoftwareSerial.h>
  SoftwareSerial softwareSerial = SoftwareSerial(rxPin, txPin);
#endif

UFM02 ufm02;

unsigned long lastUpdate;

uint8_t errorsToCheck[] = {UFM02_ERROR_FLAG_MEAS_NOT_OK, UFM02_ERROR_FLAG_BUBBLES};
uint8_t amountErrorsToCheck = (uint8_t)( sizeof(errorsToCheck) / sizeof(errorsToCheck[0]) );

void setup()
{
    Serial.begin(9600);

    Serial.println("Starting UFM02 03_Basic_UART demo on Arduino...");

    // If your board supports a second hardware serial bus (like the ESP32)...
    #ifdef ESP32
      // ...then use Serial1 or Serial2
      Serial2.begin(UFM02_BAUDRATE, SERIAL_8N1, rxPin, txPin);
      ufm02.begin(&Serial2);
    #else
      // ...otherwise go for SoftwareSerial
      softwareSerial.begin(UFM02_BAUDRATE);
      ufm02.begin(&softwareSerial);
    #endif

    if( ufm02.init() )
    {
        Serial.println("UFM02 initialized properly");
    }
    else
    {
        Serial.println("UFM02 type not recognized");
    }

    Serial.print("Product ID read: ");
    Serial.println(ufm02.getMaterialId());
    
    lastUpdate = millis();
}

void loop()
{
    unsigned long currentTimeMs = millis();
    if( (currentTimeMs - lastUpdate) > UPDATE_PERIOD_MS )
    {
        ufm02.update();
        
        // Check for errors
        uint8_t errorDetected = 0;
        for( uint8_t errorIdx = 0; errorIdx < amountErrorsToCheck; errorIdx++ )
        {
            if( ufm02.hasErrorFlag(errorsToCheck[errorIdx]) )
            {
                Serial.print("Error detected: ");
                Serial.println( ufm02.errorToString(errorsToCheck[errorIdx]) );
                errorDetected = 1;
            }
        }
        
        if( !errorDetected )
        {
            // Print the accumulated flow in cubic meters
            Serial.print("AccumulatedVolume[m3]:");
            Serial.print(ufm02.getVolumeM3(), AMOUNT_DECIMALS_VOLUME_M3);
            Serial.print("\t");
    
            // Print the current flow rate in litres per hour
            Serial.print("FlowRate[l_per_hr]:");
            Serial.print(ufm02.getInstantFlowLPerHr());
            Serial.print("\t");
            
            // Print the temperature in degrees celsius
            Serial.print("Temperature[degC]:");
            Serial.println(ufm02.getTemperatureDegC());
        }
        
        lastUpdate = currentTimeMs;
    }
}