#include <Arduino.h>
#include <SPI.h>
#include <ScioSense_AS60XX.h>

#define PIN_INTN  2     // define pin for interrupt
#define PIN_CS    5     // define pin for chip select

static AS60xx as60xx;

float measuredTemperatures[2];
AS60xx_Measure_Type measurementsReady   = AS60XX_NO_MEASUREMENT;
As60xx_ErrorFlag errorsToCheck[]        = {ERR_TOF_TMO, ERR_TM_OC_ERR, ERR_TM_SC_ERR, ERR_TM_SQC_TMO, ERR_CHP_ERR};
uint8_t amountErrorsToCheck             = (uint8_t)( sizeof(errorsToCheck) / sizeof(errorsToCheck[0]) );

void setup()
{
    Serial.begin(9600);

    Serial.println("Starting AS60xx 02_AS60xx_Example_Ultrasound_and_Temperature demo on Arduino...");

    delay(AS60XX_BOOTUP_COMM_RELEASE_TIME_MAX_MS);

    SPI.begin();
    as60xx.begin(&SPI, PIN_CS, SPISettings(14000000, MSBFIRST, SPI_MODE1));

    if( as60xx.init() )
    {
        Serial.println("AS60xx initialized properly");
    }
    else
    {
        Serial.println("AS60xx initialization failed");
    }
    
    Serial.print("High Speed Clock Frequency: ");
    Serial.print(as60xx.getHighSpeedClockFrequencyHz() / 1000000.0);
    Serial.println(" MHz");

    Serial.print("Supply Voltage: ");
    Serial.print(as60xx.getSupplyVoltageMv());
    Serial.println(" mV");

    uint32_t configurationRegistersUsAndT[] = {
        0x48DBA399,     // 0x0C0
        0x00800401,     // 0x0C1
        0x00000000,     // 0x0C2
        0x00000001,     // 0x0C3
        0x0011FFFF,     // 0x0C4
        0x4046EE08,     // 0x0C5
        0x01012100,     // 0x0C6
        0x00246008,     // 0x0C7
        0x006C07E4,     // 0x0C8
        0x60160204,     // 0x0C9
        0x010FEA10,     // 0x0CA
        0x23A0DE71,     // 0x0CB
        0x94A0C46C,     // 0x0CC
        0x401100C4,     // 0x0CD
        0x00A7400F,     // 0x0CE
        0x00000001,     // 0x0D0
        0x000015E0,     // 0x0D1
        0x000015E0,     // 0x0D2
        0x00000023,     // 0x0DA
        0x00000023      // 0x0DB
    };
    as60xx.setConfigurationRegisters(configurationRegistersUsAndT);

    if( as60xx.writeConfig() == RESULT_OK )
    {
        Serial.println("Configuration properly written");
    }
    else
    {
        Serial.println("Error! Configuration read doesn't match the values written");
    }
    
    if( as60xx.startMeasurement() == RESULT_OK )
    {
        Serial.println("Measurements started");
    }
    else
    {
        Serial.println("Error! Measurements didn't start properly");
    }

}

void loop()
{
    if( !digitalRead(PIN_INTN) )
    {
        measurementsReady |= as60xx.update();
            
        if( (measurementsReady & AS60XX_TEMPERATURE_MEASUREMENT) && (measurementsReady & AS60XX_TOF_MEASUREMENT) )
        {
            // Print the timestamp
            As60xx_Timestamp timestamp = as60xx.getTimestamp();
            Serial.print("Timestamp[hh:mm:ss]:");
            Serial.print(timestamp.hours);
            Serial.print(":");
            Serial.print(timestamp.minutes);
            Serial.print(":");
            Serial.print(timestamp.seconds);
            Serial.print("\t");

            // Print the ToF sum upstream and downstream
            uint32_t avgTofUp = as60xx.getSumTofUp() / as60xx.getAmountHitsForSumToF();
            uint32_t avgTofDn = as60xx.getSumTofDown() / as60xx.getAmountHitsForSumToF();
            float avgTofUp_ns = as60xx.calculateTimeNs(avgTofUp);
            float avgTofDn_ns = as60xx.calculateTimeNs(avgTofDn);

            Serial.print("AvgTofSumUp[ns]:");
            Serial.print(avgTofUp_ns);
            Serial.print("\t");
            
            Serial.print("AvgTofSumDn[ns]:");
            Serial.print(avgTofDn_ns);
            Serial.print("\t");

            // Print the ToF difference
            Serial.print("TofDiff[ns]:");
            Serial.print(avgTofUp_ns - avgTofDn_ns);
            Serial.print("\t");

            // Print the temperatures measured with the 2-wire RTD
            as60xx.getTemperatures2WireDegC(measuredTemperatures);
            
            Serial.print("TempH[degC]:");
            Serial.print(measuredTemperatures[1]);
            Serial.print("\t");
            Serial.print("TempC[degC]:");
            Serial.print(measuredTemperatures[0]);
            Serial.print("\t");

            // Print the internal temperature of the AS60xx sensor
            Serial.print("TempInt[degC]");
            Serial.print(as60xx.getInternalTemperatureDegC());
        }

        for( uint8_t errorIdx = 0; errorIdx < amountErrorsToCheck; errorIdx++ )
        {
            if( as60xx.hasErrorFlag(errorsToCheck[errorIdx]) )
            {
                Serial.print("\tError detected: ");
                Serial.print( as60xx.errorToString(errorsToCheck[errorIdx]) );
            }
        }

        Serial.println("");
    }
}