## SavaTime
**SavaTime Library**
  
**Простая и мощная библиотека таймеров для Arduino.**  
Позволяет выполнять действия с задержкой, мигать светодиодами и создавать автоматику без использования блокирующей функции `delay()`.

Библиотека построена по принципу промышленных функциональных блоков (как в ПЛК или FLProg): вы просто "пропускаете" сигнал через таймер, а он сам решает, когда включиться или выключиться.

---

## 🛠 Установка
1. Скачайте архив с библиотекой.
2. В Arduino IDE выберите: `Скетч` -> `Подключить библиотеку` -> `Добавить .ZIP библиотеку...`
3. Выберите скачанный файл.

---

## 🚀 Как это работает (Главный принцип)

В обычных библиотеках вы пишете условия `if` снаружи. В **SavaTime** вы передаете условие внутрь.

**❌ Как делать НЕ надо (Плохой стиль):**
```cpp
if (digitalRead(12) == HIGH) {
    timer.TON(1000, true); // Если кнопка отпущена, таймер зависнет!
}
```

**✅ Как НАДО (Правильный стиль):**
```cpp
// Вызываем функцию всегда! А управляем через второй аргумент.
timer.TON(1000, digitalRead(12)); 
```
Когда аргумент станет `false` (0), таймер сам поймет, что нужно сброситься.

---

## 📚 Описание функций

Сначала подключите библиотеку и создайте таймер:
```cpp
#include "SavaTime.h"
SavaTime myTimer; // Создаем экземпляр (можно много, для каждой задачи свой)
```

### 1. Автоматика (Реле времени)

#### 🔹 TON (Задержка включения)
Фильтр дребезга или задержка старта.
*   **Логика:** Сигнал есть -> ждем время -> включаем. Сигнал пропал -> мгновенно выключаем.
*   **Пример:** Включить насос, только если кнопку держат 2 секунды.

```cpp
// TON(время_мс, условие)
bool pumpState = myTimer.TON(2000, digitalRead(12));
digitalWrite(13, pumpState);
```

#### 🔹 TOF (Задержка выключения)
Таймер удержания или "подсветка подъезда".
*   **Логика:** Сигнал есть -> выход ВКЛ (таймер заряжается). Сигнал пропал -> выход ВКЛ еще заданное время, потом ВЫКЛ.
*   **Особенность:** Если во время отсчета снова подать сигнал, время сбросится и начнется заново.

```cpp
// TOF(время_мс, условие)
bool lightState = myTimer.TOF(5000, digitalRead(12)); 
digitalWrite(13, lightState); // Горит, пока держим, + 5 сек после отпускания
```

---

### 2. Импульсы и События

#### 🔹 Time (Одиночный выстрел)
Снайперский режим.
*   **Логика:** Сигнал появился -> ждем время -> выдаем **ОДИН** импульс -> замолкаем.
*   **Важно:** Чтобы сработать снова, нужно обязательно отпустить кнопку (сбросить сигнал в 0).

```cpp
// Time(время_мс, условие)
if (myTimer.Time(1000, digitalRead(12))) {
    Serial.println("БАХ! Импульс прошел."); 
    // Это сообщение появится только 1 раз, даже если кнопку продолжать держать
}
```

---

### 3. Генераторы (Мигалки)

#### 🔹 Gen (Генератор событий)
Выдает короткий импульс (`true`) каждые N миллисекунд.
*   **Применение:** Опрос датчиков, вывод в порт, часы.

```cpp
// Gen(период_мс, разрешение_работы)
if (myTimer.Gen(1000, true)) { // true = работает всегда
    Serial.println("Прошла еще одна секунда...");
}
```

#### 🔹 Multi (Симметричный мультивибратор)
Классическая "мигалка". Включает и выключает выход на одинаковое время.

```cpp
// Multi(период_мс, разрешение_работы)
// Мигает: 500 мс горит, 500 мс не горит, пока нажата кнопка
digitalWrite(13, myTimer.Multi(500, digitalRead(12)));
```

#### 🔹 AsMulti (Асимметричный мультивибратор)
Позволяет задать разное время для ВКЛ и ВЫКЛ.
*   **Применение:** Проблесковые маячки, короткие вспышки, долгие паузы.

```cpp
// AsMulti(время_ВКЛ, время_ВЫКЛ, разрешение_работы)
// Вспышка 50 мс, пауза 2 секунды
bool strobe = myTimer.AsMulti(50, 2000, true);
digitalWrite(13, strobe);
```

---

### 4. Управление и Сервис

#### 🛠 Reset()
Полный сброс таймера.
Используйте эту команду, если логика вашего скетча прерывается (например, вы используете `switch-case` или `goto`), и вы хотите, чтобы при возвращении таймер был "чистым", а не сработал мгновенно из-за старой памяти.

```cpp
if (emergency) {
    myTimer.Reset(); // Обнулить всё!
}
```

#### ⏱ TOF_Remaining()
Показывает, сколько миллисекунд осталось работать таймеру `TOF` (обратный отсчет).
*   Если таймер не запущен или время вышло — вернет 0.

```cpp
uint32_t left = myTimer.TOF_Remaining(5000);
Serial.print("Осталось гореть: ");
Serial.println(left);
```

---

## ⚡ Работа с микросекундами
Если нужна высокая точность (быстрее миллисекунды), используйте версии функций с суффиксом `Micros`:
*   `StartMicros()`
*   `TimeMicros(период_мкс, условие)`

---

## 📝 Пример кода (Быстрый старт)

```cpp
#include "SavaTime.h"

SavaTime ledTimer;
SavaTime printTimer;

void setup() {
  pinMode(13, OUTPUT);
  Serial.begin(9600);
}

void loop() {
  // Мигаем светодиодом каждые 500 мс
  digitalWrite(13, ledTimer.Multi(500));

  // Пишем в порт каждые 2000 мс
  if (printTimer.Gen(2000)) {
    Serial.println("Работаю...");
  }
}
```
