
# 📖 Полная документация библиотеки SavaOLED_ESP32

Библиотека предназначена для управления OLED дисплеями на базе контроллера **SSD1306** через интерфейс **I2C** на микроконтроллерах **ESP32**.

## 📑 Оглавление
1. [Подключение и Константы](#1-подключение-и-константы)
2. [Конструктор и Инициализация](#2-конструктор-и-инициализация)
3. [Настройки текста и режимов](#3-настройки-текста-и-режимов)
4. [Управление курсором](#4-управление-курсором)
5. [Печать текста (Накопление)](#5-печать-текста-накопление)
6. [Отрисовка текста (Рендеринг)](#6-отрисовка-текста-рендеринг)
7. [Графические примитивы](#7-графические-примитивы)
8. [Работа с буфером и Экраном](#8-работа-с-буфером-и-экраном)
9. [Аппаратное управление дисплеем](#9-аппаратное-управление-дисплеем)
10. [Получение информации (Getters)](#10-получение-информации-getters)

---

## 1. Подключение и Константы

Перед началом работы подключите библиотеку и необходимые шрифты.

```cpp
#include <Arduino.h>
#include "SavaOLED_ESP32.h"
#include "fonts/SavaFont_ilya_Pro_8px.h" // Пример шрифта
```

### Полезные константы (для аргументов)
**Режимы отрисовки (`mode`):**
*   `REPLACE` — Заменяет пиксели (рисует 1, стирает 0). *По умолчанию.*
*   `ADD_UP` — Сложение (рисует 1, 0 прозрачен).
*   `INV_AUTO` — Инверсия (XOR).
*   `ERASE` — Ластик (рисует 0).
*   `ERASE_BORDER` — Стирает фигуру, затем рисует её контур.

**Выравнивание текста (`align`):**
*   `StrLeft` — По левому краю.
*   `StrCenter` — По центру.
*   `StrRight` — По правому краю.
*   `StrScroll` — Режим бегущей строки.

**Заливка (`fill`):**
*   `FILL` (true) — Залитая фигура.
*   `NO_FILL` (false) — Только контур.

---

## 2. Конструктор и Инициализация

### `SavaOLED_ESP32` (Конструктор)
Создает экземпляр объекта. Вызывается глобально.

```cpp
SavaOLED_ESP32(uint8_t width = 128, uint8_t height = 64, i2c_port_t port = I2C_NUM_0);
```
*   **`width`**: Ширина экрана. *По умолчанию: 128*.
*   **`height`**: Высота экрана (64 или 32). *По умолчанию: 64*.
*   **`port`**: Порт I2C ESP32 (`I2C_NUM_0` или `I2C_NUM_1`). *По умолчанию: I2C_NUM_0*.

**Примеры:**
```cpp
SavaOLED_ESP32 oled(); // 128x64, порт 0
SavaOLED_ESP32 oled(128, 32); // 128x32, порт 0
SavaOLED_ESP32 oled(128, 64, I2C_NUM_1); // 128x64, порт 1
```

### `setAddress`
Задает I2C адрес устройства. Вызывать **до** `init()`.
```cpp
void setAddress(uint8_t address = 0x3C);
```
*   **`address`**: Адрес устройства. Обычно `0x3C` или `0x3D`. *По умолчанию: 0x3C*.

### `init`
Инициализирует шину I2C и дисплей. Обязательно вызывать в `setup()`.
```cpp
void init(uint32_t freq = 400000, int8_t _sda = 21, int8_t _scl = 22);
```
*   **`freq`**: Частота шины в Гц. *По умолчанию: 400000 (400 кГц)*.
*   **`_sda`**: GPIO пин линии данных. *По умолчанию: 21*.
*   **`_scl`**: GPIO пин линии тактирования. *По умолчанию: 22*.

**Примеры:**
```cpp
oled.init(); // Стандартные настройки
oled.init(1000000, 4, 5); // 1 МГц, SDA на 4 пине, SCL на 5
```

---

## 3. Настройки текста и режимов

### `font`
Устанавливает активный шрифт для последующих команд `print`.
```cpp
void font(const savaFont &fontPtr);
```
*   **`fontPtr`**: Имя структуры шрифта (из подключенного .h файла).

### `drawMode`
Устанавливает глобальный режим наложения пикселей для текста и примитивов (если в примитиве не указан свой).
```cpp
void drawMode(uint8_t mode = REPLACE);
```
*   **`mode`**: `REPLACE`, `ADD_UP`, `INV_AUTO` и т.д.

### `charSpacing`
Устанавливает расстояние между символами в пикселях.
```cpp
void charSpacing(uint8_t spacing);
```
*   **`spacing`**: Число пикселей отступа. *По умолчанию (в конструкторе): 1*.

### `scroll`
Включает или выключает режим прокрутки для текста, напечатанного с выравниванием `StrScroll`.
```cpp
void scroll(bool enabled);
```
*   **`enabled`**: `true` — включить, `false` — выключить.

### `scrollSpeed`
Настраивает параметры горизонтальной прокрутки.
```cpp
void scrollSpeed(uint8_t speed = 3, bool loop = true);
```
*   **`speed`**: Скорость от 1 (медленно) до 10 (очень быстро). *По умолчанию: 3*.
*   **`loop`**: `true` — бесконечная прокрутка, `false` — прокрутить один раз и остановиться. *По умолчанию: true*.

### `scrollSpeedVert`
Настраивает скорость вертикальной прокрутки (используется в `drawPrintVert`).
```cpp
void scrollSpeedVert(uint8_t speed = 3);
```
*   **`speed`**: Скорость от 1 до 10. *По умолчанию: 3*.

### `setBuffer`
Устанавливает метод передачи данных на экран.
```cpp
void setBuffer(bool enabled);
```
*   **`enabled`**:
    *   `true` (FULL) — отправлять буфер целиком за одну транзакцию (быстрее, но требует большого буфера I2C).
    *   `false` (PAGES) — отправлять постранично (надежнее, стандартно). *По умолчанию: false*.

---

## 4. Управление курсором

Перед использованием `print()` необходимо установить курсор.

### `cursor`
Задает позицию и область для вывода текста.
```cpp
void cursor(int16_t x, int16_t y, uint8_t align = StrLeft, int16_t x2 = -1);
```
*   **`x`**: Координата X (левый край или точка привязки).
*   **`y`**: Координата Y (верхний край строки).
*   **`align`**: Режим выравнивания (`StrLeft`, `StrCenter`, `StrRight`, `StrScroll`). *По умолчанию: StrLeft*.
*   **`x2`**: Правая граница области печати. Если `-1`, то используется ширина экрана. *По умолчанию: -1*.

**Примеры:**
```cpp
oled.cursor(0, 0); // Левый верхний угол
oled.cursor(0, 10, StrCenter); // Центрировать по ширине экрана
oled.cursor(10, 10, StrLeft, 60); // Писать в области от X=10 до X=60
```

---

## 5. Печать текста (Накопление)

Эти функции **не рисуют** текст сразу, а добавляют его во внутренний буфер строки. Поддерживается цепочка вызовов.

### `print` (Строка)
```cpp
void print(const char* text);
void print(const String &s);
```
*   **`text` / `s`**: Текст для вывода. Поддерживает кириллицу (CP1251 перекодируется автоматически).

### `print` (Целые числа)
```cpp
void print(int value, uint8_t min_digits = 0);
// Также перегрузки для: int8_t, uint8_t, int16_t, uint16_t, int32_t, uint32_t
```
*   **`value`**: Число для вывода.
*   **`min_digits`**: Минимальное количество цифр. Если число короче, оно дополняется нулями слева. *По умолчанию: 0 (без нулей)*.

**Пример:** `oled.print(5, 3)` выведет `"005"`.

### `print` (Дробные числа)
```cpp
void print(float value, uint8_t decimalPlaces = 2, uint8_t min_width = 0);
void print(double value, uint8_t decimalPlaces = 2, uint8_t min_width = 0);
```
*   **`value`**: Число.
*   **`decimalPlaces`**: Количество знаков после запятой. *По умолчанию: 2*.
*   **`min_width`**: Минимальная общая ширина строки (дополняется пробелами слева). *По умолчанию: 0*.

---

## 6. Отрисовка текста (Рендеринг)

Эти функции берут всё, что было накоплено через `print()`, и рисуют в видеопамять.

### `drawPrint`
Стандартная отрисовка горизонтальной строки текста.
```cpp
void drawPrint();
```
Отрисовывает текст согласно настройкам `cursor()`, `font()`, `drawMode()`.

### `drawPrintVert`
Отрисовка вертикального текста (сверху вниз).
```cpp
void drawPrintVert();
```
*Примечание:* Шрифт должен быть повернут на 90 градусов в редакторе шрифтов, либо эта функция сама укладывает буквы столбиком (зависит от реализации шрифта). Использует `scrollSpeedVert` для прокрутки.

---

## 7. Графические примитивы

Координаты начинаются с 0. Максимум (width-1, height-1).

### `dot` (Точка)
```cpp
void dot(int16_t x, int16_t y, uint8_t mode = REPLACE);
```

### `line` (Линия)
Рисует произвольную линию (алгоритм Брезенхэма).
```cpp
void line(int16_t x1, int16_t y1, int16_t x2, int16_t y2, uint8_t mode = REPLACE);
```

### `hLine` (Горизонтальная линия)
Быстрая отрисовка горизонтальной линии.
```cpp
void hLine(int16_t x, int16_t y, int16_t w, uint8_t mode = REPLACE);
```
*   **`w`**: Длина линии.

### `vLine` (Вертикальная линия)
Быстрая отрисовка вертикальной линии.
```cpp
void vLine(int16_t x, int16_t y, int16_t h, uint8_t mode = REPLACE);
```
*   **`h`**: Высота линии.

### `rect` (Прямоугольник)
```cpp
void rect(int16_t x, int16_t y, int16_t w, int16_t h, uint8_t mode = REPLACE, bool fill = NO_FILL);
```
*   **`fill`**: `true` — залить, `false` — только рамка.

### `rectR` (Скругленный прямоугольник)
```cpp
void rectR(int16_t x, int16_t y, int16_t w, int16_t h, int16_t r, uint8_t mode = REPLACE, bool fill = NO_FILL);
```
*   **`r`**: Радиус скругления углов.

### `circle` (Круг)
```cpp
void circle(int16_t x0, int16_t y0, int16_t r, uint8_t mode = REPLACE, bool fill = NO_FILL);
```

### `bezier` (Кривая Безье)
Рисует плавную кривую по трем точкам (начало, контрольная, конец).
```cpp
void bezier(int16_t x0, int16_t y0, int16_t x1, int16_t y1, int16_t x2, int16_t y2, uint8_t mode = REPLACE);
```
*   **`x1, y1`**: Контрольная точка (изгиб тянется к ней).

### `drawPeak` (Пик)
Рисует кривую, которая гарантированно проходит через вершину (пик). Удобно для графиков.
```cpp
void drawPeak(int16_t x0, int16_t y0, int16_t x_peak, int16_t y_peak, int16_t x2, int16_t y2, uint8_t mode = REPLACE);
```
*   **`x_peak, y_peak`**: Координаты вершины.

### `drawBitmap` (Картинка)
Рисует монохромное изображение из массива байт.
```cpp
void drawBitmap(int16_t x, int16_t y, const uint8_t* bitmap, int16_t w, int16_t h, uint8_t mode = REPLACE);
```
*   **`bitmap`**: Указатель на массив данных (формат: вертикальные байты, младший бит сверху).

### `fillScreen` (Заливка)
Заполняет весь экран паттерном.
```cpp
void fillScreen(uint8_t pattern);
```
*   **`pattern`**: Байт-маска.
    *   `0x00` — Очистить (черный).
    *   `0xFF` — Залить (белый).
    *   `0x55` — Черезстрочная "зебра".

---

## 8. Работа с буфером и Экраном

### `clear`
Очищает программный буфер дисплея (заполняет нулями).
```cpp
void clear();
```

### `display`
**Самая важная команда.** Отправляет содержимое буфера на физический экран по I2C.
```cpp
void display();
```

---

## 9. Аппаратное управление дисплеем

### `contrast`
Регулирует яркость дисплея.
```cpp
void contrast(uint8_t value);
```
*   **`value`**: `0` (минимум) ... `255` (максимум).

### `power`
Включает или выключает экран (Sleep mode).
```cpp
void power(bool mode);
```
*   **`mode`**: `true` — включить, `false` — выключить.

### `invertDisplay`
Аппаратная инверсия цветов (белое становится черным без перерисовки буфера).
```cpp
void invertDisplay(bool mode);
```

### `flipH` / `flipV`
Зеркальное отражение изображения.
```cpp
void flipH(bool mode); // Горизонтально
void flipV(bool mode); // Вертикально
```

### `rotation`
Поворот экрана на 180 градусов.
```cpp
void rotation(bool rotate180);
```
*   **`rotate180`**: `true` — перевернутый режим, `false` — нормальный.

---

## 10. Получение информации (Getters)

Функции для получения текущих параметров состояния.

*   `int16_t getCursorX() const` — Текущая координата X курсора.
*   `int16_t getCursorY() const` — Текущая координата Y курсора.
*   `uint16_t getTextWidth() const` — Ширина последней отрисованной строки текста в пикселях.
*   `uint16_t getTextHeight() const` — Высота текущего шрифта.
*   `uint16_t getScopeCursor() const` — Ширина текущей области курсора (значение `x2` или ширина экрана).

---

## Пример использования (Скелет скетча)

```cpp
#include <Arduino.h>
#include "SavaOLED_ESP32.h"
#include "fonts/SavaFont_ilya_Pro_8px.h" 

// 1. Конструктор
SavaOLED_ESP32 oled(128, 64);

void setup() {
    Serial.begin(115200);
    
    // 2. Инициализация
    oled.init(); // 400kHz, SDA=21, SCL=22
    oled.clear();
    oled.font(SavaFont_ilya_Pro_8px); // Установка шрифта
}

void loop() {
    // 3. Очистка буфера перед новым кадром
    oled.clear(); 
    
    // 4. Рисование примитивов
    oled.rectR(0, 0, 128, 64, 5, REPLACE, NO_FILL); // Рамка вокруг экрана
    
    // 5. Вывод текста
    oled.cursor(64, 20, StrCenter); // Курсор по центру
    oled.print("SavaOLED");
    oled.drawPrint(); // Отрисовка текста
    
    // 6. Вывод чисел
    oled.cursor(64, 40, StrCenter);
    oled.print(millis() / 1000); // Секунды
    oled.drawPrint();

    // 7. Отправка на экран
    oled.display(); 
}
```
