/**
 * @file 08_SetPixelHSV_Rainbow.ino
 * @brief Пример индивидуального управления пикселями через модель HSV.
 * 
 * Этот скетч демонстрирует использование функции `setPixelHSV(index, h, s, v)`
 * для создания плавных цветовых градиентов. HSV (Hue, Saturation, Value) — это
 * очень интуитивный способ работы с цветом для анимаций.
 * 
 * Эффект:
 * Классическая "бегущая радуга", где градиент всех цветов плавно
 * движется вдоль светодиодной ленты.
 * 
 * АРХИТЕКТУРА:
 * Мы используем правильный паттерн `if (strip.canShow()) { ... }`, синхронизируя
 * расчеты нового кадра с реальной скоростью обновления ленты. Это обеспечивает
 * максимальную производительность и плавность.
 * 
 * =========================================================================
 *                    Что такое цветовая модель HSV?
 * =========================================================================
 * 
 * В отличие от RGB (Красный, Зеленый, Синий), модель HSV описывает цвет так,
 * как его интуитивно воспринимает человек:
 * 
 * 1. H - Hue (Тон/Оттенок): 0-255
 *    Это сам цвет на цветовом круге. Плавно изменяя только этот параметр,
 *    можно пройти по всем цветам радуги.
 * 
 * 2. S - Saturation (Насыщенность): 0-255
 *    Это "сочность" или "чистота" цвета.
 *    - 255: Максимально чистый, сочный цвет.
 *    - 0: Цвет полностью "выцветает" до оттенка серого (от черного до белого).
 * 
 * 3. V - Value (Значение/Яркость): 0-255
 *    Это яркость цвета.
 *    - 255: Максимальная яркость.
 *    - 0: Черный цвет, независимо от Тона и Насыщенности.
 * 
 * -------------------------------------------------------------------------
 *   HUE | ЦВЕТ (при S=255, V=255)
 * -------------------------------------------------------------------------
 *     0 | Красный
 *    42 | Желтый
 *    85 | Зеленый
 *   127 | Голубой (Cyan)
 *   170 | Синий
 *   212 | Пурпурный (Magenta)
 *   255 | Красный (цикл замыкается)
 * -------------------------------------------------------------------------
 * 
 * HSV идеально подходит для создания радуг, так как нужно анимировать
 * всего одну переменную - Hue.
 * =========================================================================
 */

#include <SavaLED_ESP32.h>

// --- Конфигурация ---
#define LED_PIN    14
#define NUM_LEDS   100
#define BRIGHTNESS 200

SavaLED_ESP32 strip;

// Эта переменная будет хранить "сдвиг" нашей радуги.
// Увеличивая ее на каждом кадре, мы заставляем радугу двигаться.
uint8_t hue_offset = 0;

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример 08: Бегущая радуга через setPixelHSV");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    while (true);
  }
  strip.setBrightness(BRIGHTNESS);
}

void loop() {
  // Выполняем расчет и отрисовку нового кадра только тогда,
  // когда лента физически готова его принять.
  if (strip.canShow()) {
    
    // --- 1. Рисуем кадр ---
    for (uint16_t i = 0; i < strip.getNumLeds(); i++) {
      // Рассчитываем Тон (Hue) для каждого пикселя индивидуально.
      // Формула ниже растягивает одну полную радугу (256 значений Hue)
      // на всю длину ленты (NUM_LEDS).
      uint8_t pixel_hue = (i * 256 / strip.getNumLeds());
      
      // Добавляем сдвиг, чтобы анимация двигалась.
      // Результат автоматически "завернется" в диапазон 0-255, 
      // так как переменная имеет тип uint8_t.
      pixel_hue += hue_offset;
      
      // Устанавливаем цвет пикселя, используя рассчитанный Тон.
      // Насыщенность и Яркость ставим на максимум для самого яркого эффекта.
      strip.setPixelHSV(i, pixel_hue, 255, 255);
    }

    // --- 2. Обновляем состояние анимации для следующего кадра ---
    hue_offset++;
    
    // --- 3. Отправляем готовый кадр на ленту ---
    strip.show();
  }
}