/**
 * @file Night_Sky.ino
 * @brief Пример использования функций fillColor, setPixelColor и цветовых констант.
 * 
 * Этот скетч демонстрирует, как создавать сложные эффекты, комбинируя
 * управление фоном и отдельными элементами с помощью новых функций.
 * 
 * Эффект "Ночное небо":
 * 1. Вся лента залита темным синим цветом с помощью `fillColor()`.
 * 2. Несколько случайных пикселей ("звезд") плавно меняют свою яркость,
 *    мерцая белым цветом. Для этого используется `setPixelColor()`.
 *
 * =========================================================================
 *                    Таблица предопределенных цветов (SavaLED_ESP32)
 * =========================================================================
 *   КАТЕГОРИЯ          | НАЗВАНИЕ    | HEX КОД   | ОПИСАНИЕ
 * ---------------------|-------------|-----------|-------------------------
 *   ОСНОВНЫЕ (ЯРКИЕ)   | RED         | 0xFF0000  | Красный
 *                      | LIME        | 0x00FF00  | Лайм (Ярко-зеленый)
 *                      | BLUE        | 0x0000FF  | Синий
 * ---------------------|-------------|-----------|-------------------------
 *   ВТОРИЧНЫЕ (ЯРКИЕ)  | YELLOW      | 0xFFFF00  | Желтый
 *                      | CYAN        | 0x00FFFF  | Голубой (Циан)
 *                      | MAGENTA     | 0xFF00FF  | Пурпурный (Маджента)
 * ---------------------|-------------|-----------|-------------------------
 *   ТЕПЛЫЕ ОТТЕНКИ     | ORANGE      | 0xFF7F00  | Оранжевый
 *                      | GOLD        | 0xFFD700  | Золотой
 *                      | CRIMSON     | 0xDC143C  | Малиновый
 *                      | PINK        | 0xFF69B4  | Ярко-розовый
 *                      | BROWN       | 0xA52A2A  | Коричневый
 *                      | MAROON      | 0x800000  | Бордовый
 * ---------------------|-------------|-----------|-------------------------
 *   ХОЛОДНЫЕ ОТТЕНКИ   | GREEN       | 0x008000  | Зеленый (Лесной)
 *                      | OLIVE       | 0x808000  | Оливковый
 *                      | TEAL        | 0x008080  | Бирюзовый
 *                      | SKYBLUE     | 0x87CEEB  | Небесно-голубой
 *                      | NAVY        | 0x000080  | Темно-синий
 *                      | INDIGO      | 0x4B0082  | Индиго
 *                      | PURPLE      | 0x800080  | Лиловый
 *                      | VIOLET      | 0xEE82EE  | Фиалковый
 * ---------------------|-------------|-----------|-------------------------
 *   ОТТЕНКИ СЕРОГО     | WHITE       | 0xFFFFFF  | Белый
 *                      | SILVER      | 0xC0C0C0  | Серебряный
 *                      | GRAY        | 0x808080  | Серый
 *                      | BLACK       | 0x000000  | Черный (Выкл)
 * =========================================================================
 * 
 * 
 * Пример: strip.fillColor(ORANGE, 255);
 *
 *
 * 2. УПРАВЛЕНИЕ ЦВЕТОМ ЧЕРЕЗ HSV (Тон, Насыщенность, Яркость)
 * ------------------------------------------------------------------------------------
 * Идеально для плавных цветовых переходов и радужных эффектов.
 * 
 * Функция: fillHSV(hue, saturation, value)
 *   - hue (Тон): 0-255, определяет цвет на цветовом круге.
 *   - saturation (Насыщенность): 0-255. 255 = сочный цвет, 0 = белый/серый.
 *   - value (Яркость): 0-255. 255 = максимальная яркость, 0 = черный.
 *
 *   Ключевые значения `hue` для чистых цветов:
 *   --------------------------------
 *     HUE | ЦВЕТ
 *   --------------------------------
 *       0 | Красный
 *      21 | Оранжевый
 *      42 | Желтый
 *      64 | Салатовый
 *      85 | Зеленый
 *     106 | Бирюзовый
 *     127 | Голубой (Cyan)
 *     149 | Лазурный
 *     170 | Синий
 *     191 | Фиолетовый
 *     212 | Пурпурный (Magenta)
 *     234 | Розовый
 *     255 | Красный (цикл)
 *   --------------------------------
 * 
 * Пример: strip.fillHSV(170, 255, 255); // Ярко-синий
 * 
 *
 * 3. ДРУГИЕ ПОЛЕЗНЫЕ ФУНКЦИИ
 * ------------------------------------------------------------------------------------
 *   strip.Color(r, g, b);        // Создает 32-битный цвет из RGB компонентов.
 *   strip.fill(r, g, b);         // Заливает ленту цветом RGB.
 *   strip.setPixel(n, r, g, b);  // Устанавливает цвет одного пикселя.
 *   strip.setBrightness(br);     // Устанавливает ГЛОБАЛЬНУЮ яркость для всей ленты.
 *
 * ====================================================================================
 */

#include <SavaLED_ESP32.h>

// --- Конфигурация ---
#define LED_PIN    14     // GPIO пин
#define NUM_LEDS   100    // Количество светодиодов

// Количество одновременно мерцающих звезд
#define NUM_STARS 5

// Скорость анимации (кадров в секунду)
#define ANIMATION_FPS 100
const int FRAME_INTERVAL_MS = 1000 / ANIMATION_FPS;

// Создаем экземпляр библиотеки
SavaLED_ESP32 strip;

// Структура для хранения информации о каждой "звезде"
struct Star {
  int index;          // Позиция звезды на ленте
  int brightness;     // Текущая яркость звезды
  int step;           // Шаг и направление изменения яркости
};

// Создаем массив для наших звезд
Star stars[NUM_STARS];

// Переменные для управления анимацией
unsigned long lastUpdateTime = 0;

// Прототип функции. Объявляем компилятору, что такая функция существует.
void initializeStar(int starIndex);
void updateScene();


void setup() {
  Serial.begin(115200);
  Serial.println("\nПример SavaLED_ESP32: Ночное небо");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    Serial.println("Ошибка инициализации библиотеки SavaLED_ESP32!");
    while (true);
  }

  // Используем глобальную яркость для всей сцены
  strip.setBrightness(150);

  // Инициализируем наши звезды в случайных позициях
  for (int i = 0; i < NUM_STARS; i++) {
    initializeStar(i);
  }
}

void loop() {
  // Проверяем, пора ли обновить следующий кадр анимации
  if (millis() - lastUpdateTime >= FRAME_INTERVAL_MS) {
    lastUpdateTime = millis();

    // Обновляем состояние всех звезд и фона
    updateScene();
    
    // Отправляем данные на ленту, если она готова
    if (strip.canShow()) {
      strip.show();
    }
  }
}

/**
 * @brief Обновляет весь кадр: фон и все звезды. (ИСПРАВЛЕННАЯ ВЕРСИЯ)
 */
void updateScene() {
  // --- 1. Рисуем фон ---
  // Используем fillColor() с повышенной яркостью, чтобы фон был виден после гамма-коррекции
  strip.fillColor(BLUE, 50);

  // --- 2. Рисуем звезды поверх фона ---
  for (int i = 0; i < NUM_STARS; i++) {
    // Обновляем яркость звезды
    stars[i].brightness += stars[i].step;
    
    // Если звезда достигла пика яркости, она начинает гаснуть
    if (stars[i].brightness >= 255) {
      stars[i].brightness = 255; // Ограничиваем сверху
      stars[i].step = -abs(stars[i].step); // Меняем направление на "угасание"
    } 
    // Если звезда погасла, создаем новую в другом месте
    else if (stars[i].brightness <= 0) {
      initializeStar(i); // Перерождаем звезду
    }

    // Используем setPixelColor() для отрисовки звезды.
    strip.setPixelColor(stars[i].index, WHITE, stars[i].brightness);
  }
}

/**
 * @brief Инициализирует одну звезду в случайном месте с случайными параметрами.
 * @param starIndex Индекс звезды в массиве `stars`.
 */
void initializeStar(int starIndex) {
  stars[starIndex].index = random(NUM_LEDS);
  stars[starIndex].brightness = 0;
  // Устанавливаем случайную скорость "разгорания" звезды
  stars[starIndex].step = random(2, 7); 
}