/**
 * @file Breathing_Rainbow.ino
 * @brief Пример использования новых HSV функций в библиотеке SavaLED_ESP32.
 * 
 * Этот скетч демонстрирует, насколько просто создавать плавные цветовые переходы
 * с помощью цветовой модели HSV (Тон, Насыщенность, Яркость).
 * 
 * Эффект: Вся лента целиком плавно переливается всеми цветами радуги,
 * создавая эффект "дыхания".
 * 
 * Это достигается простым инкрементом переменной "Тон" (Hue) в каждой итерации.
 */
/**
 * =========================================================================
 *                    Шпаргалка по цветам для HSV
 * =========================================================================
 * 
 * Функция fillHSV(hue, saturation, value) управляет цветом всей ленты.
 * - hue (Тон): 0-255, определяет сам цвет на цветовом круге.
 * - saturation (Насыщенность): 0-255. 255 = самый сочный цвет, 0 = белый.
 * - value (Яркость): 0-255. 255 = максимальная яркость, 0 = черный.
 *
 * Ниже приведены ключевые значения `hue` для получения чистых цветов
 * при максимальной насыщенности и яркости (например, fillHSV(hue, 255, 255)).
 *
 * -------------------------------------------------------------------------
 *   HUE | ЦВЕТ
 * -------------------------------------------------------------------------
 *     0 | Красный
 *    21 | Оранжевый
 *    42 | Желтый
 *    64 | Салатовый
 *    85 | Зеленый
 *   106 | Бирюзовый
 *   127 | Голубой (Cyan)
 *   149 | Лазурный
 *   170 | Синий
 *   191 | Фиолетовый
 *   212 | Пурпурный (Magenta)
 *   234 | Розовый
 *   255 | Красный (цикл замыкается)
 * -------------------------------------------------------------------------
 *
 * Пример использования:
 * 
 *   // Установить ярко-синий цвет на всю ленту
 *   strip.fillHSV(170, 255, 255); 
 *   strip.show();
 *
 *   // Установить пастельный, ненасыщенный красный
 *   strip.fillHSV(0, 128, 255);
 *   strip.show();
 *
 * =========================================================================
 */
#include <SavaLED_ESP32.h>

// --- Конфигурация ---
#define LED_PIN    14     // GPIO пин, к которому подключена лента
#define NUM_LEDS   100    // Количество светодиодов в вашей ленте
#define BRIGHTNESS 100    // Яркость ленты

// Устанавливаем желаемую скорость смены цветов (в кадрах в секунду)
#define ANIMATION_FPS 50
const int FRAME_INTERVAL_MS = 1000 / ANIMATION_FPS;

// Создаем экземпляр библиотеки SavaLED_ESP32
SavaLED_ESP32 strip;

// Переменные для управления анимацией
unsigned long lastUpdateTime = 0; // Время последнего обновления кадра
uint8_t currentHue = 0;           // Текущий Тон (Hue), 0-255

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример SavaLED_ESP32: Дышащая Радуга (HSV)");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    Serial.println("Ошибка инициализации библиотеки SavaLED_ESP32!");
    while (true);
  }

  Serial.println("Библиотека инициализирована. Начинаем анимацию...");
  
  // Устанавливаем общую яркость.
  // Обратите внимание: у нас есть ДВЕ яркости.
  // 1. Компонент 'V' (Value) в HSV - управляет яркостью самого цвета.
  // 2. Глобальная яркость setBrightness() - масштабирует итоговый RGB цвет.
  // Это дает дополнительную гибкость. Здесь мы используем глобальную.
  strip.setBrightness(BRIGHTNESS);
  
  // Очищаем ленту перед стартом
  strip.clear();
  strip.show();
}

void loop() {
  // Проверяем, пора ли обновить следующий кадр анимации
  if (millis() - lastUpdateTime >= FRAME_INTERVAL_MS) {
    
    // Проверяем, готова ли лента принять новые данные
    if (strip.canShow()) {
      lastUpdateTime = millis(); // Запоминаем время обновления
      
      // === Вот вся магия HSV ===
      
      // Заливаем всю ленту цветом, который определяется текущим 'currentHue'.
      // Насыщенность (s=255) и Яркость цвета (v=255) ставим на максимум.
      strip.fillHSV(currentHue, 255, 255);
      
      // Отправляем данные на ленту.
      strip.show();
      
      // Увеличиваем Тон для следующего кадра.
      // Когда currentHue достигнет 255 и переполнится, он снова станет 0,
      // и радуга начнется заново - идеальный цикл!
      currentHue++;
    }
  }
}