/**
 * @file Fill_And_Pulse.ino
 * @brief Пример использования функций fill() и Color() в библиотеке SavaLED_ESP32.
 * 
 * Этот скетч демонстрирует, как легко управлять цветом всей ленты
 * с помощью новых функций-помощников.
 * 
 * Эффект:
 * 1. Вся лента заливается одним цветом (например, красным).
 * 2. Яркость ленты плавно нарастает до максимума и плавно спадает до нуля (пульсация).
 * 3. После завершения пульсации цвет меняется на следующий (зеленый, затем синий),
 *    и цикл повторяется.
 *
 * Все анимации реализованы без использования delay().
 *
 * =========================================================================
 *                    Таблица предопределенных цветов (SavaLED_ESP32)
 * =========================================================================
 *   КАТЕГОРИЯ          | НАЗВАНИЕ    | HEX КОД   | ОПИСАНИЕ
 * ---------------------|-------------|-----------|-------------------------
 *   ОСНОВНЫЕ (ЯРКИЕ)   | RED         | 0xFF0000  | Красный
 *                      | LIME        | 0x00FF00  | Лайм (Ярко-зеленый)
 *                      | BLUE        | 0x0000FF  | Синий
 * ---------------------|-------------|-----------|-------------------------
 *   ВТОРИЧНЫЕ (ЯРКИЕ)  | YELLOW      | 0xFFFF00  | Желтый
 *                      | CYAN        | 0x00FFFF  | Голубой (Циан)
 *                      | MAGENTA     | 0xFF00FF  | Пурпурный (Маджента)
 * ---------------------|-------------|-----------|-------------------------
 *   ТЕПЛЫЕ ОТТЕНКИ     | ORANGE      | 0xFF7F00  | Оранжевый
 *                      | GOLD        | 0xFFD700  | Золотой
 *                      | CRIMSON     | 0xDC143C  | Малиновый
 *                      | PINK        | 0xFF69B4  | Ярко-розовый
 *                      | BROWN       | 0xA52A2A  | Коричневый
 *                      | MAROON      | 0x800000  | Бордовый
 * ---------------------|-------------|-----------|-------------------------
 *   ХОЛОДНЫЕ ОТТЕНКИ   | GREEN       | 0x008000  | Зеленый (Лесной)
 *                      | OLIVE       | 0x808000  | Оливковый
 *                      | TEAL        | 0x008080  | Бирюзовый
 *                      | SKYBLUE     | 0x87CEEB  | Небесно-голубой
 *                      | NAVY        | 0x000080  | Темно-синий
 *                      | INDIGO      | 0x4B0082  | Индиго
 *                      | PURPLE      | 0x800080  | Лиловый
 *                      | VIOLET      | 0xEE82EE  | Фиалковый
 * ---------------------|-------------|-----------|-------------------------
 *   ОТТЕНКИ СЕРОГО     | WHITE       | 0xFFFFFF  | Белый
 *                      | SILVER      | 0xC0C0C0  | Серебряный
 *                      | GRAY        | 0x808080  | Серый
 *                      | BLACK       | 0x000000  | Черный (Выкл)
 * =========================================================================
 * 
 * Пример: strip.fillColor(ORANGE, 255);
 *
 *
 * 2. УПРАВЛЕНИЕ ЦВЕТОМ ЧЕРЕЗ HSV (Тон, Насыщенность, Яркость)
 * ------------------------------------------------------------------------------------
 * Идеально для плавных цветовых переходов и радужных эффектов.
 * 
 * Функция: fillHSV(hue, saturation, value)
 *   - hue (Тон): 0-255, определяет цвет на цветовом круге.
 *   - saturation (Насыщенность): 0-255. 255 = сочный цвет, 0 = белый/серый.
 *   - value (Яркость): 0-255. 255 = максимальная яркость, 0 = черный.
 *
 *   Ключевые значения `hue` для чистых цветов:
 *   --------------------------------
 *     HUE | ЦВЕТ
 *   --------------------------------
 *       0 | Красный
 *      21 | Оранжевый
 *      42 | Желтый
 *      64 | Салатовый
 *      85 | Зеленый
 *     106 | Бирюзовый
 *     127 | Голубой (Cyan)
 *     149 | Лазурный
 *     170 | Синий
 *     191 | Фиолетовый
 *     212 | Пурпурный (Magenta)
 *     234 | Розовый
 *     255 | Красный (цикл)
 *   --------------------------------
 * 
 * Пример: strip.fillHSV(170, 255, 255); // Ярко-синий
 * 
 *
 * 3. ДРУГИЕ ПОЛЕЗНЫЕ ФУНКЦИИ
 * ------------------------------------------------------------------------------------
 *   strip.Color(r, g, b);        // Создает 32-битный цвет из RGB компонентов.
 *   strip.fill(r, g, b);         // Заливает ленту цветом RGB.
 *   strip.setPixel(n, r, g, b);  // Устанавливает цвет одного пикселя.
 *   strip.setBrightness(br);     // Устанавливает ГЛОБАЛЬНУЮ яркость для всей ленты.
 *
 * ====================================================================================
 */

#include <SavaLED_ESP32.h>

// --- Конфигурация ---
#define LED_PIN    14                                                   // GPIO пин
#define NUM_LEDS   100                                                  // Количество светодиодов
#define MAX_BRIGHTNESS 200                                              // Максимальная яркость для эффекта пульсации (0-255)

// Устанавливаем желаемую скорость анимации (кадров в секунду)
#define ANIMATION_FPS 100
const int FRAME_INTERVAL_MS = 1000 / ANIMATION_FPS;

// Создаем экземпляр библиотеки SavaLED_ESP32
SavaLED_ESP32 strip;

// Переменные для управления анимацией
unsigned long lastUpdateTime = 0;                                       // Время последнего обновления кадра

// Массив цветов для переключения. Используем новую функцию strip.Color()
uint32_t colors[] = {
  strip.Color(255, 0, 0),   // Красный
  strip.Color(0, 255, 0),   // Зеленый
  strip.Color(0, 0, 255)    // Синий
};
int currentColorIndex = 0; // Индекс текущего цвета в массиве

// Переменные для эффекта пульсации
int brightnessStep = 5;                                                 // Шаг изменения яркости
int currentBrightness = 0;

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример SavaLED_ESP32: Заливка и Пульсация");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    Serial.println("Ошибка инициализации библиотеки SavaLED_ESP32!");
    while (true);
  }

  // Устанавливаем начальный цвет. Используем новую функцию fill().
  strip.fill(colors[currentColorIndex]);
  strip.setBrightness(0);                                                 // Начинаем с выключенной ленты
  strip.setGammaCorrection(true);                                         //гамма-корекция(true\false)
  strip.show();
}

void loop() {
  // Проверяем, пора ли обновить следующий кадр анимации
  if (millis() - lastUpdateTime >= FRAME_INTERVAL_MS) {
    lastUpdateTime = millis();
    
    // Обновляем логику пульсации
    updatePulse();

    // Обновляем ленту, если она готова
    if (strip.canShow()) {
      // Устанавливаем текущую вычисленную яркость
      strip.setBrightness(currentBrightness);
      strip.show();
    }
  }
}

/**
 * @brief Обновляет логику эффекта "пульсация".
 * Изменяет currentBrightness и переключает цвет при необходимости.
 */
void updatePulse() {
  // Двигаем яркость вверх или вниз
  currentBrightness += brightnessStep;

  // Проверяем, не достигли ли мы границ (0 или MAX_BRIGHTNESS)
  if (currentBrightness >= MAX_BRIGHTNESS) {
    currentBrightness = MAX_BRIGHTNESS; // Ограничиваем сверху
    brightnessStep = -brightnessStep;   // Меняем направление (начинаем затухать)
  } 
  else if (currentBrightness <= 0) {
    currentBrightness = 0;              // Ограничиваем снизу
    brightnessStep = -brightnessStep;   // Меняем направление (начинаем загораться)

    // === ПЕРЕКЛЮЧАЕМ ЦВЕТ ===
    // Когда яркость упала до нуля, меняем цвет для следующей пульсации
    currentColorIndex++;
    if (currentColorIndex >= (sizeof(colors) / sizeof(colors[0]))) {
      currentColorIndex = 0; // Возвращаемся к первому цвету
    }
    
    // Используем новую функцию fill(), чтобы мгновенно изменить цвет в буфере
    strip.fill(colors[currentColorIndex]);
  }
}