/**
 * @file 07_SetPixelColor_Brightness.ino
 * @brief ПРОДВИНУТЫЙ пример: Индивидуальная яркость пикселей с помощью sin().
 * 
 * Этот скетч демонстрирует мощную функцию `setPixelColor(index, COLOR, brightness)`.
 * Она позволяет взять любой чистый цвет (например, RED) и автоматически
 * "затемнить" его до нужного уровня яркости перед установкой в буфер.
 * 
 * Эффект:
 * Тот же паттерн [4R, 4G, 4B], но теперь вся лента плавно "дышит",
 * меняя свою яркость от 0% до 100% и обратно. Анимация яркости
 * создается с помощью математической функции sin(), что дает идеально
 * плавную, органическую пульсацию.
 * 
 * АРХИТЕКТУРА:
 * Мы используем правильный паттерн `if (strip.canShow()) { ... }`.
 * Внутри этого блока мы обновляем математику анимации И рисуем кадр.
 * Это гарантирует, что анимация будет плавной и синхронизированной
 * с реальной частотой обновления ленты.
 * 
 * =================================================================================
 *                    Как работает анимация через sin(millis())
 * =================================================================================
 * 
 * 1. `millis()`:
 *    Эта функция возвращает количество миллисекунд с момента запуска ESP32.
 *    Это постоянно растущее число (0, 1, 2, 3... 1000, 1001...). Оно служит
 *    нашей "временной шкалой" для анимации.
 * 
 * 2. `millis() / 1000.0`:
 *    Мы делим время в миллисекундах на 1000.0 (число с плавающей точкой),
 *    чтобы получить время в секундах. Это замедляет анимацию до комфортной
 *    скорости. Чем больше это число (например, 2000.0), тем медленнее "дыхание".
 *    
 * 3. `sin(...)`:
 *    Тригонометрическая функция синус. Если ей на вход подавать постоянно
 *    растущее число (наше время в секундах), на выходе она будет генерировать
 *    плавно колеблющееся значение в диапазоне от -1.0 до +1.0.
 *    Это и есть наша "волна" пульсации.
 * 
 *    | Время -> | 0 | ... | 1.57 | ... | 3.14 | ... | 4.71 | ... | 6.28 |
 *    |----------|---|-----|------|-----|------|-----|------|-----|------|
 *    | sin() -> | 0 |  /  |  1.0 |  \  |   0  |  \  | -1.0 |  /  |   0  |
 * 
 * 4. `(sin(...) + 1.0)`:
 *    Мы сдвигаем диапазон волны. Вместо [-1.0 ... +1.0], он становится [0.0 ... +2.0].
 *    Это нужно, чтобы убрать отрицательные значения, так как яркость не может быть
 *    отрицательной.
 * 
 * 5. `* 127.5`:
 *    Мы масштабируем наш диапазон [0.0 ... +2.0] до нужного нам диапазона
 *    яркости [0 ... 255].
 * 
 * В ИТОГЕ: `current_brightness` автоматически и плавно "плавает" между 0 и 255,
 * следуя идеальной синусоидальной кривой, что создает очень естественный
 * эффект "дыхания".
 * =================================================================================
 */

#include <SavaLED_ESP32.h>

#define LED_PIN    14
#define NUM_LEDS   100

SavaLED_ESP32 strip;

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример 06: Индивидуальная яркость (sin() версия)");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    while (true);
  }
  // Глобальную яркость ставим на максимум, чтобы она не мешала нашим эффектам
  strip.setBrightness(255);
}

void loop() {
  // ПРАВИЛЬНЫЙ ПАТТЕРН СИНХРОНИЗАЦИИ:
  // Вся логика анимации выполняется ТОЛЬКО тогда, когда лента готова её показать.
  if (strip.canShow()) {
    
    // --- 1. Рассчитываем текущую яркость для этого кадра ---
    // Вся магия плавной пульсации заключена в этой одной строке.
    // Смотрите подробное описание в шапке файла.
    // Для изменения скорости "дыхания" измените делитель (например, 500.0 - быстрее, 2000.0 - медленнее)
    uint8_t current_brightness = (uint8_t)((sin(millis() / 1000.0) + 1.0) * 127.5);
    
    // --- 2. Рисуем кадр с новой яркостью ---
    for (uint16_t i = 0; i < strip.getNumLeds(); i++) {
      int pos = i % 12;
      uint32_t baseColor;

      // Выбираем базовый чистый цвет
      if (pos < 4) baseColor = RED;
      else if (pos < 8) baseColor = GREEN;
      else baseColor = BLUE;
      
      // ГЛАВНАЯ ФИШКА: устанавливаем цвет с применением текущей яркости анимации.
      strip.setPixelColor(i, baseColor, current_brightness);
    }

    // --- 3. Отправляем кадр ---
    strip.show();
  }
}