/**
 * @file 05_SetPixel_PackedColor.ino
 * @brief Пример: Использование готовых цветовых констант.
 * 
 * Этот пример делает ТОТ ЖЕ узор (4R, 4G, 4B), но использует более
 * удобный и профессиональный способ задания цвета - через 32-битные
 * "упакованные" константы.
 * 
 * Библиотека SavaLED_ESP32 уже содержит встроенные названия для основных цветов,
 * чтобы вам не приходилось помнить их RGB коды.
 * 
 * Эффект:
 * Тот же паттерн [RRRR GGGG BBBB], но код стал чище и понятнее.
 * 
 *
 * =========================================================================
 *                    Таблица предопределенных цветов (SavaLED_ESP32)
 * =========================================================================
 *   КАТЕГОРИЯ          | НАЗВАНИЕ    | HEX КОД   | ОПИСАНИЕ
 * ---------------------|-------------|-----------|-------------------------
 *   ОСНОВНЫЕ (ЯРКИЕ)   | RED         | 0xFF0000  | Красный
 *                      | LIME        | 0x00FF00  | Лайм (Ярко-зеленый)
 *                      | BLUE        | 0x0000FF  | Синий
 * ---------------------|-------------|-----------|-------------------------
 *   ВТОРИЧНЫЕ (ЯРКИЕ)  | YELLOW      | 0xFFFF00  | Желтый
 *                      | CYAN        | 0x00FFFF  | Голубой (Циан)
 *                      | MAGENTA     | 0xFF00FF  | Пурпурный (Маджента)
 * ---------------------|-------------|-----------|-------------------------
 *   ТЕПЛЫЕ ОТТЕНКИ     | ORANGE      | 0xFF7F00  | Оранжевый
 *                      | GOLD        | 0xFFD700  | Золотой
 *                      | CRIMSON     | 0xDC143C  | Малиновый
 *                      | PINK        | 0xFF69B4  | Ярко-розовый
 *                      | BROWN       | 0xA52A2A  | Коричневый
 *                      | MAROON      | 0x800000  | Бордовый
 * ---------------------|-------------|-----------|-------------------------
 *   ХОЛОДНЫЕ ОТТЕНКИ   | GREEN       | 0x008000  | Зеленый (Лесной)
 *                      | OLIVE       | 0x808000  | Оливковый
 *                      | TEAL        | 0x008080  | Бирюзовый
 *                      | SKYBLUE     | 0x87CEEB  | Небесно-голубой
 *                      | NAVY        | 0x000080  | Темно-синий
 *                      | INDIGO      | 0x4B0082  | Индиго
 *                      | PURPLE      | 0x800080  | Лиловый
 *                      | VIOLET      | 0xEE82EE  | Фиалковый
 * ---------------------|-------------|-----------|-------------------------
 *   ОТТЕНКИ СЕРОГО     | WHITE       | 0xFFFFFF  | Белый
 *                      | SILVER      | 0xC0C0C0  | Серебряный
 *                      | GRAY        | 0x808080  | Серый
 *                      | BLACK       | 0x000000  | Черный (Выкл)
 * =========================================================================
 * 
 */

#include <SavaLED_ESP32.h>

#define LED_PIN    14
#define NUM_LEDS   100
#define BRIGHTNESS 150

SavaLED_ESP32 strip;

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример 05: Цветовые константы");

  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    while (true);
  }
  strip.setBrightness(BRIGHTNESS);
}

void loop() {
  // ПРАВИЛЬНЫЙ ПАТТЕРН:
  // 1. Ждем готовности аппаратуры.
  if (strip.canShow()) {
    
    // 2. Аппаратура готова. Рисуем кадр, используя удобные константы.
    for (uint16_t i = 0; i < strip.getNumLeds(); i++) {
      int pos = i % 12;

      if (pos < 4) {
        strip.setPixel(i, RED);   // Используем готовую константу RED
      } else if (pos < 8) {
        strip.setPixel(i, GREEN); // Используем GREEN
      } else {
        strip.setPixel(i, BLUE);  // Используем BLUE
      }
    }

    // 3. Отправляем готовый кадр.
    strip.show();
  }
}