/**
 * @file 05_SetPixel_RGB.ino
 * @brief Базовый пример: Управление пикселями через RGB.
 * 
 * Этот пример демонстрирует фундаментальный метод работы с библиотекой:
 * установку цвета отдельного светодиода с помощью трех компонентов:
 * Красного (R), Зеленого (G) и Синего (B).
 * 
 * Эффект:
 * Лента заполняется повторяющимся узором: [4 красных, 4 зеленых, 4 синих] и т.д.
 * 
 * АРХИТЕКТУРА (ВАЖНО):
 * Пример демонстрирует правильный паттерн использования неблокирующей библиотеки в loop():
 * 1. Сначала спрашиваем: готова ли лента принять данные? (strip.canShow())
 * 2. ЕСЛИ готова -> выполняем расчеты и отрисовку нового кадра.
 * 3. Сразу же отправляем этот кадр на ленту (strip.show()).
 * 
 * Это экономит ресурсы процессора, не выполняя бесполезную работу, когда лента еще занята.
 * 
 * =========================================================================
 *                    Шпаргалка по цветам RGB (0-255)
 * =========================================================================
 *   ЦВЕТ      |  R (Красный) |  G (Зеленый) |  B (Синий)
 * ------------|--------------|--------------|-------------
 *   ЧЕРНЫЙ    |      0       |      0       |      0
 *   КРАСНЫЙ   |     255      |      0       |      0
 *   ЗЕЛЕНЫЙ   |      0       |     255      |      0
 *   СИНИЙ     |      0       |      0       |     255
 *   ЖЕЛТЫЙ    |     255      |     255      |      0
 *   ГОЛУБОЙ   |      0       |     255      |     255
 *   ПУРПУРНЫЙ |     255      |      0       |     255
 *   БЕЛЫЙ     |     255      |     255      |     255
 * =========================================================================
 */

#include <SavaLED_ESP32.h>

// --- Конфигурация ---
#define LED_PIN    14     // Пин подключения (DIN)
#define NUM_LEDS   100    // Количество светодиодов
#define BRIGHTNESS 150    // Общая яркость (0-255)

SavaLED_ESP32 strip;

void setup() {
  Serial.begin(115200);
  Serial.println("\nПример 04: SetPixel RGB");

  // Инициализация библиотеки
  if (!strip.begin(NUM_LEDS, LED_PIN)) {
    Serial.println("Ошибка! Не удалось инициализировать ленту.");
    while (true);
  }
  
  strip.setBrightness(BRIGHTNESS);
}

void loop() {
  // ГЛАВНЫЙ ПАТТЕРН РАБОТЫ:
  // Сначала проверяем, свободна ли лента для приема нового кадра.
  if (strip.canShow()) {
    
    // --- 1. Лента свободна. Начинаем рисовать новый кадр. ---
    // Проходим циклом по всем светодиодам.
    for (uint16_t i = 0; i < strip.getNumLeds(); i++) {
      // Определяем позицию внутри повторяющегося блока из 12 пикселей
      int pos = i % 12;

      if (pos < 4) {
        // Первые 4 пикселя - Красные (R=255, G=0, B=0)
        strip.setPixel(i, 255, 0, 0);
      } else if (pos < 8) {
        // Следующие 4 пикселя - Зеленые (R=0, G=255, B=0)
        strip.setPixel(i, 0, 255, 0);
      } else {
        // Последние 4 пикселя - Синие (R=0, G=0, B=255)
        strip.setPixel(i, 0, 0, 255);
      }
    }

    // --- 2. Кадр нарисован. Немедленно отправляем его на ленту. ---
    strip.show();
  }

  // Здесь может выполняться любой другой код.
  // Он не будет блокироваться работой с лентой.
}