#pragma once
#include <Arduino.h>
#include <tensorflow-runtime-universal.h>

#ifdef __has_attribute
#define HAVE_ATTRIBUTE(x) __has_attribute(x)
#else
#define HAVE_ATTRIBUTE(x) 0
#endif
#if HAVE_ATTRIBUTE(aligned) || (defined(__GNUC__) && !defined(__clang__))
#define DATA_ALIGN_ATTRIBUTE __attribute__((aligned(4)))
#else
#define DATA_ALIGN_ATTRIBUTE
#endif

namespace tinyml4all {
    template<size_t tensorArenaSize>
    class CNN2D {
        public:
            const size_t numInputs = 6912;
            const size_t numOutputs = 2;
            const size_t numOps = 8;
            

            const Model *model;
            MicroMutableOpResolver<8> resolver;
            MicroInterpreter *interpreter;
            TfLiteTensor *in;
            TfLiteTensor *out;
            uint8_t arena[tensorArenaSize];
            float outputs[2];
            float confidence;
            String label;
            size_t runtime_ms;

            /**
             * Constructor
             */
            CNN2D() :
                model(nullptr),
                interpreter(nullptr),
                in(nullptr),
                out(nullptr),
                label(""),
                _registered(false),
                _error("") {

                for (uint8_t i = 0; i < numOutputs; i++)
                    outputs[i] = 0;
            }

            /**
             * Test if model failed
             */
            inline bool failed() {
                return _error.length() > 0;
            }

            /**
             * Get error message
             */
            inline String error() {
                return _error;
            }

            /**
             * Test if input is float
             */
            inline bool isFloatInput() const {
                return in->type == kTfLiteFloat32;
            }

            /**
             * Test if input is int8
             */
            inline bool isInt8Input() const {
                return in->type == kTfLiteInt8;
            }

            /**
             * Test if output is float
             */
            inline bool isFloatOutput() const {
                return out->type == kTfLiteFloat32;
            }

            /**
             * Test if output is int8
             */
            inline bool isInt8Output() const {
                return out->type == kTfLiteInt8;
            }

            /**
             * Get i-th output
             */
            float output(uint16_t i = 0) {
                if (outputs == NULL || i >= numOutputs)
                    return sqrt(-1);

                return outputs[i];
            }

            /**
             * Init model
             */
            bool begin() {
                registerOps();
                model = tflite::GetModel(data);

                if (model->version() != TFLITE_SCHEMA_VERSION)
                    return fail(String("Model version mismatch. Expected ") + TFLITE_SCHEMA_VERSION + ", got " + model->version());

                interpreter = tf::newInterpreter<8>(&resolver, model, arena, tensorArenaSize);

                if (interpreter->AllocateTensors() != kTfLiteOk)
                    return fail("AllocateTensors() failed");

                in = interpreter->input(0);
                out = interpreter->output(0);

                if (!isFloatInput() && !isInt8Input())
                    return fail("Only float and int8 inputs are supported");

                if (!isFloatOutput() && !isInt8Output())
                    return fail("Only float and int8 outputs are supported");

                return fail();
            }

            /**
             * Predict float input
             */
            bool predict(float *x) {
                const float inputScale = in->params.scale == 0 ? 1 : in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale == 0 ? 1 : out->params.scale;
                const float outputOffset = out->params.zero_point;
                const size_t startedAt = millis();

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>((float) x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.f[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();
                runtime_ms = millis() - startedAt;

                return fail();
            }

            /**
             * Predict int8 input
             */
            bool predict(int8_t *x) {
                const float inputScale = in->params.scale == 0 ? 1 : in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale == 0 ? 1 : out->params.scale;
                const float outputOffset = out->params.zero_point;
                const size_t startedAt = millis();

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>((float) x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.f[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();
                runtime_ms = millis() - startedAt;

                return fail();
            }

            /**
             * Predict uint8 (mainly images)
             */
             bool predict(uint8_t *x) {
                const float inputScale = in->params.scale == 0 ? 1 : in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale == 0 ? 1 : out->params.scale;
                const float outputOffset = out->params.zero_point;
                const size_t startedAt = millis();

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>((float) x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.f[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();
                runtime_ms = millis() - startedAt;

                return fail();
             }

             /**
             * Predict uint8 (mainly images)
             */
             bool predict(const uint8_t *x) {
                const float inputScale = in->params.scale == 0 ? 1 : in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale == 0 ? 1 : out->params.scale;
                const float outputOffset = out->params.zero_point;
                const size_t startedAt = millis();

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>((float) x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.f[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();
                runtime_ms = millis() - startedAt;

                return fail();
             }

            /**
             * Convert results to String for debug
             */
            String outputsAsString() {
              String result = "";
              result.reserve(numOutputs * 6);

              for (int i = 0; i < numOutputs; i++) {
                if (i > 0)
                  result += ", ";

                result += outputs[i];
              }

              return result;
            }

        protected:
            bool _registered;
            String _error;

            /**
             * Register ops in resolver
             */
            void registerOps() {
                if (_registered)
                    return;

                resolver.AddDequantize();
                resolver.AddMaxPool2D();
                resolver.AddSoftmax();
                resolver.AddQuantize();
                resolver.AddConv2D();
                resolver.AddFullyConnected();
                resolver.AddReshape();
                resolver.AddShape();
                
                _registered = true;
            }

            /**
             * Set error message, if any
             */
            bool fail(String message = "") {
                _error = message;

                // return true if error is empty (success)
                return _error.length() == 0;
            }

            /**
             * If classification, get most probable class
             */
            void getClassificationResult() {
                if (numOutputs < 2)
                    return;

                confidence = outputs[0];
                uint8_t ix = 0;

                for (uint16_t i = 1; i < numOutputs; i++) {
                    if (outputs[i] > confidence) {
                        ix = i;
                        confidence = outputs[i];
                    }
                }

                
                    label = String(ix);
                
            }

            const unsigned char data[13672] DATA_ALIGN_ATTRIBUTE = { 0x1c, 0x00, 0x00, 0x00, 0x54, 0x46, 0x4c, 0x33, 0x14, 0x00, 0x20, 0x00, 0x1c, 0x00, 0x18, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0xe4, 0x00, 0x00, 0x00, 0xdc, 0x22, 0x00, 0x00, 0xec, 0x22, 0x00, 0x00, 0xb8, 0x34, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x16, 0xdc, 0xff, 0xff, 0x0c, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x73, 0x65, 0x72, 0x76, 0x69, 0x6e, 0x67, 0x5f, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x90, 0xff, 0xff, 0xff, 0x14, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0xdd, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x5f, 0x31, 0x00, 0x02, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xdc, 0xff, 0xff, 0xff, 0x17, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4e, 0x56, 0x45, 0x52, 0x53, 0x49, 0x4f, 0x4e, 0x5f, 0x4d, 0x45, 0x54, 0x41, 0x44, 0x41, 0x54, 0x41, 0x00, 0x08, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x6d, 0x69, 0x6e, 0x5f, 0x72, 0x75, 0x6e, 0x74, 0x69, 0x6d, 0x65, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x18, 0x00, 0x00, 0x00, 0xf4, 0x21, 0x00, 0x00, 0xec, 0x21, 0x00, 0x00, 0xd4, 0x21, 0x00, 0x00, 0xbc, 0x21, 0x00, 0x00, 0x6c, 0x21, 0x00, 0x00, 0xdc, 0x20, 0x00, 0x00, 0xcc, 0x14, 0x00, 0x00, 0x5c, 0x14, 0x00, 0x00, 0xcc, 0x06, 0x00, 0x00, 0x7c, 0x06, 0x00, 0x00, 0xec, 0x01, 0x00, 0x00, 0xbc, 0x01, 0x00, 0x00, 0xd4, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0xc4, 0x00, 0x00, 0x00, 0xbc, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xd6, 0xdd, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0e, 0x00, 0x08, 0x00, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x04, 0x00, 0x06, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xeb, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x32, 0x2e, 0x31, 0x35, 0x2e, 0x31, 0x00, 0x00, 0x3a, 0xde, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x31, 0x2e, 0x31, 0x34, 0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0xcd, 0xff, 0xff, 0xcc, 0xcd, 0xff, 0xff, 0xd0, 0xcd, 0xff, 0xff, 0xd4, 0xcd, 0xff, 0xff, 0xd8, 0xcd, 0xff, 0xff, 0xdc, 0xcd, 0xff, 0xff, 0xe0, 0xcd, 0xff, 0xff, 0xe4, 0xcd, 0xff, 0xff, 0xe8, 0xcd, 0xff, 0xff, 0x7a, 0xde, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0xd8, 0x00, 0x00, 0x00, 0x9d, 0x1a, 0x1a, 0xb3, 0x1b, 0xe4, 0xb6, 0x68, 0xcf, 0x8d, 0x8d, 0x86, 0xe8, 0xf9, 0xc5, 0x14, 0x43, 0xe5, 0xaa, 0x87, 0x8b, 0x2a, 0xe2, 0xc9, 0x7f, 0x17, 0xec, 0xfa, 0x19, 0x96, 0x08, 0x19, 0x65, 0x71, 0xae, 0x6a, 0x5a, 0x96, 0x56, 0xea, 0xdc, 0x03, 0xc9, 0x45, 0xf3, 0x5e, 0x40, 0x9c, 0x76, 0x46, 0xbf, 0x52, 0xdf, 0x81, 0xfd, 0xdc, 0xda, 0x89, 0x04, 0x52, 0xab, 0xfd, 0xd9, 0x81, 0x97, 0x14, 0x5e, 0xe8, 0x60, 0x3b, 0x4b, 0xa7, 0x99, 0xaa, 0xbe, 0x25, 0x66, 0xfd, 0x65, 0xfe, 0x78, 0xa6, 0xa1, 0xc6, 0xaa, 0xed, 0x11, 0x00, 0x2e, 0x7f, 0xed, 0x2f, 0x46, 0x2a, 0x30, 0xaf, 0xf4, 0x51, 0xeb, 0xdb, 0xf8, 0x3e, 0xc4, 0xb8, 0xc9, 0x2c, 0xce, 0x8c, 0x05, 0xf9, 0x5a, 0xc8, 0x40, 0x77, 0x6b, 0x03, 0x8b, 0xcd, 0x21, 0xed, 0x81, 0x98, 0xc1, 0x71, 0xa5, 0xd2, 0x93, 0xa6, 0xcf, 0xc9, 0xac, 0x2d, 0xd4, 0x43, 0xe9, 0x95, 0x81, 0x43, 0xd1, 0x49, 0x3d, 0x5c, 0x66, 0x8a, 0x38, 0xa7, 0xd9, 0x83, 0xdc, 0xca, 0xbe, 0x81, 0xad, 0xf3, 0x8f, 0x6d, 0x87, 0x56, 0xa7, 0x19, 0x45, 0x8b, 0x86, 0xa3, 0x81, 0xb3, 0xb0, 0x06, 0x3d, 0x6b, 0x3a, 0x22, 0x85, 0x5b, 0xf1, 0x02, 0x29, 0x90, 0xff, 0xf0, 0x20, 0x0f, 0x90, 0xd2, 0xfb, 0x07, 0x22, 0x5c, 0x50, 0xce, 0xd6, 0x58, 0x70, 0x8d, 0x3f, 0x19, 0x24, 0xfe, 0x26, 0xde, 0x0a, 0x46, 0x6f, 0xda, 0x14, 0xcb, 0xca, 0x66, 0x39, 0x57, 0x62, 0xe4, 0x3b, 0x92, 0xf2, 0x7f, 0x5e, 0xdf, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xd4, 0x02, 0x00, 0x00, 0xc5, 0xf9, 0xff, 0xff, 0x7a, 0x05, 0x00, 0x00, 0xf9, 0x01, 0x00, 0x00, 0xe6, 0x07, 0x00, 0x00, 0xc3, 0x07, 0x00, 0x00, 0x11, 0x07, 0x00, 0x00, 0x31, 0xf6, 0xff, 0xff, 0x8a, 0xdf, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x80, 0x04, 0x00, 0x00, 0xb9, 0x3a, 0xfd, 0x2b, 0xdf, 0x14, 0xc1, 0xc0, 0x73, 0xb6, 0x30, 0x46, 0x3f, 0x6c, 0x5a, 0x1c, 0xe0, 0x55, 0x06, 0x68, 0xe2, 0x3a, 0xc7, 0xa0, 0x4c, 0xaf, 0x18, 0xb7, 0x13, 0xc3, 0xb3, 0xbc, 0xbc, 0x6c, 0x52, 0x27, 0xd2, 0x37, 0x0a, 0x6f, 0x5a, 0x51, 0xaf, 0xa9, 0x29, 0xf6, 0xe1, 0x16, 0x36, 0x65, 0xc0, 0xb3, 0x8b, 0x1b, 0x60, 0x3a, 0x51, 0xd4, 0x6f, 0xf5, 0x62, 0xb9, 0x06, 0x0e, 0x98, 0xe5, 0x13, 0x90, 0xc5, 0xc1, 0x81, 0x54, 0xca, 0xf8, 0x11, 0x2d, 0xa7, 0x06, 0x43, 0x01, 0x62, 0x19, 0x34, 0x6e, 0x47, 0xd6, 0xa2, 0x22, 0xa3, 0xdc, 0x2e, 0x5e, 0x6a, 0x35, 0xcd, 0x1e, 0xee, 0x11, 0x7a, 0x20, 0xb2, 0xc8, 0xd3, 0x9d, 0xcd, 0x53, 0xcb, 0x11, 0xa8, 0x63, 0xae, 0x51, 0x0c, 0x6f, 0xdf, 0x85, 0x81, 0x8c, 0x72, 0x45, 0x6c, 0x68, 0x7f, 0x74, 0xab, 0xab, 0x35, 0xd2, 0xca, 0x18, 0x60, 0xb6, 0x57, 0x57, 0xe2, 0xaa, 0xe4, 0x48, 0xff, 0x54, 0xc5, 0xbe, 0xab, 0x92, 0xf2, 0x7e, 0xd8, 0x03, 0xad, 0xcb, 0xf8, 0x70, 0x3a, 0x19, 0xfe, 0x44, 0x29, 0xec, 0x96, 0xd1, 0x42, 0xbc, 0xbd, 0x7f, 0x56, 0x8f, 0x9b, 0xa1, 0xbd, 0xd3, 0xe9, 0x77, 0x9f, 0xfd, 0x5c, 0xb7, 0xb3, 0x14, 0x58, 0xec, 0x5a, 0xdd, 0x8a, 0xd3, 0x9f, 0xfa, 0xc9, 0xbd, 0xfa, 0x22, 0x63, 0xcc, 0xaa, 0x8e, 0x55, 0xa5, 0x1c, 0xf3, 0x1a, 0x43, 0x7b, 0xa9, 0xb0, 0x12, 0xe6, 0x77, 0xe0, 0x09, 0xd2, 0x50, 0x1f, 0x0a, 0x44, 0x71, 0x00, 0x9f, 0x6d, 0x36, 0x62, 0xaf, 0x37, 0xce, 0x2c, 0x4d, 0xe3, 0x15, 0x51, 0xba, 0xe7, 0x60, 0x59, 0x01, 0xb2, 0x4f, 0x81, 0xee, 0x66, 0x71, 0x34, 0xac, 0xb0, 0x3f, 0x38, 0x94, 0x99, 0x56, 0x8f, 0x9b, 0xef, 0x37, 0x27, 0xfd, 0x92, 0xdf, 0x05, 0xa3, 0x71, 0xeb, 0xa9, 0xeb, 0xf7, 0xbe, 0xa2, 0x53, 0x40, 0xda, 0x74, 0xfe, 0xa7, 0x1a, 0x92, 0xea, 0x6f, 0xaa, 0x7f, 0x39, 0xfb, 0x12, 0x2c, 0x41, 0x1d, 0x74, 0xec, 0x2f, 0x0a, 0x6e, 0x36, 0x34, 0xff, 0x22, 0xd6, 0xe1, 0x20, 0x81, 0x8f, 0x79, 0x60, 0x0e, 0xab, 0x63, 0xcc, 0xf5, 0xaf, 0x0b, 0xd3, 0x62, 0xf3, 0x23, 0x02, 0xed, 0x54, 0x51, 0xa1, 0xb0, 0xb8, 0x76, 0x76, 0x5e, 0xc7, 0xf0, 0x9c, 0xed, 0xdb, 0xbb, 0x2f, 0x12, 0xa0, 0x02, 0x25, 0x56, 0x7a, 0xa9, 0x21, 0xf3, 0x92, 0x3c, 0x67, 0x25, 0xed, 0x52, 0x54, 0x6f, 0x40, 0xbb, 0x6f, 0xe1, 0x0b, 0x2d, 0x4c, 0x4f, 0xf2, 0x51, 0xd9, 0x25, 0xb5, 0x6e, 0x0b, 0x22, 0xd4, 0xd6, 0xe8, 0x25, 0x67, 0x57, 0x21, 0x10, 0xb4, 0x5e, 0xc6, 0xe8, 0x6a, 0x2e, 0xc1, 0x47, 0x62, 0x3f, 0x03, 0xc2, 0xe2, 0xa5, 0xae, 0xb0, 0xac, 0x57, 0x29, 0xe8, 0x43, 0xb8, 0xe0, 0x61, 0x22, 0x12, 0x18, 0xdf, 0x49, 0xbf, 0x74, 0x2d, 0x05, 0x06, 0xb5, 0x11, 0x0d, 0x4e, 0x14, 0xaf, 0x50, 0xa2, 0x71, 0xca, 0x54, 0xc7, 0x60, 0x6d, 0x5a, 0xe4, 0x51, 0xbf, 0x1b, 0xbc, 0xe1, 0x98, 0x7f, 0xbe, 0xb7, 0x52, 0x58, 0xe7, 0x14, 0x27, 0x67, 0xe4, 0x67, 0x4f, 0xd9, 0xa0, 0x4d, 0xdd, 0x38, 0x2e, 0x4a, 0x51, 0xcd, 0xf1, 0x00, 0xbf, 0xbc, 0x00, 0x3e, 0x10, 0x9a, 0xf5, 0xc7, 0x2e, 0x39, 0x01, 0x6a, 0x21, 0x1c, 0x3f, 0x03, 0x20, 0x3a, 0xc9, 0xa4, 0x56, 0xe1, 0x21, 0xba, 0x15, 0x89, 0xf0, 0x3e, 0xf4, 0xab, 0xee, 0x64, 0xfe, 0x31, 0x4c, 0xfa, 0xbc, 0xce, 0x0e, 0xcf, 0x51, 0xe0, 0x09, 0xc8, 0x61, 0x17, 0x93, 0x08, 0xfc, 0x81, 0x17, 0x2f, 0x29, 0x45, 0xea, 0xa5, 0xed, 0xab, 0xd2, 0x78, 0x43, 0xd3, 0x7d, 0x73, 0x16, 0xa6, 0x2b, 0x0b, 0xef, 0xcd, 0x52, 0x6f, 0x9a, 0xe9, 0x40, 0xac, 0x81, 0x34, 0xe7, 0xff, 0x24, 0x08, 0xa9, 0x23, 0x88, 0xcd, 0xd1, 0x15, 0xe2, 0x1b, 0x13, 0xf5, 0x1e, 0x16, 0x9e, 0xc7, 0xae, 0xcb, 0x56, 0xfb, 0x9d, 0x73, 0x49, 0xf8, 0x0b, 0x5e, 0xe4, 0xb4, 0x69, 0xf2, 0x98, 0x43, 0xe4, 0xb3, 0x2b, 0x76, 0xd8, 0xd5, 0x04, 0x99, 0x17, 0x72, 0x99, 0x53, 0x26, 0x45, 0xd1, 0x87, 0xc8, 0x85, 0x7b, 0xd7, 0x17, 0x4f, 0x3b, 0xee, 0xd8, 0xc7, 0xeb, 0x52, 0x0a, 0x7f, 0x66, 0xb6, 0xa7, 0x58, 0x28, 0x15, 0xf9, 0xe0, 0x0e, 0x43, 0x02, 0x07, 0xb0, 0x64, 0x2c, 0x2c, 0xc5, 0x18, 0x67, 0x36, 0xc9, 0x5f, 0x5b, 0x26, 0xb9, 0xad, 0x25, 0xe2, 0x0b, 0x67, 0x18, 0x4f, 0x31, 0x1a, 0xa8, 0x31, 0xef, 0x0d, 0xac, 0x54, 0xf2, 0xf2, 0x4c, 0xed, 0xb4, 0x41, 0x43, 0xd9, 0xac, 0x4c, 0x62, 0x23, 0xfb, 0xab, 0x27, 0x2e, 0xab, 0x2a, 0xcd, 0xf7, 0xf9, 0xc6, 0x39, 0x8c, 0x96, 0xcc, 0xf1, 0xf9, 0xe7, 0xca, 0x04, 0xd5, 0xe0, 0xf2, 0x85, 0xd7, 0x77, 0x5c, 0x2f, 0x11, 0x2c, 0x9e, 0xaf, 0x29, 0x60, 0xb9, 0xe1, 0x02, 0xa2, 0x62, 0xdd, 0x2f, 0x4d, 0xb2, 0x51, 0xf6, 0x44, 0x32, 0xe3, 0x20, 0xb3, 0x28, 0xd3, 0x5f, 0x99, 0x7f, 0x44, 0xe2, 0x36, 0xa5, 0x5e, 0xb2, 0xbd, 0x7a, 0x42, 0xe2, 0x5e, 0x46, 0xcd, 0xc3, 0xc0, 0x2e, 0x35, 0x1a, 0x5e, 0x71, 0x08, 0x13, 0x27, 0x01, 0x66, 0x9c, 0xe3, 0xf6, 0xd0, 0x29, 0xab, 0xb4, 0xee, 0x4c, 0xc8, 0xd4, 0x54, 0x74, 0x6a, 0x21, 0x49, 0x05, 0x40, 0xf7, 0x3c, 0x3b, 0xbf, 0x48, 0xe9, 0x10, 0x31, 0x39, 0xd1, 0x74, 0x22, 0x28, 0xb7, 0x69, 0x24, 0x35, 0x8f, 0xb0, 0xbe, 0x09, 0xd5, 0xaf, 0x09, 0xb2, 0xa2, 0xa7, 0x8f, 0x81, 0xa4, 0x6d, 0x0f, 0x20, 0x1e, 0x54, 0x0f, 0xd6, 0x13, 0x13, 0x06, 0xae, 0x0f, 0x11, 0x15, 0xc5, 0x0b, 0xd7, 0xbd, 0x3c, 0x09, 0xad, 0xb3, 0x8f, 0xea, 0x8e, 0x4d, 0xe0, 0xb1, 0xa0, 0xef, 0x04, 0xb9, 0xc8, 0x0c, 0xc1, 0xd1, 0xe3, 0x20, 0x4e, 0x9c, 0x6b, 0x52, 0xe3, 0xc6, 0x5d, 0xfe, 0x29, 0xba, 0xc0, 0xd7, 0x71, 0xf5, 0x19, 0xd0, 0x1a, 0xfb, 0x7b, 0xc5, 0xb0, 0x39, 0xdc, 0x0f, 0xb5, 0x22, 0x69, 0x46, 0xe0, 0xe2, 0xd3, 0x76, 0xdc, 0x31, 0xc2, 0x98, 0x5a, 0x53, 0x58, 0x08, 0xb6, 0x10, 0xc9, 0xdd, 0xac, 0x46, 0x1b, 0x13, 0xaf, 0x58, 0x37, 0x0b, 0x06, 0xf8, 0x90, 0x5f, 0x26, 0x7f, 0xba, 0x28, 0xae, 0x36, 0xc7, 0xcc, 0xd7, 0xc6, 0x1b, 0x2d, 0xa0, 0x16, 0x37, 0x1f, 0xe2, 0xbf, 0x03, 0x63, 0xf9, 0xab, 0x1e, 0xb0, 0xf4, 0xb1, 0x28, 0xf5, 0x4c, 0x42, 0x81, 0xa7, 0x05, 0xfa, 0x2b, 0x04, 0xbc, 0x55, 0x24, 0x38, 0xbc, 0xc4, 0xb5, 0x20, 0x11, 0xbb, 0xdc, 0x1c, 0x16, 0x6d, 0x17, 0x5e, 0x57, 0xc6, 0xfb, 0x15, 0x32, 0xdb, 0x26, 0xc5, 0x7b, 0x8a, 0xb6, 0xd5, 0xe3, 0x5a, 0x43, 0xde, 0x08, 0xd9, 0x4b, 0x01, 0xa3, 0xf3, 0xe6, 0x93, 0x96, 0xd4, 0x0e, 0xef, 0x4d, 0x0e, 0x9a, 0xc8, 0x6c, 0x45, 0x24, 0x45, 0x11, 0xb5, 0x3e, 0x12, 0xdd, 0x00, 0xb6, 0x87, 0xa4, 0x5e, 0x5a, 0x45, 0x76, 0x57, 0x41, 0x64, 0xc9, 0x55, 0xe9, 0x87, 0x05, 0x4a, 0x6a, 0xa9, 0xf2, 0x5d, 0x15, 0x58, 0x96, 0x6d, 0x54, 0xac, 0x2a, 0xf1, 0x02, 0xdb, 0x7f, 0x8a, 0x56, 0x06, 0xdc, 0xe3, 0x2f, 0x21, 0xfc, 0x9f, 0x41, 0x57, 0x8d, 0x08, 0x15, 0x20, 0x08, 0x61, 0x15, 0x62, 0xec, 0x38, 0xd6, 0x54, 0x7f, 0x16, 0x30, 0x02, 0x5c, 0x5d, 0xf5, 0xb9, 0x29, 0xa8, 0xf6, 0x43, 0xcc, 0xb9, 0x6b, 0x17, 0x95, 0x87, 0x45, 0xa8, 0x8c, 0x10, 0xc4, 0xc8, 0xba, 0xe6, 0x38, 0x90, 0x8c, 0x8a, 0x35, 0x14, 0x6a, 0xd6, 0x03, 0x6f, 0x51, 0xcf, 0x08, 0x3d, 0x50, 0xd8, 0x4a, 0x5c, 0x69, 0xff, 0xc1, 0x87, 0x31, 0x37, 0xf7, 0x90, 0xff, 0x02, 0xda, 0x92, 0xdb, 0x5b, 0xc0, 0xb2, 0x00, 0xc3, 0xa2, 0x31, 0xba, 0xbd, 0x01, 0x9d, 0x06, 0x74, 0x95, 0x5a, 0xe4, 0x25, 0xc0, 0x5e, 0x4f, 0x1e, 0xf6, 0xc3, 0x26, 0x1d, 0xab, 0xed, 0xc5, 0x72, 0x68, 0x8a, 0x4d, 0x42, 0x0b, 0xa0, 0x57, 0xde, 0x5b, 0xcb, 0x66, 0x36, 0x45, 0x11, 0x7f, 0xf6, 0x4b, 0xe1, 0xde, 0x4d, 0x52, 0x5c, 0x7d, 0x7b, 0x29, 0x81, 0xac, 0xd7, 0x47, 0xf7, 0x71, 0x5d, 0x67, 0x26, 0xa4, 0xf5, 0x03, 0x57, 0xdb, 0xcf, 0x0b, 0x13, 0xdb, 0x9b, 0xf0, 0x4c, 0xbf, 0x48, 0xab, 0xb6, 0xd7, 0x1f, 0x68, 0x08, 0x3b, 0x0b, 0xdc, 0xf0, 0x16, 0xe4, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x30, 0xff, 0xff, 0xff, 0xab, 0xfe, 0xff, 0xff, 0xda, 0x01, 0x00, 0x00, 0x48, 0x01, 0x00, 0x00, 0x2a, 0x00, 0x00, 0x00, 0xa1, 0x03, 0x00, 0x00, 0x24, 0xff, 0xff, 0xff, 0x30, 0xfd, 0xff, 0xff, 0x9a, 0x02, 0x00, 0x00, 0xce, 0xfc, 0xff, 0xff, 0x44, 0x00, 0x00, 0x00, 0x4b, 0xff, 0xff, 0xff, 0xb5, 0xff, 0xff, 0xff, 0xd7, 0x00, 0x00, 0x00, 0x2c, 0xff, 0xff, 0xff, 0xdc, 0x03, 0x00, 0x00, 0x62, 0xe4, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x80, 0x0d, 0x00, 0x00, 0xc0, 0x1f, 0xb6, 0xfe, 0x5e, 0x17, 0x2e, 0xe8, 0x9a, 0x76, 0x43, 0x55, 0xf3, 0x78, 0xbd, 0xfc, 0xf8, 0xd6, 0x06, 0xf3, 0xbb, 0xe8, 0xa7, 0xfa, 0x27, 0x18, 0xd8, 0x81, 0x10, 0xa6, 0xa3, 0x2a, 0x2d, 0x2a, 0x15, 0xfc, 0x27, 0xc6, 0xaf, 0x88, 0x6d, 0xa3, 0xb3, 0xcc, 0x8b, 0x24, 0x0b, 0x16, 0x44, 0x39, 0xc0, 0xec, 0xff, 0xb3, 0x31, 0xd3, 0xda, 0xed, 0x0c, 0x9f, 0x32, 0xdc, 0x57, 0x53, 0x22, 0xe8, 0x98, 0x4a, 0xf5, 0xb6, 0x9f, 0xc5, 0xed, 0x49, 0xe1, 0x83, 0xa0, 0xf7, 0x13, 0xa1, 0xb7, 0x51, 0xbb, 0xf5, 0x30, 0x04, 0x57, 0x66, 0xb6, 0x4b, 0x67, 0x56, 0x5e, 0xef, 0x28, 0x0e, 0x46, 0x9b, 0x9b, 0x2d, 0xe0, 0x44, 0xc2, 0x59, 0x3e, 0xe0, 0xb2, 0x0b, 0x7b, 0x25, 0xea, 0xef, 0x5e, 0x41, 0xd7, 0xae, 0xca, 0x40, 0x3e, 0xe9, 0x51, 0x04, 0xce, 0xff, 0xbc, 0xae, 0x04, 0xf3, 0x5e, 0xcd, 0x47, 0xa1, 0x4a, 0x92, 0x4a, 0xa8, 0x62, 0x5a, 0xf6, 0xdb, 0xd8, 0x71, 0x7a, 0xe0, 0x23, 0xa6, 0x0c, 0x24, 0xf2, 0xd3, 0xa7, 0xa2, 0x65, 0x16, 0xf9, 0xe4, 0x07, 0xc2, 0xd9, 0x52, 0xe1, 0xcc, 0xe2, 0x70, 0xf6, 0x4d, 0xcf, 0xda, 0xcb, 0xd7, 0xab, 0xce, 0xa8, 0x09, 0x9d, 0xc4, 0xa0, 0x27, 0xf3, 0x56, 0xcf, 0xef, 0xfb, 0x1e, 0xfd, 0x98, 0xf1, 0xca, 0x2b, 0xe1, 0xae, 0x2b, 0xf6, 0x23, 0x58, 0xae, 0xfe, 0x95, 0x40, 0xf3, 0xb8, 0xb1, 0x18, 0x4d, 0x4d, 0xa8, 0xf5, 0x60, 0xdc, 0xac, 0x53, 0x40, 0x8c, 0x35, 0x17, 0x33, 0x6c, 0x22, 0x06, 0x44, 0x1f, 0x1b, 0x0e, 0x79, 0x3c, 0xf9, 0x07, 0xf3, 0xc0, 0xf1, 0xcf, 0xbe, 0xf8, 0x23, 0x68, 0x97, 0x4a, 0x0b, 0xc4, 0x00, 0x81, 0x04, 0x43, 0xf5, 0x1f, 0x38, 0x92, 0x23, 0x05, 0xe3, 0xbc, 0x67, 0x65, 0x64, 0xe0, 0x28, 0xb4, 0xa5, 0xe7, 0x3b, 0x1f, 0x74, 0x5d, 0x47, 0x38, 0xf3, 0x45, 0x4b, 0x84, 0x61, 0x16, 0x15, 0xc4, 0x62, 0x2d, 0x97, 0xaf, 0xc5, 0xde, 0x09, 0x07, 0xec, 0x00, 0xa0, 0x5f, 0xa1, 0xa9, 0xcb, 0x3e, 0xef, 0xd7, 0xe9, 0x0c, 0x33, 0x01, 0xd5, 0xa7, 0xd6, 0x42, 0x58, 0xcd, 0xf0, 0x32, 0xe8, 0xf0, 0xe4, 0x1f, 0xdd, 0x0f, 0x09, 0x50, 0xeb, 0x08, 0x2a, 0xd6, 0x98, 0x02, 0xe3, 0x60, 0xbe, 0xf0, 0x42, 0xd2, 0xd8, 0x9e, 0x0e, 0x45, 0x5d, 0xde, 0xc6, 0xc2, 0x18, 0x95, 0x00, 0xb0, 0x9d, 0x2f, 0x36, 0x2d, 0x3b, 0xd9, 0xa7, 0x44, 0xf6, 0x50, 0xab, 0x17, 0xc4, 0xbd, 0xe1, 0x3e, 0x9b, 0x81, 0x0f, 0x69, 0x20, 0xbe, 0x5c, 0x19, 0x49, 0x49, 0x27, 0xab, 0x38, 0x0a, 0x50, 0xcd, 0x1b, 0xe9, 0x06, 0x11, 0xed, 0x06, 0x69, 0xa1, 0x55, 0x2d, 0x25, 0xf4, 0x61, 0xdb, 0x92, 0xda, 0xb1, 0x8f, 0x4b, 0xcd, 0x42, 0x55, 0x99, 0x06, 0xba, 0xcb, 0xba, 0xe8, 0xd6, 0x6e, 0x29, 0xdf, 0x33, 0x3a, 0xfd, 0x02, 0xbc, 0xe5, 0xfe, 0x57, 0xd9, 0xa9, 0xda, 0xa1, 0x09, 0x4c, 0xf2, 0x53, 0x14, 0xc8, 0x38, 0xf0, 0x35, 0xc3, 0xce, 0x96, 0x10, 0x29, 0xf4, 0xd8, 0x07, 0xa2, 0xef, 0x60, 0x30, 0x17, 0x5f, 0x45, 0xd9, 0xee, 0x57, 0xfc, 0xc2, 0x30, 0xb1, 0xcf, 0x34, 0x7f, 0xde, 0xcc, 0x16, 0x03, 0x08, 0x3d, 0x65, 0x1e, 0xc2, 0xf1, 0x24, 0x5c, 0xb6, 0xca, 0xef, 0x45, 0xdc, 0xa3, 0xfb, 0xa5, 0xc3, 0x53, 0x05, 0xa5, 0x1a, 0x9e, 0x04, 0x9b, 0x5f, 0x15, 0x27, 0xe6, 0xbf, 0xba, 0x36, 0x99, 0x50, 0xa2, 0xa3, 0xc1, 0x3e, 0xc1, 0xc0, 0x51, 0x19, 0xb1, 0x6d, 0xdd, 0xec, 0xd7, 0x39, 0x2f, 0x91, 0x49, 0xc4, 0xf8, 0x51, 0xb7, 0xc9, 0x7f, 0x45, 0xba, 0x90, 0x68, 0x20, 0xc6, 0x73, 0x5f, 0x5e, 0xab, 0x67, 0xd5, 0xbc, 0xa7, 0xec, 0x72, 0x42, 0x11, 0x14, 0xd3, 0xac, 0x98, 0xa7, 0xc9, 0x0d, 0x5f, 0x61, 0x59, 0x15, 0x49, 0xed, 0x00, 0x26, 0x01, 0xdd, 0x4f, 0x97, 0x4c, 0x57, 0x9f, 0xf8, 0x68, 0x10, 0x34, 0x65, 0x39, 0x2f, 0xfe, 0x4d, 0xb7, 0xa4, 0xd0, 0x39, 0x34, 0x40, 0xc5, 0x4e, 0xc0, 0xdb, 0xe1, 0x33, 0x19, 0x05, 0xd6, 0x11, 0x33, 0xcc, 0x27, 0xd6, 0xd6, 0x53, 0xd3, 0xe5, 0x40, 0x24, 0xda, 0x1e, 0x60, 0x64, 0xfd, 0x81, 0xc2, 0x1f, 0x68, 0x2c, 0xfe, 0x27, 0xa4, 0xe5, 0x21, 0x1a, 0xbf, 0x20, 0x41, 0xe5, 0x34, 0x4b, 0x9e, 0xf6, 0x83, 0x99, 0xa9, 0x47, 0x01, 0x68, 0xb2, 0xca, 0xbf, 0xdd, 0x2c, 0x6c, 0x06, 0x0c, 0x5c, 0xe8, 0xf5, 0x43, 0xe7, 0xe4, 0x61, 0x07, 0x81, 0xb8, 0x67, 0x02, 0x9a, 0x32, 0x67, 0x3a, 0x4a, 0x66, 0x9b, 0x4d, 0xb4, 0x98, 0xb4, 0x5b, 0xf5, 0xf4, 0xb3, 0xde, 0x3d, 0x85, 0xb0, 0x22, 0x56, 0xe2, 0x5e, 0x07, 0x99, 0xb7, 0x02, 0x88, 0x42, 0x38, 0x33, 0xe0, 0x38, 0x1a, 0x30, 0x1a, 0x03, 0x04, 0x27, 0xfb, 0xdd, 0x0c, 0x82, 0x28, 0xfd, 0xdc, 0xc4, 0xe5, 0x92, 0x60, 0xc3, 0xd2, 0x9c, 0x30, 0x1c, 0xfa, 0xf6, 0xc1, 0xda, 0xe9, 0xc6, 0x30, 0x23, 0x3c, 0xa8, 0x9b, 0xac, 0x2b, 0xd6, 0x59, 0x4b, 0x56, 0xdf, 0x35, 0x70, 0x2f, 0xb6, 0xde, 0x53, 0xf5, 0x4b, 0xe3, 0x6e, 0xbc, 0xad, 0x61, 0xa9, 0xd8, 0xf0, 0x25, 0x27, 0x4f, 0xaf, 0xe2, 0x96, 0xc5, 0x0c, 0xbb, 0xf2, 0x63, 0x51, 0x11, 0xf0, 0xb3, 0x49, 0xd0, 0xb8, 0xc9, 0xa0, 0x24, 0xc5, 0x52, 0x26, 0x16, 0x49, 0xf3, 0xa4, 0xec, 0xd7, 0xa6, 0x14, 0xfb, 0x29, 0x3f, 0x25, 0x13, 0xfb, 0xd4, 0x34, 0x31, 0x0e, 0x56, 0x6f, 0x46, 0xf4, 0xc8, 0x67, 0xed, 0x22, 0x61, 0x07, 0xc0, 0x21, 0xaf, 0x9b, 0xc1, 0x93, 0xf7, 0x5d, 0x10, 0x5f, 0x40, 0x95, 0x37, 0xcf, 0x0b, 0x2c, 0x65, 0x95, 0xb5, 0x81, 0x14, 0xfd, 0xfd, 0xa3, 0x9d, 0xe2, 0x9c, 0x77, 0x1f, 0x59, 0x5c, 0xd0, 0xe9, 0x1f, 0xac, 0xb0, 0xbb, 0xa8, 0x54, 0xc3, 0xcb, 0xb6, 0x2b, 0x6f, 0x59, 0x96, 0xb2, 0x28, 0xa4, 0x2b, 0x1f, 0x2b, 0xd8, 0x2d, 0x75, 0xb0, 0x11, 0xf7, 0x61, 0x5f, 0x96, 0x3c, 0x72, 0xf4, 0x24, 0xc7, 0x45, 0xac, 0xa4, 0x2e, 0x05, 0xe1, 0x17, 0xdf, 0xdc, 0x18, 0xd2, 0x22, 0x52, 0xd6, 0xab, 0xf5, 0x38, 0x1c, 0x7a, 0xa8, 0xa1, 0x27, 0xb0, 0xbf, 0x47, 0x65, 0xb4, 0x9d, 0xdf, 0xbe, 0x30, 0x1d, 0x25, 0xd3, 0xf8, 0xd7, 0x34, 0xfb, 0xaf, 0x3a, 0x17, 0x4b, 0xe4, 0x6b, 0x55, 0x44, 0x1c, 0x66, 0x01, 0xd4, 0x5d, 0xe0, 0xd2, 0x81, 0xc6, 0x38, 0x1b, 0x14, 0xea, 0x98, 0x36, 0xea, 0xe8, 0x01, 0x67, 0x5a, 0x62, 0xb8, 0xab, 0x37, 0x01, 0x15, 0x0c, 0xdb, 0xd3, 0xd0, 0xef, 0x34, 0x3f, 0x3a, 0xfd, 0x3c, 0xb6, 0xd5, 0x7f, 0x4b, 0xfb, 0x48, 0xfa, 0xd3, 0xeb, 0xba, 0x1e, 0x12, 0x01, 0xc0, 0xd0, 0x9f, 0xec, 0x15, 0xdd, 0xd1, 0xb9, 0x3f, 0xcc, 0x06, 0x20, 0xd8, 0xff, 0x49, 0xe1, 0x70, 0x77, 0x24, 0x12, 0x62, 0xae, 0x44, 0x97, 0xf0, 0x44, 0xb9, 0xae, 0xd7, 0xbd, 0x27, 0x36, 0x24, 0xeb, 0xa2, 0xee, 0x19, 0xbf, 0x66, 0xbd, 0xb3, 0x03, 0xc6, 0xdf, 0x09, 0x17, 0xfb, 0x38, 0xc7, 0xb8, 0x11, 0x0a, 0x2d, 0xbf, 0x0d, 0x4c, 0x32, 0xc5, 0xf5, 0x0f, 0x22, 0x1c, 0x1d, 0xa8, 0x22, 0x67, 0x25, 0x40, 0xe1, 0x11, 0xb6, 0xd7, 0xc3, 0x42, 0x3b, 0xfe, 0xb3, 0x43, 0x45, 0x07, 0x77, 0x8e, 0x51, 0x47, 0xac, 0x3f, 0x3e, 0xe6, 0xba, 0xf0, 0x82, 0x89, 0xb9, 0x26, 0xc8, 0x2b, 0x1d, 0x46, 0x01, 0x52, 0x2d, 0xf9, 0x86, 0x7e, 0xfd, 0xd7, 0xef, 0x71, 0x44, 0xf3, 0x08, 0x30, 0xc0, 0x0f, 0x7c, 0x49, 0xe6, 0xce, 0xe9, 0x01, 0x16, 0x24, 0xe6, 0xad, 0x30, 0xd6, 0x19, 0xda, 0x17, 0x3e, 0xcc, 0xcc, 0xf2, 0x09, 0x45, 0x97, 0x17, 0xa5, 0xe1, 0xc1, 0x3e, 0xbc, 0x46, 0x71, 0x13, 0x18, 0x1c, 0x2b, 0xc7, 0x24, 0xdc, 0xe0, 0x36, 0x4a, 0xf8, 0x68, 0x7f, 0x6f, 0x99, 0x48, 0x54, 0x7b, 0x1b, 0x2a, 0x41, 0xa5, 0xa9, 0x1f, 0xe7, 0xc4, 0xe3, 0x60, 0x3b, 0xf3, 0x2a, 0x2f, 0x4d, 0xf5, 0x3a, 0xc9, 0xaf, 0x93, 0xdf, 0x5f, 0xa0, 0x4f, 0x3b, 0xed, 0x45, 0x1e, 0x10, 0xdb, 0xc5, 0x68, 0x8c, 0xc8, 0x13, 0x58, 0xf7, 0xf7, 0x61, 0xd9, 0x4c, 0xcc, 0xa9, 0x3c, 0xc0, 0x40, 0x24, 0x07, 0xe6, 0xe5, 0x23, 0x8f, 0xdf, 0xe8, 0x48, 0xd8, 0xb5, 0x18, 0x54, 0xac, 0x22, 0x08, 0xcf, 0xb3, 0xbf, 0x4a, 0x47, 0x40, 0xf2, 0xfa, 0x5f, 0xd1, 0x71, 0xaa, 0xbc, 0xc8, 0xdd, 0xee, 0xc4, 0x8c, 0x27, 0x64, 0xc6, 0xbe, 0x21, 0xfc, 0x5e, 0x05, 0x00, 0xed, 0x54, 0x6e, 0xb7, 0x2d, 0x10, 0x3c, 0xd6, 0x36, 0xb8, 0xf7, 0xfa, 0xf4, 0xae, 0xce, 0x1b, 0xb7, 0x8d, 0x2a, 0xfc, 0x47, 0xc2, 0xeb, 0x1b, 0x4e, 0x31, 0xc0, 0x16, 0xc9, 0x12, 0x83, 0x3f, 0x81, 0x96, 0xfa, 0x69, 0x00, 0x9b, 0xf2, 0xb6, 0xb3, 0xdc, 0x03, 0xe4, 0x0d, 0x0f, 0xac, 0xc3, 0xf7, 0x93, 0x31, 0x40, 0x58, 0xdf, 0x70, 0x51, 0xff, 0x57, 0x3b, 0x0d, 0xf0, 0x4f, 0xc9, 0x47, 0xb6, 0x9c, 0x28, 0xe3, 0x0f, 0x6e, 0x0b, 0xf8, 0x25, 0xe4, 0x67, 0x2b, 0x4e, 0xc0, 0xd7, 0x9d, 0xa4, 0x0a, 0x43, 0x18, 0x1a, 0x4a, 0xb3, 0x55, 0x2f, 0x03, 0xc1, 0xd5, 0xad, 0x01, 0x2c, 0x9c, 0xdd, 0xd2, 0x4c, 0x3e, 0x3e, 0xa5, 0xba, 0xd2, 0x23, 0xc7, 0xdd, 0xfa, 0x0d, 0x21, 0xfb, 0xbc, 0x27, 0x37, 0xb8, 0xa7, 0xb6, 0x11, 0xde, 0x9b, 0xb6, 0xf8, 0xe4, 0xb4, 0x1b, 0x23, 0x0a, 0xe4, 0xa3, 0x10, 0xb6, 0xf1, 0xa7, 0xe1, 0x65, 0xd4, 0xec, 0xc7, 0xf4, 0x68, 0x3c, 0xb5, 0xa9, 0xb7, 0xe3, 0x19, 0xde, 0x8e, 0x09, 0x61, 0xf6, 0xea, 0x31, 0xff, 0x67, 0x7a, 0xc2, 0xc5, 0xf1, 0xdd, 0xf5, 0xe6, 0xc8, 0x42, 0x07, 0x9b, 0x18, 0x2e, 0x42, 0x28, 0xd3, 0x7c, 0xfa, 0xdf, 0xde, 0xfd, 0xb3, 0x71, 0xac, 0x1c, 0x0c, 0x61, 0x27, 0xae, 0x2e, 0xa5, 0x58, 0xa2, 0x36, 0xaa, 0x5e, 0xa3, 0xb5, 0xbb, 0x45, 0xc8, 0x62, 0xeb, 0x53, 0xc8, 0x21, 0xc1, 0xc1, 0x28, 0x17, 0xb3, 0x02, 0xc7, 0x2b, 0xfd, 0x06, 0xf3, 0x3e, 0xc1, 0xaf, 0x30, 0x56, 0x2f, 0xf1, 0xf7, 0x19, 0x72, 0xfd, 0x16, 0xf4, 0xd8, 0xf3, 0x45, 0xdd, 0x29, 0x65, 0x99, 0x02, 0xdd, 0x0b, 0xab, 0xe0, 0xaa, 0xdc, 0x23, 0x9c, 0x08, 0x0b, 0x4b, 0xf1, 0xb2, 0x7f, 0xce, 0x92, 0x22, 0x1e, 0xbb, 0xaf, 0xf6, 0x4a, 0x1c, 0x41, 0xa2, 0x20, 0x30, 0x2c, 0x65, 0x4e, 0xf1, 0xac, 0x48, 0xe9, 0x92, 0x2f, 0x12, 0xb6, 0xf8, 0xce, 0xe7, 0xe1, 0x8d, 0xaf, 0x36, 0x34, 0x67, 0xba, 0xde, 0xe0, 0x4e, 0x2d, 0x0c, 0x7f, 0xe4, 0x64, 0xe3, 0x99, 0xd5, 0x2f, 0xb9, 0xa1, 0x26, 0xde, 0xd5, 0x31, 0x2f, 0xd6, 0x42, 0x53, 0xbf, 0x8a, 0xfa, 0xe5, 0x3b, 0x3b, 0x25, 0xd0, 0xd1, 0xdd, 0x6b, 0x39, 0xab, 0x27, 0x04, 0x32, 0xc5, 0x49, 0x85, 0xcf, 0xf4, 0x0a, 0xe7, 0x14, 0x74, 0x35, 0xae, 0x19, 0x1a, 0x21, 0xbc, 0xaa, 0xfd, 0xa4, 0x19, 0x0f, 0xde, 0xb2, 0xa9, 0x33, 0x43, 0x75, 0x8e, 0x2b, 0x9c, 0xd2, 0x2c, 0x51, 0x4f, 0xe3, 0xc6, 0xc7, 0xa1, 0x2a, 0x37, 0x17, 0xa0, 0x6e, 0x74, 0x26, 0xb9, 0x1e, 0x6e, 0x45, 0xd0, 0xfd, 0xf4, 0xa9, 0x0e, 0xe8, 0xbd, 0xb3, 0xf0, 0x44, 0xd1, 0x1e, 0xe2, 0x3e, 0x16, 0xc5, 0x3e, 0x8e, 0xa1, 0x6c, 0x2c, 0xe2, 0x65, 0x40, 0x6a, 0x8f, 0x9f, 0x17, 0x2f, 0x07, 0x32, 0xb0, 0x76, 0xbd, 0x3a, 0xaa, 0xfe, 0xd3, 0xf3, 0x7f, 0xe3, 0xb9, 0x08, 0xa7, 0x38, 0xec, 0x57, 0x39, 0xbc, 0xba, 0xc4, 0x48, 0xbb, 0x56, 0xd3, 0xb0, 0x4d, 0xe9, 0x22, 0xce, 0x05, 0x55, 0x12, 0xb9, 0x3b, 0x25, 0xca, 0x16, 0x2e, 0x22, 0xbb, 0xff, 0x03, 0x25, 0xe9, 0x3c, 0x02, 0xee, 0x07, 0x1c, 0xc4, 0xf7, 0x04, 0x5a, 0xb7, 0xa7, 0x09, 0xf4, 0x09, 0xad, 0x4f, 0xda, 0xb2, 0xd2, 0x35, 0xfc, 0xfd, 0x59, 0x33, 0xd4, 0x23, 0x0b, 0xfe, 0xbb, 0xcb, 0xd6, 0x03, 0xe7, 0xb2, 0xe7, 0xa3, 0xca, 0xaf, 0xf4, 0xcf, 0xb8, 0x15, 0xe4, 0xe5, 0xcb, 0xcb, 0x31, 0x17, 0x23, 0xec, 0x5e, 0x58, 0x41, 0x00, 0x54, 0x61, 0x14, 0xe2, 0xfb, 0x1a, 0x23, 0x2e, 0xf0, 0xbe, 0xa7, 0x17, 0x24, 0x4d, 0x5b, 0x74, 0x17, 0x4c, 0x16, 0xb3, 0xae, 0x5c, 0xb5, 0x5b, 0xe4, 0x64, 0xe5, 0xf1, 0xe8, 0xa1, 0x25, 0xa0, 0x4c, 0x46, 0xe7, 0xc9, 0xee, 0xb9, 0xfc, 0x7d, 0x29, 0x28, 0x30, 0x29, 0x44, 0x9c, 0x43, 0xad, 0xda, 0xb8, 0xe0, 0xd7, 0x34, 0x50, 0xfd, 0xf0, 0x12, 0x50, 0xb0, 0xfe, 0x2b, 0xc4, 0xdd, 0x5f, 0xda, 0xff, 0x16, 0xd4, 0xbc, 0x44, 0x1d, 0x9c, 0x17, 0xe6, 0x04, 0xdf, 0xb5, 0xce, 0xfe, 0x21, 0xee, 0xc5, 0x0e, 0x19, 0x07, 0xf1, 0xc0, 0x34, 0xb7, 0xe6, 0x1a, 0x4c, 0xbd, 0x3b, 0xbe, 0x0f, 0xbd, 0xbc, 0x08, 0x39, 0xa3, 0xa8, 0xd9, 0x4b, 0xea, 0xf3, 0xf2, 0x01, 0x47, 0xe3, 0xf4, 0x0d, 0x04, 0x19, 0x37, 0xcd, 0x2a, 0x27, 0x42, 0xe9, 0x2b, 0xe7, 0xe1, 0x2f, 0x05, 0xcf, 0xb3, 0x0b, 0x08, 0x2e, 0x30, 0xe6, 0xb6, 0xec, 0x47, 0x13, 0x2b, 0x3b, 0x0b, 0xcf, 0x4f, 0xc6, 0xbe, 0x04, 0xda, 0xad, 0x58, 0xdc, 0x21, 0x23, 0xaf, 0x4a, 0x39, 0xb5, 0xd8, 0x49, 0x60, 0xd9, 0xe4, 0x12, 0xc1, 0x7f, 0xf3, 0xb8, 0x12, 0xe7, 0x0d, 0xee, 0x20, 0x1f, 0xdb, 0xd4, 0xd2, 0xed, 0xe5, 0xc2, 0xe4, 0x2e, 0x21, 0x41, 0x02, 0xd5, 0x21, 0x22, 0x2b, 0x30, 0xfe, 0xe4, 0xe1, 0x3e, 0xcc, 0xe5, 0x15, 0x53, 0xb1, 0x28, 0x93, 0x14, 0xca, 0xbf, 0x5c, 0xda, 0xf5, 0x37, 0xec, 0x4a, 0x2c, 0x22, 0x8e, 0x52, 0xe6, 0xb2, 0x11, 0xbc, 0xca, 0xff, 0x59, 0xb5, 0x61, 0x57, 0xb0, 0x10, 0x61, 0x7f, 0x43, 0xae, 0x0d, 0x57, 0x3d, 0x06, 0x13, 0xb1, 0xa4, 0x63, 0x46, 0x1b, 0x32, 0x42, 0xf3, 0xd6, 0x76, 0x6c, 0x39, 0xb8, 0x5d, 0xff, 0x49, 0x01, 0x38, 0xac, 0x46, 0xf4, 0x34, 0xca, 0x47, 0xa1, 0x02, 0xdf, 0xe4, 0x55, 0x0b, 0x5e, 0xc0, 0x07, 0x24, 0x24, 0x3a, 0xea, 0x09, 0x49, 0x98, 0x99, 0xdc, 0xae, 0x3d, 0xcb, 0xe3, 0x9b, 0x9e, 0x5a, 0xeb, 0xbb, 0xaf, 0xcc, 0xb0, 0x4c, 0x59, 0x51, 0xb1, 0x00, 0x49, 0x50, 0x7a, 0xe2, 0x71, 0x52, 0x12, 0x02, 0xab, 0x37, 0x53, 0x16, 0xec, 0xb3, 0xcf, 0x34, 0xce, 0xd2, 0x4b, 0x26, 0xdb, 0xc8, 0x6a, 0xde, 0xa3, 0x20, 0x8f, 0x58, 0xb2, 0x47, 0x43, 0xe4, 0xb3, 0x53, 0xa1, 0x04, 0xf0, 0xdd, 0x41, 0x3e, 0xf7, 0xe0, 0x0f, 0xea, 0xe0, 0x26, 0xf3, 0xf0, 0x0b, 0x01, 0xfb, 0x01, 0x28, 0x1a, 0x2e, 0x54, 0xb8, 0x88, 0x95, 0xb1, 0xad, 0xd2, 0x48, 0xf2, 0xbd, 0xd7, 0x2a, 0x32, 0x26, 0x26, 0xbb, 0x6d, 0x81, 0x20, 0xfd, 0xd2, 0xef, 0x24, 0x10, 0x2e, 0xff, 0xca, 0x2d, 0xf3, 0xf3, 0x1f, 0x47, 0x74, 0x9f, 0x70, 0xc4, 0x07, 0x2f, 0x30, 0xc7, 0x1d, 0x11, 0x4d, 0xa6, 0xfd, 0x50, 0xd3, 0x42, 0xb8, 0x09, 0x0b, 0x55, 0x00, 0x18, 0x14, 0xd8, 0x0f, 0x45, 0x12, 0x13, 0x43, 0x25, 0x26, 0x16, 0x62, 0x03, 0xdf, 0xeb, 0xe2, 0xdf, 0x41, 0xdc, 0xdc, 0xba, 0x91, 0x17, 0x93, 0xa7, 0x13, 0xa0, 0x32, 0xe1, 0x04, 0xf7, 0xca, 0x1c, 0x2c, 0xa5, 0x24, 0x47, 0x5f, 0x3b, 0x57, 0xa5, 0xf4, 0x25, 0x56, 0x34, 0x5b, 0x3d, 0x45, 0xc5, 0x11, 0x35, 0x53, 0x43, 0xd3, 0xc8, 0xbf, 0x5a, 0xf9, 0xb8, 0x11, 0xda, 0x54, 0x33, 0x15, 0x36, 0x31, 0x4d, 0xe5, 0x27, 0x4a, 0xc7, 0x33, 0x17, 0x19, 0x38, 0x30, 0xf9, 0xef, 0xa3, 0x38, 0x14, 0x04, 0xeb, 0x04, 0x31, 0x5c, 0x2b, 0xef, 0x52, 0x5f, 0x5e, 0xd5, 0xce, 0x6b, 0xa6, 0x90, 0xb4, 0x98, 0x60, 0xa7, 0x56, 0x10, 0xf4, 0x25, 0x45, 0x6c, 0xa8, 0x5f, 0x48, 0x18, 0x42, 0xeb, 0x08, 0x01, 0x07, 0x0d, 0x45, 0x72, 0xcc, 0x38, 0x41, 0xa8, 0x12, 0x48, 0x4b, 0xa4, 0x4c, 0x09, 0x25, 0x91, 0xb4, 0x97, 0x1f, 0xc2, 0x1a, 0xf5, 0x75, 0x4d, 0x1f, 0x2f, 0xdd, 0x1c, 0x30, 0x7f, 0x5a, 0xbf, 0x36, 0x33, 0xec, 0xcb, 0xd5, 0x04, 0xff, 0xb2, 0x02, 0x33, 0x50, 0xc8, 0x7f, 0x73, 0xd4, 0x0f, 0x30, 0x64, 0xad, 0x0c, 0x00, 0xab, 0x50, 0x2a, 0xf0, 0x8d, 0x5e, 0xe6, 0x5e, 0xa9, 0x8c, 0xfe, 0x64, 0x97, 0x50, 0xad, 0xe2, 0xc8, 0xcf, 0xa3, 0xa6, 0x5c, 0xf4, 0xce, 0xf9, 0x45, 0x4a, 0x4c, 0xf4, 0x3c, 0x53, 0x0f, 0xe6, 0xf0, 0x11, 0xd2, 0x09, 0x30, 0xf2, 0x62, 0xb0, 0x62, 0x27, 0xe0, 0xc4, 0x7e, 0xd7, 0x56, 0xac, 0xd8, 0x42, 0x13, 0xc9, 0x15, 0x9c, 0xf0, 0x25, 0x1d, 0x04, 0xb7, 0xec, 0xc6, 0xec, 0xbc, 0xf6, 0x3e, 0x3e, 0xcb, 0x1f, 0x13, 0x08, 0xfe, 0x0c, 0x58, 0xdf, 0xf4, 0x16, 0x2a, 0x09, 0x29, 0x01, 0x3e, 0x16, 0xba, 0x00, 0x91, 0x4d, 0x01, 0x46, 0xe1, 0x23, 0xdc, 0xa9, 0x0f, 0xe5, 0x2b, 0x8a, 0xca, 0x19, 0x05, 0xe1, 0xe8, 0x33, 0xdc, 0x1a, 0xb8, 0xe0, 0x31, 0x1e, 0xe9, 0x41, 0xf3, 0xfa, 0x2f, 0xac, 0xe7, 0xb1, 0x1d, 0xc9, 0xe0, 0x77, 0xdc, 0x5f, 0x30, 0x11, 0x0e, 0xe4, 0x00, 0xc7, 0x47, 0xe1, 0x3f, 0x11, 0xe7, 0xf0, 0xd4, 0x62, 0xb3, 0x1e, 0x45, 0x19, 0xa8, 0x0c, 0xdb, 0x34, 0x45, 0x1a, 0x48, 0x61, 0xaa, 0xfa, 0xc1, 0xb8, 0xf6, 0xd1, 0x0e, 0xbd, 0xd2, 0x0e, 0x21, 0x6a, 0x55, 0x48, 0x7f, 0x45, 0xeb, 0xe2, 0xc4, 0xe6, 0xb7, 0xa9, 0xfe, 0x34, 0xb5, 0xea, 0xc6, 0x4c, 0x51, 0xe2, 0xb2, 0x5f, 0x3b, 0x60, 0xc7, 0xbe, 0x1d, 0xb6, 0xb6, 0xf8, 0xdd, 0x5a, 0x37, 0xf4, 0xda, 0x30, 0x39, 0x20, 0xbb, 0x10, 0x3a, 0x0f, 0x59, 0x4b, 0x28, 0xa5, 0x51, 0x5e, 0xcf, 0x11, 0x31, 0xc2, 0xcb, 0xdd, 0xf8, 0x31, 0x04, 0xf3, 0x62, 0x1c, 0xe0, 0x9b, 0x19, 0x0c, 0x18, 0xa8, 0xbf, 0xa6, 0xbc, 0x20, 0x6a, 0x5b, 0x35, 0xc9, 0xf2, 0x33, 0xc1, 0x24, 0x54, 0x7f, 0x98, 0x23, 0x63, 0xce, 0xc9, 0xba, 0x2b, 0x4a, 0x03, 0xdc, 0x32, 0x07, 0x12, 0x9b, 0x3d, 0xb7, 0x4a, 0xf3, 0x1a, 0xd1, 0xf3, 0x0b, 0x25, 0x2a, 0xe2, 0xb6, 0xf7, 0x0f, 0xa2, 0x8f, 0xf7, 0x97, 0x25, 0x4d, 0xc8, 0xb1, 0xbe, 0x08, 0xee, 0x49, 0x02, 0xb6, 0x14, 0xb0, 0xdb, 0x22, 0x34, 0xf1, 0x5b, 0xa9, 0x69, 0xd7, 0xcc, 0x22, 0x30, 0x14, 0x31, 0x0e, 0xae, 0x5b, 0xe3, 0xa6, 0x5d, 0xbc, 0xf4, 0xf3, 0x0a, 0x1a, 0x6f, 0xf1, 0x98, 0x1a, 0x3a, 0x08, 0x3c, 0x58, 0xdf, 0xf0, 0x7f, 0xa5, 0xd0, 0x6a, 0xb1, 0xb4, 0x13, 0x6a, 0x49, 0xae, 0xf3, 0xd4, 0x0c, 0xb8, 0x06, 0xaf, 0x6b, 0xe8, 0xa6, 0x46, 0x15, 0xf1, 0x3e, 0x35, 0xdd, 0x40, 0x52, 0x99, 0x90, 0x49, 0x0f, 0x64, 0x6e, 0x2f, 0xb8, 0x91, 0xab, 0xc3, 0x2d, 0xe3, 0xce, 0x1b, 0x09, 0x45, 0xea, 0x64, 0xbc, 0xe5, 0xa4, 0xc0, 0xda, 0x48, 0x03, 0xc0, 0x59, 0xb9, 0x14, 0x3b, 0x47, 0x5d, 0xfc, 0x04, 0xbd, 0xfb, 0x39, 0xf6, 0xd1, 0xcd, 0xc0, 0x60, 0x21, 0xc3, 0xda, 0xc6, 0xf3, 0xc7, 0xb8, 0x37, 0xc7, 0xea, 0x25, 0x74, 0x1c, 0xbb, 0x3b, 0x05, 0xf5, 0xe7, 0x98, 0xcf, 0x19, 0x4f, 0x05, 0xb7, 0xb9, 0x97, 0x36, 0xab, 0x1a, 0xac, 0xa6, 0xe1, 0xda, 0xea, 0xce, 0xd5, 0xf0, 0x58, 0x08, 0x05, 0x6a, 0x3c, 0xe6, 0x34, 0xc5, 0xa5, 0x3b, 0xa8, 0xb4, 0xbd, 0x41, 0x40, 0xf2, 0xb4, 0xaa, 0x01, 0xe6, 0x0a, 0x81, 0xa9, 0x9c, 0xce, 0x7d, 0xcc, 0xf7, 0x04, 0xe1, 0x7b, 0x14, 0x2f, 0xa1, 0x5c, 0xd9, 0xf3, 0x01, 0x28, 0x2d, 0x9b, 0xb6, 0x32, 0xdc, 0xb0, 0xba, 0x52, 0xfa, 0x09, 0xef, 0x40, 0xc8, 0x2e, 0xbc, 0xce, 0x41, 0x46, 0x8c, 0x3a, 0x70, 0x5a, 0x9c, 0xde, 0xbd, 0x9c, 0x92, 0x07, 0x76, 0xa3, 0xc8, 0xc8, 0x00, 0x4b, 0xbd, 0xa3, 0x5c, 0xbe, 0xc6, 0xe1, 0x29, 0xda, 0x50, 0x26, 0xee, 0xaf, 0xff, 0x37, 0xa1, 0x1f, 0xfc, 0x5d, 0x46, 0x3f, 0x28, 0xe6, 0x3d, 0xa2, 0x16, 0x1b, 0x01, 0xc6, 0x2d, 0x9f, 0xd1, 0x06, 0x23, 0xb7, 0x93, 0xaa, 0x11, 0x1b, 0xbd, 0x66, 0x4a, 0x64, 0xb3, 0xb2, 0xa8, 0x57, 0x13, 0x63, 0x7f, 0xfe, 0xb2, 0x7d, 0xd5, 0x0c, 0xfc, 0x8a, 0xad, 0xa3, 0xa4, 0x59, 0xf9, 0x02, 0x11, 0x53, 0x04, 0xd0, 0xff, 0x35, 0xae, 0xa3, 0x37, 0xa6, 0xb5, 0xcc, 0x39, 0xf6, 0xe5, 0x73, 0xf8, 0x39, 0x4c, 0xae, 0x1a, 0xbb, 0x67, 0x68, 0x24, 0x5b, 0x41, 0x5e, 0xc3, 0x4d, 0xbd, 0x46, 0x3e, 0x0d, 0xff, 0x0f, 0x99, 0x59, 0x12, 0x24, 0xb7, 0xd3, 0x25, 0x5c, 0x45, 0x48, 0x4a, 0x96, 0x42, 0x9d, 0x13, 0xf8, 0x45, 0xa9, 0x4b, 0xb3, 0x91, 0xc4, 0xac, 0xa9, 0x97, 0xcc, 0xef, 0xaa, 0xd6, 0x18, 0x2d, 0x4a, 0x45, 0xd8, 0x05, 0x39, 0x3a, 0x3c, 0x64, 0x24, 0x63, 0xe4, 0x64, 0x07, 0x92, 0x42, 0x22, 0x1d, 0xc8, 0x42, 0x29, 0x36, 0x32, 0xd3, 0x9d, 0x13, 0xcf, 0x6d, 0x98, 0xca, 0x91, 0xb0, 0xd2, 0xcf, 0xdf, 0x7f, 0x35, 0x98, 0xc8, 0x46, 0x37, 0xd0, 0x57, 0xae, 0xae, 0x17, 0x4b, 0x1d, 0x1e, 0xe3, 0x27, 0x44, 0x04, 0x9f, 0xce, 0xcb, 0x8d, 0xef, 0xd6, 0xb5, 0xd7, 0xd4, 0xee, 0xdb, 0x04, 0xb4, 0x7d, 0xf1, 0xf7, 0xa0, 0x0e, 0x54, 0xe2, 0x96, 0x6a, 0x0b, 0x34, 0x9c, 0xbb, 0xad, 0xd5, 0x6f, 0x2c, 0xdb, 0xd9, 0x44, 0x95, 0x1e, 0x24, 0x54, 0x51, 0x5d, 0x2d, 0x2f, 0x17, 0xdc, 0xca, 0x30, 0x10, 0xf1, 0xd3, 0x26, 0xd3, 0xcd, 0x5b, 0xd2, 0x91, 0x3d, 0x6e, 0x35, 0xd0, 0x25, 0x29, 0xb6, 0x57, 0xc6, 0xac, 0xb8, 0x9e, 0xcf, 0x11, 0xd7, 0xb0, 0x52, 0xe9, 0xfa, 0xd9, 0x66, 0x57, 0x10, 0xf8, 0x26, 0xf7, 0xf6, 0x06, 0x92, 0x39, 0x05, 0xd6, 0xa2, 0x2d, 0x1a, 0x11, 0xbc, 0x3a, 0x6c, 0xcc, 0x5b, 0xa2, 0x9c, 0xb2, 0x68, 0xa9, 0x63, 0x36, 0xa5, 0x9b, 0x01, 0x46, 0x3d, 0xc4, 0x13, 0x11, 0x9e, 0xe7, 0xa4, 0x1e, 0x57, 0xf6, 0xa8, 0x0d, 0x5b, 0xac, 0xb8, 0x01, 0x2a, 0xe7, 0xce, 0xb6, 0x14, 0xdf, 0x99, 0x99, 0xdb, 0xd9, 0x3d, 0xad, 0xec, 0xdb, 0x10, 0x93, 0x2d, 0xff, 0xfc, 0x62, 0xd5, 0x3a, 0x81, 0x47, 0x4c, 0xcd, 0x4d, 0x08, 0xb0, 0x27, 0xad, 0xc8, 0xba, 0x00, 0x3d, 0x0e, 0xf9, 0x03, 0xab, 0x14, 0x50, 0x25, 0xb5, 0x31, 0x15, 0x63, 0x0a, 0x1b, 0x0a, 0xbb, 0xd4, 0xd7, 0xde, 0x60, 0x23, 0x2e, 0x23, 0x94, 0x28, 0x11, 0xac, 0x1e, 0x2a, 0xa1, 0xc6, 0xbc, 0x3c, 0xdd, 0x60, 0xa4, 0xd3, 0x2e, 0xbc, 0xb7, 0xd1, 0x57, 0x16, 0xf2, 0x20, 0x13, 0x59, 0x2d, 0x52, 0xce, 0x2d, 0xb9, 0xef, 0x38, 0x3e, 0xf4, 0x46, 0x54, 0x3c, 0x0f, 0x0e, 0xfd, 0xde, 0xbb, 0x54, 0x33, 0x26, 0x63, 0x45, 0x0c, 0x0c, 0xa0, 0x27, 0xe0, 0x2b, 0xba, 0x20, 0xc0, 0x0f, 0x38, 0xd4, 0x3f, 0x00, 0x10, 0x21, 0x51, 0x95, 0xe1, 0xeb, 0x8a, 0x2b, 0x28, 0x00, 0xaf, 0x76, 0x5e, 0x3d, 0x26, 0x49, 0xc9, 0x6b, 0xc1, 0xf8, 0x99, 0x44, 0xaf, 0x03, 0x49, 0xd1, 0xce, 0x0f, 0x0c, 0xdd, 0xde, 0xf2, 0x26, 0xb0, 0x01, 0x43, 0x2e, 0x37, 0xab, 0x5b, 0x4f, 0x9b, 0xc2, 0xdb, 0x19, 0xe3, 0x37, 0xbf, 0xbf, 0x57, 0xb0, 0xef, 0xfb, 0xae, 0xe2, 0xa3, 0xa1, 0x52, 0x5e, 0x2c, 0x00, 0x4b, 0x2e, 0x39, 0x24, 0x33, 0xa9, 0xb3, 0x91, 0xcb, 0x22, 0xd7, 0xcd, 0xec, 0x4c, 0x1b, 0xcc, 0xa6, 0xfc, 0xc0, 0x59, 0xe2, 0x32, 0x38, 0xce, 0x1d, 0x06, 0x61, 0xb5, 0xe0, 0x34, 0x04, 0xe2, 0x56, 0x3a, 0xf3, 0x3d, 0xec, 0xc8, 0x14, 0x10, 0xf6, 0x1a, 0x5d, 0x1d, 0xd3, 0x1e, 0x2c, 0x06, 0xc7, 0x57, 0x2d, 0x23, 0x72, 0xad, 0xb9, 0xa5, 0x54, 0xf6, 0xe0, 0x45, 0xa4, 0x86, 0x36, 0xbf, 0x81, 0xed, 0xbb, 0x26, 0xfb, 0xb9, 0xe3, 0xbb, 0xce, 0xb4, 0x33, 0xa7, 0xd8, 0xb8, 0xa4, 0x04, 0xce, 0xc2, 0x32, 0xaa, 0xe3, 0x0c, 0xa2, 0x27, 0xd4, 0x34, 0x3c, 0xdb, 0xee, 0xad, 0xd7, 0x1f, 0x29, 0xd5, 0x07, 0x94, 0xc5, 0x3b, 0xf7, 0x03, 0xb8, 0x24, 0xeb, 0x90, 0x16, 0x90, 0xe1, 0xe1, 0x4b, 0xa2, 0x4b, 0x2a, 0xdc, 0x45, 0xf9, 0xca, 0xb9, 0x43, 0xb6, 0xb1, 0xb9, 0xc6, 0x59, 0x69, 0x52, 0x04, 0x2b, 0x0c, 0x46, 0x6e, 0x6d, 0x29, 0xa3, 0x4a, 0xe4, 0x55, 0x49, 0x66, 0x2d, 0xf9, 0xfb, 0x56, 0xb6, 0xc3, 0xe8, 0x55, 0xe0, 0xa9, 0x50, 0xda, 0x2d, 0x4d, 0xdb, 0xc4, 0x08, 0xb6, 0xa9, 0x3c, 0xe7, 0x1c, 0x3c, 0xba, 0x0e, 0x0b, 0xfe, 0xd2, 0x48, 0x18, 0x3f, 0x2c, 0x5f, 0x79, 0xda, 0x7f, 0x8e, 0x64, 0xf1, 0x8f, 0x2b, 0x53, 0xc5, 0x09, 0x04, 0xef, 0xee, 0x33, 0xcb, 0xf0, 0x59, 0x0d, 0x91, 0x4a, 0x21, 0xeb, 0xe2, 0xee, 0xdd, 0x14, 0xf6, 0x07, 0x90, 0xf9, 0xfa, 0x58, 0xbf, 0xf9, 0xc1, 0xe4, 0x39, 0x47, 0xf8, 0xd0, 0x9f, 0xea, 0x19, 0x6a, 0xa8, 0x24, 0x4b, 0xe1, 0x23, 0x8d, 0xa7, 0x32, 0x26, 0xd6, 0xce, 0xe5, 0x24, 0x36, 0x4d, 0xee, 0xf1, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x0f, 0xff, 0xff, 0xff, 0x94, 0x00, 0x00, 0x00, 0x12, 0xff, 0xff, 0xff, 0x24, 0x01, 0x00, 0x00, 0x1d, 0xff, 0xff, 0xff, 0x6a, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x7e, 0xff, 0xff, 0xff, 0xe6, 0xff, 0xff, 0xff, 0xdf, 0x00, 0x00, 0x00, 0x9e, 0x00, 0x00, 0x00, 0xeb, 0x00, 0x00, 0x00, 0xae, 0xff, 0xff, 0xff, 0x3e, 0xfe, 0xff, 0xff, 0xa1, 0xff, 0xff, 0xff, 0x68, 0x02, 0x00, 0x00, 0x69, 0xff, 0xff, 0xff, 0x3a, 0x00, 0x00, 0x00, 0x8e, 0x00, 0x00, 0x00, 0x19, 0x01, 0x00, 0x00, 0x9c, 0x01, 0x00, 0x00, 0xb0, 0x01, 0x00, 0x00, 0x03, 0xff, 0xff, 0xff, 0xd3, 0xff, 0xff, 0xff, 0x5a, 0xf2, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0xb1, 0xf8, 0xb3, 0x4f, 0xb3, 0x18, 0xf0, 0x42, 0xec, 0x00, 0x61, 0x43, 0x91, 0x9e, 0xe0, 0xe0, 0x56, 0x36, 0x22, 0xa0, 0x2c, 0x12, 0x0e, 0xb5, 0x3d, 0x34, 0xa5, 0x6c, 0xb4, 0x28, 0xf5, 0xe7, 0xa7, 0xfb, 0x03, 0xdd, 0xd3, 0x16, 0xb6, 0xa3, 0x3d, 0xe6, 0xd5, 0x34, 0x5d, 0x53, 0x5f, 0x25, 0x46, 0x59, 0xe7, 0x2d, 0x0b, 0x05, 0xf8, 0xa0, 0x0e, 0xde, 0x26, 0x11, 0xe7, 0x29, 0xcf, 0x04, 0xc1, 0x1c, 0xad, 0xf2, 0x07, 0x38, 0x09, 0x2d, 0x60, 0x2a, 0xa3, 0x39, 0xd2, 0xb6, 0x45, 0xfe, 0x50, 0x50, 0xd2, 0x39, 0x04, 0x26, 0x55, 0x15, 0xec, 0x1b, 0xd4, 0xdb, 0x4d, 0x9e, 0xa1, 0x24, 0xfb, 0x56, 0x23, 0xcf, 0x55, 0xec, 0x0a, 0x92, 0x58, 0x50, 0x02, 0x02, 0xd0, 0xbc, 0x99, 0xed, 0x4c, 0x65, 0x1b, 0x1f, 0xc9, 0x08, 0xec, 0x00, 0x15, 0x4b, 0x1c, 0xec, 0x06, 0x13, 0x5b, 0x4b, 0x18, 0xb5, 0x51, 0x2a, 0x1f, 0xb2, 0xfa, 0x3f, 0x25, 0x21, 0xb8, 0xf1, 0xc9, 0xfd, 0x57, 0xbf, 0x09, 0xcf, 0xd9, 0x3d, 0xad, 0xe0, 0x42, 0x4f, 0x0c, 0x32, 0x32, 0x31, 0xdf, 0xda, 0x63, 0x34, 0xb6, 0x9d, 0x14, 0x4d, 0x60, 0x51, 0xb2, 0x51, 0xeb, 0xe3, 0x03, 0x4d, 0x31, 0x43, 0xa1, 0xc9, 0xa0, 0x20, 0x4a, 0x4e, 0xab, 0xb7, 0x9e, 0x1c, 0x0b, 0x1e, 0x62, 0xc0, 0x62, 0x4e, 0xd8, 0xb7, 0x1e, 0xe5, 0x09, 0x34, 0x08, 0x54, 0x3d, 0xb5, 0x58, 0x69, 0x0f, 0x4d, 0xbd, 0x3e, 0x87, 0x3b, 0x0e, 0x5b, 0xd8, 0x66, 0x13, 0x36, 0x91, 0x0c, 0xd8, 0x38, 0xc7, 0x1d, 0x97, 0x15, 0x19, 0x76, 0x0f, 0x6c, 0x19, 0x99, 0x6f, 0xd4, 0xe3, 0xb5, 0xe5, 0xe6, 0x1f, 0x51, 0x3d, 0xd9, 0xdb, 0x2e, 0xfe, 0x23, 0xcc, 0x29, 0x02, 0x91, 0x10, 0x32, 0xbb, 0xdf, 0xf9, 0x47, 0x84, 0x37, 0x1d, 0x2f, 0x9c, 0x3c, 0x09, 0xad, 0xc1, 0x41, 0x99, 0xb1, 0xba, 0xb0, 0xa9, 0x5a, 0xa4, 0xc6, 0xbc, 0x9c, 0x36, 0xa4, 0x55, 0x8e, 0x6a, 0xae, 0xdb, 0x36, 0x2f, 0x43, 0xc9, 0x33, 0x63, 0x6d, 0x05, 0xa1, 0x50, 0x4e, 0x32, 0xc5, 0x33, 0xb8, 0x2c, 0x0a, 0xbd, 0x2f, 0x69, 0xf0, 0x1c, 0x47, 0x41, 0xcf, 0x59, 0x12, 0xc1, 0x1a, 0x1d, 0x3a, 0x13, 0xb9, 0x05, 0xec, 0xd2, 0x3e, 0xe0, 0xfc, 0xce, 0xd9, 0xd0, 0xbe, 0xf8, 0xbe, 0x3f, 0x61, 0xcb, 0xfe, 0xab, 0x5d, 0xf5, 0xf5, 0x9f, 0xaf, 0xf2, 0x23, 0x06, 0x06, 0x4a, 0x28, 0xd0, 0xde, 0x3f, 0xc2, 0x26, 0xd6, 0x3e, 0x43, 0x2c, 0xd6, 0xef, 0x57, 0x1f, 0x29, 0x71, 0x5d, 0x23, 0xdc, 0x54, 0x5a, 0xe9, 0x44, 0xfc, 0x08, 0x05, 0xcf, 0x2b, 0x54, 0xfc, 0xd5, 0xb1, 0x69, 0x0c, 0xc4, 0x35, 0x61, 0x3e, 0xf7, 0x1a, 0xd4, 0x3c, 0xcc, 0x9e, 0xba, 0x19, 0x96, 0x64, 0x18, 0xd6, 0x9a, 0x64, 0x34, 0xef, 0xc9, 0x48, 0xda, 0x66, 0x18, 0xbb, 0x55, 0x03, 0xa1, 0xb4, 0xbe, 0x58, 0x79, 0x61, 0x5c, 0x9d, 0xc4, 0x68, 0x00, 0x2d, 0xfe, 0x51, 0xd7, 0x03, 0xda, 0x0c, 0x19, 0x33, 0x01, 0xf6, 0xf3, 0xf4, 0x55, 0x68, 0xc8, 0xca, 0x43, 0x50, 0x0c, 0x33, 0x0e, 0xcd, 0x19, 0xea, 0x39, 0xf5, 0x6c, 0x27, 0xb3, 0xaa, 0xf0, 0x01, 0xe9, 0xfe, 0x95, 0x33, 0xc9, 0x30, 0x2d, 0xc3, 0x81, 0x29, 0xa7, 0x08, 0x00, 0xf9, 0x1b, 0xce, 0xd0, 0x92, 0xf7, 0x64, 0xd9, 0xff, 0x4b, 0xde, 0xd4, 0xa5, 0x54, 0xf8, 0xa9, 0x01, 0xf9, 0x5f, 0xcc, 0x3e, 0x58, 0xde, 0xeb, 0xd0, 0xa8, 0x0f, 0x32, 0xb6, 0x67, 0x4f, 0x50, 0x34, 0xe1, 0xa8, 0xf1, 0x28, 0x1a, 0xdc, 0xa5, 0xd6, 0xf5, 0x49, 0xc4, 0x09, 0x31, 0x08, 0x18, 0x4d, 0x42, 0xad, 0xa8, 0xa9, 0xc6, 0x5f, 0x4c, 0x1a, 0x47, 0x97, 0x27, 0xf9, 0x08, 0x5b, 0xa0, 0x18, 0x3e, 0xc1, 0xc0, 0xcc, 0xc6, 0x2c, 0x25, 0x36, 0xac, 0xb7, 0x15, 0x0b, 0x5a, 0xb7, 0xf0, 0xdb, 0xae, 0x5d, 0x1c, 0x21, 0x6a, 0x5b, 0xca, 0xa4, 0xfa, 0x3a, 0xce, 0xa9, 0x67, 0xe8, 0x30, 0xdf, 0xb8, 0xa8, 0xf5, 0xac, 0xbe, 0x40, 0xef, 0xdb, 0x2f, 0xbb, 0x25, 0x48, 0xdc, 0xa1, 0xf3, 0xd2, 0xcc, 0xbd, 0x54, 0xed, 0xdc, 0xe2, 0xfb, 0x59, 0xff, 0x4f, 0x2d, 0xb5, 0x17, 0xae, 0xf5, 0xf1, 0x57, 0x03, 0x09, 0x96, 0x25, 0xc5, 0xe6, 0xed, 0x05, 0xad, 0xdf, 0x1d, 0xe5, 0xe6, 0x46, 0x0b, 0x04, 0x4b, 0xbc, 0x29, 0x57, 0x00, 0xa6, 0xf8, 0xca, 0x46, 0x2a, 0xc5, 0x63, 0xcf, 0x00, 0x1f, 0x42, 0x3f, 0xb8, 0x13, 0x6c, 0xd8, 0x1f, 0x5f, 0x00, 0x12, 0x42, 0x94, 0x20, 0xc1, 0xa0, 0xbd, 0xb6, 0x56, 0x02, 0x15, 0xe5, 0xef, 0x21, 0xad, 0x2c, 0xa4, 0x4c, 0xf1, 0xb1, 0xf3, 0x2a, 0x4a, 0x66, 0x47, 0xba, 0x4a, 0x46, 0xa6, 0xfa, 0xb2, 0x1b, 0x26, 0x39, 0xce, 0xbe, 0x0e, 0xb6, 0x41, 0x41, 0xb5, 0x3d, 0x50, 0xe2, 0xc6, 0xb4, 0xac, 0xc0, 0x33, 0x13, 0xa2, 0x23, 0xd5, 0xb8, 0x1d, 0x29, 0x42, 0xd2, 0x4d, 0x04, 0xa2, 0x48, 0xc7, 0x4d, 0x35, 0xc5, 0xce, 0xe7, 0x1f, 0x57, 0x9e, 0x23, 0xb5, 0x97, 0xd7, 0x18, 0xeb, 0x14, 0x5f, 0x52, 0x30, 0xa5, 0x2c, 0x60, 0xf7, 0x22, 0xcb, 0xf9, 0xa2, 0x1f, 0xde, 0x52, 0x65, 0xd8, 0x21, 0x1f, 0x04, 0x06, 0x35, 0xb4, 0xbc, 0xe9, 0xf7, 0xb1, 0xcf, 0x14, 0x09, 0xa0, 0x58, 0xcd, 0xaf, 0xb4, 0x98, 0x0a, 0x2c, 0x24, 0x4a, 0x69, 0x9f, 0xe4, 0x27, 0x2c, 0xf7, 0xd2, 0x0d, 0xd1, 0x24, 0x3b, 0x1a, 0xdd, 0xf2, 0xdb, 0x45, 0x20, 0xbd, 0xa8, 0x4d, 0xc7, 0x18, 0x2f, 0x51, 0x04, 0x27, 0x05, 0xf8, 0xac, 0xa7, 0x5c, 0x40, 0xdb, 0xd8, 0xfa, 0x29, 0x59, 0x84, 0x47, 0xf8, 0x21, 0x5d, 0xfa, 0x41, 0x32, 0x02, 0x51, 0xe1, 0x58, 0x23, 0xb2, 0xde, 0x2a, 0xea, 0xef, 0xcd, 0xdf, 0x45, 0x07, 0x63, 0x21, 0xe7, 0x3e, 0x45, 0xe2, 0xe9, 0x04, 0x03, 0x55, 0x38, 0xfd, 0xe6, 0x4f, 0xf1, 0x53, 0x0a, 0x24, 0xe7, 0xd2, 0xae, 0x0a, 0xec, 0x4a, 0x30, 0x2d, 0x77, 0x58, 0x2c, 0xbd, 0x0c, 0xa7, 0x06, 0x07, 0xdd, 0xc2, 0xc3, 0x31, 0x71, 0x08, 0xde, 0xf4, 0x33, 0x38, 0x31, 0x52, 0x2b, 0xd2, 0x1d, 0x3e, 0x69, 0xf3, 0xd7, 0x1b, 0x48, 0x59, 0x4e, 0xb2, 0x18, 0x5f, 0xfc, 0x2f, 0xbf, 0xd3, 0xca, 0x53, 0xe9, 0x16, 0xff, 0x56, 0x11, 0x1e, 0x4f, 0xfb, 0xd1, 0xc2, 0xe7, 0x3c, 0x4d, 0x2f, 0xdf, 0xa8, 0xc1, 0x9c, 0x43, 0xa9, 0xaf, 0x20, 0x11, 0xdc, 0x50, 0x4d, 0xdc, 0x30, 0x2e, 0x1d, 0x50, 0x65, 0x48, 0x30, 0x40, 0x00, 0x3c, 0x3d, 0xaa, 0x20, 0xc8, 0x91, 0x36, 0x4e, 0xb1, 0xd6, 0xf3, 0xf1, 0x98, 0xb8, 0x06, 0xcc, 0x34, 0x49, 0x00, 0x10, 0xe1, 0xc9, 0x9f, 0x33, 0x1f, 0x4f, 0xb5, 0x65, 0xc3, 0x5d, 0xe0, 0x60, 0x1f, 0x53, 0xd2, 0xba, 0x6d, 0x2d, 0x16, 0xfe, 0x98, 0x16, 0xf5, 0x21, 0x18, 0xb0, 0xa0, 0x1a, 0x01, 0x36, 0xa3, 0x35, 0x45, 0xce, 0x51, 0x06, 0x50, 0x65, 0xf3, 0xd4, 0xdb, 0x13, 0xea, 0xe9, 0x6a, 0x3c, 0x0e, 0xa3, 0x4d, 0x45, 0xe4, 0x1d, 0xa3, 0x14, 0x37, 0x4a, 0x0a, 0x01, 0x21, 0xab, 0x09, 0xc8, 0x70, 0x64, 0x36, 0x37, 0x02, 0xdf, 0x14, 0xd7, 0xed, 0xa4, 0xa3, 0x55, 0x6a, 0xd1, 0x66, 0xf8, 0x0f, 0xef, 0x12, 0x49, 0xed, 0xb2, 0xdb, 0xbf, 0x6f, 0x2e, 0xb2, 0xf2, 0xde, 0x61, 0xaf, 0x0c, 0xfa, 0xd7, 0x24, 0xd2, 0xc5, 0x0e, 0xd7, 0xf4, 0x39, 0xa4, 0x2b, 0x63, 0x0f, 0x0d, 0x42, 0x11, 0x28, 0xc4, 0xa4, 0x1d, 0xa4, 0xa4, 0xa1, 0x73, 0x5a, 0x34, 0x1c, 0x62, 0xec, 0x51, 0xc3, 0xb9, 0x4f, 0x88, 0xb0, 0xb4, 0x3e, 0xd5, 0xbe, 0x09, 0xd0, 0x32, 0xcf, 0x29, 0x19, 0xe9, 0x30, 0xc6, 0x65, 0x3f, 0xe4, 0x9e, 0x62, 0x4d, 0x18, 0xc6, 0xa4, 0x35, 0xb0, 0x5e, 0x14, 0x2b, 0x27, 0x06, 0xef, 0x17, 0xba, 0xf0, 0x3e, 0xc8, 0x30, 0x28, 0xdf, 0xb8, 0xe5, 0x32, 0x18, 0xa8, 0x42, 0x64, 0x05, 0xc0, 0x03, 0xbe, 0xcc, 0xab, 0x30, 0xda, 0xb0, 0x2d, 0x32, 0x8e, 0x3a, 0x30, 0x96, 0xce, 0xa6, 0xf2, 0x71, 0x21, 0x44, 0x0a, 0x44, 0xe2, 0x9d, 0x69, 0x9b, 0xa5, 0xd2, 0xe7, 0x53, 0xf1, 0xb3, 0xf5, 0xe0, 0x4e, 0x22, 0xb1, 0xcc, 0xe2, 0x4d, 0xdf, 0xe7, 0xbb, 0xa9, 0x33, 0x4a, 0x04, 0xce, 0x9d, 0xc1, 0xbf, 0x5c, 0x47, 0x71, 0xf6, 0xc2, 0x5d, 0xad, 0x95, 0x42, 0x10, 0x14, 0x2b, 0xde, 0xcd, 0xe9, 0x0a, 0x49, 0x24, 0x6e, 0x29, 0x58, 0x0b, 0xb5, 0x10, 0x42, 0xc1, 0xda, 0xd9, 0x13, 0xac, 0x35, 0x0e, 0x9c, 0xdd, 0x12, 0x58, 0x1e, 0xa9, 0x13, 0x1c, 0xbe, 0x59, 0x21, 0x10, 0xed, 0x42, 0x24, 0x31, 0x10, 0xf5, 0xb6, 0xb5, 0xc0, 0x9b, 0x62, 0xdb, 0xcd, 0x1d, 0x5d, 0x53, 0x4a, 0xd9, 0xe8, 0x50, 0xc0, 0xd7, 0x53, 0xa9, 0xbc, 0x2d, 0x98, 0xba, 0xe5, 0x13, 0x24, 0x53, 0x46, 0x1a, 0x6a, 0x38, 0xed, 0xdb, 0xa2, 0x4e, 0xf2, 0x0b, 0xb2, 0xd4, 0xd1, 0xbb, 0x40, 0x47, 0xbf, 0x25, 0x62, 0x1d, 0xeb, 0x5e, 0x2e, 0xdd, 0x4c, 0xf3, 0x9b, 0x3b, 0xbd, 0xbd, 0x5d, 0xf7, 0xed, 0x12, 0xff, 0xa0, 0x00, 0x01, 0xc1, 0x19, 0xf8, 0xa4, 0x2a, 0x48, 0xd8, 0xbf, 0x0e, 0x52, 0xb2, 0xdd, 0x10, 0xc1, 0xfb, 0x2d, 0x2b, 0xfa, 0xa8, 0x66, 0xa0, 0xe9, 0x32, 0x4e, 0xd0, 0x3f, 0xc0, 0x2c, 0x37, 0x67, 0x38, 0x3a, 0x43, 0xdb, 0xc8, 0xa8, 0xda, 0x05, 0x19, 0xa4, 0xc0, 0x5f, 0x13, 0x0b, 0xbc, 0x64, 0xc1, 0x5c, 0xda, 0xf7, 0xa4, 0x29, 0xa3, 0x51, 0x97, 0xd9, 0xda, 0x11, 0xfe, 0x1c, 0x60, 0x9f, 0xae, 0x08, 0x5c, 0x65, 0xf6, 0x28, 0x98, 0x1f, 0x51, 0xe6, 0xae, 0xeb, 0x9d, 0x57, 0xc4, 0x29, 0x08, 0xab, 0x60, 0xeb, 0xb3, 0x5a, 0xb9, 0x12, 0xa3, 0x0d, 0x1b, 0x57, 0xb0, 0x5a, 0x7c, 0xb7, 0x46, 0x5a, 0x2d, 0xe1, 0xd8, 0x6d, 0x1a, 0x9e, 0x0d, 0x01, 0x3f, 0x6d, 0x62, 0xe1, 0xd8, 0xea, 0x2c, 0x61, 0xb7, 0xaa, 0x05, 0x4a, 0xb3, 0x55, 0x91, 0xfb, 0x55, 0xa4, 0xb3, 0x04, 0x29, 0xb9, 0x46, 0xb7, 0x5c, 0xdc, 0x55, 0xba, 0xe5, 0x54, 0x1c, 0x67, 0x4f, 0x42, 0x38, 0x50, 0x96, 0xd7, 0x2f, 0xb2, 0x4d, 0xae, 0xe7, 0x2f, 0x5a, 0xb9, 0x2e, 0xd3, 0x3e, 0x37, 0xf1, 0x60, 0x1b, 0x3c, 0x27, 0x9a, 0x2c, 0x35, 0xe0, 0xed, 0x4f, 0x36, 0xf6, 0x4c, 0xbd, 0x64, 0x02, 0x9d, 0x8e, 0x59, 0xa2, 0xaa, 0xfc, 0xbe, 0xd0, 0x36, 0xc6, 0xc0, 0xc2, 0x4c, 0x58, 0xf5, 0x2a, 0xde, 0xf1, 0xa2, 0xf8, 0x16, 0x51, 0x44, 0x1a, 0xac, 0xf1, 0xd1, 0x30, 0xa4, 0x99, 0x96, 0x2c, 0xab, 0x4c, 0x4a, 0x3e, 0xf8, 0xcf, 0xbe, 0xe8, 0x42, 0xef, 0xc1, 0x45, 0xf4, 0xd8, 0xf1, 0x53, 0x1e, 0x26, 0xaa, 0x63, 0x29, 0x1c, 0xd7, 0xd2, 0x35, 0x42, 0x03, 0x2d, 0x0d, 0xfa, 0x0d, 0x50, 0xf0, 0x46, 0x0c, 0x11, 0x22, 0xcb, 0x33, 0x57, 0x0f, 0x0b, 0x28, 0xca, 0xb6, 0xd5, 0x4e, 0x51, 0x50, 0x43, 0xb2, 0xfc, 0x5d, 0x0b, 0xa2, 0x2d, 0xc5, 0xe2, 0xc4, 0x69, 0x49, 0x1b, 0x5f, 0xa5, 0x09, 0x37, 0x2d, 0xe6, 0x52, 0xa3, 0x5c, 0xda, 0x1d, 0x10, 0x0e, 0xa6, 0x28, 0xe5, 0x45, 0xea, 0xa8, 0x56, 0x22, 0xb6, 0xd9, 0x05, 0x9d, 0xed, 0xf8, 0xde, 0xdd, 0xde, 0x34, 0xd8, 0x50, 0xac, 0xb2, 0x33, 0xda, 0xb7, 0x99, 0x01, 0x09, 0x5e, 0x3e, 0x4c, 0xcb, 0x41, 0xb3, 0x06, 0x32, 0x46, 0xe8, 0x63, 0xa0, 0xc0, 0x2a, 0x39, 0x17, 0x1f, 0x06, 0x9d, 0xd5, 0x51, 0xf9, 0x21, 0xbc, 0x2d, 0xdb, 0xbe, 0x9b, 0x58, 0x02, 0x40, 0x9c, 0xdb, 0x41, 0x20, 0xf2, 0x26, 0x97, 0xf5, 0xdf, 0x56, 0xc5, 0x23, 0x12, 0x28, 0xee, 0x97, 0xd2, 0x20, 0xea, 0xeb, 0xa2, 0xfc, 0xfb, 0xf4, 0xec, 0xc6, 0xc1, 0xe1, 0xf7, 0x42, 0xc2, 0xda, 0x60, 0xbe, 0x48, 0xac, 0x25, 0xf7, 0xb0, 0xec, 0xdb, 0x08, 0xc0, 0xbb, 0xa0, 0x35, 0x09, 0xdc, 0xc8, 0x47, 0x4e, 0xba, 0x23, 0x64, 0x4c, 0x52, 0xd8, 0xb5, 0x1b, 0xee, 0x14, 0x99, 0x05, 0xe2, 0x0e, 0xcb, 0xc1, 0xa5, 0x55, 0xb3, 0x11, 0x2a, 0xe6, 0x47, 0xd6, 0x15, 0x35, 0xb3, 0x37, 0xd0, 0x04, 0xb2, 0xdb, 0x46, 0xe8, 0xdb, 0xfe, 0xf3, 0x05, 0xd1, 0x67, 0xfa, 0xfe, 0x0a, 0xe8, 0x2b, 0x0b, 0xef, 0xf0, 0x17, 0x7d, 0xdc, 0x50, 0xce, 0x95, 0xf1, 0x2b, 0xad, 0x46, 0x48, 0x0f, 0x09, 0x2a, 0xd2, 0xfe, 0xc0, 0xc2, 0xc4, 0x46, 0x67, 0x27, 0x1d, 0xdc, 0xa3, 0xb4, 0xc9, 0xa9, 0x07, 0xbe, 0xcb, 0x62, 0xec, 0x4d, 0x0a, 0x22, 0x97, 0x11, 0x5c, 0x40, 0x07, 0xef, 0x14, 0x99, 0x0c, 0x18, 0x2e, 0x31, 0xa4, 0x25, 0x14, 0xa7, 0xc5, 0x9b, 0xa9, 0x13, 0x76, 0xbe, 0xd6, 0x4d, 0xae, 0xb7, 0xb2, 0xb2, 0xd2, 0x10, 0xa8, 0x3d, 0x3b, 0x3c, 0x5d, 0xd9, 0x45, 0x4e, 0xe5, 0xca, 0x16, 0xb1, 0xdb, 0xe3, 0x06, 0xf1, 0xf0, 0xb5, 0x61, 0x33, 0xe3, 0x29, 0x22, 0x68, 0xcc, 0xd3, 0xfe, 0x94, 0x06, 0xde, 0xa8, 0xaa, 0x4a, 0xf0, 0x4c, 0x70, 0x9f, 0xcf, 0xc5, 0x14, 0x4d, 0xcf, 0xd2, 0x02, 0xcf, 0xf9, 0x5b, 0xcc, 0x29, 0x09, 0x11, 0x67, 0xa1, 0xb5, 0xf0, 0x02, 0x5d, 0x02, 0xd6, 0x50, 0x38, 0xfe, 0xb4, 0x2e, 0x3d, 0x06, 0xae, 0x32, 0xe0, 0xe9, 0x0f, 0x42, 0x68, 0xf8, 0xb5, 0x05, 0xac, 0x33, 0xd5, 0xe7, 0x6a, 0x45, 0x4f, 0xb0, 0xb0, 0x21, 0x46, 0xa8, 0x66, 0xfb, 0xc5, 0xda, 0xc2, 0x35, 0xa8, 0x62, 0x5b, 0xd5, 0x4c, 0xbe, 0x17, 0x67, 0xdc, 0xb6, 0xc9, 0x4f, 0xfe, 0x20, 0x43, 0x69, 0x4f, 0x32, 0x05, 0xb4, 0xf9, 0xf5, 0xf7, 0xef, 0x54, 0x53, 0xb2, 0x51, 0xd9, 0x4d, 0xaa, 0x2f, 0xea, 0xc9, 0x4f, 0xdb, 0x0f, 0xce, 0x1a, 0xdf, 0x3c, 0xdb, 0x41, 0xfb, 0x0b, 0x3c, 0xde, 0xb3, 0xea, 0x4a, 0x4a, 0xc7, 0xdc, 0xc5, 0xa6, 0xde, 0x1f, 0x49, 0x5f, 0x5c, 0xb0, 0x46, 0x16, 0xab, 0x29, 0xee, 0x36, 0xec, 0x46, 0x4e, 0xe9, 0xcb, 0x10, 0x37, 0xb7, 0xcf, 0x66, 0x13, 0x55, 0x14, 0xda, 0x0a, 0xde, 0xad, 0xea, 0x56, 0x56, 0x9f, 0x4b, 0x4a, 0x3c, 0x53, 0x0c, 0x29, 0x4c, 0x67, 0x34, 0x5e, 0x73, 0x58, 0x94, 0xe0, 0xf9, 0x3d, 0x5a, 0x24, 0xb2, 0xb7, 0xb8, 0xf0, 0x43, 0xc5, 0x94, 0xfd, 0xbc, 0xfe, 0x30, 0x01, 0xe1, 0x39, 0x00, 0x2f, 0x68, 0x9e, 0xb2, 0x51, 0x08, 0x59, 0xc0, 0x1c, 0x07, 0xb7, 0x74, 0xc4, 0x94, 0xf2, 0x34, 0x17, 0x5c, 0x1c, 0x19, 0xe1, 0x3b, 0xaf, 0x58, 0x14, 0x60, 0x27, 0x0e, 0x4b, 0x00, 0xf3, 0x0e, 0xfa, 0x19, 0x4e, 0x46, 0x65, 0xdc, 0xe3, 0xc0, 0xb6, 0x60, 0x98, 0xee, 0xf0, 0x3c, 0xf6, 0x3b, 0x1b, 0xbe, 0xa2, 0xc0, 0xc7, 0xaa, 0xcb, 0xe5, 0x01, 0xed, 0x3f, 0xcb, 0x3b, 0x35, 0x65, 0x68, 0xd5, 0x33, 0x07, 0x65, 0xac, 0x59, 0xb1, 0xc7, 0xb6, 0xb6, 0xae, 0x09, 0xec, 0x98, 0xe4, 0xf3, 0x1b, 0xe6, 0x43, 0xf0, 0x46, 0x68, 0xc0, 0xb4, 0x61, 0x1d, 0x20, 0x98, 0x16, 0x35, 0xd4, 0xd7, 0xb2, 0xf4, 0xec, 0x44, 0xfc, 0xe1, 0xe7, 0x5a, 0xcb, 0x9b, 0x04, 0x26, 0xf5, 0xb4, 0x0b, 0xc0, 0x2f, 0x4d, 0xf1, 0xce, 0xbd, 0x9a, 0xe3, 0x3d, 0xbf, 0xa0, 0x07, 0xb0, 0xe3, 0xc2, 0x32, 0xeb, 0xa4, 0xf7, 0xf3, 0xb1, 0x0e, 0x22, 0x9e, 0x03, 0x17, 0xd8, 0x5b, 0x44, 0x2a, 0x53, 0xd1, 0x19, 0x24, 0xff, 0xd0, 0x4d, 0xbd, 0x19, 0xd7, 0xc6, 0xc7, 0xf5, 0xd3, 0x57, 0x47, 0x41, 0xb6, 0xa5, 0x05, 0xe6, 0x2a, 0xf7, 0xd5, 0x4a, 0x10, 0x02, 0xea, 0x0a, 0x29, 0x47, 0x18, 0x4d, 0x1d, 0x19, 0xb4, 0x30, 0x36, 0x41, 0x99, 0x09, 0x4c, 0x6c, 0x09, 0xd2, 0x2a, 0xcd, 0xd1, 0x3c, 0x51, 0xe1, 0x28, 0x62, 0x12, 0x06, 0x28, 0xa4, 0x12, 0x5b, 0x40, 0x6a, 0x55, 0x0d, 0x90, 0x51, 0xdd, 0x60, 0xa1, 0x00, 0xf7, 0xce, 0x4f, 0x45, 0xce, 0x97, 0x1d, 0x21, 0xf1, 0xd5, 0x43, 0xef, 0xb6, 0x9a, 0x44, 0xcd, 0xda, 0x18, 0xff, 0x24, 0xd6, 0xd8, 0xce, 0xcf, 0x1f, 0x35, 0xc5, 0xb6, 0xdc, 0x4d, 0xcc, 0x05, 0xaf, 0x58, 0xfc, 0xc6, 0x71, 0x3a, 0x6d, 0xd6, 0xb6, 0xd7, 0xa6, 0x0f, 0x06, 0x63, 0x38, 0xef, 0xce, 0x01, 0xda, 0xc4, 0x5d, 0x3d, 0x62, 0xa1, 0x1a, 0x28, 0x35, 0x42, 0x60, 0x65, 0xe9, 0xad, 0x18, 0x27, 0x44, 0x2b, 0x27, 0xb5, 0xd6, 0x6b, 0xcb, 0xcf, 0x9d, 0x79, 0x13, 0x43, 0xda, 0xd6, 0x0d, 0xda, 0x09, 0x54, 0x35, 0xe0, 0xf4, 0xcc, 0x9b, 0xd8, 0xd0, 0x0e, 0x9e, 0x9d, 0xca, 0x03, 0x06, 0xc4, 0x65, 0x30, 0x55, 0xa0, 0x10, 0xd2, 0x53, 0x08, 0x67, 0x3f, 0x20, 0xc5, 0xb1, 0xd8, 0x07, 0xf5, 0x5f, 0x94, 0x32, 0x05, 0x4e, 0x91, 0x19, 0x11, 0x40, 0xab, 0xb6, 0x53, 0xd6, 0x5b, 0xbd, 0x5a, 0x61, 0xf6, 0x07, 0x08, 0xa3, 0xeb, 0xc3, 0x0d, 0x0e, 0x66, 0xc6, 0xc8, 0xea, 0x24, 0xe7, 0xbb, 0xa8, 0x31, 0xa2, 0xe6, 0x47, 0x0e, 0x3f, 0x17, 0xc2, 0x9a, 0xe6, 0xc7, 0x10, 0xaa, 0xbc, 0xbe, 0x35, 0x19, 0xca, 0x4b, 0xb0, 0xc9, 0xfc, 0x32, 0x9a, 0xed, 0xf5, 0x9c, 0x3e, 0xff, 0x4d, 0x53, 0x26, 0xc0, 0x1e, 0x30, 0x28, 0x4e, 0xe5, 0xb6, 0xc4, 0x3d, 0x63, 0xd0, 0x29, 0x5c, 0xaa, 0x2e, 0x23, 0x44, 0x01, 0x1d, 0x59, 0x4c, 0xbf, 0x31, 0xd2, 0xe5, 0x58, 0xe3, 0x0c, 0xb4, 0x12, 0x42, 0x2e, 0xab, 0xb2, 0xfc, 0xeb, 0x00, 0xa6, 0xc5, 0x1f, 0xc0, 0x0d, 0xb1, 0xe3, 0xa0, 0x10, 0xfd, 0xbf, 0xa8, 0xaa, 0x62, 0xac, 0xc4, 0x1a, 0x06, 0xd0, 0xc6, 0xae, 0xb6, 0xc7, 0x03, 0x14, 0xa9, 0xfc, 0x3f, 0xf2, 0xd8, 0xab, 0xb0, 0x64, 0xde, 0x0d, 0x25, 0xa8, 0xf5, 0xf1, 0x50, 0x1a, 0xb2, 0x0d, 0xd0, 0xc1, 0x64, 0x5c, 0xe2, 0x3f, 0xee, 0xe7, 0xe3, 0x22, 0xfb, 0x46, 0x4b, 0xb9, 0xc0, 0x67, 0x54, 0xa8, 0xb6, 0xe3, 0xf0, 0x3d, 0x32, 0xd0, 0x18, 0xe4, 0xb3, 0x3c, 0x62, 0x32, 0xb5, 0xdf, 0x9d, 0xe2, 0xe7, 0x19, 0x2c, 0x9d, 0x42, 0x37, 0x23, 0x33, 0x05, 0xa9, 0x14, 0x35, 0xe5, 0xd9, 0xaf, 0xce, 0x62, 0x07, 0xe9, 0x41, 0x79, 0x46, 0xb0, 0x3e, 0xca, 0xab, 0x43, 0xe6, 0xfd, 0x0c, 0xc7, 0x34, 0x2c, 0x4e, 0x3a, 0xcc, 0xe2, 0xe0, 0x29, 0xf4, 0x5e, 0xd5, 0x6a, 0x16, 0xd2, 0x9a, 0xbb, 0xfe, 0xa7, 0xb9, 0xb6, 0xaa, 0xbc, 0xb5, 0xc8, 0x66, 0x21, 0xda, 0xe7, 0xb6, 0xff, 0xed, 0xc5, 0x0e, 0xbc, 0x64, 0xd7, 0xc3, 0xc8, 0x99, 0x5a, 0x29, 0x98, 0x2f, 0x55, 0x52, 0xa7, 0x9e, 0x18, 0xce, 0xe4, 0x07, 0x5b, 0xbe, 0xb8, 0x06, 0xdf, 0x63, 0x5d, 0x65, 0xaf, 0xee, 0xe6, 0x15, 0xe8, 0xd5, 0x22, 0xc2, 0x3b, 0x51, 0x49, 0xc3, 0xcd, 0x1f, 0x85, 0x30, 0x39, 0x1b, 0x62, 0x57, 0xdd, 0x2e, 0xfa, 0x01, 0x2b, 0x11, 0x46, 0x56, 0x07, 0x01, 0xf0, 0x67, 0x06, 0xc0, 0xf1, 0x10, 0x3f, 0xd3, 0xe7, 0x58, 0x04, 0xd5, 0x04, 0x1d, 0x5a, 0x49, 0x46, 0x40, 0xa7, 0x5a, 0x04, 0x4f, 0xcc, 0xb5, 0xae, 0x9b, 0x58, 0x2d, 0x0b, 0xf5, 0x3a, 0x57, 0x13, 0x55, 0x2a, 0x22, 0x06, 0xe2, 0xe7, 0xb0, 0xc3, 0xfd, 0xc1, 0xb0, 0xb9, 0xc1, 0xa8, 0x61, 0xbf, 0xea, 0x61, 0x9e, 0x9a, 0x33, 0xdf, 0x67, 0x45, 0x43, 0x2f, 0x27, 0x2c, 0x07, 0xdc, 0x66, 0x2f, 0x2e, 0xc7, 0xe3, 0x38, 0x48, 0x6f, 0x18, 0xce, 0x34, 0x0f, 0xfe, 0x4f, 0x3b, 0x27, 0x28, 0xb4, 0xda, 0x19, 0xcf, 0xbd, 0x14, 0xd1, 0xf0, 0xda, 0x02, 0xc9, 0x64, 0xe1, 0xc9, 0xf9, 0x57, 0xc2, 0xc4, 0x1f, 0xd2, 0x46, 0xb8, 0x61, 0x06, 0xf4, 0x41, 0xa4, 0x16, 0x53, 0x53, 0xb9, 0xf5, 0xf6, 0xca, 0x13, 0xbd, 0x1a, 0xb5, 0xa5, 0x29, 0x3f, 0x39, 0xda, 0x9a, 0xd2, 0x0c, 0x65, 0xb8, 0xea, 0x12, 0x39, 0xc4, 0xe5, 0x9a, 0xac, 0xed, 0x23, 0x5f, 0x1b, 0x05, 0x5f, 0xc0, 0xb1, 0xa6, 0xd4, 0xb0, 0xa7, 0xf6, 0x21, 0x19, 0xbf, 0xb7, 0x34, 0xec, 0x55, 0xaa, 0xd1, 0x97, 0xa4, 0xbb, 0xbe, 0xea, 0xa0, 0x16, 0xb9, 0xc4, 0xba, 0xaa, 0xb3, 0xdf, 0xaa, 0xd4, 0xe0, 0x99, 0x1b, 0x10, 0x4f, 0x40, 0xc1, 0xc1, 0x42, 0xb1, 0x00, 0x60, 0x47, 0x1c, 0x0c, 0xa8, 0x29, 0xec, 0x34, 0xe0, 0x1a, 0x0d, 0x03, 0x13, 0x1d, 0x3d, 0xcc, 0x41, 0x1c, 0xaa, 0xd4, 0x25, 0x3f, 0x6a, 0x04, 0xb9, 0x10, 0x9d, 0x1f, 0xdf, 0xb5, 0x99, 0xba, 0xb1, 0x1c, 0x56, 0x20, 0x19, 0x5e, 0xc3, 0x43, 0x53, 0xed, 0x43, 0xb9, 0x50, 0x44, 0x14, 0xa4, 0x04, 0xad, 0x23, 0x0a, 0x54, 0x12, 0xcc, 0x56, 0xeb, 0xd4, 0xd8, 0xef, 0x42, 0x1f, 0xea, 0xde, 0x95, 0xe8, 0x5f, 0xc2, 0x20, 0x37, 0xc2, 0x50, 0xba, 0x14, 0xa9, 0x43, 0xce, 0x5a, 0x29, 0xf1, 0x46, 0x34, 0xe2, 0xa5, 0xfd, 0x3d, 0x41, 0xeb, 0xb0, 0x44, 0x3f, 0x63, 0x19, 0xf9, 0x54, 0x40, 0x03, 0x41, 0x6a, 0xe7, 0x32, 0xe6, 0xe2, 0xc9, 0x33, 0x6b, 0xe4, 0xdd, 0x98, 0x47, 0x9a, 0xb9, 0x1b, 0xf0, 0x9a, 0x38, 0x1f, 0x57, 0xa9, 0x61, 0x28, 0x2f, 0xdb, 0x67, 0xaf, 0x47, 0xd1, 0x5e, 0xec, 0x50, 0xda, 0xa9, 0xc4, 0x55, 0x13, 0x3f, 0x35, 0xdb, 0xcc, 0x07, 0x9b, 0x70, 0x0c, 0xf6, 0x57, 0x14, 0x19, 0xfc, 0xfd, 0xaa, 0x5f, 0xb0, 0x46, 0xc1, 0x51, 0x4a, 0x50, 0xdf, 0xf1, 0xe2, 0x9d, 0xe5, 0x0d, 0x6c, 0xae, 0x9b, 0x09, 0xbd, 0xa1, 0x08, 0xff, 0x59, 0xe1, 0xab, 0x27, 0x18, 0x52, 0x59, 0xca, 0x09, 0x35, 0x15, 0x17, 0xaa, 0x28, 0xd9, 0xac, 0x48, 0xe8, 0x15, 0x13, 0xc3, 0xf1, 0xb0, 0xc6, 0xa0, 0xbb, 0xa9, 0x3d, 0xb1, 0xc8, 0xd9, 0x4f, 0x2e, 0xf9, 0x20, 0xbd, 0x35, 0xa2, 0xbf, 0x17, 0xbe, 0x42, 0x38, 0x35, 0x23, 0x96, 0x96, 0xa8, 0xb6, 0x5c, 0xde, 0x9c, 0x2b, 0x5f, 0x02, 0x21, 0xe8, 0xc5, 0xf5, 0x3d, 0x56, 0x4f, 0x9f, 0xd7, 0xdc, 0x32, 0x05, 0xd2, 0x0f, 0xf9, 0xd5, 0x42, 0x15, 0xe1, 0xeb, 0xc0, 0x37, 0x2e, 0x14, 0xb6, 0xfa, 0x3d, 0xd7, 0x47, 0x35, 0x5d, 0x4c, 0xb2, 0x0f, 0xb4, 0xb1, 0xaa, 0xef, 0x5a, 0xd5, 0xa6, 0x24, 0xab, 0xbc, 0x4d, 0x24, 0xc4, 0x67, 0xd1, 0x5a, 0x17, 0x9e, 0xdb, 0xdb, 0xfb, 0x37, 0x65, 0x66, 0xfe, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x85, 0xfe, 0xff, 0xff, 0xd9, 0x00, 0x00, 0x00, 0xd1, 0x00, 0x00, 0x00, 0x95, 0x02, 0x00, 0x00, 0xbd, 0x00, 0x00, 0x00, 0xef, 0xff, 0xff, 0xff, 0x6e, 0xff, 0xff, 0xff, 0xdf, 0xfe, 0xff, 0xff, 0x83, 0x00, 0x00, 0x00, 0xc2, 0x00, 0x00, 0x00, 0x82, 0x01, 0x00, 0x00, 0xf1, 0xff, 0xff, 0xff, 0x41, 0x01, 0x00, 0x00, 0x48, 0xff, 0xff, 0xff, 0x61, 0xff, 0xff, 0xff, 0x9d, 0x00, 0x00, 0x00, 0x24, 0xff, 0xff, 0xff, 0xe0, 0xff, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x93, 0x00, 0x00, 0x00, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x74, 0xff, 0xff, 0xff, 0xed, 0x00, 0x00, 0x00, 0xb4, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x00, 0xd7, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xb6, 0xff, 0xff, 0xff, 0x7a, 0xff, 0xff, 0xff, 0x8b, 0xff, 0xff, 0xff, 0xf2, 0xfe, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xac, 0xf3, 0xae, 0x46, 0xe7, 0x1c, 0x60, 0xf2, 0xf5, 0x84, 0x78, 0xb6, 0x41, 0xfa, 0x61, 0x3f, 0x41, 0x33, 0xab, 0x66, 0xf9, 0x6f, 0x76, 0x1a, 0x75, 0x91, 0x5d, 0x20, 0xfd, 0x5f, 0x72, 0x34, 0xc4, 0x51, 0x3a, 0x27, 0x94, 0x5c, 0xb6, 0xca, 0x4e, 0x45, 0xf6, 0xb8, 0x59, 0x60, 0x39, 0x57, 0xa6, 0x82, 0x01, 0x05, 0x52, 0xf8, 0x39, 0x99, 0x44, 0x3a, 0x1e, 0x7f, 0x49, 0xf7, 0xaf, 0xbe, 0x3e, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x4b, 0xff, 0xff, 0xff, 0xb5, 0x00, 0x00, 0x00, 0x52, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x60, 0x00, 0x00, 0x00, 0xd8, 0xee, 0xff, 0xff, 0xdc, 0xee, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x4d, 0x4c, 0x49, 0x52, 0x20, 0x43, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x74, 0x65, 0x64, 0x2e, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x18, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x8c, 0x02, 0x00, 0x00, 0x90, 0x02, 0x00, 0x00, 0x94, 0x02, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x6d, 0x61, 0x69, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x28, 0x02, 0x00, 0x00, 0xc4, 0x01, 0x00, 0x00, 0x74, 0x01, 0x00, 0x00, 0x34, 0x01, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x78, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x6a, 0xfe, 0xff, 0xff, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x1c, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x04, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f, 0x01, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0xa2, 0xfe, 0xff, 0xff, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xa8, 0xef, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xd6, 0xfe, 0xff, 0xff, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x1c, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x07, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x46, 0xff, 0xff, 0xff, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x24, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x36, 0xff, 0xff, 0xff, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x26, 0xff, 0xff, 0xff, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x18, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x16, 0xff, 0xff, 0xff, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x62, 0xff, 0xff, 0xff, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x18, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x52, 0xff, 0xff, 0xff, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x1a, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x34, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x18, 0x00, 0x17, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x16, 0x00, 0x00, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x24, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x10, 0x00, 0x0f, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x90, 0x0e, 0x00, 0x00, 0x28, 0x0e, 0x00, 0x00, 0xa4, 0x0d, 0x00, 0x00, 0x28, 0x0d, 0x00, 0x00, 0xb4, 0x0c, 0x00, 0x00, 0x48, 0x0c, 0x00, 0x00, 0xbc, 0x0a, 0x00, 0x00, 0x30, 0x09, 0x00, 0x00, 0x04, 0x08, 0x00, 0x00, 0xd8, 0x06, 0x00, 0x00, 0x0c, 0x06, 0x00, 0x00, 0x40, 0x05, 0x00, 0x00, 0x8c, 0x04, 0x00, 0x00, 0x10, 0x04, 0x00, 0x00, 0x54, 0x03, 0x00, 0x00, 0x98, 0x02, 0x00, 0x00, 0x1c, 0x02, 0x00, 0x00, 0xb0, 0x01, 0x00, 0x00, 0x14, 0x01, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xda, 0xf1, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0xc4, 0xf1, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3b, 0x19, 0x00, 0x00, 0x00, 0x53, 0x74, 0x61, 0x74, 0x65, 0x66, 0x75, 0x6c, 0x50, 0x61, 0x72, 0x74, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x64, 0x43, 0x61, 0x6c, 0x6c, 0x3a, 0x30, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x42, 0xf2, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x7c, 0x00, 0x00, 0x00, 0x2c, 0xf2, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x93, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2e, 0x0d, 0x5a, 0x3c, 0x4c, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xe2, 0xf2, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x74, 0x00, 0x00, 0x00, 0xcc, 0xf2, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x21, 0x6f, 0xe9, 0x3b, 0x46, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x52, 0x65, 0x6c, 0x75, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x7a, 0xf3, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x12, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0x64, 0xf3, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xf9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0xa5, 0xe4, 0x7a, 0x3c, 0x1a, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x66, 0x6c, 0x61, 0x74, 0x74, 0x65, 0x6e, 0x2f, 0x52, 0x65, 0x73, 0x68, 0x61, 0x70, 0x65, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xe2, 0xf3, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x11, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x4c, 0x00, 0x00, 0x00, 0xcc, 0xf3, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xf9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0xa5, 0xe4, 0x7a, 0x3c, 0x22, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x6d, 0x61, 0x78, 0x5f, 0x70, 0x6f, 0x6f, 0x6c, 0x69, 0x6e, 0x67, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x4d, 0x61, 0x78, 0x50, 0x6f, 0x6f, 0x6c, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x5a, 0xf4, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x8c, 0x00, 0x00, 0x00, 0x44, 0xf4, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xf9, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0xa5, 0xe4, 0x7a, 0x3c, 0x61, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x32, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x32, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x32, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x12, 0xf5, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x8c, 0x00, 0x00, 0x00, 0xfc, 0xf4, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xa4, 0x93, 0x83, 0x3c, 0x61, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xca, 0xf5, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x4c, 0x00, 0x00, 0x00, 0xb4, 0xf5, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4b, 0x37, 0x3e, 0x3c, 0x20, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x6d, 0x61, 0x78, 0x5f, 0x70, 0x6f, 0x6f, 0x6c, 0x69, 0x6e, 0x67, 0x32, 0x64, 0x2f, 0x4d, 0x61, 0x78, 0x50, 0x6f, 0x6f, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x42, 0xf6, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x84, 0x00, 0x00, 0x00, 0x2c, 0xf6, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x17, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x4b, 0x37, 0x3e, 0x3c, 0x5b, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xf2, 0xf6, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x9c, 0x00, 0x00, 0x00, 0xdc, 0xf6, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x61, 0x9e, 0xfe, 0x3a, 0x27, 0xe1, 0x03, 0x3b, 0xa4, 0x83, 0x02, 0x3b, 0x18, 0x25, 0x00, 0x3b, 0x13, 0xec, 0x01, 0x3b, 0xfd, 0x2b, 0x04, 0x3b, 0x1d, 0x00, 0xda, 0x3a, 0x03, 0x54, 0x03, 0x3b, 0x18, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xba, 0xf7, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0xa8, 0x00, 0x00, 0x00, 0xa4, 0xf7, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x9e, 0xff, 0x36, 0x8d, 0x65, 0x04, 0x37, 0xab, 0x06, 0x03, 0x37, 0xbe, 0xa5, 0x00, 0x37, 0x82, 0x6e, 0x02, 0x37, 0xae, 0xb0, 0x04, 0x37, 0xf9, 0xda, 0xda, 0x36, 0xdb, 0xd7, 0x03, 0x37, 0x28, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x82, 0xf8, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0xe8, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0xfc, 0x00, 0x00, 0x00, 0x6c, 0xf8, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x25, 0xd6, 0xc3, 0x3a, 0x4a, 0x44, 0xae, 0x3a, 0x3e, 0x5a, 0xb7, 0x3a, 0xdc, 0x89, 0xbc, 0x3a, 0x36, 0x64, 0xb9, 0x3a, 0xe7, 0x85, 0xc2, 0x3a, 0xd7, 0x52, 0xd0, 0x3a, 0x12, 0x17, 0xae, 0x3a, 0x3b, 0x1d, 0xe3, 0x3a, 0xd6, 0x2f, 0xba, 0x3a, 0xc2, 0x6d, 0xb8, 0x3a, 0x3b, 0x94, 0xc2, 0x3a, 0x15, 0xf4, 0xbe, 0x3a, 0xf7, 0x61, 0xc2, 0x3a, 0x64, 0xbc, 0xb5, 0x3a, 0x2e, 0x90, 0xd1, 0x3a, 0x1a, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xaa, 0xf9, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0xe4, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x08, 0x01, 0x00, 0x00, 0x94, 0xf9, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x3c, 0x83, 0x91, 0x37, 0x53, 0x7c, 0x81, 0x37, 0x94, 0x3c, 0x88, 0x37, 0x0a, 0x17, 0x8c, 0x37, 0x6b, 0xc0, 0x89, 0x37, 0x65, 0x89, 0x90, 0x37, 0x7a, 0xca, 0x9a, 0x37, 0xb9, 0x5a, 0x81, 0x37, 0xc0, 0xc0, 0xa8, 0x37, 0xb8, 0x57, 0x8a, 0x37, 0x4c, 0x09, 0x89, 0x37, 0x0b, 0x94, 0x90, 0x37, 0x66, 0xe2, 0x8d, 0x37, 0xb1, 0x6e, 0x90, 0x37, 0x13, 0x09, 0x87, 0x37, 0x46, 0xb6, 0x9b, 0x37, 0x2a, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x31, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xd2, 0xfa, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x48, 0x01, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x5c, 0x01, 0x00, 0x00, 0xbc, 0xfa, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0xcc, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0xbf, 0xa4, 0x97, 0x3a, 0xce, 0x7c, 0x98, 0x3a, 0x9f, 0x9b, 0x9e, 0x3a, 0x8e, 0x2d, 0x9b, 0x3a, 0xb8, 0xe3, 0x8e, 0x3a, 0xf6, 0x45, 0xa1, 0x3a, 0x45, 0x20, 0xa7, 0x3a, 0xd1, 0x8d, 0x95, 0x3a, 0xfe, 0x01, 0xa8, 0x3a, 0x21, 0x00, 0x99, 0x3a, 0xdc, 0x67, 0x8d, 0x3a, 0xce, 0xdd, 0xaa, 0x3a, 0xcc, 0x6c, 0xc5, 0x3a, 0x8a, 0x69, 0x99, 0x3a, 0xa8, 0x87, 0xb1, 0x3a, 0xde, 0x44, 0x90, 0x3a, 0x62, 0x6f, 0xab, 0x3a, 0x57, 0xb2, 0xa4, 0x3a, 0x6a, 0x22, 0x9d, 0x3a, 0xbb, 0x12, 0x9b, 0x3a, 0x63, 0x4d, 0x9a, 0x3a, 0xf0, 0x10, 0xb7, 0x3a, 0x8a, 0xaa, 0xa2, 0x3a, 0xc1, 0x8c, 0x9e, 0x3a, 0x1a, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x32, 0x2f, 0x43, 0x6f, 0x6e, 0x76, 0x32, 0x44, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x5a, 0xfc, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x44, 0x01, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x68, 0x01, 0x00, 0x00, 0x44, 0xfc, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0xc8, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x85, 0xe1, 0x9b, 0x37, 0x9e, 0xbf, 0x9c, 0x37, 0x37, 0x0a, 0xa3, 0x37, 0x9c, 0x83, 0x9f, 0x37, 0xdf, 0xe1, 0x92, 0x37, 0x9f, 0xc7, 0xa5, 0x37, 0xcc, 0xcb, 0xab, 0x37, 0xa4, 0xbb, 0x99, 0x37, 0xd4, 0xb3, 0xac, 0x37, 0x9c, 0x46, 0x9d, 0x37, 0x66, 0x5b, 0x91, 0x37, 0x16, 0xa4, 0xaf, 0x37, 0x11, 0xf1, 0xca, 0x37, 0xf7, 0xb2, 0x9d, 0x37, 0x9b, 0x7d, 0xb6, 0x37, 0xe3, 0x4c, 0x94, 0x37, 0xbc, 0x39, 0xb0, 0x37, 0x7d, 0x4c, 0xa9, 0x37, 0x77, 0x86, 0xa1, 0x37, 0x0a, 0x68, 0x9f, 0x37, 0x2e, 0x9d, 0x9e, 0x37, 0x7e, 0x2e, 0xbc, 0x37, 0x2a, 0x36, 0xa7, 0x37, 0xee, 0xfa, 0xa2, 0x37, 0x2a, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6f, 0x6e, 0x76, 0x32, 0x64, 0x5f, 0x32, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0xe2, 0xfd, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0xcc, 0xfd, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xe9, 0xc0, 0x06, 0x3b, 0x17, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x4a, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x50, 0x00, 0x00, 0x00, 0x34, 0xfe, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xbe, 0x10, 0x04, 0x38, 0x27, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x01, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xba, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x54, 0x00, 0x00, 0x00, 0xa4, 0xfe, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc4, 0x6e, 0x5b, 0x3b, 0x25, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x32, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x60, 0x00, 0x00, 0x00, 0x1c, 0xff, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x12, 0x17, 0xc8, 0x37, 0x35, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0xb2, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x18, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x2c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x66, 0x6c, 0x61, 0x74, 0x74, 0x65, 0x6e, 0x2f, 0x43, 0x6f, 0x6e, 0x73, 0x74, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x00, 0x1c, 0x00, 0x18, 0x00, 0x17, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x50, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x81, 0x80, 0x80, 0x3b, 0x19, 0x00, 0x00, 0x00, 0x73, 0x65, 0x72, 0x76, 0x69, 0x6e, 0x67, 0x5f, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x5f, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x5f, 0x31, 0x3a, 0x30, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x68, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xb8, 0xff, 0xff, 0xff, 0x19, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0xc8, 0xff, 0xff, 0xff, 0x09, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x0c, 0x00, 0x0c, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0xf0, 0xff, 0xff, 0xff, 0x11, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x0c, 0x00, 0x10, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03 };
    };
}