#pragma once
#include <Arduino.h>
#include <tensorflow-runtime-universal.h>

#ifdef __has_attribute
#define HAVE_ATTRIBUTE(x) __has_attribute(x)
#else
#define HAVE_ATTRIBUTE(x) 0
#endif
#if HAVE_ATTRIBUTE(aligned) || (defined(__GNUC__) && !defined(__clang__))
#define DATA_ALIGN_ATTRIBUTE __attribute__((aligned(4)))
#else
#define DATA_ALIGN_ATTRIBUTE
#endif

namespace tinyml4all {
    template<size_t tensorArenaSize>
    class MLP {
        public:
            const size_t numInputs = 4;
            const size_t numOutputs = 3;
            const size_t numOps = 6;
            
                const String targetNames[3] = { "setosa", "versicolor", "virginica" };
            

            const Model *model;
            MicroMutableOpResolver<6> resolver;
            MicroInterpreter *interpreter;
            TfLiteTensor *in;
            TfLiteTensor *out;
            uint8_t arena[tensorArenaSize];
            float outputs[3];
            float confidence;
            String label;

            /**
             * Constructor
             */
            MLP() :
                model(nullptr),
                interpreter(nullptr),
                in(nullptr),
                out(nullptr),
                label(""),
                _registered(false),
                _error("") {

                for (uint8_t i = 0; i < numOutputs; i++)
                    outputs[i] = 0;
            }

            /**
             * Test if model failed
             */
            inline bool failed() {
                return _error.length() > 0;
            }

            /**
             * Get error message
             */
            inline String error() {
                return _error;
            }

            /**
             * Test if input is float
             */
            inline bool isFloatInput() const {
                return in->type == kTfLiteFloat32;
            }

            /**
             * Test if input is int8
             */
            inline bool isInt8Input() const {
                return in->type == kTfLiteInt8;
            }

            /**
             * Test if output is float
             */
            inline bool isFloatOutput() const {
                return out->type == kTfLiteFloat32;
            }

            /**
             * Test if output is int8
             */
            inline bool isInt8Output() const {
                return out->type == kTfLiteInt8;
            }

            /**
             * Get i-th output
             */
            float output(uint16_t i = 0) {
                if (outputs == NULL || i >= numOutputs)
                    return sqrt(-1);

                return outputs[i];
            }

            /**
             * Init model
             */
            bool begin() {
                registerOps();
                model = tflite::GetModel(data);

                if (model->version() != TFLITE_SCHEMA_VERSION)
                    return fail(String("Model version mismatch. Expected ") + TFLITE_SCHEMA_VERSION + ", got " + model->version());

                interpreter = tf::newInterpreter<6>(&resolver, model, arena, tensorArenaSize);

                if (interpreter->AllocateTensors() != kTfLiteOk)
                    return fail("AllocateTensors() failed");

                in = interpreter->input(0);
                out = interpreter->output(0);

                if (!isFloatInput() && !isInt8Input())
                    return fail("Only float and int8 inputs are supported");

                if (!isFloatOutput() && !isInt8Output())
                    return fail("Only float and int8 outputs are supported");

                return fail();
            }

            /**
             * Predict float input
             */
            bool predict(float *x) {
                const float inputScale = in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale;
                const float outputOffset = out->params.zero_point;

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>(x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.int8[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();

                return fail();
            }

            /**
             * Predict int8 input
             */
            bool predict(int8_t *x) {
                const float inputScale = in->params.scale;
                const float inputOffset = in->params.zero_point;
                const float outputScale = out->params.scale;
                const float outputOffset = out->params.zero_point;

                if (model == nullptr && !begin())
                    return false;

                if (failed())
                    return false;

                // set (quantized) input
                if (isFloatInput())
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.f[i] = x[i];
                else
                  for (uint16_t i = 0; i < numInputs; i++)
                      in->data.int8[i] = static_cast<int8_t>(x[i] / inputScale + inputOffset);

                if (interpreter->Invoke() != kTfLiteOk)
                    return fail("Invoke() failed");

                // get (quantized) output
                if (isFloatOutput())
                    for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = out->data.int8[i];
                else
                  for (uint16_t i = 0; i < numOutputs; i++)
                      outputs[i] = ((float) out->data.int8[i] - outputOffset) * outputScale;

                getClassificationResult();

                return fail();
            }

            /**
             * Convert results to String for debug
             */
            String outputsAsString() {
              String result = "";
              result.reserve(numOutputs * 6);

              for (int i = 0; i < numOutputs; i++) {
                if (i > 0)
                  result += ", ";

                result += outputs[i];
              }

              return result;
            }

        protected:
            bool _registered;
            String _error;

            /**
             * Register ops in resolver
             */
            void registerOps() {
                if (_registered)
                    return;

                resolver.AddShape();
                resolver.AddFullyConnected();
                resolver.AddDequantize();
                resolver.AddQuantize();
                resolver.AddSoftmax();
                resolver.AddReshape();
                
                _registered = true;
            }

            /**
             * Set error message, if any
             */
            bool fail(String message = "") {
                _error = message;

                // return true if error is empty (success)
                return _error.length() == 0;
            }

            /**
             * If classification, get most probable class
             */
            void getClassificationResult() {
                if (numOutputs < 2)
                    return;

                confidence = outputs[0];
                uint8_t ix = 0;

                for (uint16_t i = 1; i < numOutputs; i++) {
                    if (outputs[i] > confidence) {
                        ix = i;
                        confidence = outputs[i];
                    }
                }

                
                    label = targetNames[ix];
                
            }

            const unsigned char data[3760] DATA_ALIGN_ATTRIBUTE = { 0x1c, 0x00, 0x00, 0x00, 0x54, 0x46, 0x4c, 0x33, 0x14, 0x00, 0x20, 0x00, 0x1c, 0x00, 0x18, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x98, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x64, 0x05, 0x00, 0x00, 0x74, 0x05, 0x00, 0x00, 0x44, 0x0e, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x73, 0x65, 0x72, 0x76, 0x69, 0x6e, 0x67, 0x5f, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x90, 0xff, 0xff, 0xff, 0x0d, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x86, 0xfa, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x5f, 0x31, 0x00, 0x02, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xdc, 0xff, 0xff, 0xff, 0x10, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x43, 0x4f, 0x4e, 0x56, 0x45, 0x52, 0x53, 0x49, 0x4f, 0x4e, 0x5f, 0x4d, 0x45, 0x54, 0x41, 0x44, 0x41, 0x54, 0x41, 0x00, 0x08, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x6d, 0x69, 0x6e, 0x5f, 0x72, 0x75, 0x6e, 0x74, 0x69, 0x6d, 0x65, 0x5f, 0x76, 0x65, 0x72, 0x73, 0x69, 0x6f, 0x6e, 0x00, 0x11, 0x00, 0x00, 0x00, 0x70, 0x04, 0x00, 0x00, 0x68, 0x04, 0x00, 0x00, 0x4c, 0x04, 0x00, 0x00, 0xf4, 0x03, 0x00, 0x00, 0x84, 0x03, 0x00, 0x00, 0xf4, 0x01, 0x00, 0x00, 0xa4, 0x01, 0x00, 0x00, 0x14, 0x01, 0x00, 0x00, 0xe4, 0x00, 0x00, 0x00, 0xb4, 0x00, 0x00, 0x00, 0xac, 0x00, 0x00, 0x00, 0xa4, 0x00, 0x00, 0x00, 0x9c, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x8c, 0x00, 0x00, 0x00, 0x6c, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x3e, 0xfb, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x0e, 0x00, 0x08, 0x00, 0x04, 0x00, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x04, 0x00, 0x06, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xeb, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x32, 0x2e, 0x31, 0x35, 0x2e, 0x31, 0x00, 0x00, 0xa2, 0xfb, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x31, 0x2e, 0x31, 0x34, 0x2e, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x8c, 0xfb, 0xff, 0xff, 0x90, 0xfb, 0xff, 0xff, 0x94, 0xfb, 0xff, 0xff, 0x98, 0xfb, 0xff, 0xff, 0x9c, 0xfb, 0xff, 0xff, 0xd2, 0xfb, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x5a, 0x81, 0xef, 0x6b, 0x97, 0xcd, 0x14, 0xaa, 0xb2, 0xa2, 0x15, 0xe0, 0x74, 0xf7, 0x56, 0x78, 0x5e, 0x36, 0xe5, 0x8f, 0xa5, 0xfb, 0xeb, 0x51, 0xd1, 0xc1, 0x51, 0x7d, 0xfb, 0xd5, 0x15, 0xf9, 0xfe, 0xfb, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xc7, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x74, 0xff, 0xff, 0xff, 0x6e, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x2a, 0xfc, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xf7, 0x27, 0xdb, 0xa2, 0x19, 0x58, 0x09, 0xe9, 0xeb, 0xad, 0xa6, 0xfb, 0xe4, 0x17, 0xa6, 0x29, 0xac, 0xfd, 0xf3, 0x05, 0x5f, 0x08, 0x72, 0x20, 0xff, 0xdc, 0x54, 0x0f, 0x47, 0xfd, 0x0b, 0x3b, 0xc6, 0x52, 0x14, 0x50, 0x5a, 0xf8, 0xeb, 0x11, 0x28, 0xf5, 0xde, 0xc1, 0x47, 0xf1, 0x42, 0x22, 0x34, 0xeb, 0x50, 0xf7, 0xfb, 0x23, 0x7d, 0x24, 0x25, 0x1f, 0x26, 0xe3, 0x3b, 0x04, 0xb0, 0xee, 0xf6, 0xe7, 0x39, 0xf4, 0x71, 0x24, 0xea, 0x20, 0x5b, 0x23, 0xd1, 0xb3, 0xb5, 0x36, 0x07, 0x22, 0x3c, 0x5c, 0x34, 0x62, 0x8f, 0xb5, 0x53, 0xff, 0xd9, 0x05, 0xa0, 0x2f, 0xa0, 0x13, 0x7f, 0x57, 0x6f, 0x1b, 0x17, 0x4b, 0xad, 0x1e, 0xbc, 0xa6, 0xc0, 0x26, 0x1b, 0xf7, 0xf6, 0xeb, 0x1f, 0x01, 0x56, 0x05, 0xcc, 0x50, 0xa2, 0xd9, 0x21, 0xeb, 0x3a, 0x3c, 0xce, 0x2c, 0xdf, 0x5e, 0x3f, 0xd6, 0xb6, 0xfc, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x00, 0x44, 0x01, 0x00, 0x00, 0x3d, 0xff, 0xff, 0xff, 0xb6, 0xff, 0xff, 0xff, 0x12, 0xff, 0xff, 0xff, 0x8f, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd4, 0xfe, 0xff, 0xff, 0xec, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xd8, 0xff, 0xff, 0xff, 0x48, 0xfe, 0xff, 0xff, 0x02, 0xfd, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x80, 0x01, 0x00, 0x00, 0xd4, 0xdd, 0x2b, 0x3f, 0x21, 0x48, 0x4b, 0xd0, 0xaf, 0xee, 0x00, 0xd0, 0x16, 0x0f, 0x3d, 0x3f, 0xc1, 0x22, 0x68, 0x56, 0xed, 0xf7, 0x73, 0x63, 0x21, 0xc4, 0xc0, 0x7f, 0xa9, 0xbc, 0x45, 0xe7, 0x4c, 0xb3, 0xb4, 0x20, 0xe0, 0xb4, 0x0f, 0x1c, 0xe4, 0xe3, 0xe4, 0x31, 0x33, 0xef, 0xc0, 0xec, 0x37, 0xb1, 0x15, 0x50, 0x5e, 0xb7, 0x12, 0xd1, 0xfd, 0xb3, 0xb2, 0x4a, 0xd6, 0x46, 0xaa, 0xad, 0x45, 0xc0, 0x12, 0x0a, 0xc6, 0x2b, 0xc2, 0x49, 0xac, 0x2f, 0x0e, 0xe4, 0x4c, 0xb8, 0xcf, 0xd3, 0x17, 0xca, 0xef, 0xc6, 0x17, 0x20, 0x57, 0xd0, 0xf9, 0xaa, 0xe7, 0x51, 0x3f, 0x4d, 0xb8, 0xb1, 0xa8, 0xb4, 0x34, 0xf8, 0x52, 0xd7, 0xe2, 0xcd, 0xc6, 0xda, 0x4b, 0xe4, 0x4c, 0xe0, 0x4e, 0x03, 0x18, 0x42, 0xc1, 0xae, 0x3b, 0xf2, 0x9c, 0x07, 0x12, 0x2c, 0x2e, 0xa9, 0x36, 0x03, 0x59, 0x2a, 0x13, 0xfb, 0x0e, 0xa9, 0xfa, 0xae, 0x04, 0x2f, 0xad, 0xea, 0xd7, 0xcc, 0xbc, 0xac, 0xd1, 0x43, 0x04, 0x2e, 0xe3, 0x48, 0x5c, 0x03, 0xe2, 0x40, 0x6a, 0x25, 0xd5, 0x81, 0x1b, 0x06, 0xfc, 0xb5, 0x39, 0xf0, 0xa8, 0x21, 0xab, 0x20, 0x07, 0xf6, 0xef, 0x23, 0x4f, 0x46, 0xba, 0x3b, 0xee, 0x2a, 0x05, 0x36, 0xbf, 0x51, 0x43, 0xe4, 0x0e, 0xbd, 0x06, 0xcc, 0x36, 0xb6, 0xa2, 0xb0, 0x07, 0xf9, 0xe3, 0xe8, 0xb3, 0x3b, 0x4f, 0xb2, 0x7e, 0x28, 0x8a, 0xd7, 0x6a, 0x7e, 0x4c, 0x10, 0x1f, 0xc5, 0x2f, 0x30, 0x69, 0x12, 0x58, 0xbd, 0x2c, 0xf3, 0x45, 0xc9, 0xa1, 0xfc, 0xb4, 0xdd, 0x3a, 0xeb, 0x36, 0xe9, 0x2b, 0x45, 0x32, 0xb5, 0xf8, 0xfb, 0x2e, 0xe5, 0xd1, 0xe9, 0x18, 0x06, 0x02, 0x02, 0xb2, 0x4d, 0xc9, 0x2d, 0xcf, 0xe1, 0xc0, 0x07, 0x54, 0xd2, 0xec, 0x46, 0xe9, 0xcd, 0xe2, 0x09, 0xad, 0xea, 0x13, 0xcf, 0xe9, 0x11, 0xac, 0xc2, 0xe0, 0x47, 0xc1, 0x45, 0x2d, 0x0b, 0x03, 0xd1, 0x30, 0x23, 0x4a, 0xfe, 0xc5, 0xe4, 0x5a, 0x11, 0x14, 0x1b, 0x2d, 0x19, 0x4b, 0xf5, 0x16, 0x15, 0x13, 0x11, 0xac, 0x13, 0xcd, 0x42, 0x4e, 0x45, 0xba, 0xf1, 0xfd, 0xaf, 0xdd, 0x50, 0xd8, 0xb5, 0x0b, 0x01, 0xe4, 0x25, 0x0e, 0x52, 0x46, 0x4f, 0xe3, 0x58, 0xd4, 0x1c, 0xf9, 0x0f, 0x00, 0xaa, 0xfa, 0xfe, 0x22, 0xa7, 0xfd, 0x47, 0xa7, 0x23, 0xc5, 0x46, 0x34, 0x09, 0x38, 0xb2, 0x00, 0xb6, 0xa8, 0x1f, 0x54, 0xe0, 0xdf, 0xdf, 0xdf, 0xb1, 0xfc, 0x23, 0xad, 0x4a, 0x57, 0xc0, 0x1d, 0xd9, 0xf7, 0xce, 0xf9, 0xf4, 0xb3, 0x3d, 0x51, 0xcc, 0x58, 0xee, 0xdd, 0xe2, 0x28, 0x27, 0xb9, 0x1e, 0xf1, 0x47, 0xc3, 0x38, 0x2c, 0x37, 0xcc, 0x3f, 0xf4, 0x1b, 0xaa, 0x39, 0xc3, 0x08, 0x40, 0xda, 0x8e, 0xfe, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xe5, 0xfc, 0xff, 0xff, 0xa7, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xff, 0xff, 0xff, 0x11, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x62, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x64, 0x03, 0x00, 0x00, 0x42, 0xfc, 0xff, 0xff, 0x53, 0x04, 0x00, 0x00, 0x98, 0x03, 0x00, 0x00, 0x81, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xe1, 0xfc, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0xbe, 0xfe, 0xff, 0xff, 0x96, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0xff, 0xff, 0xff, 0xfa, 0xfe, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0xdd, 0x52, 0xca, 0x25, 0x9f, 0x25, 0xad, 0x81, 0xb4, 0x0a, 0xf2, 0x13, 0xba, 0x18, 0x5b, 0xcb, 0x29, 0xa5, 0xb1, 0xc1, 0x5a, 0xe7, 0xa0, 0x0f, 0xd9, 0x85, 0xf9, 0xa8, 0x34, 0x9f, 0x0f, 0x13, 0xcc, 0x32, 0x56, 0xfc, 0xe2, 0xcb, 0x3a, 0x47, 0x24, 0xed, 0x63, 0xd7, 0x45, 0xba, 0x61, 0xc7, 0x49, 0x3a, 0xa0, 0x2e, 0xc6, 0x0c, 0xc0, 0xd1, 0xb4, 0xa8, 0x01, 0xe9, 0x57, 0xa5, 0x9a, 0x41, 0xcd, 0x69, 0x09, 0xe3, 0xfc, 0x1c, 0xee, 0x41, 0x4e, 0xff, 0xff, 0xff, 0x04, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x59, 0x02, 0x00, 0x00, 0xd9, 0xff, 0xff, 0xff, 0x8a, 0xfd, 0xff, 0xff, 0x34, 0xff, 0xff, 0xff, 0x38, 0xff, 0xff, 0xff, 0x0f, 0x00, 0x00, 0x00, 0x4d, 0x4c, 0x49, 0x52, 0x20, 0x43, 0x6f, 0x6e, 0x76, 0x65, 0x72, 0x74, 0x65, 0x64, 0x2e, 0x00, 0x01, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x18, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x60, 0x01, 0x00, 0x00, 0x64, 0x01, 0x00, 0x00, 0x68, 0x01, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x6d, 0x61, 0x69, 0x6e, 0x00, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x08, 0x01, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x88, 0x00, 0x00, 0x00, 0x50, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x1a, 0x00, 0x14, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x1c, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x04, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3f, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x62, 0xff, 0xff, 0xff, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x04, 0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x96, 0xff, 0xff, 0xff, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x86, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xca, 0xff, 0xff, 0xff, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0xba, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x16, 0x00, 0x00, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x0b, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x18, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x08, 0x00, 0x07, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0xc4, 0x06, 0x00, 0x00, 0x24, 0x06, 0x00, 0x00, 0xa8, 0x05, 0x00, 0x00, 0x2c, 0x05, 0x00, 0x00, 0xc0, 0x04, 0x00, 0x00, 0x44, 0x04, 0x00, 0x00, 0xd8, 0x03, 0x00, 0x00, 0x64, 0x03, 0x00, 0x00, 0xf8, 0x02, 0x00, 0x00, 0x5c, 0x02, 0x00, 0x00, 0xb8, 0x01, 0x00, 0x00, 0x14, 0x01, 0x00, 0x00, 0x70, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x8a, 0xf9, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0x74, 0xf9, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x3b, 0x19, 0x00, 0x00, 0x00, 0x53, 0x74, 0x61, 0x74, 0x65, 0x66, 0x75, 0x6c, 0x50, 0x61, 0x72, 0x74, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x65, 0x64, 0x43, 0x61, 0x6c, 0x6c, 0x3a, 0x30, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0xf2, 0xf9, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x7c, 0x00, 0x00, 0x00, 0xdc, 0xf9, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2a, 0xfe, 0x1f, 0x3d, 0x4c, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x92, 0xfa, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x7c, 0x00, 0x00, 0x00, 0x7c, 0xfa, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x32, 0x81, 0xd4, 0x3c, 0x4c, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x32, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x32, 0x2f, 0x52, 0x65, 0x6c, 0x75, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x32, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x32, 0xfb, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x7c, 0x00, 0x00, 0x00, 0x1c, 0xfb, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x88, 0xfe, 0xaf, 0x3c, 0x4c, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x31, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x31, 0x2f, 0x52, 0x65, 0x6c, 0x75, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x31, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0xd2, 0xfb, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x74, 0x00, 0x00, 0x00, 0xbc, 0xfb, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xe5, 0xff, 0x29, 0x3d, 0x46, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x52, 0x65, 0x6c, 0x75, 0x3b, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x6a, 0xfc, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0x54, 0xfc, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x69, 0x58, 0xc8, 0x3b, 0x17, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xd2, 0xfc, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x50, 0x00, 0x00, 0x00, 0xbc, 0xfc, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xec, 0x9d, 0x46, 0x39, 0x27, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x42, 0xfd, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0x2c, 0xfd, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6d, 0xc1, 0xa9, 0x3b, 0x19, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x31, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0xaa, 0xfd, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x00, 0x94, 0xfd, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xc1, 0x74, 0x61, 0x39, 0x29, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x31, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x22, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x44, 0x00, 0x00, 0x00, 0x0c, 0xfe, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x21, 0x99, 0x8f, 0x3b, 0x19, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x32, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x8a, 0xfe, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x00, 0x74, 0xfe, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xe8, 0x70, 0xc5, 0x38, 0x29, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x64, 0x65, 0x6e, 0x73, 0x65, 0x5f, 0x32, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x54, 0x00, 0x00, 0x00, 0xec, 0xfe, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x2b, 0x02, 0x9a, 0x3b, 0x25, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x4d, 0x61, 0x74, 0x4d, 0x75, 0x6c, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x7a, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01, 0x14, 0x00, 0x00, 0x00, 0x34, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x64, 0x00, 0x00, 0x00, 0x64, 0xff, 0xff, 0xff, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x0a, 0xaf, 0xff, 0x38, 0x35, 0x00, 0x00, 0x00, 0x73, 0x65, 0x71, 0x75, 0x65, 0x6e, 0x74, 0x69, 0x61, 0x6c, 0x2f, 0x63, 0x6c, 0x61, 0x73, 0x73, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x5f, 0x68, 0x65, 0x61, 0x64, 0x2f, 0x42, 0x69, 0x61, 0x73, 0x41, 0x64, 0x64, 0x2f, 0x52, 0x65, 0x61, 0x64, 0x56, 0x61, 0x72, 0x69, 0x61, 0x62, 0x6c, 0x65, 0x4f, 0x70, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x00, 0x1c, 0x00, 0x18, 0x00, 0x17, 0x00, 0x10, 0x00, 0x0c, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x20, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x54, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x98, 0xca, 0xfd, 0x3c, 0x19, 0x00, 0x00, 0x00, 0x73, 0x65, 0x72, 0x76, 0x69, 0x6e, 0x67, 0x5f, 0x64, 0x65, 0x66, 0x61, 0x75, 0x6c, 0x74, 0x5f, 0x69, 0x6e, 0x70, 0x75, 0x74, 0x5f, 0x31, 0x3a, 0x30, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0xf0, 0xff, 0xff, 0xff, 0x19, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x19, 0x0c, 0x00, 0x10, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09 };
    };
}