/*
 * @file EZ_USB_MIDI_HOST_Config.h
 * @brief Arduino MIDI Library compatible wrapper for usb_midi_host
 *        application driver
 *
 * This library manages all USB MIDI devices connected to the
 * single USB Root hub for TinyUSB for the whole plug in,
 * operate, unplug lifecycle. It makes each virtual MIDI cable for
 * each connected USB MIDI device behave as if it were a serial port
 * MIDI device and enables applications to use the Arduino MIDI Library
 * (formally called the FortySevenEffects MIDI library) to send and
 * receive MIDI messages between the application and the device.
 *
 * Most applications should only instantiate the EZ_USB_MIDI_HOST
 * class by calling EZ_USB_MIDI_HOST::instance(); e.g.
 *   auto usbhMIDI = EZ_USB_MIDI_HOST::instance();
 *
 * Please see the CONFIGURATION section below to allow your application
 * to tailor the memory utilization of this class
 *
 * The MIT License (MIT)
 *
 * Copyright (c) 2023 rppicomidi
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#pragma once
#include "midi_Settings.h"
#include "midi_Namespace.h"
#include "EZ_USB_MIDI_HOST_namespace.h"
/// Default maximum number of connected MIDI devices supported
#define RPPICOMIDI_TUH_MIDI_MAX_DEV_DEFAULT CFG_TUH_DEVICE_MAX

#ifndef RPPICOMIDI_TUH_MIDI_MAX_DEV
#define RPPICOMIDI_TUH_MIDI_MAX_DEV RPPICOMIDI_TUH_MIDI_MAX_DEV_DEFAULT
#endif

/// Because the MIDI Library send() and sendSysEx() libraries
/// will try to send every byte in the sysex message all at once,
/// and because the  USB transmitter system can only send as many
/// bytes at once as the endpoint can support (usually 64 bytes),
/// and because the write() method of the Transport class can't
/// report an error that the MIDI Library will do anything about,
/// the USB transmitter FIFO has to be able to buffer an entire
/// maximum size system exclusive message. The buffer must be a
/// multiple of 4 bytes. The sysex buffer from messages may not
/// have the F0 and F7 bytes, but the USB packet needs to send them.
#define RPPICOMIDI_EZ_USB_MIDI_HOST_GET_BUFSIZE(MaxSysExPayload)  (((((MaxSysExPayload) + 2) / 3) + 1) * 4)
BEGIN_EZ_USB_MIDI_HOST_NAMESPACE
/// This structure contains the default settings class
/// for the EZ_USB_MIDI_HOST class and the Arduino MIDI class.
/// If you want to use the defaults, pass MidiHostSettingsDefault to
/// the second argument of the RPPICOMIDI_EZ_USB_MIDI_HOST_INSTANCE()
/// macro. If you want to modify the settings, do not edit this file.
/// Instead, make a subclass of this structure and assign new values.
/// That is what this structure does to override some settings in the
/// Arduino MIDI Library DefaultSettings class.
struct MidiHostSettingsDefault : public MIDI_NAMESPACE::DefaultSettings
{
    /// Note Off is a different message than Note On with 0 velocity,
    ///    especially for Mackie Control protocol
    static const bool HandleNullVelocityNoteOnAsNoteOff = false;

    /// USB messages always come in packets, so 1 byte parsing is a bad match.
    /// However, if don't use 1 byte parsing, then it appears parse() can
    /// be called recursively RPPICOMIDI_TUH_MIDI_MAX_SYSEX times, which
    /// will likely cause issues.
    static const bool Use1ByteParsing = true;

    /// The maximum length of a System Exclusive message payload (excluding 0xF0 and 0xF7 bytes)
    static const unsigned SysExMaxSize = 128;
    /// The buffer size for receiving and transmitting messages should be large enough to
    /// hold and entire system exclusive message. If your attached devices do not use
    /// system exclusive messages, you can save system memory by overriding the bufsize
    /// values in a subclass of this struct, but you should make the buffers no shorter than
    /// 64 bytes each.
    static const unsigned MidiRxBufsize = RPPICOMIDI_EZ_USB_MIDI_HOST_GET_BUFSIZE(SysExMaxSize);
    static const unsigned MidiTxBufsize = RPPICOMIDI_EZ_USB_MIDI_HOST_GET_BUFSIZE(SysExMaxSize);
    /// USB MIDI packets can be routed to one of up to 16 virtual cables. Each virtual cable
    /// consumes a MIDI Library transport class object. Most MIDI devices have fewer than 16
    /// virtual cables. You can save memory by overriding this value in a new subclass of
    /// this struct, but MaxCables must be at least 1.
    static const unsigned MaxCables = 16;
};
END_EZ_USB_MIDI_HOST_NAMESPACE
