///
/// @file Screen_EPD.h
/// @brief Library for Pervasive Displays screens and boards - Basic edition
///
/// @details Project Pervasive Displays Library Suite
/// @n Based on highView technology
///
/// @date 21 Dec 2025
/// @version 1001
///
/// @copyright (c) Pervasive Displays Inc., 2021-2025
/// @copyright (c) Etigues, 2010-2025
/// @copyright All rights reserved
/// @copyright For exclusive use with Pervasive Displays screens
///
/// * Basic edition: for hobbyists and for basic usage
/// @n Creative Commons Attribution-ShareAlike 4.0 International (CC BY-SA 4.0)
/// @see https://creativecommons.org/licenses/by-sa/4.0/
///
/// @n Consider the Evaluation or Commercial editions for professionals or organisations and for commercial usage
///
/// * Evaluation edition: for professionals or organisations, evaluation only, no commercial usage
/// @n All rights reserved
///
/// * Commercial edition: for professionals or organisations, commercial usage
/// @n All rights reserved
///
/// * Viewer edition: for professionals or organisations
/// @n All rights reserved
///
/// * Documentation
/// @n All rights reserved
///

// SDK and configuration
#include "PDLS_Common.h"

#if (PDLS_COMMON_RELEASE < 1000)
#error Required PDLS_COMMON_RELEASE 1000
#endif // PDLS_COMMON_RELEASE

// Other libraries
#include "hV_Screen_Buffer.h"

#if (hV_SCREEN_BUFFER_RELEASE < 1000)
#error Required hV_SCREEN_BUFFER_RELEASE 1000
#endif // hV_SCREEN_BUFFER_RELEASE

#ifndef SCREEN_EPD_RELEASE
///
/// @brief Library release number
///
#define SCREEN_EPD_RELEASE 1001

#include "Driver_EPD_Virtual.h"

///
/// @brief Library variant
///
#define SCREEN_EPD_VARIANT "Basic"

// Objects
//
///
/// @brief Class for Pervasive Displays iTC screens
/// @details Screen controllers
/// * LCD: Proprietary, SPI
/// * Touch: Proprietary, I2C
/// * Fonts: No external Flash
///
/// @note All commands work on the frame-buffer,
/// to be displayed on screen with flush()
///
class Screen_EPD final : public hV_Screen_Buffer
{
  public:
    ///
    /// @brief Constructor with default pins
    /// @param driver &driver to link Screen_EPD to
    /// @note Frame-buffer generated by the class
    /// @note To be used with begin() with no parameter
    ///
    Screen_EPD(Driver_EPD_Virtual * driver);

    ///
    /// @brief Initialisation
    /// @note Frame-buffer generated internally, not suitable for FRAM
    /// @warning begin() initialises GPIOs and reads OTP
    ///
    void begin();

    ///
    /// @brief Clear the screen
    /// @param colour default = white
    /// @note Clear next frame-buffer
    ///
    void clear(uint16_t colour = myColours.white);

    ///
    /// @brief Update the display, normal update
    /// @note
    /// 1. Send the frame-buffer to the screen
    /// 2. Refresh the screen
    /// @warning When normal update not available, proxy for fast update
    ///
    void flush();

    ///
    /// @brief Update the display, fast update
    /// @note
    /// 1. Send the frame-buffer to the screen
    /// 2. Refresh the screen
    /// 3. Copy next frame-buffer into old frame-buffer
    /// @warning When fast update not available, proxy for normal update
    ///
    void flushFast();

    ///
    /// @brief Regenerate the panel
    /// @details White-to-black-to-white cycle to reduce ghosting
    /// @param mode default = UPDATE_NORMAL = normal mode
    ///
    void regenerate(uint8_t mode = UPDATE_NORMAL);

    //
    // === Power section
    //
    /// @brief Set the power profile
    /// @param mode default = POWER_MODE_AUTO, otherwise POWER_MODE_MANUAL
    /// @param scope default = POWER_SCOPE_GPIO_ONLY, otherwise POWER_SCOPE_GPIO_BUS, POWER_SCOPE_NONE
    /// @note If panelPower is NOT_CONNECTED, (POWER_MODE_AUTO, POWER_SCOPE_GPIO_ONLY) defaults to (POWER_MODE_MANUAL, POWER_SCOPE_NONE)
    /// @note Call suspend(POWER_SCOPE_GPIO_BUS) manually
    /// @note Advanced edition only
    ///
    void setPowerProfile(uint8_t mode = POWER_MODE_AUTO, uint8_t scope = POWER_SCOPE_GPIO_ONLY);

    ///
    /// @brief Suspend
    /// @param bus include SPI bus, default = POWER_SCOPE_GPIO_ONLY, otherwise POWER_SCOPE_BUS_GPIO or POWER_SCOPE_NONE
    /// @details Power off and set all GPIOs low, POWER_SCOPE_BUS_GPIO also turns SPI off
    /// @note If panelPower is NOT_CONNECTED, POWER_SCOPE_GPIO_ONLY defaults to POWER_SCOPE_NONE
    /// @note Advanced edition only
    ///
    void suspend(uint8_t suspendScope = POWER_SCOPE_GPIO_ONLY);

    ///
    /// @brief Resume after suspend()
    /// @details Turn SPI on and set all GPIOs levels
    /// @note Advanced edition only
    ///
    void resume();
    //
    // === End of Power section
    //

    //
    // === Temperature section
    //
    ///
    /// @brief Set panelPower pin
    /// @param panelPowerPin panelPower pin
    /// @note EXT4 requires panelPower
    /// @note If flashCS defined with panelPowerPin, then flashCS set to NOT_CONNECTED
    /// @warning setPanelPowerPin() should be called before begin()
    ///
    void setPanelPowerPin(uint8_t panelPowerPin = NOT_CONNECTED);

    ///
    /// @brief Set temperature in Celsius
    /// @details Set the temperature for update
    /// @param temperatureC temperature in °C, default = 25 °C
    /// @note Refer to data-sheets for authorised operating temperatures
    ///
    void setTemperatureC(int8_t temperatureC = 25);

    ///
    /// @brief Set temperature in Fahrenheit
    /// @details Set the temperature for update
    /// @param temperatureF temperature in °F, default = 77 °F = 25 °C
    /// @note Refer to data-sheets for authorised operating temperatures
    ///
    void setTemperatureF(int16_t temperatureF = 77);

    ///
    /// @brief Check the mode against the temperature
    ///
    /// @param updateMode expected update mode
    /// @return uint8_t recommended mode
    /// @note If required, defaulting to `UPDATE_NORMAL` or `UPDATE_NONE`
    /// @warning Default temperature is 25 °C, otherwise set by setTemperatureC() or setTemperatureF()
    ///
    uint8_t checkTemperatureMode(uint8_t updateMode);
    //
    // === End of Temperature section
    //

    //
    // === Miscellaneous section
    //
    ///
    /// @brief Who Am I
    /// @return Who Am I string
    ///
    virtual STRING_TYPE WhoAmI();

    ///
    /// @brief Library reference
    ///
    /// @return STRING_CONST_TYPE scope and release number
    /// @note Example `Advanced v9.0.0`
    ///
    virtual STRING_CONST_TYPE reference();

    ///
    /// @brief Get number of colours
    ///
    /// @return uint8_t number of colours
    /// * `2` = monochrome
    /// * `3` = black-white-red or black-white-yellow
    /// * `4` = black-white-red-yellow
    ///
    uint8_t screenColours();

    ///
    /// @brief Screen number
    /// @return Screen number as string
    /// @note Example `266-KS-0C`
    ///
    virtual STRING_TYPE screenNumber();

    ///
    /// @brief Recommend variant for film
    /// @param uint8_t Context film
    /// @note exit() called after
    ///
    void debugVariant(uint8_t contextFilm);
    //
    // === End of Miscellaneous section
    //

  protected:

    /// @cond NOT_PUBLIC

    // Orientation
    ///
    /// @brief Set orientation
    /// @param orientation 1..3, 6, 7
    ///
    void s_setOrientation(uint8_t orientation); // compulsory

    ///
    /// @brief Check and orient coordinates, logical coordinates
    /// @param x x-axis coordinate, modified
    /// @param y y-axis coordinate, modified
    /// @return `RESULT_SUCCESS` =  false = success, `RESULT_ERROR` =  true = error
    ///
    bool s_orientCoordinates(uint16_t & x, uint16_t & y); // compulsory

    // Write and Read
    /// @brief Set point
    /// @param x1 x coordinate
    /// @param y1 y coordinate
    /// @param colour 16-bit colour
    /// @n @b More: @ref Colour, @ref Coordinate
    ///
    void s_setPoint(uint16_t x1, uint16_t y1, uint16_t colour);

    /// @brief Get point
    /// @param x1 x coordinate
    /// @param y1 y coordinate
    /// @return colour 16-bit colour
    /// @n @b More: @ref Colour, @ref Coordinate
    ///
    uint16_t s_getPoint(uint16_t x1, uint16_t y1);

    ///
    /// @brief Update the screen
    /// @param updateMode update mode, default = `UPDATE_NORMAL`, otherwise `UPDATE_FAST`
    ///
    void s_flush(uint8_t updateMode = UPDATE_NORMAL);

    // Position
    ///
    /// @brief Convert
    /// @param x1 x-axis coordinate
    /// @param y1 y-axis coordinate
    /// @return index for s_newImage[]
    ///
    uint32_t s_getZ(uint16_t x1, uint16_t y1);

    ///
    /// @brief Convert
    /// @param x1 x-axis coordinate
    /// @param y1 y-axis coordinate
    /// @return bit for s_newImage[]
    ///
    uint16_t s_getB(uint16_t x1, uint16_t y1);

    //
    // === Energy section
    //

    //
    // === End of Energy section
    //

    // Variables specific to the screen
    Driver_EPD_Virtual * s_driver;

    uint16_t u_codeSize;
    uint8_t u_codeFilm;
    uint8_t u_codeDriver;
    uint8_t u_codeExtra;
    uint16_t u_bufferSizeV, u_bufferSizeH, u_bufferDepth;
    uint32_t u_pageColourSize;

    uint8_t u_suspendMode = POWER_MODE_AUTO;
    uint8_t u_suspendScope = POWER_SCOPE_GPIO_ONLY;

    //
    // === Touch section
    //
    virtual void s_getRawTouch(touch_t & touch);
    virtual bool s_getInterruptTouch();
    //
    // === End of Touch section
    //

    /// @endcond
};

#endif // SCREEN_EPD_RELEASE

