///
/// @file Touch_GUI.ino
/// @brief Example of features for Evaluation edition
///
/// @details Example for Pervasive Displays Library Suite
/// @n Based on highView technology
///
/// @date 21 Nov 2025
/// @version 1000
///
/// @copyright (c) Pervasive Displays Inc., 2021-2025
/// @copyright All rights reserved
/// @copyright For exclusive use with Pervasive Displays screens
///
/// * Basic edition: for hobbyists and for basic usage
/// @n Creative Commons Attribution-ShareAlike 4.0 International (CC BY-SA 4.0)
///
/// * Evaluation edition: for professionals or organisations, no commercial usage
/// @n All rights reserved
///
/// * Commercial edition: for professionals or organisations, commercial usage
/// @n All rights reserved
///

// Set parameters
#define DISPLAY_GUI 1

// SDK and configuration
// #include <Arduino.h>
#include "PDLS_Common.h"

// Board
pins_t myBoard = boardRaspberryPiPico_RP2040;

// Driver
#include "Pervasive_Touch_Small.h"
// Pervasive_Touch_Small myDriver(eScreen_EPD_370_PS_0C_Touch, myBoard);
Pervasive_Touch_Small myDriver(eScreen_EPD_271_KS_09_Touch, myBoard);

// Screen
#include "PDLS_Basic.h"
Screen_EPD myScreen(&myDriver);

// Checks
#if (SCREEN_EPD_RELEASE < 1000)
#error Required SCREEN_EPD_RELEASE 1000
#endif // SCREEN_EPD_RELEASE

// Fonts
uint8_t fontSmall, fontMedium, fontLarge, fontVery;

// GUI
#include "hV_GUI.h"

// Checks
#ifndef WITH_TOUCH
#error Required WITH_TOUCH
#endif // WITH_TOUCH

// Prototypes

// Utilities

// Functions
#if (DISPLAY_GUI == 1)

GUI myGUI(&myScreen);

void displayGUI()
{
    myScreen.setOrientation(ORIENTATION_LANDSCAPE);
    myScreen.selectFont(fontMedium);

    myScreen.clear();

    myGUI.begin();

    Button myButtonNormal(&myGUI);
    Button myButtonInstant(&myGUI);
    Text myText(&myGUI);

    uint16_t x = myScreen.screenSizeX();
    uint16_t y = myScreen.screenSizeY();

    uint16_t dx = x / 7;
    uint16_t dy = y / 5;

    myGUI.delegate(false);

    myButtonNormal.dStringDefine(dx * 1, dy * 3, dx * 2, dy, "Normal", fontMedium);
    myButtonInstant.dStringDefine(dx * 4, dy * 3, dx * 2, dy, "Instant", fontMedium);
    myText.dDefine(0, dy, x, dy, fontMedium);

    myButtonNormal.draw();
    myButtonInstant.draw();
    myText.draw("Empty");

    myScreen.flush();

    myGUI.delegate(true);

    uint8_t k = 8;
    uint32_t chrono32;
    while (k > 0)
    {
        if (myScreen.getTouchInterrupt())
        {
            chrono32 = hV_HAL_getMilliseconds();
            if (myButtonNormal.check(checkNormal))
            {
                k -= 1;
                chrono32 = hV_HAL_getMilliseconds() - chrono32;
                myText.draw(formatString("%s in %i ms (%i left)", "Normal", chrono32, k));
                hV_HAL_log(LEVEL_INFO, "%3i: %s in %i ms", k, "Normal", chrono32);
            }

            chrono32 = hV_HAL_getMilliseconds();
            if (myButtonInstant.check(checkInstant))
            {
                k -= 1;
                chrono32 = hV_HAL_getMilliseconds() - chrono32;
                myText.draw(formatString("%s in %i ms (%i left)", "Instant", chrono32, k));
                hV_HAL_log(LEVEL_INFO, "%3i: %s in %i ms", k, "Instant", chrono32);
            }
        } // getTouchInterrupt

        hV_HAL_delayMilliseconds(100);
    }

    myScreen.clear();
}
#endif // DISPLAY_GUI

///
/// @brief Setup
///
void setup()
{
    hV_HAL_begin();

    hV_HAL_Serial_crlf();
    hV_HAL_log(LEVEL_INFO, __FILE__);
    hV_HAL_log(LEVEL_INFO, __DATE__ " " __TIME__);
    hV_HAL_Serial_crlf();

    // Screen
    myScreen.begin();

    myScreen.regenerate();

    // Fonts
#if (FONT_MODE == USE_FONT_TERMINAL)

    fontSmall = Font_Terminal6x8;
    fontMedium = Font_Terminal8x12;
    fontLarge = Font_Terminal12x16;
    fontVery = Font_Terminal16x24;

#else // FONT_MODE

    fontSmall = myScreen.addFont(Font_Latin_DejaVuSans12);
    fontSmall -= ((fontSmall > 0) ? 1 : 0);
    fontMedium = myScreen.addFont(Font_Latin_DejaVuSans16);
    fontMedium -= ((fontMedium > 0) ? 1 : 0);
    fontLarge = myScreen.addFont(Font_Latin_DejaVuSans24);
    fontLarge -= ((fontLarge > 0) ? 1 : 0);
    fontVery = myScreen.addFont(Font_Latin_DejaVuMono48);
    fontVery -= ((fontVery > 0) ? 1 : 0);

#endif // FONT_MODE

    // Example
#if (DISPLAY_GUI == 1)

    hV_HAL_log(LEVEL_INFO, "DISPLAY_GUI");
    myScreen.clear();
    displayGUI();
    hV_HAL_delayMilliseconds(8000);

#endif // DISPLAY_GUI

    hV_HAL_log(LEVEL_INFO, "Regenerate");
    myScreen.regenerate();

    hV_HAL_exit(0);
}

///
/// @brief Loop, empty
///
void loop()
{
    hV_HAL_delayMilliseconds(1000);
}

