# STM32 HAL — Quick Guide (DWIN Unified)

Short steps to run the DWIN library on STM32 (HAL). Works with **USART1/USART2** etc. using the **ISR RX queue**.

---

## 1) Load the HMI *(optional)*
This step is **not required** to use the library. The sample **DWIN_SET** image simply helps you *see* the provided examples on the screen and better understand the API. If you already have your own HMI project, you can skip this step.
- Download **DWIN_SET** from Releases and update the panel:
  - Latest: https://github.com/recepsenbas/DWIN-Unified/releases/latest
- Power‑on update via microSD (FAT32). Use the HMI’s **default baud 115200** unless your image says otherwise — STM32 HW UART handles 115200 well; **no need to drop to 9600** in most cases.

---

## 2) Wire the UART
- Example (Blue Pill): **USART1** → Panel TX→**PA10**, Panel RX→**PA9**, **GND ↔ GND**
- Any UART works (e.g., **USART2** on other boards). Match the baud with your HMI (typically **115200**).

> Tip: Keep TX/RX short and clean; common GND is mandatory.

---

## 3) Add the library to your CubeIDE project
**Copy files into your project**
- Headers → `Core/Inc/`
  - `dwin.h`
  - `dwin_port_stm32.h`
- Sources → `Core/Src/`
  - `dwin.c`
  - `dwin_port_stm32.c`

**Enable the sources in CMake** (project root `CMakeLists.txt`):
```cmake
# Add DWIN Unified sources
# (keep other Core/Src files as generated by Cube)
target_sources(${CMAKE_PROJECT_NAME} PRIVATE
    # Add user sources here
    ${CMAKE_SOURCE_DIR}/Core/Src/dwin.c
    ${CMAKE_SOURCE_DIR}/Core/Src/dwin_port_stm32.c
)
```

No extra include path is needed since headers are in `Core/Inc/`.

---

## 4) Enable the UART interrupt (critical)
To receive bytes with `HAL_UART_Receive_IT`, the **global IRQ** for your UART **must be enabled**:
```c
// Example for USART1 (Blue Pill). For USART2 use USART2_IRQn.
HAL_NVIC_SetPriority(USART1_IRQn, 0, 0);
HAL_NVIC_EnableIRQ(USART1_IRQn);
```
Place this after `HAL_UART_Init()` in your UART init.

---

## 5) Minimal binding (ISR RX queue)
```c
#include "dwin.h"
#include "dwin_port_stm32.h"

extern UART_HandleTypeDef huart1;   // or huart2
static Dwin dwin; static DwinStm32Port dport;

static size_t tx_write(const uint8_t* d, size_t n){ return dwin_stm32_tx(&dport, d, n, 1000); }
static int     rx_read(uint8_t*, size_t){ return 0; }   // not used in ISR mode
static int     rx_avail(void){ return 0; }
static uint32_t tick_ms(void){ return HAL_GetTick(); }

void app_init(void)
{
    dwin_init(&dwin, tx_write, rx_read, rx_avail, tick_ms, /*log*/NULL);
    dwin_use_isr_queue(&dwin, 1);            // enable ISR RX ring buffer

    dwin_stm32_attach(&dport, &dwin, &huart1); // or &huart2
    dwin_stm32_begin(&dport);                 // arms HAL_UART_Receive_IT(...,1)
}

void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
    dwin_stm32_rx_byte_isr(huart);            // feed byte + re-arm RX IT
}

void loop_tick(void)
{
    dwin_poll(&dwin);                         // parse frames & run callbacks
}
```

---

## 6) Quick sanity actions
- `dwin_set_page(&dwin, 0);`
- `dwin_set_brightness(&dwin, 80);`
- `dwin_write_text(&dwin, 0x5000, "Hello STM32");`

If callbacks are set, touches will arrive in your `on_touch()`.

---

## Troubleshooting
- **No frames / callbacks** → Did you enable `USARTx_IRQn` (NVIC)? Is `HAL_UART_RxCpltCallback()` calling `dwin_stm32_rx_byte_isr()`?
- **Nothing updates** → Panel didn’t load `DWIN_SET/` or wrong VP addresses.
- **CRC/Timeout** → Mismatched baud or CRC; for STM32 prefer **115200** with clean wiring.

---

## License
Part of **DWIN Unified** (MIT). See repository root for details.