/**
 * @file MCP9808SensorLib.h
 *
 * @author positronic57
 *
 * @brief Arduino library for communication with Microchip MCP9808 temperature sensor.
 *
 * @details Arduino Wire library used for I2C I/O operations.
 * Implemented functionalities:
 *  - reading temperature;
 *  - changing the resolution of the temperatue measurement;
 *  - swithing between the operational modes of the senosr.
 * Alarms or interrupts are not supported.
 *
 * Supported architectures:
 * - AVR (Arduino boards based on Atmega328p microcontroller);
 * - ESP32C6 (boards included in latest Arduino ESP32 by Espressif Systems).
 *
 * @date: 27.07.2025
 *
 * @copyright GNU General Public License v2
 */


#if !defined(ARDUINO_ARCH_ESP32) && !defined(ARDUINO_ARCH_AVR)
  #pragma GCC error "Unspported architecture"
#endif

#include <inttypes.h>

#include <Wire.h>


#define MCP9808_SENSOR_LIB_VERSION    "0.1.0"

#define MCP9808_DEFAULT_I2C_ADDRESS   0x18

#define MCP9808_CONFIG_REG            0x01
#define MCP9808_TEMPERATURE_REG       0x05
#define MCP9808_MANU_ID_REG           0x06
#define MCP9808_DEV_ID_REG            0x07
#define MCP9808_RESOLUTION_REG        0x08

#define MCP9808_MANU_DEV_ID           0x0454

#define MCP9808_CONFIG_REG_DEF_VALUE  0x001F
#define MCP9808_CONGIF_REG_SHDN_BIT   8

#define ABSOLUTE_ZERO                 -273.15



class MCP9808Sensor {

public:
  MCP9808Sensor() = default;

  enum class MODE_OF_OPERATION : int {
    CONTINUOUS_CONVERSION = 0,    // The sensor continuously measures the temperature
    SHUTDOWN                      // Low-power mode. No temperature conversion. Communication with the sensor is still possible
  };

  enum class RESOLUTION {
    RES_0_5_DEG = 0,          // Represents temperature resolution of 0.5°C
    RES_0_25_DEG,             // Represents temperature resolution of 0.25°C
    RES_0_125_DEG,            // Represents temperature resolution of 0.125°C
    RES_0_0625_DEG            // Represents temperature resolution of 0.0625°C
  };

private:
  TwoWire &m_I2C_bus = Wire; // An instance of TwoWire class from the Wire library is used for I2C I/O operations
  uint8_t m_I2C_address = MCP9808_DEFAULT_I2C_ADDRESS;
  bool m_ready = false;
  uint16_t m_config_register = MCP9808_CONFIG_REG_DEF_VALUE;
  MCP9808Sensor::MODE_OF_OPERATION m_current_state = MCP9808Sensor::MODE_OF_OPERATION::SHUTDOWN;

private:
  /**
   * @brief Reads data from the sensor at specific address, using
   * TwoWire::read() function from Arduino Wire library.
   */
  bool I2CRead(uint8_t I2C_register_address, uint8_t rd_buffer[], size_t buffer_size);

  /**
   * @brief Sends data to the sensor, at specific address, using
   * TwoWire::write() function from Arduino Wire library.
   */
  bool I2CWrite(uint8_t I2C_register_address, const uint8_t tx_buffer[], size_t buffer_size);


public:
  /**
   * @brief Assigns the instance of TwoWire class from Arduino Wire Library as I2C I/O backbone.
   * Proves the presence of the sensor using the manufaturer ID and the device ID info from the
   * device with the provided I2C address.
   * Configures the mode of operation and the resolution for the temperature measurements.
   *
   * Call this function before calling the other methods from this class.
   */
  bool begin(uint8_t i2c_address = MCP9808_DEFAULT_I2C_ADDRESS,
             MCP9808Sensor::MODE_OF_OPERATION mode_of_op = MCP9808Sensor::MODE_OF_OPERATION::CONTINUOUS_CONVERSION,
             MCP9808Sensor::RESOLUTION resolution = MCP9808Sensor::RESOLUTION::RES_0_25_DEG
             );

  /**
   * @brief Switch between the two modes of operations of the sensor.
   */
  bool setModOfOperation(MCP9808Sensor::MODE_OF_OPERATION op_mode);

  /**
   * @brief Function for setting one of the temperature resolutions
   * supported by the sensor.
   */
  bool setResolution(MCP9808Sensor::RESOLUTION resolution);

  /**
   * @brief Reads the temperature measurement from the abient temperature register
   * and returns the temperature in degrees Celsius as a floating point number.
   */
  float readTemperature();
};
