/**
*
* @file MCP9808SensorLib.cpp
* 
* @brief MCP9808 Sensor Library source.
*
* @author positronic57
*
* @date: 27.07.2025
*
* @copyright GNU General Public License v2
* 
*
*/

#include "MCP9808SensorLib.h"

bool MCP9808Sensor::begin(uint8_t i2c_address,
           MCP9808Sensor::MODE_OF_OPERATION mode_of_op,
           MCP9808Sensor::RESOLUTION resolution
)
{
  m_I2C_address = i2c_address;

  do
  {
#if defined (ARDUINO_ARCH_ESP32)
    m_ready = m_I2C_bus.begin();
    if (!m_ready) {
      break;
    }
#elif defined (ARDUINO_ARCH_AVR)
    m_I2C_bus.begin();
#endif
    uint16_t dev_ID = 0;
    uint16_t manu_ID = 0;

    if (!I2CRead(MCP9808_DEV_ID_REG, (uint8_t *)(&dev_ID), sizeof(dev_ID))) {
      return false;
    }

    if (!I2CRead(MCP9808_MANU_ID_REG, (uint8_t *)(&manu_ID), sizeof(manu_ID))) {
      return false;
    }

    if ((dev_ID | manu_ID) == MCP9808_MANU_DEV_ID) {
      m_ready = false;
      break;
    }
    m_ready = true;

    m_ready = setModOfOperation(mode_of_op);
    if (!m_ready) {
      break;
    }

    m_ready = setResolution(resolution);

  } while(0);

  return m_ready;
}


bool MCP9808Sensor::setModOfOperation(MCP9808Sensor::MODE_OF_OPERATION op_mode)
{
  if (!m_ready) {
      return m_ready;
  }

  switch(op_mode)
  {
    case MODE_OF_OPERATION::CONTINUOUS_CONVERSION:
      // Set shutdown bit to 0
      m_config_register &= ~(1 << MCP9808_CONGIF_REG_SHDN_BIT);
      break;
    case MODE_OF_OPERATION::SHUTDOWN:
      // Set shutdown bit to value 1
      m_config_register |= (1 << MCP9808_CONGIF_REG_SHDN_BIT);
      break;
  }

  uint8_t tx_buffer[2] = { 0 };

  tx_buffer[0] = m_config_register >> 8;
  tx_buffer[1] = m_config_register & 0xFF;

  if (!I2CWrite(MCP9808_CONFIG_REG, tx_buffer, sizeof(tx_buffer))) {
    return false;
  }
  m_current_state = op_mode;

  return true;
}


bool MCP9808Sensor::setResolution(MCP9808Sensor::RESOLUTION resolution)
{
  if (!m_ready) {
    return m_ready;
  }
  uint8_t res = static_cast<uint8_t>(resolution);

  return I2CWrite(MCP9808_RESOLUTION_REG, &res, sizeof(res));
}

/**
 * The temperature value is represented with 13 bits, where one bit determines the sign,
 * and the rest 12 bits are fixed point representation of a decimal number,
 * with four bits for the fractional part.
 *
 * The conversion from a fixed to a floating point representation is done by clearing
 * the sign bit, shifting the MSB part for 4 positions to the left and adding the new
 * value to the LSB divided by 16.0 (2^number of fractional bits).
 *
 * The alternative way is to take all 12 bits as integer value and divide it with
 * 2^(number of fractional bits) as decimal number.
 *
 */
float MCP9808Sensor::readTemperature()
{
  float temperature = ABSOLUTE_ZERO;

  if ((!m_ready) || (m_current_state == MCP9808Sensor::MODE_OF_OPERATION::SHUTDOWN)) {
    return temperature;
  }

  uint8_t ambient_temperature[2] = { 0 };
  if (!I2CRead(MCP9808_TEMPERATURE_REG, ambient_temperature, sizeof(ambient_temperature))) {
    return false;
  }

  // Clear flag bits
  ambient_temperature[0] &= 0x1F;

  // Check if a temperature < 0°C
  bool negative_value = (ambient_temperature[0] & 0x10) == 0x10;

  // Clear the sign bit
  ambient_temperature[0] &= 0x0F;

  temperature = (ambient_temperature[0] << 4) + ambient_temperature[1] / 16.0f;

  if (negative_value) {
    temperature -= 256;
  }

  return temperature;
}


bool MCP9808Sensor::I2CRead(uint8_t I2C_register_address, uint8_t rd_buffer[], size_t buffer_size)
{
  m_I2C_bus.beginTransmission(m_I2C_address);
  // Send the register address to the I2C slave
  m_I2C_bus.write(I2C_register_address);
  if (m_I2C_bus.endTransmission() != 0) {
    return false;
  }
  // Get the data from the I2C slave
  m_I2C_bus.requestFrom(m_I2C_address, buffer_size);
  size_t elem = 0;//buffer_size;
  while(m_I2C_bus.available() && (elem < buffer_size)) {
    //rd_buffer[--elem] = m_I2C_bus.read();
    rd_buffer[elem++] = m_I2C_bus.read();
  }

  return true;
}


bool MCP9808Sensor::I2CWrite(uint8_t I2C_register_address, const uint8_t tx_buffer[], size_t buffer_size)
{

  m_I2C_bus.beginTransmission(m_I2C_address);
  // Send register address to the I2C slave
  m_I2C_bus.write(I2C_register_address);
  // Send the data
  if (m_I2C_bus.write(tx_buffer, buffer_size) != buffer_size) {
    return false;
  }

  if (m_I2C_bus.endTransmission() != 0) {
    return false;
  }

  return true;
}
