#include "I2CSlaveHandler.h"
#include <Wire.h>

I2CSlaveHandler::I2CSlaveHandler(I2CSlaveInterface* ioProcessor) {
  _ioProcessor = ioProcessor;
}

void I2CSlaveHandler::begin(byte slaveAddress, void (*receiveEvent)(int), void (*requestEvent)(void)) {
  Wire.begin(slaveAddress);

  if (receiveEvent != NULL) {
    // register function to be called when a message arrives
    Wire.onReceive(receiveEvent);
  }

  if (requestEvent != NULL) {
    // register function to be called when a request arrives
    Wire.onRequest(requestEvent);
  }
}

void I2CSlaveHandler::receiveEvent(int numBytesReceived) {
  if (_ioProcessor == NULL) {
    logNullProcessorError_();
    return;
  }
  if (_rxDataReady == false) {
    void* _rxBuffer = _ioProcessor->getRxBuffer();
    int _rxBufferSize = _ioProcessor->getRxBufferSize();
    // copy the data to the buffer
    Wire.readBytes((byte*)_rxBuffer, min(numBytesReceived, _rxBufferSize));
    _rxDataReady = true;
  } else {
    // dump the data
    while(Wire.available() > 0) {
      byte c = Wire.read();
    }
  }
}

void I2CSlaveHandler::next() {
  if (_ioProcessor == NULL) {
    logNullProcessorError_();
    return;
  }
  // checks if a message has been received
  if (_rxDataReady == true) {
    _ioProcessor->processRxData();
    _rxDataReady = false;
    _txDataReady = true;
  }

  // prepares the data to response
  if (_txDataReady == true && _requestSent == true) {
    _ioProcessor->prepareTxData();
    _txDataReady = false;
    _requestSent = false;
  }
}

void I2CSlaveHandler::requestEvent() {
  if (_ioProcessor == NULL) {
    logNullProcessorError_();
    return;
  }
  void* _txBuffer = _ioProcessor->getTxBuffer();
  int _txBufferSize = _ioProcessor->getTxBufferSize();
  if (_txBuffer != NULL && _txBufferSize > 0) {
    Wire.write((byte*)_txBuffer, _txBufferSize);
  }
  _requestSent = true;
}

void I2CSlaveHandler::reset() {
  debugReset_();
  _txDataReady = false;
  _rxDataReady = false;
  _requestSent = false;
}

void I2CSlaveHandler::debugReset_() {
  Serial.print("LOG: "), Serial.println("Reset the I2CSlaveHandler instance");
}

void I2CSlaveHandler::logNullProcessorError_() {
  Serial.print("ERR: "), Serial.println("_ioProcessor is NULL");
}
