#include "I2CMasterHandler.h"
#include <Wire.h>

I2CMasterHandler::I2CMasterHandler() {
}

void I2CMasterHandler::begin(byte masterAddress) {
  Wire.begin(masterAddress);
}

void I2CMasterHandler::call(byte slaveAddress, I2CMasterInterface* ioProcessor) {
  call_(slaveAddress, NULL, 0, NULL, 0, ioProcessor);
}

void I2CMasterHandler::call(byte slaveAddress,
    void* txBuffer, int txBufferSize, void* rxBuffer, int rxBufferSize) {
  call_(slaveAddress, txBuffer, txBufferSize, rxBuffer, rxBufferSize, NULL);
}

void I2CMasterHandler::call_(byte slaveAddress,
    void* txBuffer, int txBufferSize, void* rxBuffer, int rxBufferSize,
    I2CMasterInterface* ioProcessor) {
  if (ioProcessor != NULL) {
    rxBuffer = ioProcessor->getRxBuffer();
    rxBufferSize = ioProcessor->getRxBufferSize();
    txBuffer = ioProcessor->getTxBuffer();
    txBufferSize = ioProcessor->getTxBufferSize();
  }

  // updates the data in txDataBuffer
  if (millis() - _prevUpdateTime >= _updateInterval) {
    _prevUpdateTime = millis();
    if (_txDataReady == false) { // ensure previous message has been sent
      if (ioProcessor != NULL) {
        ioProcessor->prepareTxData();
      }
      _txDataReady = true;
    }
  }

  // sends the data if it is ready to be sent
  if (_txDataReady == true) {
    send(slaveAddress, txBuffer, txBufferSize);
    _txDataReady = false;
    _rqDataReady = true;
  }

  // just one request following every transmission
  if (_rqDataReady == true) {
    request(slaveAddress, rxBuffer, rxBufferSize);
    _rqDataReady = false;
    _rxDataReady = true;
  }

  // checks if a message has been received
  if (_rxDataReady == true) {
    if (ioProcessor != NULL) {
      ioProcessor->processRxData();
    }
    _rxDataReady = false;
  }
}

int8_t I2CMasterHandler::send(byte slaveAddress, void* buffer, int bufferSize) {
  if (buffer == NULL || bufferSize == 0) return 0;
  Wire.beginTransmission(slaveAddress);
  Wire.write((byte*)buffer, bufferSize);
  return Wire.endTransmission(); // actually sends the data
}

int8_t I2CMasterHandler::request(byte slaveAddress, void* buffer, int bufferSize) {
  if (buffer == NULL || bufferSize == 0) return 0;
  size_t numBytes = bufferSize;
  // the request is immediately followed by the read for the response
  size_t numBytesRequested = Wire.requestFrom(slaveAddress, numBytes);
  size_t numBytesReceived = Wire.readBytes((byte*)buffer, min(numBytes, numBytesRequested));
  return numBytesReceived;
}
