/******************************************************************************
BQ25672.h
BQ25672 Arduino / PlatformIO Library

Author: Piotr Malek
Repository: https://github.com/piotrmalek/BQ25672

Public interface definition for the BQ25672 battery charger library.

License: MIT
******************************************************************************/

#ifndef bq25672_h_
#define bq25672_h_

#include <Arduino.h>
#include <Wire.h>
#include <math.h>

// ===== Compile-time configuration =====

#if __has_include("BQ25672_config.h")
#include "BQ25672_config.h"
#endif

// Enable floating-point helper API (V / A)
#ifndef BQ25672_ENABLE_FLOAT_API
#define BQ25672_ENABLE_FLOAT_API 0
#endif


// Enable enum-to-string helper functions
#ifndef BQ25672_ENABLE_STRINGS
#define BQ25672_ENABLE_STRINGS 0
#endif

class BQ25672
{
public:
    explicit BQ25672(TwoWire &wire = Wire, uint8_t addr = 0x6B)
        : m_wire(&wire), m_addr(addr) {}

    enum class Error : uint8_t
    {
        OK = 0,
        I2C_READ_FAIL,
        I2C_WRITE_FAIL,
        PART_INFO_READ_FAIL,
        WRONG_PART_NUMBER,
        WRONG_DEVICE_REV,
        ADC_ENABLE_FAIL,
        VBUS_ADC_ENABLE_FAIL
    };

    enum class PrechargeVbatThreshold : uint8_t
    {
        VREG_15_0pct = 0, // 15%  * VREG
        VREG_62_2pct = 1, // 62.2%* VREG
        VREG_66_7pct = 2, // 66.7%* VREG
        VREG_71_4pct = 3  // 71.4%* VREG
    };

    enum class CellCount : uint8_t
    {
        S1 = 0, // 1s
        S2 = 1, // 2s
        S3 = 2, // 3s
        S4 = 3  // 4s
    };

    enum class RechargeDeglitch : uint8_t
    {
        MS_64 = 0,   // 64 ms
        MS_256 = 1,  // 256 ms
        MS_1024 = 2, // 1024 ms (default)
        MS_2048 = 3  // 2048 ms
    };

    enum class PrechargeSafetyTimer : uint8_t
    {
        HRS_2 = 0,  // 2 hrs (default)
        HRS_0_5 = 1 // 0.5 hrs
    };

    enum class TopOffTimer : uint8_t
    {
        Disabled = 0, // 00b
        Min15 = 1,    // 01b
        Min30 = 2,    // 10b
        Min45 = 3     // 11b
    };

    enum class FastChargeTimer : uint8_t
    {
        Hrs5 = 0,  // 00b
        Hrs8 = 1,  // 01b
        Hrs12 = 2, // 10b (default)
        Hrs24 = 3  // 11b
    };

    enum class VacOvpThreshold : uint8_t
    {
        V26 = 0, // 26V (default)
        V22 = 1, // 22V
        V12 = 2, // 12V
        V7 = 3   // 7V
    };

    enum class WatchdogTimer : uint8_t
    {
        Disabled = 0,
        S0_5 = 1,
        S1 = 2,
        S2 = 3,
        S20 = 4,
        S40 = 5, // default
        S80 = 6,
        S160 = 7
    };

    enum class SdrvCtrl : uint8_t
    {
        Idle = 0,            // 00b
        ShutdownMode = 1,    // 01b
        ShipMode = 2,        // 10b
        SystemPowerReset = 3 // 11b
    };

    enum class PwmFrequency : uint8_t
    {
        MHz1_5 = 0,
        kHz750 = 1
    };

    enum class IbatRegOtg : uint8_t
    {
        A3 = 0,      // 3A
        A4 = 1,      // 4A
        A5 = 2,      // 5A
        Disabled = 3 // Disable (default)
    };

    enum class VocPct : uint8_t
    {
        P0_5625 = 0,
        P0_6250 = 1,
        P0_6875 = 2,
        P0_7500 = 3,
        P0_8125 = 4,
        P0_8750 = 5, // default
        P0_9375 = 6,
        P1_0000 = 7
    };

    enum class VocDelay : uint8_t
    {
        Ms50 = 0,
        Ms300 = 1, // default
        S2 = 2,
        S5 = 3
    };

    enum class VocRate : uint8_t
    {
        S30 = 0,
        Min2 = 1, // default
        Min10 = 2,
        Min30 = 3
    };

    enum class ThermalRegThreshold : uint8_t
    {
        C60 = 0,
        C80 = 1,
        C100 = 2,
        C120 = 3 // default
    };

    enum class ThermalShutdownThreshold : uint8_t
    {
        C150 = 0, // default
        C130 = 1,
        C120 = 2,
        C85 = 3
    };

    enum class JeitaVsetHigh : uint8_t
    {
        ChargeSuspend = 0, // Suspend charging
        VregMinus800 = 1,  // VREG - 800 mV
        VregMinus600 = 2,  // VREG - 600 mV
        VregMinus400 = 3,  // VREG - 400 mV (default)
        VregMinus300 = 4,  // VREG - 300 mV
        VregMinus200 = 5,  // VREG - 200 mV
        VregMinus100 = 6,  // VREG - 100 mV
        VregUnchanged = 7  // VREG unchanged
    };

    enum class JeitaIsetHigh : uint8_t
    {
        ChargeSuspend = 0,
        Ichg20pct = 1,    // 20% * ICHG
        Ichg40pct = 2,    // 40% * ICHG
        IchgUnchanged = 3 // default
    };

    enum class JeitaIsetCold : uint8_t
    {
        ChargeSuspend = 0,
        Ichg20pct = 1, // default
        Ichg40pct = 2,
        IchgUnchanged = 3
    };

    enum class TsCoolThreshold : uint8_t
    {
        P71_1 = 0, // 71.1%  (~5°C)
        P68_4 = 1, // 68.4%  (~10°C) default
        P65_5 = 2, // 65.5%  (~15°C)
        P62_4 = 3  // 62.4%  (~20°C)
    };

    enum class TsWarmThreshold : uint8_t
    {
        P48_4 = 0, // 48.4%  (~40°C)
        P44_8 = 1, // 44.8%  (~45°C) default
        P41_2 = 2, // 41.2%  (~50°C)
        P37_7 = 3  // 37.7%  (~55°C)
    };

    enum class OtgHotThreshold : uint8_t
    {
        C55 = 0,
        C60 = 1, // default
        C65 = 2,
        Disabled = 3
    };

    enum class OtgColdThreshold : uint8_t
    {
        Cneg10 = 0, // default
        Cneg20 = 1
    };

    struct Stat0
    {
        bool iindpm_or_iotg = false;
        bool vindpm_or_votg = false;
        bool watchdog_expired = false;
        bool power_good = false;
        bool vac2_present = false;
        bool vac1_present = false;
        bool vbus_present = false;
    };

    enum class ChargeStatus : uint8_t
    {
        NotCharging = 0,
        TrickleCharge = 1,
        PreCharge = 2,
        FastCharge_CC = 3,
        TaperCharge_CV = 4,
        Reserved5 = 5,
        TopOffActive = 6,
        TerminationDone = 7
    };

    enum class VbusStatus : uint8_t
    {
        NoInputOrBhotOrBcold = 0x0,
        UsbSdp = 0x1,     // 500mA
        UsbCdp = 0x2,     // 1.5A
        UsbDcp = 0x3,     // 3.25A
        Hvdcp_1_5A = 0x4, // Adjustable High Voltage DCP (HVDCP)
        UnknownAdapter_3A = 0x5,
        NonStandardAdapter = 0x6,
        OtgMode = 0x7,
        NotQualifiedAdapter = 0x8,
        Reserved9 = 0x9,
        ReservedA = 0xA,
        VbusPowered = 0xB, // Device directly powered from VBUS
        BackupMode = 0xC,
        ReservedD = 0xD,
        ReservedE = 0xE,
        ReservedF = 0xF
    };

    struct Stat1
    {
        ChargeStatus chg_status = ChargeStatus::NotCharging;
        VbusStatus vbus_status = VbusStatus::NoInputOrBhotOrBcold;
        bool bc12_done = false;
    };

    struct Stat3
    {
        bool acrb2_placed = false;
        bool acrb1_placed = false;
        bool adc_done = false;
        bool vsysmin_regulation = false;
        bool chg_timer_expired = false;
        bool trickle_timer_expired = false;
        bool precharge_timer_expired = false;
    };

    struct Stat4
    {
        bool vbat_otg_too_low = false;
        bool ts_cold = false;
        bool ts_cool = false;
        bool ts_warm = false;
        bool ts_hot = false;
    };

    enum class TsRange : uint8_t
    {
        Normal = 0,
        Cold,
        Cool,
        Warm,
        Hot,
        Invalid // multiple bits set (shouldn't happen)
    };

    struct Fault0
    {
        bool ibat_regulation = false;
        bool vbus_ovp = false;
        bool vbat_ovp = false;
        bool ibus_ocp = false;
        bool ibat_ocp = false;
        bool conv_ocp = false;
        bool vac2_ovp = false;
        bool vac1_ovp = false;
    };

    struct Fault1
    {
        bool vsys_short = false;
        bool vsys_ovp = false;
        bool otg_ovp = false;
        bool otg_uvp = false;
        bool tshut = false;
    };

    enum class IcoStatus : uint8_t
    {
        Disabled = 0,
        OptimizationRunning = 1,
        MaxInputDetected = 2,
        Reserved = 3
    };

    struct Stat2
    {
        IcoStatus ico_status = IcoStatus::Disabled;
        bool thermal_regulation = false;
        bool dpdm_ongoing = false;
        bool vbat_present = false;
    };

    struct Flag0
    {
        bool iindpm = false;
        bool vindpm = false;
        bool watchdog = false;
        bool poor_source = false;
        bool power_good_change = false;
        bool vac2_present_change = false;
        bool vac1_present_change = false;
        bool vbus_present_change = false;
    };

    struct Flag1
    {
        bool charge_status_change = false;
        bool ico_status_change = false;
        bool vbus_status_change = false;
        bool thermal_reg_event = false;
        bool vbat_present_change = false;
        bool bc12_done_change = false;
    };

    struct Flag2
    {
        bool dpdm_done = false;
        bool adc_done = false;
        bool vsysmin_reg_change = false;
        bool fast_charge_timer_expired = false;
        bool trickle_timer_expired = false;
        bool precharge_timer_expired = false;
        bool topoff_timer_expired = false;
    };

    struct Flag3
    {
        bool vbat_otg_low = false;
        bool ts_cold = false;
        bool ts_cool = false;
        bool ts_warm = false;
        bool ts_hot = false;
    };

    struct FaultFlag0
    {
        bool ibat_reg = false;
        bool vbus_ovp = false;
        bool vbat_ovp = false;
        bool ibus_ocp = false;
        bool ibat_ocp = false;
        bool conv_ocp = false;
        bool vac2_ovp = false;
        bool vac1_ovp = false;
    };

    struct FaultFlag1
    {
        bool vsys_short = false;
        bool vsys_ovp = false;
        bool otg_ovp = false;
        bool otg_uvp = false;
        bool tshut = false;
    };

    struct Mask0
    {
        bool iindpm_masked = false;
        bool vindpm_masked = false;
        bool wd_masked = false;
        bool poor_source_masked = false;
        bool pg_masked = false;
        bool vac2_present_masked = false;
        bool vac1_present_masked = false;
        bool vbus_present_masked = false;
    };

    struct Mask1
    {
        bool chg_masked = false;
        bool ico_masked = false;
        bool vbus_masked = false;
        bool treg_masked = false;
        bool vbat_present_masked = false;
        bool bc12_done_masked = false;
    };

    struct Mask2
    {
        bool dpdm_done_masked = false;
        bool adc_done_masked = false;
        bool vsysmin_masked = false;
        bool fast_charge_timer_masked = false;
        bool trickle_timer_masked = false;
        bool precharge_timer_masked = false;
        bool topoff_timer_masked = false;
    };

    struct Mask3
    {
        bool vbat_otg_low_masked = false;
        bool ts_cold_masked = false;
        bool ts_cool_masked = false;
        bool ts_warm_masked = false;
        bool ts_hot_masked = false;
    };

    struct FaultMask0
    {
        bool ibat_reg_masked = false;
        bool vbus_ovp_masked = false;
        bool vbat_ovp_masked = false;
        bool ibus_ocp_masked = false;
        bool ibat_ocp_masked = false;
        bool conv_ocp_masked = false;
        bool vac2_ovp_masked = false;
        bool vac1_ovp_masked = false;
    };

    struct FaultMask1
    {
        bool vsys_short_masked = false;
        bool vsys_ovp_masked = false;
        bool otg_ovp_masked = false;
        bool otg_uvp_masked = false;
        bool tshut_masked = false;
    };

    enum class AdcRate : uint8_t
    {
        Continuous = 0,
        OneShot = 1
    };

    enum class AdcSample : uint8_t
    {
        Bit15 = 0, // 15-bit effective resolution
        Bit14 = 1, // 14-bit effective resolution
        Bit13 = 2, // 13-bit effective resolution
        Bit12 = 3  // 12-bit effective resolution (not recommended)
    };

    struct AdcCtrl
    {
        bool adc_enabled = false;
        AdcRate rate = AdcRate::Continuous;
        AdcSample sample = AdcSample::Bit12;
        bool running_average = false;
        bool avg_init_new_conversion = false;
    };

    struct AdcDis0
    {
        bool ibus_disabled = false;
        bool ibat_disabled = false;
        bool vbus_disabled = false;
        bool vbat_disabled = false;
        bool vsys_disabled = false;
        bool ts_disabled = false;
        bool tdie_disabled = false;
    };

    struct AdcDis1
    {
        bool dp_disabled = false;
        bool dm_disabled = false;
        bool vac2_disabled = false;
        bool vac1_disabled = false;
    };

    enum class DpDac : uint8_t
    {
        HiZ = 0,      // HIZ
        Zero = 1,     // 0
        V0_6 = 2,     // 0.6V
        V1_2 = 3,     // 1.2V
        V2_0 = 4,     // 2.0V
        V2_7 = 5,     // 2.7V
        V3_3 = 6,     // 3.3V
        DpDmShort = 7 // D+/D- short
    };

    enum class DmDac : uint8_t
    {
        HiZ = 0,  // HIZ
        Zero = 1, // 0
        V0_6 = 2, // 0.6V
        V1_2 = 3, // 1.2V
        V2_0 = 4, // 2.0V
        V2_7 = 5, // 2.7V
        V3_3 = 6, // 3.3V
        // 7 reserved
    };

    struct DpdmDriver
    {
        DpDac dp = DpDac::HiZ;
        DmDac dm = DmDac::HiZ;
    };

public:
    Error begin(bool enable_adc = true, bool enable_vbus_adc = true);

    // --- MIN_SYS_V (REG 0x00) ---
    // Minimal System Voltage .

    bool getMinSystemVoltageRaw(uint8_t &out);
    bool getMinSystemVoltage_mV(uint16_t &mv);
    bool setMinSystemVoltage_mV(uint16_t mv);

    // --- CHG_VLIM (REG 0x01) ---
    // VREG_10:0: 0 mV offset, 10 mV step, range 3000..18800 mV, clamped low.

    bool getChargeVoltageLimitRaw(uint16_t &out);
    bool getChargeVoltageLimit_mV(uint16_t &mv);
    bool setChargeVoltageLimit_mV(uint16_t mv);

    // --- CHG_ILIM (REG 0x03) ---
    // ICHG_8:0: 0 mA offset, 10 mA step, range 50..3000 mA, clamped low.

    bool getChargeCurrentLimitRaw(uint16_t &out);
    bool getChargeCurrentLimit_mA(uint16_t &ma);
    bool setChargeCurrentLimit_mA(uint16_t ma);

    // --- IN_VLIM (REG 0x05) ---
    // VINDPM_7:0: 0 mV offset, 100 mV step, range 3600..22000 mV, clamped low.
    // Note: This register is not reset by REG_RST/WATCHDOG and may change on adapter plug/unplug.

    bool getInputVoltageLimitRaw(uint8_t &out);
    bool getInputVoltageLimit_mV(uint16_t &mv);
    bool setInputVoltageLimit_mV(uint16_t mv);

    // --- IN_ILIM (REG 0x06) ---
    // IINDPM_8:0: 0 mA offset, 10 mA step, range 100..3300 mA, clamped low.

    bool getInputCurrentLimitRaw(uint16_t &out);
    bool getInputCurrentLimit_mA(uint16_t &ma);
    bool setInputCurrentLimit_mA(uint16_t ma);

    // --- PRECHG_CTRL (REG 0x08) ---
    // [7:6] VBAT_LOWV_1:0 - Precharge->Fast charge threshold as ratio of VREG
    // [5:0] IPRECHG_5:0  - Precharge current limit (40 mA step), range 40..2000 mA, clamped low.

    bool getPrechargeControlRaw(uint8_t &out);
    bool getPrechargeVbatThreshold(PrechargeVbatThreshold &thr);
    bool setPrechargeVbatThreshold(PrechargeVbatThreshold thr);
    bool getPrechargeCurrentLimit_mA(uint16_t &ma);
    bool setPrechargeCurrentLimit_mA(uint16_t ma);

    // --- TERM_CTRL (REG 0x09) ---
    // [6]   REG_RST      - Reset registers to default values and reset timer
    // [5]   STOP_WD_CHG  - Watchdog expiration sets EN_CHG=0 when set
    // [4:0] ITERM_4:0    - Termination current (40 mA step), range 40..1000 mA, clamped low.

    bool getTerminationControlRaw(uint8_t &out);
    bool getRegResetBit(bool &enabled);        // Bit 6: REG_RST
    bool triggerRegisterReset();               // Writing 1 triggers reset (datasheet: 1h = Reset). Typically self-clears.
    bool getStopWatchdogCharge(bool &enabled); // Bit 5: STOP_WD_CHG
    bool setStopWatchdogCharge(bool enable);
    bool getTerminationCurrent_mA(uint16_t &ma); // Bits 4:0: ITERM
    bool setTerminationCurrent_mA(uint16_t ma);

    // --- RECHG_CTRL (REG 0x0A) ---
    // [7:6] CELL_1:0   - Battery cell count (1S..4S)
    // [5:4] TRECHG_1:0 - Battery recharge deglitch time
    // [3:0] VRECHG_3:0 - Recharge threshold offset below VREG
    //                   Range 50..800 mV, offset 50 mV, step 50 mV.

    bool getRechargeControlRaw(uint8_t &out);
    bool getCellCount(CellCount &cells); // [7:6] CELL
    bool setCellCount(CellCount cells);

    // [5:4] TRECHG
    bool getRechargeDeglitch(RechargeDeglitch &t);
    bool setRechargeDeglitch(RechargeDeglitch t);

    // [3:0] VRECHG offset below VREG
    bool getRechargeThresholdOffset_mV(uint16_t &mv);
    bool setRechargeThresholdOffset_mV(uint16_t mv);

    // --- VOTG (REG 0x0B) ---
    // VOTG_10:0: 2800 mV offset, 10 mV step, range 2800..22000 mV, clamped high.

    bool getOtgVoltageRaw(uint16_t &out);
    bool getOtgVoltage_mV(uint16_t &mv);
    bool setOtgVoltage_mV(uint16_t mv);

    // --- IOTG (REG 0x0D) ---
    // [7]   PRECHG_TMR - Pre-charge safety timer (0: 2 hrs, 1: 0.5 hrs)
    // [6:0] IOTG_6:0   - OTG current limit (40 mA step), range 160..3360 mA, clamped low.

    bool getIotgControlRaw(uint8_t &out);
    bool getPrechargeSafetyTimer(PrechargeSafetyTimer &t); // Bit 7: PRECHG_TMR
    bool setPrechargeSafetyTimer(PrechargeSafetyTimer t);
    bool getOtgCurrentLimit_mA(uint16_t &ma); // Bits 6:0: IOTG current limit
    bool setOtgCurrentLimit_mA(uint16_t ma);

    // --- TIMER_CTRL (REG 0x0E) ---
    // [7:6] TOPOFF_TMR_1:0  - Top-off timer control
    // [5]   EN_TRICHG_TMR   - Enable trickle charge timer (fixed 1 hr)
    // [4]   EN_PRECHG_TMR   - Enable pre-charge timer
    // [3]   EN_CHG_TMR      - Enable fast charge timer
    // [2:1] CHG_TMR_1:0     - Fast charge timer setting
    // [0]   TMR2X_EN        - Slow timers by 2X during input DPM or thermal regulation

    bool getTimerControlRaw(uint8_t &out);
    bool getTopOffTimer(TopOffTimer &t); // [7:6] TOPOFF_TMR
    bool setTopOffTimer(TopOffTimer t);
    bool getTrickleChargeTimerEnabled(bool &enabled); // Bit 5: EN_TRICHG_TMR
    bool setTrickleChargeTimerEnabled(bool enable);
    bool getPrechargeTimerEnabled(bool &enabled); // Bit 4: EN_PRECHG_TMR
    bool setPrechargeTimerEnabled(bool enable);
    bool getFastChargeTimerEnabled(bool &enabled); // Bit 3: EN_CHG_TMR
    bool setFastChargeTimerEnabled(bool enable);
    bool getFastChargeTimer(FastChargeTimer &t); // [2:1] CHG_TMR
    bool setFastChargeTimer(FastChargeTimer t);
    bool getTimer2xEnabled(bool &enabled); // Bit 0: TMR2X_EN
    bool setTimer2xEnabled(bool enable);

    // --- Charger control ---

    // --- CHG_CTRL0 (REG 0x0F) ---
    // [7] EN_AUTO_IBATDIS - Auto battery discharging during BAT OVP fault
    // [6] FORCE_IBATDIS   - Force battery discharging current
    // [5] EN_CHG          - Charger enable
    // [4] EN_ICO          - Input Current Optimizer enable
    // [3] FORCE_ICO       - Force start ICO (valid only when EN_ICO=1, self-clears)
    // [2] EN_HIZ          - Enable HIZ mode (also reset to 0 when adapter plugged)
    // [1] EN_TERM         - Enable termination
    // [0] (not defined here) - preserved

    bool getChargerControl0Raw(uint8_t &out);
    bool getAutoIbatDischargeEnabled(bool &enabled);
    bool setAutoIbatDischargeEnabled(bool enable);
    bool getForceIbatDischarge(bool &enabled);
    bool setForceIbatDischarge(bool enable);
    bool getChargeEnabled(bool &enabled);
    bool setChargeEnabled(bool enable);
    bool getIcoEnabled(bool &enabled);
    bool setIcoEnabled(bool enable);
    bool triggerIcoStart(); // FORCE_ICO: can be written 1 to start ICO; returns to 0 after ICO starts.
    bool triggerIcoStartSafe();
    bool getHizEnabled(bool &enabled);
    bool setHizEnabled(bool enable);
    bool getTerminationEnabled(bool &enabled);
    bool setTerminationEnabled(bool enable);

    // --- CHG_CTRL1 (REG 0x10) ---
    // [5:4] VAC_OVP_1:0  - VAC OVP thresholds
    // [3]   WD_RST       - I2C watchdog timer reset (self-clearing)
    // [2:0] WATCHDOG_2:0 - Watchdog timer settings
    // [7:6] reserved - preserved

    bool getChargerControl1Raw(uint8_t &out);
    bool getVacOvpThreshold(VacOvpThreshold &t); // [5:4] VAC_OVP
    bool setVacOvpThreshold(VacOvpThreshold t);
    bool triggerWatchdogReset(); // [3] WD_RST - write 1 to reset watchdog timer, bit returns to 0 after reset.
    bool getWatchdogTimer(WatchdogTimer &t);
    bool setWatchdogTimer(WatchdogTimer t);

    // --- CHG_CTRL2 (REG 0x11) ---
    // [7]   FORCE_INDET    - Force D+/D- detection (self-clearing)
    // [6]   AUTO_INDET_EN  - Automatic D+/D- detection enable on VBUS plug-in
    // [5]   EN_12V         - Enable 12V mode in HVDCP
    // [4]   EN_9V          - Enable 9V mode in HVDCP
    // [3]   HVDCP_EN       - Enable HVDCP handshake
    // [2:1] SDRV_CTRL_1:0  - SFET (ship FET) control
    // [0]   SDRV_DLY       - Delay for SDRV_CTRL action (0: add 10s delay, 1: no delay)

    bool getChargerControl2Raw(uint8_t &out);
    bool triggerDpDmDetection();                     // [7] FORCE_INDET (write 1 to trigger, then it resets to 0)
    bool getAutoDpDmDetectionEnabled(bool &enabled); // [6] AUTO_INDET_EN
    bool setAutoDpDmDetectionEnabled(bool enable);
    bool getHvdcp12VEnabled(bool &enabled); // [5] EN_12V
    bool setHvdcp12VEnabled(bool enable);
    bool getHvdcp9VEnabled(bool &enabled); // [4] EN_9V
    bool setHvdcp9VEnabled(bool enable);
    bool getHvdcpEnabled(bool &enabled); // [3] HVDCP_EN
    bool setHvdcpEnabled(bool enable);
    bool getSdrvControl(SdrvCtrl &mode); // [2:1] SDRV_CTRL
    bool setSdrvControl(SdrvCtrl mode);
    bool getSdrvDelayNo10s(bool &no_delay); // [0] SDRV_DLY
    bool setSdrvDelayNo10s(bool no_delay);  // true = do NOT add 10s delay, false = add 10s delay

    // --- CHG_CTRL3 (REG 0x12) ---
    // [7] DIS_ACDRV    - Force EN_ACDRV1=0 and EN_ACDRV2=0
    // [6] EN_OTG       - OTG enable
    // [5] PFM_OTG_DIS  - Disable PFM in OTG mode (0: enable PFM, 1: disable PFM)
    // [4] PFM_FWD_DIS  - Disable PFM in forward mode (0: enable PFM, 1: disable PFM)
    // [3] WKUP_DLY     - Ship mode wake-up delay (0: 1s, 1: 15ms)
    // [2] DIS_LDO      - Disable BATFET LDO mode in pre-charge stage (0: enable, 1: disable)
    // [1] DIS_OTG_OOA  - Disable OOA in OTG mode (0: enable, 1: disable)
    // [0] DIS_FWD_OOA  - Disable OOA in forward mode (0: enable, 1: disable)

    bool getChargerControl3Raw(uint8_t &out);
    bool getAcdrvDisabled(bool &disabled);
    bool setAcdrvDisabled(bool disable);
    bool getOtgEnabled(bool &enabled);
    bool setOtgEnabled(bool enable);
    bool getPfmOtgDisabled(bool &disabled);
    bool setPfmOtgDisabled(bool disable);
    bool getPfmForwardDisabled(bool &disabled);
    bool setPfmForwardDisabled(bool disable);
    bool getWakeupDelay15ms(bool &is_15ms); // WKUP_DLY: 0 = 1s (default), 1 = 15ms
    bool setWakeupDelay15ms(bool is_15ms);
    bool getBatfetLdoDisabled(bool &disabled);
    bool setBatfetLdoDisabled(bool disable);
    bool getOoaOtgDisabled(bool &disabled);
    bool setOoaOtgDisabled(bool disable);
    bool getOoaForwardDisabled(bool &disabled);
    bool setOoaForwardDisabled(bool disable);

    // --- CHG_CTRL4 (REG 0x13) ---
    // [7] EN_ACDRV2        - External ACFET2-RBFET2 gate driver control (may be locked at 0)
    // [6] EN_ACDRV1        - External ACFET1-RBFET1 gate driver control (may be locked at 0)
    // [5] PWM_FREQ         - Switching frequency selection (0: 1.5 MHz, 1: 750 kHz)
    // [4] DIS_STAT         - Disable STAT pin output (0: enable STAT, 1: disable STAT)
    // [3] DIS_VSYS_SHORT   - Disable forward mode VSYS short hiccup protection (0: enable, 1: disable)
    // [2] DIS_VOTG_UVP     - Disable OTG mode VOTG UVP hiccup protection (0: enable, 1: disable)
    // [1] FORCE_VINDPM_DET - Force VINDPM detection (self-clearing)
    // [0] EN_IBUS_OCP      - Enable IBUS_OCP in forward mode

    bool getChargerControl4Raw(uint8_t &out);
    bool getAcdrv2Enabled(bool &enabled);
    bool setAcdrv2Enabled(bool enable);
    bool getAcdrv1Enabled(bool &enabled);
    bool setAcdrv1Enabled(bool enable);
    bool getPwmFrequency(PwmFrequency &f);
    bool setPwmFrequency(PwmFrequency f);
    bool getStatPinDisabled(bool &disabled);
    bool setStatPinDisabled(bool disable);
    bool getVsysShortProtectionDisabled(bool &disabled);
    bool setVsysShortProtectionDisabled(bool disable);
    bool getVotgUvpProtectionDisabled(bool &disabled);
    bool setVotgUvpProtectionDisabled(bool disable);
    bool triggerVindpmDetection(); // Bit 1: FORCE_VINDPM_DET (write 1 to trigger, then it resets to 0)
    bool getIbusOcpEnabled(bool &enabled);
    bool setIbusOcpEnabled(bool enable);

    // --- CHG_CTRL5 (REG 0x14) ---
    // [7]   SFET_PRESENT   - Ship FET populated indicator (enables ship-FET related features)
    // [6]   reserved
    // [5]   EN_IBAT        - IBAT discharge current sensing enable
    // [4:3] IBAT_REG_1:0   - Battery discharging current regulation in OTG mode
    // [2]   EN_IINDPM      - Enable internal IINDPM regulation
    // [1]   EN_EXTILIM     - Enable external ILIM_HIZ pin input current regulation
    // [0]   EN_BATOC       - Enable battery discharging current OCP
    //
    // Note: If SFET_PRESENT=0, some ship-FET related bits in other registers may be locked to 0.

    bool getChargerControl5Raw(uint8_t &out);
    bool getShipFetPresent(bool &present); // [7] SFET_PRESENT
    bool setShipFetPresent(bool present);
    bool getIbatDischargeSensingEnabled(bool &enabled); // [5] EN_IBAT
    bool setIbatDischargeSensingEnabled(bool enable);
    bool getIbatRegulationOtg(IbatRegOtg &r); // [4:3] IBAT_REG
    bool setIbatRegulationOtg(IbatRegOtg r);
    bool getInternalIindpmEnabled(bool &enabled); // [2] EN_IINDPM
    bool setInternalIindpmEnabled(bool enable);
    bool getExternalIlimEnabled(bool &enabled); // [1] EN_EXTILIM
    bool setExternalIlimEnabled(bool enable);
    bool getBatteryDischargeOcpEnabled(bool &enabled); // [0] EN_BATOC
    bool setBatteryDischargeOcpEnabled(bool enable);

    // --- MPPT / temperature / NTC ---

    // --- MPPT_CTRL (REG 0x15) ---
    // [7:5] VOC_PCT_2:0  - VINDPM as a percentage of VBUS open-circuit voltage (VOC)
    // [4:3] VOC_DLY_1:0  - Delay before VOC measurement after converter stops switching
    // [2:1] VOC_RATE_1:0 - Interval between VOC measurements
    // [0]   EN_MPPT      - Enable MPPT VOC measurement

    bool getMpptControlRaw(uint8_t &out);

    // [0] EN_MPPT
    bool getMpptEnabled(bool &enabled);
    bool setMpptEnabled(bool enable);

    // [7:5] VOC_PCT
    bool getVocPct(VocPct &p);
    bool setVocPct(VocPct p);
    bool getVocPct_permille(uint16_t &permille);

    // [4:3] VOC_DLY
    bool getVocDelay(VocDelay &d);
    bool setVocDelay(VocDelay d);
    bool getVocDelay_ms(uint32_t &ms);

    // [2:1] VOC_RATE
    bool getVocRate(VocRate &r);
    bool setVocRate(VocRate r);
    bool getVocRate_s(uint32_t &s);

    // --- TEMP_CTRL (REG 0x16) ---
    // [7:6] TREG_1:0    - Thermal regulation thresholds
    // [5:4] TSHUT_1:0   - Thermal shutdown thresholds
    // [3]   VBUS_PD_EN  - Enable VBUS pull-down resistor (6k Ohm)
    // [2]   VAC1_PD_EN  - Enable VAC1 pull-down resistor
    // [1]   VAC2_PD_EN  - Enable VAC2 pull-down resistor
    // [0]   reserved    - preserved

    bool getTemperatureControlRaw(uint8_t &out);

    // [7:6] TREG
    bool getThermalRegThreshold(ThermalRegThreshold &t);
    bool setThermalRegThreshold(ThermalRegThreshold t);
    bool getThermalRegThreshold_C(uint16_t &celsius);

    // [5:4] TSHUT
    bool getThermalShutdownThreshold(ThermalShutdownThreshold &t);
    bool setThermalShutdownThreshold(ThermalShutdownThreshold t);
    bool getThermalShutdownThreshold_C(uint16_t &celsius);

    // [3] VBUS_PD_EN
    bool getVbusPullDownEnabled(bool &enabled);
    bool setVbusPullDownEnabled(bool enable);

    // [2] VAC1_PD_EN
    bool getVac1PullDownEnabled(bool &enabled);
    bool setVac1PullDownEnabled(bool enable);

    // [1] VAC2_PD_EN
    bool getVac2PullDownEnabled(bool &enabled);
    bool setVac2PullDownEnabled(bool enable);

    // --- NTC_CTRL0 (REG 0x17) ---
    // [7:5] JEITA_VSET_2:0   - JEITA high temp (TWARN–THOT) charge voltage setting
    // [4:3] JEITA_ISETH_1:0  - JEITA high temp (TWARN–THOT) charge current setting
    // [2:1] JEITA_ISETC_1:0  - JEITA low temp (TCOLD–TCOOL) charge current setting
    // [0]   reserved         - preserved

    bool getNtcControl0Raw(uint8_t &out);

    // [7:5] JEITA_VSET
    bool getJeitaVsetHigh(JeitaVsetHigh &v);
    bool setJeitaVsetHigh(JeitaVsetHigh v);

    // [4:3] JEITA_ISETH
    bool getJeitaIsetHigh(JeitaIsetHigh &i);
    bool setJeitaIsetHigh(JeitaIsetHigh i);

    // [2:1] JEITA_ISETC
    bool getJeitaIsetCold(JeitaIsetCold &i);
    bool setJeitaIsetCold(JeitaIsetCold i);

    // --- NTC_CTRL1 (REG 0x18) ---
    // [7:6] TS_COOL_1:0 - JEITA VT2 rising threshold (%REGN), approx temp depends on NTC network
    // [5:4] TS_WARM_1:0 - JEITA VT3 falling threshold (%REGN), approx temp depends on NTC network
    // [3:2] BHOT_1:0    - OTG mode TS HOT temperature threshold
    // [1]   BCOLD       - OTG mode TS COLD temperature threshold
    // [0]   TS_IGNORE   - Ignore TS feedback (TS always considered good)

    bool getNtcControl1Raw(uint8_t &out);

    // [7:6] TS_COOL
    bool getTsCoolThreshold(TsCoolThreshold &t);
    bool setTsCoolThreshold(TsCoolThreshold t);

    // Optional helper: approximate temp in °C for TS_COOL (based on the datasheet note)
    bool getTsCoolApprox_C(int16_t &celsius);

    // [5:4] TS_WARM
    bool getTsWarmThreshold(TsWarmThreshold &t);
    bool setTsWarmThreshold(TsWarmThreshold t);

    // Optional helper: approximate temp in °C for TS_WARM (based on the datasheet note)
    bool getTsWarmApprox_C(int16_t &celsius);

    // [3:2] BHOT
    bool getOtgHotThreshold(OtgHotThreshold &t);
    bool setOtgHotThreshold(OtgHotThreshold t);
    bool getOtgHotThreshold_C(int16_t &celsius, bool &disabled);

    // [1] BCOLD
    bool getOtgColdThreshold(OtgColdThreshold &t);
    bool setOtgColdThreshold(OtgColdThreshold t);

    // [0] TS_IGNORE
    bool getTsIgnoreEnabled(bool &enabled);
    bool setTsIgnoreEnabled(bool enable);

    // --- ICO_ILIM (REG 0x19) ---
    // ICO_ILIM_8:0 (read-only): Input current limit obtained from ICO or ILIM_HIZ pin setting
    // 0 mA offset, 10 mA step, range 100..3300 mA, clamped low.

    bool getIcoInputCurrentLimit_mA(uint16_t &ma);

    // --- Status registers ---

    // --- STAT0 (REG 0x1B) ---
    // [7] IINDPM_STAT        - In IINDPM regulation (forward) / IOTG regulation (OTG)
    // [6] VINDPM_STAT        - In VINDPM regulation (forward) / VOTG regulation (OTG)
    // [5] WD_STAT            - Watchdog timer expired
    // [4] reserved
    // [3] PG_STAT            - Power good
    // [2] AC2_PRESENT_STAT   - VAC2 present
    // [1] AC1_PRESENT_STAT   - VAC1 present
    // [0] VBUS_PRESENT_STAT  - VBUS present

    bool getStatus0Raw(uint8_t &out);
    bool getStatus0(Stat0 &s);

    // --- STAT1 (REG 0x1C) ---
    // [7:5] CHG_STAT_2:0     - Charge status
    // [4:1] VBUS_STAT_3:0    - VBUS status
    // [0]   BC1_2_DONE_STAT  - BC1.2 / non-standard detection done

    bool getStatus1Raw(uint8_t &out);
    bool getStatus1(Stat1 &s);

    // --- STAT2 (REG 0x1D) ---
    // [7:6] ICO_STAT_1:0        - ICO status
    // [5:3] reserved
    // [2]   TREG_STAT           - Thermal regulation active
    // [1]   DPDM_STAT           - D+/D- detection ongoing
    // [0]   VBAT_PRESENT_STAT   - Battery present (VBAT > VBAT_UVLOZ)

    bool getStatus2Raw(uint8_t &out);
    bool getStatus2(Stat2 &s);

    // --- STAT3 (REG 0x1E) ---
    // [7] ACRB2_STAT        - ACFET2-RBFET2 placed
    // [6] ACRB1_STAT        - ACFET1-RBFET1 placed
    // [5] ADC_DONE_STAT     - ADC conversion complete (one-shot mode only)
    // [4] VSYS_STAT         - VSYSMIN regulation active (VBAT < VSYSMIN)
    // [3] CHG_TMR_STAT      - Fast charge safety timer expired
    // [2] TRICHG_TMR_STAT   - Trickle charge safety timer expired
    // [1] PRECHG_TMR_STAT   - Pre-charge safety timer expired
    // [0] reserved

    bool getStatus3Raw(uint8_t &out);
    bool getStatus3(Stat3 &s);

    // --- STAT4 (REG 0x1F) ---
    // [7:5] reserved
    // [4] VBATOTG_LOW_STAT - Battery too low to enable OTG
    // [3] TS_COLD_STAT     - TS in cold range
    // [2] TS_COOL_STAT     - TS in cool range
    // [1] TS_WARM_STAT     - TS in warm range
    // [0] TS_HOT_STAT      - TS in hot range

    bool getStatus4Raw(uint8_t &out);
    bool getStatus4(Stat4 &s);
    TsRange getTsRangeFromStat4(const Stat4 &s);

    // --- Fault status ---

    // --- FAULT0 (REG 0x20) ---
    // [7] IBAT_REG_STAT   - Battery discharging current regulation active
    // [6] VBUS_OVP_STAT   - VBUS over-voltage protection active
    // [5] VBAT_OVP_STAT   - VBAT over-voltage protection active
    // [4] IBUS_OCP_STAT   - IBUS over-current protection active
    // [3] IBAT_OCP_STAT   - IBAT over-current protection active
    // [2] CONV_OCP_STAT   - Converter over-current protection active
    // [1] VAC2_OVP_STAT   - VAC2 over-voltage protection active
    // [0] VAC1_OVP_STAT   - VAC1 over-voltage protection active

    bool getFault0Raw(uint8_t &out);
    bool getFault0(Fault0 &f);
    bool anyFault0Active(const Fault0 &f);

    // --- FAULT1 (REG 0x21) ---
    // [7] VSYS_SHORT_STAT  - VSYS short circuit protection active
    // [6] VSYS_OVP_STAT    - VSYS over-voltage protection active
    // [5] OTG_OVP_STAT     - OTG over-voltage protection active
    // [4] OTG_UVP_STAT     - OTG under-voltage protection active
    // [3] reserved
    // [2] TSHUT_STAT       - Thermal shutdown protection active
    // [1:0] reserved

    bool getFault1Raw(uint8_t &out);
    bool getFault1(Fault1 &f);
    bool anyFault1Active(const Fault1 &f);

    // --- Charger flags ---

    // --- FLAG0 (REG 0x22) ---
    // [7] IINDPM_FLAG        - IINDPM/IOTG rising edge detected
    // [6] VINDPM_FLAG        - VINDPM/VOTG rising edge detected
    // [5] WD_FLAG            - Watchdog timer rising edge detected
    // [4] POORSRC_FLAG       - Poor source status rising edge detected
    // [3] PG_FLAG            - Any change in PG_STAT detected
    // [2] AC2_PRESENT_FLAG   - VAC2 present status changed
    // [1] AC1_PRESENT_FLAG   - VAC1 present status changed
    // [0] VBUS_PRESENT_FLAG  - VBUS present status changed

    bool getFlag0Raw(uint8_t &out);
    bool getFlag0(Flag0 &f);
    bool anyFlag0Active(const Flag0 &f);

    // --- FLAG1 (REG 0x23) ---
    // [7] CHG_FLAG           - Charge status changed
    // [6] ICO_FLAG           - ICO status changed
    // [5] reserved
    // [4] VBUS_FLAG          - VBUS status changed
    // [3] reserved
    // [2] TREG_FLAG          - Thermal regulation threshold event
    // [1] VBAT_PRESENT_FLAG  - VBAT present status changed
    // [0] BC1_2_DONE_FLAG    - BC1.2 detection status changed

    bool getFlag1Raw(uint8_t &out);
    bool getFlag1(Flag1 &f);
    bool anyFlag1Active(const Flag1 &f);

    // --- FLAG2 (REG 0x24) ---
    // [7] reserved
    // [6] DPDM_DONE_FLAG     - D+/D- detection completed
    // [5] ADC_DONE_FLAG      - ADC conversion completed (one-shot mode only)
    // [4] VSYS_FLAG          - Entered/exited VSYSMIN regulation
    // [3] CHG_TMR_FLAG       - Fast charge timer expired event
    // [2] TRICHG_TMR_FLAG    - Trickle charge timer expired event
    // [1] PRECHG_TMR_FLAG    - Pre-charge timer expired event
    // [0] TOPOFF_TMR_FLAG    - Top-off timer expired event

    bool getFlag2Raw(uint8_t &out);
    bool getFlag2(Flag2 &f);
    bool anyFlag2Active(const Flag2 &f);

    // --- FLAG3 (REG 0x25) ---
    // [7:5] reserved
    // [4] VBATOTG_LOW_FLAG  - VBAT too low to enable OTG mode
    // [3] TS_COLD_FLAG      - TS crossed cold threshold (T1)
    // [2] TS_COOL_FLAG      - TS crossed cool threshold (T2)
    // [1] TS_WARM_FLAG      - TS crossed warm threshold (T3)
    // [0] TS_HOT_FLAG       - TS crossed hot threshold (T5)

    bool getFlag3Raw(uint8_t &out);
    bool getFlag3(Flag3 &f);
    bool anyFlag3Active(const Flag3 &f);

    // --- Fault flags ---

    // --- FAULT_FLAG0 (REG 0x26) ---
    // [7] IBAT_REG_FLAG   - Enter/exit IBAT regulation event
    // [6] VBUS_OVP_FLAG   - Enter VBUS OVP event
    // [5] VBAT_OVP_FLAG   - Enter VBAT OVP event
    // [4] IBUS_OCP_FLAG   - Enter IBUS OCP event
    // [3] IBAT_OCP_FLAG   - Enter discharged OCP event
    // [2] CONV_OCP_FLAG   - Enter converter OCP event
    // [1] VAC2_OVP_FLAG   - Enter VAC2 OVP event
    // [0] VAC1_OVP_FLAG   - Enter VAC1 OVP event

    bool getFaultFlag0Raw(uint8_t &out);
    bool getFaultFlag0(FaultFlag0 &f);
    bool anyFaultFlag0Active(const FaultFlag0 &f);

    // --- FAULT_FLAG1 (REG 0x27) ---
    // [7] VSYS_SHORT_FLAG  - Stop switching due to VSYS short circuit
    // [6] VSYS_OVP_FLAG    - Stop switching due to VSYS over-voltage
    // [5] OTG_OVP_FLAG     - Stop OTG due to VBUS over-voltage
    // [4] OTG_UVP_FLAG     - Stop OTG due to VBUS under-voltage
    // [3] reserved
    // [2] TSHUT_FLAG       - Thermal shutdown threshold event
    // [1:0] reserved

    bool getFaultFlag1Raw(uint8_t &out);
    bool getFaultFlag1(FaultFlag1 &f);
    bool anyFaultFlag1Active(const FaultFlag1 &f);

    // --- Masks ---

    // --- MASK0 (REG 0x28) ---
    // Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
    //
    // [7] IINDPM_MASK        - Mask IINDPM/IOTG event
    // [6] VINDPM_MASK        - Mask VINDPM/VOTG event
    // [5] WD_MASK            - Mask watchdog expired event
    // [4] POORSRC_MASK       - Mask poor source event
    // [3] PG_MASK            - Mask PG toggle event
    // [2] AC2_PRESENT_MASK   - Mask VAC2 present change event
    // [1] AC1_PRESENT_MASK   - Mask VAC1 present change event
    // [0] VBUS_PRESENT_MASK  - Mask VBUS present change event

    bool getMask0Raw(uint8_t &out);
    bool setMask0Raw(uint8_t value);
    bool getMask0(Mask0 &m);
    bool setMask0(const Mask0 &m);

    // MASK0 setters (1 = masked, 0 = unmasked)
    bool setIindpmMask(bool masked);
    bool setVindpmMask(bool masked);
    bool setWatchdogMask(bool masked);
    bool setPoorSourceMask(bool masked);
    bool setPowerGoodMask(bool masked);
    bool setVac2PresentMask(bool masked);
    bool setVac1PresentMask(bool masked);
    bool setVbusPresentMask(bool masked);

    // --- MASK1 (REG 0x29) ---
    // Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
    //
    // [7] CHG_MASK           - Charge status change mask
    // [6] ICO_MASK           - ICO status change mask
    // [5] reserved
    // [4] VBUS_MASK          - VBUS status change mask
    // [3] reserved
    // [2] TREG_MASK          - Thermal regulation entry mask
    // [1] VBAT_PRESENT_MASK  - VBAT present status change mask
    // [0] BC1_2_DONE_MASK    - BC1.2 detection status change mask

    bool getMask1Raw(uint8_t &out);
    bool getMask1(Mask1 &m);
    bool setMask1(const Mask1 &m);

    // Per-bit setters (1 = masked, 0 = unmasked)
    bool setChgMask(bool masked);
    bool setIcoMask(bool masked);
    bool setVbusMask(bool masked);
    bool setTregMask(bool masked);
    bool setVbatPresentMask(bool masked);
    bool setBc12DoneMask(bool masked);

    // --- MASK2 (REG 0x2A) ---
    // Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
    //
    // [7] reserved
    // [6] DPDM_DONE_MASK     - D+/D- detection done mask
    // [5] ADC_DONE_MASK      - ADC conversion done mask (one-shot mode)
    // [4] VSYS_MASK          - VSYSMIN regulation enter/exit mask
    // [3] CHG_TMR_MASK       - Fast charge timer expire mask
    // [2] TRICHG_TMR_MASK    - Trickle charge timer expire mask
    // [1] PRECHG_TMR_MASK    - Pre-charge timer expire mask
    // [0] TOPOFF_TMR_MASK    - Top-off timer expire mask

    bool getMask2Raw(uint8_t &out);
    bool getMask2(Mask2 &m);
    bool setMask2(const Mask2 &m);
    bool setDpdmDoneMask(bool masked);
    bool setAdcDoneMask(bool masked);
    bool setVsysMinMask(bool masked);
    bool setFastChargeTimerMask(bool masked);
    bool setTrickleTimerMask(bool masked);
    bool setPrechargeTimerMask(bool masked);
    bool setTopoffTimerMask(bool masked);

    // --- MASK3 (REG 0x2B) ---
    // Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
    //
    // [7:5] reserved
    // [4] VBATOTG_LOW_MASK  - VBAT too low for OTG event mask
    // [3] TS_COLD_MASK      - TS crossed cold threshold (T1) mask
    // [2] TS_COOL_MASK      - TS crossed cool threshold (T2) mask
    // [1] TS_WARM_MASK      - TS crossed warm threshold (T3) mask
    // [0] TS_HOT_MASK       - TS crossed hot threshold (T5) mask

    bool getMask3Raw(uint8_t &out);
    bool getMask3(Mask3 &m);
    bool setMask3(const Mask3 &m);
    bool setVbatOtgLowMask(bool masked);
    bool setTsColdMask(bool masked);
    bool setTsCoolMask(bool masked);
    bool setTsWarmMask(bool masked);
    bool setTsHotMask(bool masked);

    // --- FAULT_MASK0 (REG 0x2C) ---
    // Mask bits: 0 = fault event produces INT pulse, 1 = fault event does NOT produce INT pulse (masked)
    //
    // [7] IBAT_REG_MASK   - IBAT regulation enter/exit mask
    // [6] VBUS_OVP_MASK   - VBUS OVP entry mask
    // [5] VBAT_OVP_MASK   - VBAT OVP entry mask
    // [4] IBUS_OCP_MASK   - IBUS OCP fault mask
    // [3] IBAT_OCP_MASK   - IBAT OCP fault mask
    // [2] CONV_OCP_MASK   - Converter OCP fault mask
    // [1] VAC2_OVP_MASK   - VAC2 OVP entry mask
    // [0] VAC1_OVP_MASK   - VAC1 OVP entry mask

    bool getFaultMask0Raw(uint8_t &out);
    bool getFaultMask0(FaultMask0 &m);
    bool setFaultMask0(const FaultMask0 &m);
    bool setIbatRegFaultMask(bool masked);
    bool setVbusOvpFaultMask(bool masked);
    bool setVbatOvpFaultMask(bool masked);
    bool setIbusOcpFaultMask(bool masked);
    bool setIbatOcpFaultMask(bool masked);
    bool setConvOcpFaultMask(bool masked);
    bool setVac2OvpFaultMask(bool masked);
    bool setVac1OvpFaultMask(bool masked);

    // --- FAULT_MASK1 (REG 0x2D) ---
    // Mask bits: 0 = fault event produces INT pulse, 1 = fault event does NOT produce INT pulse (masked)
    //
    // [7] VSYS_SHORT_MASK  - VSYS short circuit fault mask
    // [6] VSYS_OVP_MASK    - VSYS over-voltage fault mask
    // [5] OTG_OVP_MASK     - OTG VBUS over-voltage fault mask
    // [4] OTG_UVP_MASK     - OTG VBUS under-voltage fault mask
    // [3] reserved
    // [2] TSHUT_MASK       - Thermal shutdown fault mask
    // [1:0] reserved

    bool getFaultMask1Raw(uint8_t &out);
    bool getFaultMask1(FaultMask1 &m);
    bool setFaultMask1(const FaultMask1 &m);
    bool setVsysShortFaultMask(bool masked);
    bool setVsysOvpFaultMask(bool masked);
    bool setOtgOvpFaultMask(bool masked);
    bool setOtgUvpFaultMask(bool masked);
    bool setTshutFaultMask(bool masked);

    // --- ADC configuration ---

    // --- ADC_CTRL (REG 0x2E) ---
    // [7]   ADC_EN          - ADC enable
    // [6]   ADC_RATE        - 0=continuous, 1=one-shot
    // [5:4] ADC_SAMPLE_1:0  - sample speed / effective resolution
    // [3]   ADC_AVG         - 0=single, 1=running average
    // [2]   ADC_AVG_INIT    - 0=use existing value, 1=start with new conversion
    // [1:0] reserved

    bool getAdcCtrlRaw(uint8_t &out);
    bool getAdcCtrl(AdcCtrl &a);
    bool setAdcCtrl(const AdcCtrl &a);
    bool setAdcEnabled(bool enable);
    bool setAdcRate(AdcRate rate);
    bool setAdcSample(AdcSample sample);
    bool setAdcRunningAverage(bool enable);
    bool setAdcAvgInitNewConversion(bool enable);

    // --- ADC_DIS0 (REG 0x2F) ---
    // Note: 0 = enabled, 1 = disabled
    //
    // [7] IBUS_ADC_DIS  - IBUS ADC disable
    // [6] IBAT_ADC_DIS  - IBAT ADC disable
    // [5] VBUS_ADC_DIS  - VBUS ADC disable
    // [4] VBAT_ADC_DIS  - VBAT ADC disable
    // [3] VSYS_ADC_DIS  - VSYS ADC disable
    // [2] TS_ADC_DIS    - TS ADC disable
    // [1] TDIE_ADC_DIS  - TDIE ADC disable
    // [0] reserved

    bool getAdcDis0Raw(uint8_t &out);
    bool getAdcDis0(AdcDis0 &a);
    bool setAdcDis0(const AdcDis0 &a);
    bool setIbusAdcEnabled(bool enable = true);
    bool setIbatAdcEnabled(bool enable = true);
    bool setVbusAdcEnabled(bool enable = true); // 0=disabled, 1=enabled
    bool setVbatAdcEnabled(bool enable = true);
    bool setVsysAdcEnabled(bool enable = true);
    bool setTsAdcEnabled(bool enable = true);
    bool setTdieAdcEnabled(bool enable = true);

    // --- ADC_DIS1 (REG 0x30) ---
    // Note: 0 = enabled, 1 = disabled
    //
    // [7] DP_ADC_DIS    - D+ ADC disable
    // [6] DM_ADC_DIS    - D- ADC disable
    // [5] VAC2_ADC_DIS  - VAC2 ADC disable
    // [4] VAC1_ADC_DIS  - VAC1 ADC disable
    // [3:0] reserved

    bool getAdcDis1Raw(uint8_t &out);
    bool getAdcDis1(AdcDis1 &a);
    bool setAdcDis1(const AdcDis1 &a);
    bool setDpAdcEnabled(bool enable = true);
    bool setDmAdcEnabled(bool enable = true);
    bool setVac2AdcEnabled(bool enable = true);
    bool setVac1AdcEnabled(bool enable = true);

    // --- DPDM ---

    // --- DPDM_DRIVER (REG 0x47) ---
    // [7:5] DPLUS_DAC_2:0   - D+ output driver level
    // [4:2] DMINUS_DAC_2:0  - D- output driver level
    // [1:0] reserved

    bool getDpdmDriverRaw(uint8_t &out);
    bool getDpdmDriver(DpdmDriver &d);
    bool setDpdmDriver(const DpdmDriver &d);
    bool setDplusDac(DpDac dp);
    bool setDminusDac(DmDac dm);

    // --- Part information ---

    bool getPartInfo(uint8_t &out);

    // --- Measurements ---

    bool readIbus_mA(uint16_t &ma); // IBUS ADC (1 LSB = 1 mA)
    bool readIbat_mA(uint16_t &ma); // IBAT ADC (1 LSB = 1 mA)
    bool readVbus_mV(uint16_t &mv); // VBUS ADC (1 LSB = 1 mV)
    bool readVac1_mV(uint16_t &mv); // VAC1 ADC (1 LSB = 1 mV)
    bool readVac2_mV(uint16_t &mv); // VAC2 ADC (1 LSB = 1 mV)
    bool readVbat_mV(uint16_t &mv); // VBAT ADC (1 LSB = 1 mV)
    bool readVsys_mV(uint16_t &mv); // VSYS ADC (1 LSB = 1 mV)
    uint16_t readTsRaw();           // TS ADC (raw value, scaling TBD)
    uint16_t readTdieRaw();         // TDIE ADC (raw value, scaling TBD)
    uint16_t readDpRaw();           // D+ ADC (raw value)
    uint16_t readDmRaw();           // D- ADC (raw value)

    // --- Float API ---

#if defined(BQ25672_ENABLE_FLOAT_API) && BQ25672_ENABLE_FLOAT_API
    bool getMinSystemVoltage_V(float &v);
    bool setMinSystemVoltage_V(float v);
    bool getChargeVoltageLimit_V(float &v);
    bool setChargeVoltageLimit_V(float v);
    bool getChargeCurrentLimit_A(float &a);
    bool setChargeCurrentLimit_A(float a);
    bool getInputVoltageLimit_V(float &v);
    bool setInputVoltageLimit_V(float v);
    bool getInputCurrentLimit_A(float &a);
    bool setInputCurrentLimit_A(float a);
    bool getPrechargeCurrentLimit_A(float &a);
    bool getTerminationCurrent_A(float &a);
    bool getOtgVoltage_V(float &v);
    bool setOtgVoltage_V(float v);
    bool getOtgCurrentLimit_A(float &a);
    bool getVocPctFactor(float &factor);
    bool getIcoInputCurrentLimit_A(float &a);
    bool readIbus_A(float &a);
    bool readIbat_A(float &a);
    bool readVbus_V(float &v);
    bool readVac1_V(float &v);
    bool readVac2_V(float &v);
    bool readVbat_V(float &v);
    bool readVsys_V(float &v);
#endif

    // --- String API ---

#if defined(BQ25672_ENABLE_STRINGS) && BQ25672_ENABLE_STRINGS
    static const char *errorToString(Error e);
    static const char *prechargeThresholdToString(BQ25672::PrechargeVbatThreshold t);
    static const char *cellToString(BQ25672::CellCount c);
    static const char *rechgDeglitchToString(BQ25672::RechargeDeglitch t);
    static const char *topOffToString(BQ25672::TopOffTimer t);
    static const char *fastChgToString(BQ25672::FastChargeTimer t);
    static const char *vacOvpToString(BQ25672::VacOvpThreshold t);
    static const char *watchdogToString(BQ25672::WatchdogTimer t);
    static const char *ibatRegToString(BQ25672::IbatRegOtg r);
    static const char *chargeStatusToString(BQ25672::ChargeStatus s);
    static const char *vbusStatusToString(BQ25672::VbusStatus s);
    static const char *icoStatusToString(BQ25672::IcoStatus s);
    static const char *tsRangeToString(BQ25672::TsRange r);
#endif

private:
    TwoWire *m_wire;
    uint8_t m_addr;

    enum class REG : uint8_t
    {
        MIN_SYS_V = 0x00,   // 0x00 Minimal System Voltage
        CHG_VLIM = 0x01,    // 0x01 Charge Voltage Limit
                            // 0x02 reserved
        CHG_ILIM = 0x03,    // 0x03 Charge Current Limit
                            // 0x04 reserved
        IN_VLIM = 0x05,     // 0x05 Input Voltage Limit
        IN_ILIM = 0x06,     // 0x06 Input Current Limit
                            // 0x07 reserved
        PRECHG_CTRL = 0x08, // 0x08 Precharge Control
        TERM_CTRL = 0x09,   // 0x09 Termination Control
        RECHG_CTRL = 0x0A,  // 0x0A Re-charge Control
        VOTG = 0x0B,        // 0x0B VOTG Regulation
                            // 0x0C reserved
        IOTG = 0x0D,        // 0x0D IOTG Regulation
        TIMER_CTRL = 0x0E,  // 0x0E Timer Control

        CHG_CTRL0 = 0x0F, // 0x0F Charger Control 0
        CHG_CTRL1 = 0x10, // 0x10 Charger Control 1
        CHG_CTRL2 = 0x11, // 0x11 Charger Control 2
        CHG_CTRL3 = 0x12, // 0x12 Charger Control 3
        CHG_CTRL4 = 0x13, // 0x13 Charger Control 4
        CHG_CTRL5 = 0x14, // 0x14 Charger Control 5

        MPPT_CTRL = 0x15, // 0x15 MPPT Control
        TEMP_CTRL = 0x16, // 0x16 Temperature Control
        NTC_CTRL0 = 0x17, // 0x17 NTC Control 0
        NTC_CTRL1 = 0x18, // 0x18 NTC Control 1
        ICO_ILIM = 0x19,  // 0x19 ICO Current Limit
                          // 0x1A reserved

        STAT0 = 0x1B, // 0x1B Charger Status 0
        STAT1 = 0x1C, // 0x1C Charger Status 1
        STAT2 = 0x1D, // 0x1D Charger Status 2
        STAT3 = 0x1E, // 0x1E Charger Status 3
        STAT4 = 0x1F, // 0x1F Charger Status 4

        FAULT0 = 0x20, // 0x20 FAULT Status 0
        FAULT1 = 0x21, // 0x21 FAULT Status 1

        FLAG0 = 0x22, // 0x22 Charger Flag 0
        FLAG1 = 0x23, // 0x23 Charger Flag 1
        FLAG2 = 0x24, // 0x24 Charger Flag 2
        FLAG3 = 0x25, // 0x25 Charger Flag 3

        FAULT_FLAG0 = 0x26, // 0x26 FAULT Flag 0
        FAULT_FLAG1 = 0x27, // 0x27 FAULT Flag 1

        MASK0 = 0x28, // 0x28 Charger Mask 0
        MASK1 = 0x29, // 0x29 Charger Mask 1
        MASK2 = 0x2A, // 0x2A Charger Mask 2
        MASK3 = 0x2B, // 0x2B Charger Mask 3

        FAULT_MASK0 = 0x2C, // 0x2C FAULT Mask 0
        FAULT_MASK1 = 0x2D, // 0x2D FAULT Mask 1

        ADC_CTRL = 0x2E, // 0x2E ADC Control
        ADC_DIS0 = 0x2F, // 0x2F ADC Function Disable 0
        ADC_DIS1 = 0x30, // 0x30 ADC Function Disable 1

        IBUS_ADC = 0x31, // 0x31 IBUS ADC
                         // 0x32 reserved
        IBAT_ADC = 0x33, // 0x33 IBAT ADC
                         // 0x34 reserved
        VBUS_ADC = 0x35, // 0x35 VBUS ADC
                         // 0x36 reserved
        VAC1_ADC = 0x37, // 0x37 VAC1 ADC
                         // 0x38 reserved
        VAC2_ADC = 0x39, // 0x39 VAC2 ADC
                         // 0x3A reserved
        VBAT_ADC = 0x3B, // 0x3B VBAT ADC
                         // 0x3C reserved
        VSYS_ADC = 0x3D, // 0x3D VSYS ADC
                         // 0x3E reserved
        TS_ADC = 0x3F,   // 0x3F TS ADC
                         // 0x40 reserved
        TDIE_ADC = 0x41, // 0x41 TDIE ADC
                         // 0x42 reserved
        DP_ADC = 0x43,   // 0x43 D+ ADC
                         // 0x44 reserved
        DM_ADC = 0x45,   // 0x45 D- ADC
                         // 0x46 reserved

        DPDM_DRIVER = 0x47, // 0x47 DPDM Driver
        PART_INFO = 0x48    // 0x48 Part Information
    };

private:
    bool readBytes(REG reg, uint8_t *buf, uint8_t len);
    bool read8(REG reg, uint8_t &out);
    bool write8(REG reg, uint8_t val);
    bool updateBits(REG reg, uint8_t mask, uint8_t value);
    bool read16(REG reg_msb, uint16_t &out);
    bool write16(REG reg_msb, uint16_t val);
    bool isBQ25672();
};

#endif
