/******************************************************************************
BQ25672.cpp
BQ25672 Arduino / PlatformIO Library

Author: Piotr Malek
Created: 2026
Repository: https://github.com/piotrmalek/BQ25672

This file implements the BQ25672 class for the Texas Instruments BQ25672
battery charger and power-path management IC.

The library provides:
- Full register-level access to the device
- High-level helpers using mV / mA units
- Optional floating-point API
- Optional string helpers for enums, status, and fault decoding

The code is designed to be portable across Arduino-compatible platforms
using the Wire (I2C) interface.

License:
This library is distributed under the MIT License.
See the LICENSE file for details.

Distributed as-is; no warranty is given.
******************************************************************************/

#include "BQ25672.h"

BQ25672::Error BQ25672::begin(bool enable_adc, bool enable_vbus_adc)
{
    uint8_t v;
    if (!read8(REG::PART_INFO, v))
        return BQ25672::Error::PART_INFO_READ_FAIL;

    const uint8_t pn = (v >> 3) & 0x07;
    const uint8_t rev = v & 0x07;

    if (pn != 0b100)
        return BQ25672::Error::WRONG_PART_NUMBER;
    if (rev != 0b001)
        return BQ25672::Error::WRONG_DEVICE_REV;

    if (enable_adc)
    {
        if (!setAdcEnabled(true))
            return BQ25672::Error::ADC_ENABLE_FAIL;
    }

    if (enable_vbus_adc)
    {
        if (!setVbusAdcEnabled(true))
            return BQ25672::Error::VBUS_ADC_ENABLE_FAIL;
    }

    return BQ25672::Error::OK;
}

// --- Configuration / limits ---

// --- MIN_SYS_V (REG 0x00) ---
// Minimal System Voltage .

bool BQ25672::getMinSystemVoltageRaw(uint8_t &out)
{
    return read8(REG::MIN_SYS_V, out);
}

bool BQ25672::getMinSystemVoltage_mV(uint16_t &mv)
{
    uint8_t reg;
    if (!BQ25672::getMinSystemVoltageRaw(reg))
        return false;

    const uint8_t code = reg & 0x3F; // bits 5..0
    mv = uint16_t(2500) + uint16_t(code) * 250;
    if (mv > 16000)
        mv = 16000; // safety clamp
    return true;
}

bool BQ25672::setMinSystemVoltage_mV(uint16_t mv)
{
    // Clamp to datasheet range
    if (mv < 2500)
        mv = 2500;
    if (mv > 16000)
        mv = 16000;

    // Convert to code (rounded to nearest step)
    // code = (mv - 2500) / 250
    uint16_t code = (mv - 2500 + 125) / 250; // +125 for rounding
    if (code > 0x3F)
        code = 0x3F;

    const uint8_t mask = 0x3F; // VSYSMIN_5:0
    return updateBits(REG::MIN_SYS_V, mask, static_cast<uint8_t>(code));
}

// --- CHG_VLIM (REG 0x01) ---
// VREG_10:0: 0 mV offset, 10 mV step, range 3000..18800 mV, clamped low.

bool BQ25672::getChargeVoltageLimitRaw(uint16_t &out)
{
    return read16(REG::CHG_VLIM, out);
}

bool BQ25672::getChargeVoltageLimit_mV(uint16_t &mv)
{
    uint16_t raw;
    if (!BQ25672::getChargeVoltageLimitRaw(raw))
        return false;

    const uint16_t code = raw & 0x07FF; // bits 10..0
    mv = static_cast<uint16_t>(code * 10);

    // safety clamp
    if (mv < 3000)
        mv = 3000;
    if (mv > 18800)
        mv = 18800;

    return true;
}

bool BQ25672::setChargeVoltageLimit_mV(uint16_t mv)
{
    // Clamp to datasheet range
    if (mv < 3000)
        mv = 3000;
    if (mv > 18800)
        mv = 18800;

    // Convert to register code (10 mV per LSB)
    uint16_t code = (mv + 5) / 10; // +5 for rounding
    if (code > 0x07FF)
        code = 0x07FF;

    // Write full 16-bit register (no reserved bits below bit 11)
    return write16(REG::CHG_VLIM, code);
}

// --- CHG_ILIM (REG 0x03) ---
// ICHG_8:0: 0 mA offset, 10 mA step, range 50..3000 mA, clamped low.

bool BQ25672::getChargeCurrentLimitRaw(uint16_t &out)
{
    return read16(REG::CHG_ILIM, out);
}

bool BQ25672::getChargeCurrentLimit_mA(uint16_t &ma)
{
    uint16_t raw;
    if (!getChargeCurrentLimitRaw(raw))
        return false;

    const uint16_t code = raw & 0x01FF; // bits 8..0
    ma = static_cast<uint16_t>(code * 10);

    // safety clamp
    if (ma < 50)
        ma = 50; // datasheet: clamped low
    if (ma > 3000)
        ma = 3000; // datasheet range

    return true;
}

bool BQ25672::setChargeCurrentLimit_mA(uint16_t ma)
{
    // Clamp to datasheet range
    if (ma < 50)
        ma = 50;
    if (ma > 3000)
        ma = 3000;

    // Convert to code (10 mA per LSB), rounded to nearest step
    uint16_t code = (ma + 5) / 10;
    if (code > 0x01FF)
        code = 0x01FF;

    // Write 16-bit register (bits 15..9 are reserved/RO)
    return write16(REG::CHG_ILIM, code);
}

// --- IN_VLIM (REG 0x05) ---
// VINDPM_7:0: 0 mV offset, 100 mV step, range 3600..22000 mV, clamped low.
// Note: This register is not reset by REG_RST/WATCHDOG and may change on adapter plug/unplug.

bool BQ25672::getInputVoltageLimitRaw(uint8_t &out)
{
    return read8(REG::IN_VLIM, out);
}

bool BQ25672::getInputVoltageLimit_mV(uint16_t &mv)
{
    uint8_t reg;
    if (!getInputVoltageLimitRaw(reg))
        return false;

    mv = static_cast<uint16_t>(reg) * 100u;

    // safety clamp
    if (mv < 3600)
        mv = 3600; // datasheet: clamped low
    if (mv > 22000)
        mv = 22000; // datasheet range

    return true;
}

bool BQ25672::setInputVoltageLimit_mV(uint16_t mv)
{
    // Clamp to datasheet range
    if (mv < 3600)
        mv = 3600;
    if (mv > 22000)
        mv = 22000;

    // Convert to code (100 mV per LSB), rounded to nearest step
    uint16_t code = (mv + 50) / 100;
    if (code > 0xFF)
        code = 0xFF;

    return write8(REG::IN_VLIM, static_cast<uint8_t>(code));
}

// --- IN_ILIM (REG 0x06) ---
// IINDPM_8:0: 0 mA offset, 10 mA step, range 100..3300 mA, clamped low.

bool BQ25672::getInputCurrentLimitRaw(uint16_t &out)
{
    return read16(REG::IN_ILIM, out);
}

bool BQ25672::getInputCurrentLimit_mA(uint16_t &ma)
{
    uint16_t raw;
    if (!getInputCurrentLimitRaw(raw))
        return false;

    const uint16_t code = raw & 0x01FF; // bits 8..0
    ma = static_cast<uint16_t>(code * 10);

    // safety clamp
    if (ma < 100)
        ma = 100; // datasheet: clamped low
    if (ma > 3300)
        ma = 3300; // datasheet range

    return true;
}

bool BQ25672::setInputCurrentLimit_mA(uint16_t ma)
{
    // Clamp to datasheet range
    if (ma < 100)
        ma = 100;
    if (ma > 3300)
        ma = 3300;

    // Convert to code (10 mA per LSB), rounded to nearest step
    uint16_t code = (ma + 5) / 10;
    if (code > 0x01FF)
        code = 0x01FF;

    return write16(REG::IN_ILIM, code);
}

// --- PRECHG_CTRL (REG 0x08) ---
// [7:6] VBAT_LOWV_1:0 - Precharge->Fast charge threshold as ratio of VREG
// [5:0] IPRECHG_5:0  - Precharge current limit (40 mA step), range 40..2000 mA, clamped low.

bool BQ25672::getPrechargeControlRaw(uint8_t &out)
{
    return read8(REG::PRECHG_CTRL, out);
}

bool BQ25672::getPrechargeVbatThreshold(PrechargeVbatThreshold &thr)
{
    uint8_t reg;
    if (!getPrechargeControlRaw(reg))
        return false;
    thr = static_cast<PrechargeVbatThreshold>((reg >> 6) & 0x03);
    return true;
}

bool BQ25672::setPrechargeVbatThreshold(PrechargeVbatThreshold thr)
{
    const uint8_t mask = 0b11000000; // bits 7..6
    const uint8_t val = (static_cast<uint8_t>(thr) & 0x03) << 6;
    return updateBits(REG::PRECHG_CTRL, mask, val);
}

bool BQ25672::getPrechargeCurrentLimit_mA(uint16_t &ma)
{
    uint8_t reg;
    if (!read8(REG::PRECHG_CTRL, reg))
        return false;

    const uint8_t code = reg & 0x3F; // bits 5..0
    ma = uint16_t(code) * 40;

    if (ma < 40)
        ma = 40; // clamped low
    if (ma > 2000)
        ma = 2000; // range clamp
    return true;
}

bool BQ25672::setPrechargeCurrentLimit_mA(uint16_t ma)
{
    // Clamp to datasheet range
    if (ma < 40)
        ma = 40;
    if (ma > 2000)
        ma = 2000;

    // Convert to code (40 mA per LSB), rounded to nearest step
    uint16_t code = (ma + 20) / 40; // +20 for rounding
    if (code > 0x3F)
        code = 0x3F;

    const uint8_t mask = 0x3F; // bits 5..0
    return updateBits(REG::PRECHG_CTRL, mask, static_cast<uint8_t>(code));
}

// --- TERM_CTRL (REG 0x09) ---
// [6]   REG_RST      - Reset registers to default values and reset timer
// [5]   STOP_WD_CHG  - Watchdog expiration sets EN_CHG=0 when set
// [4:0] ITERM_4:0    - Termination current (40 mA step), range 40..1000 mA, clamped low.

bool BQ25672::getTerminationControlRaw(uint8_t &out)
{
    return read8(REG::TERM_CTRL, out);
}

// Bit 6: REG_RST
bool BQ25672::getRegResetBit(bool &enabled)
{
    uint8_t reg;
    if (!getTerminationControlRaw(reg))
        return false;
    enabled = (reg & (1u << 6)) != 0;
    return true;
}

// Writing 1 triggers reset (datasheet: 1h = Reset). Typically self-clears.
bool BQ25672::triggerRegisterReset()
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::TERM_CTRL, mask, mask); // write 1
}

// Bit 5: STOP_WD_CHG
bool BQ25672::getStopWatchdogCharge(bool &enabled)
{
    uint8_t reg;
    if (!getTerminationControlRaw(reg))
        return false;
    enabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setStopWatchdogCharge(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::TERM_CTRL, mask, enable ? mask : 0);
}

// Bits 4:0: ITERM
bool BQ25672::getTerminationCurrent_mA(uint16_t &ma)
{
    uint8_t reg;
    if (!getTerminationControlRaw(reg))
        return false;

    const uint8_t code = reg & 0x1F; // bits 4..0
    ma = uint16_t(code) * 40;

    if (ma < 40)
        ma = 40; // clamped low
    if (ma > 1000)
        ma = 1000; // range clamp
    return true;
}

bool BQ25672::setTerminationCurrent_mA(uint16_t ma)
{
    // Clamp to datasheet range
    if (ma < 40)
        ma = 40;
    if (ma > 1000)
        ma = 1000;

    // Convert to code (40 mA per LSB), rounded to nearest step
    uint16_t code = (ma + 20) / 40;
    if (code > 0x1F)
        code = 0x1F;

    const uint8_t mask = 0x1F; // bits 4..0
    return updateBits(REG::TERM_CTRL, mask, static_cast<uint8_t>(code));
}

// --- RECHG_CTRL (REG 0x0A) ---
// [7:6] CELL_1:0   - Battery cell count (1S..4S)
// [5:4] TRECHG_1:0 - Battery recharge deglitch time
// [3:0] VRECHG_3:0 - Recharge threshold offset below VREG
//                   Range 50..800 mV, offset 50 mV, step 50 mV.

bool BQ25672::getRechargeControlRaw(uint8_t &out)
{
    return read8(REG::RECHG_CTRL, out);
}

// [7:6] CELL
bool BQ25672::getCellCount(CellCount &cells)
{
    uint8_t reg;
    if (!getRechargeControlRaw(reg))
        return false;
    cells = static_cast<CellCount>((reg >> 6) & 0x03);
    return true;
}

bool BQ25672::setCellCount(CellCount cells)
{
    const uint8_t mask = 0b11000000;
    const uint8_t val = (static_cast<uint8_t>(cells) & 0x03) << 6;
    return updateBits(REG::RECHG_CTRL, mask, val);
}

// [5:4] TRECHG

bool BQ25672::getRechargeDeglitch(RechargeDeglitch &t)
{
    uint8_t reg;
    if (!getRechargeControlRaw(reg))
        return false;
    t = static_cast<RechargeDeglitch>((reg >> 4) & 0x03);
    return true;
}

bool BQ25672::setRechargeDeglitch(RechargeDeglitch t)
{
    const uint8_t mask = 0b00110000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 4;
    return updateBits(REG::RECHG_CTRL, mask, val);
}

// [3:0] VRECHG offset below VREG
bool BQ25672::getRechargeThresholdOffset_mV(uint16_t &mv)
{
    uint8_t reg;
    if (!getRechargeControlRaw(reg))
        return false;

    const uint8_t code = reg & 0x0F;         // bits 3..0
    mv = uint16_t(50) + uint16_t(code) * 50; // offset 50 mV, step 50 mV

    // safety clamp (per datasheet range)
    if (mv < 50)
        mv = 50;
    if (mv > 800)
        mv = 800;
    return true;
}

bool BQ25672::setRechargeThresholdOffset_mV(uint16_t mv)
{
    // Clamp to datasheet range
    if (mv < 50)
        mv = 50;
    if (mv > 800)
        mv = 800;

    // code = (mv - 50) / 50, rounded to nearest step
    uint16_t code = (mv - 50 + 25) / 50; // +25 for rounding
    if (code > 0x0F)
        code = 0x0F;

    const uint8_t mask = 0x0F;
    return updateBits(REG::RECHG_CTRL, mask, static_cast<uint8_t>(code));
}

// --- VOTG (REG 0x0B) ---
// VOTG_10:0: 2800 mV offset, 10 mV step, range 2800..22000 mV, clamped high.

bool BQ25672::getOtgVoltageRaw(uint16_t &out)
{
    return read16(REG::VOTG, out);
}

bool BQ25672::getOtgVoltage_mV(uint16_t &mv)
{
    uint16_t raw;
    if (!getOtgVoltageRaw(raw))
        return false;

    const uint16_t code = raw & 0x07FF; // bits 10..0
    mv = uint16_t(2800) + code * 10;

    // safety clamp
    if (mv < 2800)
        mv = 2800;
    if (mv > 22000)
        mv = 22000;
    return true;
}

bool BQ25672::setOtgVoltage_mV(uint16_t mv)
{
    // Clamp to datasheet range
    if (mv < 2800)
        mv = 2800;
    if (mv > 22000)
        mv = 22000;

    // code = (mv - 2800) / 10, rounded to nearest step
    uint16_t code = (mv - 2800 + 5) / 10;
    if (code > 0x07FF)
        code = 0x07FF;

    return write16(REG::VOTG, code);
}

// --- IOTG (REG 0x0D) ---
// [7]   PRECHG_TMR - Pre-charge safety timer (0: 2 hrs, 1: 0.5 hrs)
// [6:0] IOTG_6:0   - OTG current limit (40 mA step), range 160..3360 mA, clamped low.

bool BQ25672::getIotgControlRaw(uint8_t &out)
{
    return read8(REG::IOTG, out);
}

// Bit 7: PRECHG_TMR
bool BQ25672::getPrechargeSafetyTimer(PrechargeSafetyTimer &t)
{
    uint8_t reg;
    if (!getIotgControlRaw(reg))
        return false;
    t = ((reg & (1u << 7)) != 0) ? PrechargeSafetyTimer::HRS_0_5
                                 : PrechargeSafetyTimer::HRS_2;
    return true;
}

bool BQ25672::setPrechargeSafetyTimer(PrechargeSafetyTimer t)
{
    const uint8_t mask = (1u << 7);
    const uint8_t val = (t == PrechargeSafetyTimer::HRS_0_5) ? mask : 0;
    return updateBits(REG::IOTG, mask, val);
}

// Bits 6:0: IOTG current limit
bool BQ25672::getOtgCurrentLimit_mA(uint16_t &ma)
{
    uint8_t reg;
    if (!getIotgControlRaw(reg))
        return false;

    const uint8_t code = reg & 0x7F; // bits 6..0
    ma = uint16_t(code) * 40;

    if (ma < 160)
        ma = 160; // clamped low + range min
    if (ma > 3360)
        ma = 3360; // range max
    return true;
}

bool BQ25672::setOtgCurrentLimit_mA(uint16_t ma)
{
    // Clamp to datasheet range
    if (ma < 160)
        ma = 160;
    if (ma > 3360)
        ma = 3360;

    // Convert to code (40 mA per LSB), rounded to nearest step
    uint16_t code = (ma + 20) / 40;
    if (code > 0x7F)
        code = 0x7F;

    const uint8_t mask = 0x7F; // bits 6..0
    return updateBits(REG::IOTG, mask, static_cast<uint8_t>(code));
}

// --- TIMER_CTRL (REG 0x0E) ---
// [7:6] TOPOFF_TMR_1:0  - Top-off timer control
// [5]   EN_TRICHG_TMR   - Enable trickle charge timer (fixed 1 hr)
// [4]   EN_PRECHG_TMR   - Enable pre-charge timer
// [3]   EN_CHG_TMR      - Enable fast charge timer
// [2:1] CHG_TMR_1:0     - Fast charge timer setting
// [0]   TMR2X_EN        - Slow timers by 2X during input DPM or thermal regulation

bool BQ25672::getTimerControlRaw(uint8_t &out)
{
    return read8(REG::TIMER_CTRL, out);
}

// [7:6] TOPOFF_TMR
bool BQ25672::getTopOffTimer(TopOffTimer &t)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    t = static_cast<TopOffTimer>((reg >> 6) & 0x03);
    return true;
}

bool BQ25672::setTopOffTimer(TopOffTimer t)
{
    const uint8_t mask = 0b11000000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 6;
    return updateBits(REG::TIMER_CTRL, mask, val);
}

// Bit 5: EN_TRICHG_TMR
bool BQ25672::getTrickleChargeTimerEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    enabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setTrickleChargeTimerEnabled(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::TIMER_CTRL, mask, enable ? mask : 0);
}

// Bit 4: EN_PRECHG_TMR
bool BQ25672::getPrechargeTimerEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    enabled = (reg & (1u << 4)) != 0;
    return true;
}

bool BQ25672::setPrechargeTimerEnabled(bool enable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::TIMER_CTRL, mask, enable ? mask : 0);
}

// Bit 3: EN_CHG_TMR

bool BQ25672::getFastChargeTimerEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    enabled = (reg & (1u << 3)) != 0;
    return true;
}

bool BQ25672::setFastChargeTimerEnabled(bool enable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::TIMER_CTRL, mask, enable ? mask : 0);
}

// [2:1] CHG_TMR
bool BQ25672::getFastChargeTimer(FastChargeTimer &t)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    t = static_cast<FastChargeTimer>((reg >> 1) & 0x03);
    return true;
}

bool BQ25672::setFastChargeTimer(FastChargeTimer t)
{
    const uint8_t mask = 0b00000110;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 1;
    return updateBits(REG::TIMER_CTRL, mask, val);
}

// Bit 0: TMR2X_EN
bool BQ25672::getTimer2xEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTimerControlRaw(reg))
        return false;
    enabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setTimer2xEnabled(bool enable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::TIMER_CTRL, mask, enable ? mask : 0);
}

// --- Charger control ---

// --- CHG_CTRL0 (REG 0x0F) ---
// [7] EN_AUTO_IBATDIS - Auto battery discharging during BAT OVP fault
// [6] FORCE_IBATDIS   - Force battery discharging current
// [5] EN_CHG          - Charger enable
// [4] EN_ICO          - Input Current Optimizer enable
// [3] FORCE_ICO       - Force start ICO (valid only when EN_ICO=1, self-clears)
// [2] EN_HIZ          - Enable HIZ mode (also reset to 0 when adapter plugged)
// [1] EN_TERM         - Enable termination
// [0] (not defined here) - preserved

bool BQ25672::getChargerControl0Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL0, out);
}

bool BQ25672::getAutoIbatDischargeEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 7)) != 0;
    return true;
}

bool BQ25672::setAutoIbatDischargeEnabled(bool enable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

bool BQ25672::getForceIbatDischarge(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 6)) != 0;
    return true;
}

bool BQ25672::setForceIbatDischarge(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

bool BQ25672::getChargeEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setChargeEnabled(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

bool BQ25672::getIcoEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 4)) != 0;
    return true;
}

bool BQ25672::setIcoEnabled(bool enable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

// FORCE_ICO: can be written 1 to start ICO; returns to 0 after ICO starts.
bool BQ25672::triggerIcoStart()
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::CHG_CTRL0, mask, mask); // write 1
}

bool BQ25672::triggerIcoStartSafe()
{
    bool en;
    if (!getIcoEnabled(en))
        return false;
    if (!en)
        return false;
    return triggerIcoStart();
}

bool BQ25672::getHizEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 2)) != 0;
    return true;
}

bool BQ25672::setHizEnabled(bool enable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

bool BQ25672::getTerminationEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl0Raw(reg))
        return false;
    enabled = (reg & (1u << 1)) != 0;
    return true;
}

bool BQ25672::setTerminationEnabled(bool enable)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::CHG_CTRL0, mask, enable ? mask : 0);
}

// --- CHG_CTRL1 (REG 0x10) ---
// [5:4] VAC_OVP_1:0  - VAC OVP thresholds
// [3]   WD_RST       - I2C watchdog timer reset (self-clearing)
// [2:0] WATCHDOG_2:0 - Watchdog timer settings
// [7:6] reserved - preserved

bool BQ25672::getChargerControl1Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL1, out);
}

// [5:4] VAC_OVP
bool BQ25672::getVacOvpThreshold(VacOvpThreshold &t)
{
    uint8_t reg;
    if (!getChargerControl1Raw(reg))
        return false;
    t = static_cast<VacOvpThreshold>((reg >> 4) & 0x03);
    return true;
}

bool BQ25672::setVacOvpThreshold(VacOvpThreshold t)
{
    const uint8_t mask = 0b00110000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 4;
    return updateBits(REG::CHG_CTRL1, mask, val);
}

// [3] WD_RST - write 1 to reset watchdog timer, bit returns to 0 after reset.
bool BQ25672::triggerWatchdogReset()
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::CHG_CTRL1, mask, mask);
}

// [2:0] WATCHDOG

bool BQ25672::getWatchdogTimer(WatchdogTimer &t)
{
    uint8_t reg;
    if (!getChargerControl1Raw(reg))
        return false;
    t = static_cast<WatchdogTimer>(reg & 0x07);
    return true;
}

bool BQ25672::setWatchdogTimer(WatchdogTimer t)
{
    const uint8_t mask = 0x07;
    return updateBits(REG::CHG_CTRL1, mask, static_cast<uint8_t>(t) & 0x07);
}

// --- CHG_CTRL2 (REG 0x11) ---
// [7]   FORCE_INDET    - Force D+/D- detection (self-clearing)
// [6]   AUTO_INDET_EN  - Automatic D+/D- detection enable on VBUS plug-in
// [5]   EN_12V         - Enable 12V mode in HVDCP
// [4]   EN_9V          - Enable 9V mode in HVDCP
// [3]   HVDCP_EN       - Enable HVDCP handshake
// [2:1] SDRV_CTRL_1:0  - SFET (ship FET) control
// [0]   SDRV_DLY       - Delay for SDRV_CTRL action (0: add 10s delay, 1: no delay)

bool BQ25672::getChargerControl2Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL2, out);
}

// [7] FORCE_INDET (write 1 to trigger, then it resets to 0)
bool BQ25672::triggerDpDmDetection()
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::CHG_CTRL2, mask, mask);
}

// [6] AUTO_INDET_EN
bool BQ25672::getAutoDpDmDetectionEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    enabled = (reg & (1u << 6)) != 0;
    return true;
}

bool BQ25672::setAutoDpDmDetectionEnabled(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::CHG_CTRL2, mask, enable ? mask : 0);
}

// [5] EN_12V
bool BQ25672::getHvdcp12VEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    enabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setHvdcp12VEnabled(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::CHG_CTRL2, mask, enable ? mask : 0);
}

// [4] EN_9V
bool BQ25672::getHvdcp9VEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    enabled = (reg & (1u << 4)) != 0;
    return true;
}

bool BQ25672::setHvdcp9VEnabled(bool enable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::CHG_CTRL2, mask, enable ? mask : 0);
}

// [3] HVDCP_EN
bool BQ25672::getHvdcpEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    enabled = (reg & (1u << 3)) != 0;
    return true;
}

bool BQ25672::setHvdcpEnabled(bool enable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::CHG_CTRL2, mask, enable ? mask : 0);
}

// [2:1] SDRV_CTRL
bool BQ25672::getSdrvControl(SdrvCtrl &mode)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    mode = static_cast<SdrvCtrl>((reg >> 1) & 0x03);
    return true;
}

bool BQ25672::setSdrvControl(SdrvCtrl mode)
{
    const uint8_t mask = 0b00000110;
    const uint8_t val = (static_cast<uint8_t>(mode) & 0x03) << 1;
    return updateBits(REG::CHG_CTRL2, mask, val);
}

// [0] SDRV_DLY
bool BQ25672::getSdrvDelayNo10s(bool &no_delay)
{
    uint8_t reg;
    if (!getChargerControl2Raw(reg))
        return false;
    no_delay = (reg & 0x01) != 0;
    return true;
}

// true = do NOT add 10s delay, false = add 10s delay
bool BQ25672::setSdrvDelayNo10s(bool no_delay)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::CHG_CTRL2, mask, no_delay ? mask : 0);
}

// --- CHG_CTRL3 (REG 0x12) ---
// [7] DIS_ACDRV    - Force EN_ACDRV1=0 and EN_ACDRV2=0
// [6] EN_OTG       - OTG enable
// [5] PFM_OTG_DIS  - Disable PFM in OTG mode (0: enable PFM, 1: disable PFM)
// [4] PFM_FWD_DIS  - Disable PFM in forward mode (0: enable PFM, 1: disable PFM)
// [3] WKUP_DLY     - Ship mode wake-up delay (0: 1s, 1: 15ms)
// [2] DIS_LDO      - Disable BATFET LDO mode in pre-charge stage (0: enable, 1: disable)
// [1] DIS_OTG_OOA  - Disable OOA in OTG mode (0: enable, 1: disable)
// [0] DIS_FWD_OOA  - Disable OOA in forward mode (0: enable, 1: disable)

bool BQ25672::getChargerControl3Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL3, out);
}

bool BQ25672::getAcdrvDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & (1u << 7)) != 0;
    return true;
}

bool BQ25672::setAcdrvDisabled(bool disable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

bool BQ25672::getOtgEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    enabled = (reg & (1u << 6)) != 0;
    return true;
}

bool BQ25672::setOtgEnabled(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::CHG_CTRL3, mask, enable ? mask : 0);
}

bool BQ25672::getPfmOtgDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setPfmOtgDisabled(bool disable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

bool BQ25672::getPfmForwardDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & (1u << 4)) != 0;
    return true;
}

bool BQ25672::setPfmForwardDisabled(bool disable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

// WKUP_DLY: 0 = 1s (default), 1 = 15ms
bool BQ25672::getWakeupDelay15ms(bool &is_15ms)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    is_15ms = (reg & (1u << 3)) != 0;
    return true;
}

bool BQ25672::setWakeupDelay15ms(bool is_15ms)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::CHG_CTRL3, mask, is_15ms ? mask : 0);
}

bool BQ25672::getBatfetLdoDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & (1u << 2)) != 0;
    return true;
}

bool BQ25672::setBatfetLdoDisabled(bool disable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

bool BQ25672::getOoaOtgDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & (1u << 1)) != 0;
    return true;
}

bool BQ25672::setOoaOtgDisabled(bool disable)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

bool BQ25672::getOoaForwardDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl3Raw(reg))
        return false;
    disabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setOoaForwardDisabled(bool disable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::CHG_CTRL3, mask, disable ? mask : 0);
}

// --- CHG_CTRL4 (REG 0x13) ---
// [7] EN_ACDRV2        - External ACFET2-RBFET2 gate driver control (may be locked at 0)
// [6] EN_ACDRV1        - External ACFET1-RBFET1 gate driver control (may be locked at 0)
// [5] PWM_FREQ         - Switching frequency selection (0: 1.5 MHz, 1: 750 kHz)
// [4] DIS_STAT         - Disable STAT pin output (0: enable STAT, 1: disable STAT)
// [3] DIS_VSYS_SHORT   - Disable forward mode VSYS short hiccup protection (0: enable, 1: disable)
// [2] DIS_VOTG_UVP     - Disable OTG mode VOTG UVP hiccup protection (0: enable, 1: disable)
// [1] FORCE_VINDPM_DET - Force VINDPM detection (self-clearing)
// [0] EN_IBUS_OCP      - Enable IBUS_OCP in forward mode

bool BQ25672::getChargerControl4Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL4, out);
}

bool BQ25672::getAcdrv2Enabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    enabled = (reg & (1u << 7)) != 0;
    return true;
}

bool BQ25672::setAcdrv2Enabled(bool enable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::CHG_CTRL4, mask, enable ? mask : 0);
}

bool BQ25672::getAcdrv1Enabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    enabled = (reg & (1u << 6)) != 0;
    return true;
}

bool BQ25672::setAcdrv1Enabled(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::CHG_CTRL4, mask, enable ? mask : 0);
}

bool BQ25672::getPwmFrequency(PwmFrequency &f)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    f = ((reg & (1u << 5)) != 0) ? PwmFrequency::kHz750 : PwmFrequency::MHz1_5;
    return true;
}

bool BQ25672::setPwmFrequency(PwmFrequency f)
{
    const uint8_t mask = (1u << 5);
    const uint8_t val = (f == PwmFrequency::kHz750) ? mask : 0;
    return updateBits(REG::CHG_CTRL4, mask, val);
}

bool BQ25672::getStatPinDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    disabled = (reg & (1u << 4)) != 0;
    return true;
}

bool BQ25672::setStatPinDisabled(bool disable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::CHG_CTRL4, mask, disable ? mask : 0);
}

bool BQ25672::getVsysShortProtectionDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    disabled = (reg & (1u << 3)) != 0;
    return true;
}

bool BQ25672::setVsysShortProtectionDisabled(bool disable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::CHG_CTRL4, mask, disable ? mask : 0);
}

bool BQ25672::getVotgUvpProtectionDisabled(bool &disabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    disabled = (reg & (1u << 2)) != 0;
    return true;
}

bool BQ25672::setVotgUvpProtectionDisabled(bool disable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::CHG_CTRL4, mask, disable ? mask : 0);
}

// Bit 1: FORCE_VINDPM_DET (write 1 to trigger, then it resets to 0)
bool BQ25672::triggerVindpmDetection()
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::CHG_CTRL4, mask, mask);
}

bool BQ25672::getIbusOcpEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl4Raw(reg))
        return false;
    enabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setIbusOcpEnabled(bool enable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::CHG_CTRL4, mask, enable ? mask : 0);
}

// --- CHG_CTRL5 (REG 0x14) ---
// [7]   SFET_PRESENT   - Ship FET populated indicator (enables ship-FET related features)
// [6]   reserved
// [5]   EN_IBAT        - IBAT discharge current sensing enable
// [4:3] IBAT_REG_1:0   - Battery discharging current regulation in OTG mode
// [2]   EN_IINDPM      - Enable internal IINDPM regulation
// [1]   EN_EXTILIM     - Enable external ILIM_HIZ pin input current regulation
// [0]   EN_BATOC       - Enable battery discharging current OCP
//
// Note: If SFET_PRESENT=0, some ship-FET related bits in other registers may be locked to 0.

bool BQ25672::getChargerControl5Raw(uint8_t &out)
{
    return read8(REG::CHG_CTRL5, out);
}

// [7] SFET_PRESENT
bool BQ25672::getShipFetPresent(bool &present)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    present = (reg & (1u << 7)) != 0;
    return true;
}

bool BQ25672::setShipFetPresent(bool present)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::CHG_CTRL5, mask, present ? mask : 0);
}

// [5] EN_IBAT
bool BQ25672::getIbatDischargeSensingEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    enabled = (reg & (1u << 5)) != 0;
    return true;
}

bool BQ25672::setIbatDischargeSensingEnabled(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::CHG_CTRL5, mask, enable ? mask : 0);
}

// [4:3] IBAT_REG
bool BQ25672::getIbatRegulationOtg(IbatRegOtg &r)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    r = static_cast<IbatRegOtg>((reg >> 3) & 0x03);
    return true;
}

bool BQ25672::setIbatRegulationOtg(IbatRegOtg r)
{
    const uint8_t mask = 0b00011000;
    const uint8_t val = (static_cast<uint8_t>(r) & 0x03) << 3;
    return updateBits(REG::CHG_CTRL5, mask, val);
}

// [2] EN_IINDPM
bool BQ25672::getInternalIindpmEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    enabled = (reg & (1u << 2)) != 0;
    return true;
}

bool BQ25672::setInternalIindpmEnabled(bool enable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::CHG_CTRL5, mask, enable ? mask : 0);
}

// [1] EN_EXTILIM
bool BQ25672::getExternalIlimEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    enabled = (reg & (1u << 1)) != 0;
    return true;
}

bool BQ25672::setExternalIlimEnabled(bool enable)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::CHG_CTRL5, mask, enable ? mask : 0);
}

// [0] EN_BATOC
bool BQ25672::getBatteryDischargeOcpEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getChargerControl5Raw(reg))
        return false;
    enabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setBatteryDischargeOcpEnabled(bool enable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::CHG_CTRL5, mask, enable ? mask : 0);
}

// --- MPPT / temperature / NTC ---

// --- MPPT_CTRL (REG 0x15) ---
// [7:5] VOC_PCT_2:0  - VINDPM as a percentage of VBUS open-circuit voltage (VOC)
// [4:3] VOC_DLY_1:0  - Delay before VOC measurement after converter stops switching
// [2:1] VOC_RATE_1:0 - Interval between VOC measurements
// [0]   EN_MPPT      - Enable MPPT VOC measurement

bool BQ25672::getMpptControlRaw(uint8_t &out)
{
    return read8(REG::MPPT_CTRL, out);
}

// [0] EN_MPPT
bool BQ25672::getMpptEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getMpptControlRaw(reg))
        return false;
    enabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setMpptEnabled(bool enable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::MPPT_CTRL, mask, enable ? mask : 0);
}

// [7:5] VOC_PCT
bool BQ25672::getVocPct(VocPct &p)
{
    uint8_t reg;
    if (!getMpptControlRaw(reg))
        return false;
    p = static_cast<VocPct>((reg >> 5) & 0x07);
    return true;
}

bool BQ25672::setVocPct(VocPct p)
{
    const uint8_t mask = 0b11100000;
    const uint8_t val = (static_cast<uint8_t>(p) & 0x07) << 5;
    return updateBits(REG::MPPT_CTRL, mask, val);
}

bool BQ25672::getVocPct_permille(uint16_t &permille)
{
    VocPct p;
    if (!getVocPct(p))
        return false;

    switch (p)
    {
    case VocPct::P0_5625:
        permille = 562;
        return true;
    case VocPct::P0_6250:
        permille = 625;
        return true;
    case VocPct::P0_6875:
        permille = 688;
        return true;
    case VocPct::P0_7500:
        permille = 750;
        return true;
    case VocPct::P0_8125:
        permille = 813;
        return true;
    case VocPct::P0_8750:
        permille = 875;
        return true;
    case VocPct::P0_9375:
        permille = 938;
        return true;
    case VocPct::P1_0000:
        permille = 1000;
        return true;
    default:
        return false;
    }
}

// [4:3] VOC_DLY
bool BQ25672::getVocDelay(VocDelay &d)
{
    uint8_t reg;
    if (!getMpptControlRaw(reg))
        return false;
    d = static_cast<VocDelay>((reg >> 3) & 0x03);
    return true;
}

bool BQ25672::setVocDelay(VocDelay d)
{
    const uint8_t mask = 0b00011000;
    const uint8_t val = (static_cast<uint8_t>(d) & 0x03) << 3;
    return updateBits(REG::MPPT_CTRL, mask, val);
}

bool BQ25672::getVocDelay_ms(uint32_t &ms)
{
    VocDelay d;
    if (!getVocDelay(d))
        return false;

    switch (d)
    {
    case VocDelay::Ms50:
        ms = 50;
        return true;
    case VocDelay::Ms300:
        ms = 300;
        return true;
    case VocDelay::S2:
        ms = 2000;
        return true;
    case VocDelay::S5:
        ms = 5000;
        return true;
    default:
        return false;
    }
}

// [2:1] VOC_RATE
bool BQ25672::getVocRate(VocRate &r)
{
    uint8_t reg;
    if (!getMpptControlRaw(reg))
        return false;
    r = static_cast<VocRate>((reg >> 1) & 0x03);
    return true;
}

bool BQ25672::setVocRate(VocRate r)
{
    const uint8_t mask = 0b00000110;
    const uint8_t val = (static_cast<uint8_t>(r) & 0x03) << 1;
    return updateBits(REG::MPPT_CTRL, mask, val);
}

bool BQ25672::getVocRate_s(uint32_t &s)
{
    VocRate r;
    if (!getVocRate(r))
        return false;

    switch (r)
    {
    case VocRate::S30:
        s = 30;
        return true;
    case VocRate::Min2:
        s = 120;
        return true;
    case VocRate::Min10:
        s = 600;
        return true;
    case VocRate::Min30:
        s = 1800;
        return true;
    default:
        return false;
    }
}

// --- TEMP_CTRL (REG 0x16) ---
// [7:6] TREG_1:0    - Thermal regulation thresholds
// [5:4] TSHUT_1:0   - Thermal shutdown thresholds
// [3]   VBUS_PD_EN  - Enable VBUS pull-down resistor (6k Ohm)
// [2]   VAC1_PD_EN  - Enable VAC1 pull-down resistor
// [1]   VAC2_PD_EN  - Enable VAC2 pull-down resistor
// [0]   reserved    - preserved

bool BQ25672::getTemperatureControlRaw(uint8_t &out)
{
    return read8(REG::TEMP_CTRL, out);
}

// [7:6] TREG
bool BQ25672::getThermalRegThreshold(ThermalRegThreshold &t)
{
    uint8_t reg;
    if (!getTemperatureControlRaw(reg))
        return false;
    t = static_cast<ThermalRegThreshold>((reg >> 6) & 0x03);
    return true;
}

bool BQ25672::setThermalRegThreshold(ThermalRegThreshold t)
{
    const uint8_t mask = 0b11000000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 6;
    return updateBits(REG::TEMP_CTRL, mask, val);
}

bool BQ25672::getThermalRegThreshold_C(uint16_t &celsius)
{
    ThermalRegThreshold t;
    if (!getThermalRegThreshold(t))
        return false;

    switch (t)
    {
    case ThermalRegThreshold::C60:
        celsius = 60;
        return true;
    case ThermalRegThreshold::C80:
        celsius = 80;
        return true;
    case ThermalRegThreshold::C100:
        celsius = 100;
        return true;
    case ThermalRegThreshold::C120:
        celsius = 120;
        return true;
    default:
        return false;
    }
}

// [5:4] TSHUT
bool BQ25672::getThermalShutdownThreshold(ThermalShutdownThreshold &t)
{
    uint8_t reg;
    if (!getTemperatureControlRaw(reg))
        return false;
    t = static_cast<ThermalShutdownThreshold>((reg >> 4) & 0x03);
    return true;
}

bool BQ25672::setThermalShutdownThreshold(ThermalShutdownThreshold t)
{
    const uint8_t mask = 0b00110000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 4;
    return updateBits(REG::TEMP_CTRL, mask, val);
}

bool BQ25672::getThermalShutdownThreshold_C(uint16_t &celsius)
{
    ThermalShutdownThreshold t;
    if (!getThermalShutdownThreshold(t))
        return false;

    switch (t)
    {
    case ThermalShutdownThreshold::C150:
        celsius = 150;
        return true;
    case ThermalShutdownThreshold::C130:
        celsius = 130;
        return true;
    case ThermalShutdownThreshold::C120:
        celsius = 120;
        return true;
    case ThermalShutdownThreshold::C85:
        celsius = 85;
        return true;
    default:
        return false;
    }
}

// [3] VBUS_PD_EN
bool BQ25672::getVbusPullDownEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTemperatureControlRaw(reg))
        return false;
    enabled = (reg & (1u << 3)) != 0;
    return true;
}

bool BQ25672::setVbusPullDownEnabled(bool enable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::TEMP_CTRL, mask, enable ? mask : 0);
}

// [2] VAC1_PD_EN
bool BQ25672::getVac1PullDownEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTemperatureControlRaw(reg))
        return false;
    enabled = (reg & (1u << 2)) != 0;
    return true;
}

bool BQ25672::setVac1PullDownEnabled(bool enable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::TEMP_CTRL, mask, enable ? mask : 0);
}

// [1] VAC2_PD_EN
bool BQ25672::getVac2PullDownEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getTemperatureControlRaw(reg))
        return false;
    enabled = (reg & (1u << 1)) != 0;
    return true;
}

bool BQ25672::setVac2PullDownEnabled(bool enable)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::TEMP_CTRL, mask, enable ? mask : 0);
}

// --- NTC_CTRL0 (REG 0x17) ---
// [7:5] JEITA_VSET_2:0   - JEITA high temp (TWARN–THOT) charge voltage setting
// [4:3] JEITA_ISETH_1:0  - JEITA high temp (TWARN–THOT) charge current setting
// [2:1] JEITA_ISETC_1:0  - JEITA low temp (TCOLD–TCOOL) charge current setting
// [0]   reserved         - preserved

bool BQ25672::getNtcControl0Raw(uint8_t &out)
{
    return read8(REG::NTC_CTRL0, out);
}

// [7:5] JEITA_VSET
bool BQ25672::getJeitaVsetHigh(JeitaVsetHigh &v)
{
    uint8_t reg;
    if (!getNtcControl0Raw(reg))
        return false;
    v = static_cast<JeitaVsetHigh>((reg >> 5) & 0x07);
    return true;
}

bool BQ25672::setJeitaVsetHigh(JeitaVsetHigh v)
{
    const uint8_t mask = 0b11100000;
    const uint8_t val = (static_cast<uint8_t>(v) & 0x07) << 5;
    return updateBits(REG::NTC_CTRL0, mask, val);
}

// [4:3] JEITA_ISETH
bool BQ25672::getJeitaIsetHigh(JeitaIsetHigh &i)
{
    uint8_t reg;
    if (!getNtcControl0Raw(reg))
        return false;
    i = static_cast<JeitaIsetHigh>((reg >> 3) & 0x03);
    return true;
}

bool BQ25672::setJeitaIsetHigh(JeitaIsetHigh i)
{
    const uint8_t mask = 0b00011000;
    const uint8_t val = (static_cast<uint8_t>(i) & 0x03) << 3;
    return updateBits(REG::NTC_CTRL0, mask, val);
}

// [2:1] JEITA_ISETC
bool BQ25672::getJeitaIsetCold(JeitaIsetCold &i)
{
    uint8_t reg;
    if (!getNtcControl0Raw(reg))
        return false;
    i = static_cast<JeitaIsetCold>((reg >> 1) & 0x03);
    return true;
}

bool BQ25672::setJeitaIsetCold(JeitaIsetCold i)
{
    const uint8_t mask = 0b00000110;
    const uint8_t val = (static_cast<uint8_t>(i) & 0x03) << 1;
    return updateBits(REG::NTC_CTRL0, mask, val);
}

// --- NTC_CTRL1 (REG 0x18) ---
// [7:6] TS_COOL_1:0 - JEITA VT2 rising threshold (%REGN), approx temp depends on NTC network
// [5:4] TS_WARM_1:0 - JEITA VT3 falling threshold (%REGN), approx temp depends on NTC network
// [3:2] BHOT_1:0    - OTG mode TS HOT temperature threshold
// [1]   BCOLD       - OTG mode TS COLD temperature threshold
// [0]   TS_IGNORE   - Ignore TS feedback (TS always considered good)

bool BQ25672::getNtcControl1Raw(uint8_t &out)
{
    return read8(REG::NTC_CTRL1, out);
}

// [7:6] TS_COOL
bool BQ25672::getTsCoolThreshold(TsCoolThreshold &t)
{
    uint8_t reg;
    if (!getNtcControl1Raw(reg))
        return false;
    t = static_cast<TsCoolThreshold>((reg >> 6) & 0x03);
    return true;
}

bool BQ25672::setTsCoolThreshold(TsCoolThreshold t)
{
    const uint8_t mask = 0b11000000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 6;
    return updateBits(REG::NTC_CTRL1, mask, val);
}

// Optional helper: approximate temp in °C for TS_COOL (based on the datasheet note)
bool BQ25672::getTsCoolApprox_C(int16_t &celsius)
{
    TsCoolThreshold t;
    if (!getTsCoolThreshold(t))
        return false;
    switch (t)
    {
    case TsCoolThreshold::P71_1:
        celsius = 5;
        return true;
    case TsCoolThreshold::P68_4:
        celsius = 10;
        return true;
    case TsCoolThreshold::P65_5:
        celsius = 15;
        return true;
    case TsCoolThreshold::P62_4:
        celsius = 20;
        return true;
    default:
        return false;
    }
}

// [5:4] TS_WARM
bool BQ25672::getTsWarmThreshold(TsWarmThreshold &t)
{
    uint8_t reg;
    if (!getNtcControl1Raw(reg))
        return false;
    t = static_cast<TsWarmThreshold>((reg >> 4) & 0x03);
    return true;
}

bool BQ25672::setTsWarmThreshold(TsWarmThreshold t)
{
    const uint8_t mask = 0b00110000;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 4;
    return updateBits(REG::NTC_CTRL1, mask, val);
}

// Optional helper: approximate temp in °C for TS_WARM (based on the datasheet note)
bool BQ25672::getTsWarmApprox_C(int16_t &celsius)
{
    TsWarmThreshold t;
    if (!getTsWarmThreshold(t))
        return false;
    switch (t)
    {
    case TsWarmThreshold::P48_4:
        celsius = 40;
        return true;
    case TsWarmThreshold::P44_8:
        celsius = 45;
        return true;
    case TsWarmThreshold::P41_2:
        celsius = 50;
        return true;
    case TsWarmThreshold::P37_7:
        celsius = 55;
        return true;
    default:
        return false;
    }
}

// [3:2] BHOT
bool BQ25672::getOtgHotThreshold(OtgHotThreshold &t)
{
    uint8_t reg;
    if (!getNtcControl1Raw(reg))
        return false;
    t = static_cast<OtgHotThreshold>((reg >> 2) & 0x03);
    return true;
}

bool BQ25672::setOtgHotThreshold(OtgHotThreshold t)
{
    const uint8_t mask = 0b00001100;
    const uint8_t val = (static_cast<uint8_t>(t) & 0x03) << 2;
    return updateBits(REG::NTC_CTRL1, mask, val);
}

bool BQ25672::getOtgHotThreshold_C(int16_t &celsius, bool &disabled)
{
    OtgHotThreshold t;
    if (!getOtgHotThreshold(t))
        return false;

    disabled = (t == OtgHotThreshold::Disabled);
    if (disabled)
    {
        celsius = 0;
        return true;
    }

    switch (t)
    {
    case OtgHotThreshold::C55:
        celsius = 55;
        return true;
    case OtgHotThreshold::C60:
        celsius = 60;
        return true;
    case OtgHotThreshold::C65:
        celsius = 65;
        return true;
    default:
        return false;
    }
}

// [1] BCOLD
bool BQ25672::getOtgColdThreshold(OtgColdThreshold &t)
{
    uint8_t reg;
    if (!getNtcControl1Raw(reg))
        return false;
    t = ((reg & (1u << 1)) != 0) ? OtgColdThreshold::Cneg20 : OtgColdThreshold::Cneg10;
    return true;
}

bool BQ25672::setOtgColdThreshold(OtgColdThreshold t)
{
    const uint8_t mask = (1u << 1);
    const uint8_t val = (t == OtgColdThreshold::Cneg20) ? mask : 0;
    return updateBits(REG::NTC_CTRL1, mask, val);
}

// [0] TS_IGNORE
bool BQ25672::getTsIgnoreEnabled(bool &enabled)
{
    uint8_t reg;
    if (!getNtcControl1Raw(reg))
        return false;
    enabled = (reg & 0x01) != 0;
    return true;
}

bool BQ25672::setTsIgnoreEnabled(bool enable)
{
    const uint8_t mask = 0x01;
    return updateBits(REG::NTC_CTRL1, mask, enable ? mask : 0);
}

// --- ICO_ILIM (REG 0x19) ---
// ICO_ILIM_8:0 (read-only): Input current limit obtained from ICO or ILIM_HIZ pin setting
// 0 mA offset, 10 mA step, range 100..3300 mA, clamped low.

bool BQ25672::getIcoInputCurrentLimit_mA(uint16_t &ma)
{
    uint16_t raw;
    if (!read16(REG::ICO_ILIM, raw))
        return false;

    const uint16_t code = raw & 0x01FF; // bits 8..0
    ma = code * 10;

    if (ma < 100)
        ma = 100; // clamped low + range min
    if (ma > 3300)
        ma = 3300; // range max
    return true;
}

// --- Status registers ---

// --- STAT0 (REG 0x1B) ---
// [7] IINDPM_STAT        - In IINDPM regulation (forward) / IOTG regulation (OTG)
// [6] VINDPM_STAT        - In VINDPM regulation (forward) / VOTG regulation (OTG)
// [5] WD_STAT            - Watchdog timer expired
// [4] reserved
// [3] PG_STAT            - Power good
// [2] AC2_PRESENT_STAT   - VAC2 present
// [1] AC1_PRESENT_STAT   - VAC1 present
// [0] VBUS_PRESENT_STAT  - VBUS present

bool BQ25672::getStatus0Raw(uint8_t &out)
{
    return read8(REG::STAT0, out);
}

bool BQ25672::getStatus0(Stat0 &s)
{
    uint8_t v;
    if (!getStatus0Raw(v))
        return false;

    s.iindpm_or_iotg = (v & (1u << 7)) != 0;
    s.vindpm_or_votg = (v & (1u << 6)) != 0;
    s.watchdog_expired = (v & (1u << 5)) != 0;
    s.power_good = (v & (1u << 3)) != 0;
    s.vac2_present = (v & (1u << 2)) != 0;
    s.vac1_present = (v & (1u << 1)) != 0;
    s.vbus_present = (v & 0x01) != 0;

    return true;
}

// --- STAT1 (REG 0x1C) ---
// [7:5] CHG_STAT_2:0     - Charge status
// [4:1] VBUS_STAT_3:0    - VBUS status
// [0]   BC1_2_DONE_STAT  - BC1.2 / non-standard detection done

bool BQ25672::getStatus1Raw(uint8_t &out)
{
    return read8(REG::STAT1, out);
}

bool BQ25672::getStatus1(Stat1 &s)
{
    uint8_t v;
    if (!getStatus1Raw(v))
        return false;

    s.chg_status = static_cast<ChargeStatus>((v >> 5) & 0x07);
    s.vbus_status = static_cast<VbusStatus>((v >> 1) & 0x0F);
    s.bc12_done = (v & 0x01) != 0;
    return true;
}

// --- STAT2 (REG 0x1D) ---
// [7:6] ICO_STAT_1:0        - ICO status
// [5:3] reserved
// [2]   TREG_STAT           - Thermal regulation active
// [1]   DPDM_STAT           - D+/D- detection ongoing
// [0]   VBAT_PRESENT_STAT   - Battery present (VBAT > VBAT_UVLOZ)

bool BQ25672::getStatus2Raw(uint8_t &out)
{
    return read8(REG::STAT2, out);
}

bool BQ25672::getStatus2(Stat2 &s)
{
    uint8_t v;
    if (!getStatus2Raw(v))
        return false;

    s.ico_status = static_cast<IcoStatus>((v >> 6) & 0x03);
    s.thermal_regulation = (v & (1u << 2)) != 0;
    s.dpdm_ongoing = (v & (1u << 1)) != 0;
    s.vbat_present = (v & 0x01) != 0;

    return true;
}

// --- STAT3 (REG 0x1E) ---
// [7] ACRB2_STAT        - ACFET2-RBFET2 placed
// [6] ACRB1_STAT        - ACFET1-RBFET1 placed
// [5] ADC_DONE_STAT     - ADC conversion complete (one-shot mode only)
// [4] VSYS_STAT         - VSYSMIN regulation active (VBAT < VSYSMIN)
// [3] CHG_TMR_STAT      - Fast charge safety timer expired
// [2] TRICHG_TMR_STAT   - Trickle charge safety timer expired
// [1] PRECHG_TMR_STAT   - Pre-charge safety timer expired
// [0] reserved

bool BQ25672::getStatus3Raw(uint8_t &out)
{
    return read8(REG::STAT3, out);
}

bool BQ25672::getStatus3(Stat3 &s)
{
    uint8_t v;
    if (!getStatus3Raw(v))
        return false;

    s.acrb2_placed = (v & (1u << 7)) != 0;
    s.acrb1_placed = (v & (1u << 6)) != 0;
    s.adc_done = (v & (1u << 5)) != 0;
    s.vsysmin_regulation = (v & (1u << 4)) != 0;
    s.chg_timer_expired = (v & (1u << 3)) != 0;
    s.trickle_timer_expired = (v & (1u << 2)) != 0;
    s.precharge_timer_expired = (v & (1u << 1)) != 0;

    return true;
}

// --- STAT4 (REG 0x1F) ---
// [7:5] reserved
// [4] VBATOTG_LOW_STAT - Battery too low to enable OTG
// [3] TS_COLD_STAT     - TS in cold range
// [2] TS_COOL_STAT     - TS in cool range
// [1] TS_WARM_STAT     - TS in warm range
// [0] TS_HOT_STAT      - TS in hot range

bool BQ25672::getStatus4Raw(uint8_t &out)
{
    return read8(REG::STAT4, out);
}

bool BQ25672::getStatus4(Stat4 &s)
{
    uint8_t v;
    if (!read8(REG::STAT4, v))
        return false;

    s.vbat_otg_too_low = (v & (1u << 4)) != 0;
    s.ts_cold = (v & (1u << 3)) != 0;
    s.ts_cool = (v & (1u << 2)) != 0;
    s.ts_warm = (v & (1u << 1)) != 0;
    s.ts_hot = (v & 0x01) != 0;

    return true;
}

BQ25672::TsRange BQ25672::getTsRangeFromStat4(const Stat4 &s)
{
    const uint8_t bits =
        (s.ts_cold ? 1u : 0u) |
        (s.ts_cool ? 2u : 0u) |
        (s.ts_warm ? 4u : 0u) |
        (s.ts_hot ? 8u : 0u);

    switch (bits)
    {
    case 0:
        return TsRange::Normal;
    case 1:
        return TsRange::Cold;
    case 2:
        return TsRange::Cool;
    case 4:
        return TsRange::Warm;
    case 8:
        return TsRange::Hot;
    default:
        return TsRange::Invalid;
    }
}

// --- Fault status ---

// --- FAULT0 (REG 0x20) ---
// [7] IBAT_REG_STAT   - Battery discharging current regulation active
// [6] VBUS_OVP_STAT   - VBUS over-voltage protection active
// [5] VBAT_OVP_STAT   - VBAT over-voltage protection active
// [4] IBUS_OCP_STAT   - IBUS over-current protection active
// [3] IBAT_OCP_STAT   - IBAT over-current protection active
// [2] CONV_OCP_STAT   - Converter over-current protection active
// [1] VAC2_OVP_STAT   - VAC2 over-voltage protection active
// [0] VAC1_OVP_STAT   - VAC1 over-voltage protection active

bool BQ25672::getFault0Raw(uint8_t &out)
{
    return read8(REG::FAULT0, out);
}

bool BQ25672::getFault0(Fault0 &f)
{
    uint8_t v;
    if (!getFault0Raw(v))
        return false;

    f.ibat_regulation = (v & (1u << 7)) != 0;
    f.vbus_ovp = (v & (1u << 6)) != 0;
    f.vbat_ovp = (v & (1u << 5)) != 0;
    f.ibus_ocp = (v & (1u << 4)) != 0;
    f.ibat_ocp = (v & (1u << 3)) != 0;
    f.conv_ocp = (v & (1u << 2)) != 0;
    f.vac2_ovp = (v & (1u << 1)) != 0;
    f.vac1_ovp = (v & 0x01) != 0;

    return true;
}

bool BQ25672::anyFault0Active(const Fault0 &f)
{
    return f.ibat_regulation || f.vbus_ovp || f.vbat_ovp || f.ibus_ocp ||
           f.ibat_ocp || f.conv_ocp || f.vac2_ovp || f.vac1_ovp;
}

// --- FAULT1 (REG 0x21) ---
// [7] VSYS_SHORT_STAT  - VSYS short circuit protection active
// [6] VSYS_OVP_STAT    - VSYS over-voltage protection active
// [5] OTG_OVP_STAT     - OTG over-voltage protection active
// [4] OTG_UVP_STAT     - OTG under-voltage protection active
// [3] reserved
// [2] TSHUT_STAT       - Thermal shutdown protection active
// [1:0] reserved

bool BQ25672::getFault1Raw(uint8_t &out)
{
    return read8(REG::FAULT1, out);
}

bool BQ25672::getFault1(Fault1 &f)
{
    uint8_t v;
    if (!getFault1Raw(v))
        return false;

    f.vsys_short = (v & (1u << 7)) != 0;
    f.vsys_ovp = (v & (1u << 6)) != 0;
    f.otg_ovp = (v & (1u << 5)) != 0;
    f.otg_uvp = (v & (1u << 4)) != 0;
    f.tshut = (v & (1u << 2)) != 0;

    return true;
}

bool BQ25672::anyFault1Active(const Fault1 &f)
{
    return f.vsys_short || f.vsys_ovp || f.otg_ovp ||
           f.otg_uvp || f.tshut;
}

// --- Charger flags ---

// --- FLAG0 (REG 0x22) ---
// [7] IINDPM_FLAG        - IINDPM/IOTG rising edge detected
// [6] VINDPM_FLAG        - VINDPM/VOTG rising edge detected
// [5] WD_FLAG            - Watchdog timer rising edge detected
// [4] POORSRC_FLAG       - Poor source status rising edge detected
// [3] PG_FLAG            - Any change in PG_STAT detected
// [2] AC2_PRESENT_FLAG   - VAC2 present status changed
// [1] AC1_PRESENT_FLAG   - VAC1 present status changed
// [0] VBUS_PRESENT_FLAG  - VBUS present status changed

bool BQ25672::getFlag0Raw(uint8_t &out)
{
    return read8(REG::FLAG0, out);
}

bool BQ25672::getFlag0(Flag0 &f)
{
    uint8_t v;
    if (!getFlag0Raw(v))
        return false;

    f.iindpm = (v & (1u << 7)) != 0;
    f.vindpm = (v & (1u << 6)) != 0;
    f.watchdog = (v & (1u << 5)) != 0;
    f.poor_source = (v & (1u << 4)) != 0;
    f.power_good_change = (v & (1u << 3)) != 0;
    f.vac2_present_change = (v & (1u << 2)) != 0;
    f.vac1_present_change = (v & (1u << 1)) != 0;
    f.vbus_present_change = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::anyFlag0Active(const Flag0 &f)
{
    return f.iindpm || f.vindpm || f.watchdog || f.poor_source ||
           f.power_good_change || f.vac2_present_change ||
           f.vac1_present_change || f.vbus_present_change;
}

// --- FLAG1 (REG 0x23) ---
// [7] CHG_FLAG           - Charge status changed
// [6] ICO_FLAG           - ICO status changed
// [5] reserved
// [4] VBUS_FLAG          - VBUS status changed
// [3] reserved
// [2] TREG_FLAG          - Thermal regulation threshold event
// [1] VBAT_PRESENT_FLAG  - VBAT present status changed
// [0] BC1_2_DONE_FLAG    - BC1.2 detection status changed

bool BQ25672::getFlag1Raw(uint8_t &out)
{
    return read8(REG::FLAG1, out);
}

bool BQ25672::getFlag1(Flag1 &f)
{
    uint8_t v;
    if (!getFlag1Raw(v))
        return false;

    f.charge_status_change = (v & (1u << 7)) != 0;
    f.ico_status_change = (v & (1u << 6)) != 0;
    f.vbus_status_change = (v & (1u << 4)) != 0;
    f.thermal_reg_event = (v & (1u << 2)) != 0;
    f.vbat_present_change = (v & (1u << 1)) != 0;
    f.bc12_done_change = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::anyFlag1Active(const Flag1 &f)
{
    return f.charge_status_change ||
           f.ico_status_change ||
           f.vbus_status_change ||
           f.thermal_reg_event ||
           f.vbat_present_change ||
           f.bc12_done_change;
}

// --- FLAG2 (REG 0x24) ---
// [7] reserved
// [6] DPDM_DONE_FLAG     - D+/D- detection completed
// [5] ADC_DONE_FLAG      - ADC conversion completed (one-shot mode only)
// [4] VSYS_FLAG          - Entered/exited VSYSMIN regulation
// [3] CHG_TMR_FLAG       - Fast charge timer expired event
// [2] TRICHG_TMR_FLAG    - Trickle charge timer expired event
// [1] PRECHG_TMR_FLAG    - Pre-charge timer expired event
// [0] TOPOFF_TMR_FLAG    - Top-off timer expired event

bool BQ25672::getFlag2Raw(uint8_t &out)
{
    return read8(REG::FLAG2, out);
}

bool BQ25672::getFlag2(Flag2 &f)
{
    uint8_t v;
    if (!getFlag2Raw(v))
        return false;

    f.dpdm_done = (v & (1u << 6)) != 0;
    f.adc_done = (v & (1u << 5)) != 0;
    f.vsysmin_reg_change = (v & (1u << 4)) != 0;
    f.fast_charge_timer_expired = (v & (1u << 3)) != 0;
    f.trickle_timer_expired = (v & (1u << 2)) != 0;
    f.precharge_timer_expired = (v & (1u << 1)) != 0;
    f.topoff_timer_expired = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::anyFlag2Active(const Flag2 &f)
{
    return f.dpdm_done || f.adc_done || f.vsysmin_reg_change ||
           f.fast_charge_timer_expired || f.trickle_timer_expired ||
           f.precharge_timer_expired || f.topoff_timer_expired;
}

// --- FLAG3 (REG 0x25) ---
// [7:5] reserved
// [4] VBATOTG_LOW_FLAG  - VBAT too low to enable OTG mode
// [3] TS_COLD_FLAG      - TS crossed cold threshold (T1)
// [2] TS_COOL_FLAG      - TS crossed cool threshold (T2)
// [1] TS_WARM_FLAG      - TS crossed warm threshold (T3)
// [0] TS_HOT_FLAG       - TS crossed hot threshold (T5)

bool BQ25672::getFlag3Raw(uint8_t &out)
{
    return read8(REG::FLAG3, out);
}

bool BQ25672::getFlag3(Flag3 &f)
{
    uint8_t v;
    if (!getFlag3Raw(v))
        return false;

    f.vbat_otg_low = (v & (1u << 4)) != 0;
    f.ts_cold = (v & (1u << 3)) != 0;
    f.ts_cool = (v & (1u << 2)) != 0;
    f.ts_warm = (v & (1u << 1)) != 0;
    f.ts_hot = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::anyFlag3Active(const Flag3 &f)
{
    return f.vbat_otg_low ||
           f.ts_cold || f.ts_cool ||
           f.ts_warm || f.ts_hot;
}

// --- Fault flags ---

// --- FAULT_FLAG0 (REG 0x26) ---
// [7] IBAT_REG_FLAG   - Enter/exit IBAT regulation event
// [6] VBUS_OVP_FLAG   - Enter VBUS OVP event
// [5] VBAT_OVP_FLAG   - Enter VBAT OVP event
// [4] IBUS_OCP_FLAG   - Enter IBUS OCP event
// [3] IBAT_OCP_FLAG   - Enter discharged OCP event
// [2] CONV_OCP_FLAG   - Enter converter OCP event
// [1] VAC2_OVP_FLAG   - Enter VAC2 OVP event
// [0] VAC1_OVP_FLAG   - Enter VAC1 OVP event

bool BQ25672::getFaultFlag0Raw(uint8_t &out)
{
    return read8(REG::FAULT_FLAG0, out);
}

bool BQ25672::getFaultFlag0(FaultFlag0 &f)
{
    uint8_t v;
    if (!getFaultFlag0Raw(v))
        return false;

    f.ibat_reg = (v & (1u << 7)) != 0;
    f.vbus_ovp = (v & (1u << 6)) != 0;
    f.vbat_ovp = (v & (1u << 5)) != 0;
    f.ibus_ocp = (v & (1u << 4)) != 0;
    f.ibat_ocp = (v & (1u << 3)) != 0;
    f.conv_ocp = (v & (1u << 2)) != 0;
    f.vac2_ovp = (v & (1u << 1)) != 0;
    f.vac1_ovp = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::anyFaultFlag0Active(const FaultFlag0 &f)
{
    return f.ibat_reg || f.vbus_ovp || f.vbat_ovp || f.ibus_ocp ||
           f.ibat_ocp || f.conv_ocp || f.vac2_ovp || f.vac1_ovp;
}

// --- FAULT_FLAG1 (REG 0x27) ---
// [7] VSYS_SHORT_FLAG  - Stop switching due to VSYS short circuit
// [6] VSYS_OVP_FLAG    - Stop switching due to VSYS over-voltage
// [5] OTG_OVP_FLAG     - Stop OTG due to VBUS over-voltage
// [4] OTG_UVP_FLAG     - Stop OTG due to VBUS under-voltage
// [3] reserved
// [2] TSHUT_FLAG       - Thermal shutdown threshold event
// [1:0] reserved

bool BQ25672::getFaultFlag1Raw(uint8_t &out)
{
    return read8(REG::FAULT_FLAG1, out);
}

bool BQ25672::getFaultFlag1(FaultFlag1 &f)
{
    uint8_t v;
    if (!getFaultFlag1Raw(v))
        return false;

    f.vsys_short = (v & (1u << 7)) != 0;
    f.vsys_ovp = (v & (1u << 6)) != 0;
    f.otg_ovp = (v & (1u << 5)) != 0;
    f.otg_uvp = (v & (1u << 4)) != 0;
    f.tshut = (v & (1u << 2)) != 0;

    return true;
}

bool BQ25672::anyFaultFlag1Active(const FaultFlag1 &f)
{
    return f.vsys_short || f.vsys_ovp ||
           f.otg_ovp || f.otg_uvp ||
           f.tshut;
}

// --- Masks ---

// --- MASK0 (REG 0x28) ---
// Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
//
// [7] IINDPM_MASK        - Mask IINDPM/IOTG event
// [6] VINDPM_MASK        - Mask VINDPM/VOTG event
// [5] WD_MASK            - Mask watchdog expired event
// [4] POORSRC_MASK       - Mask poor source event
// [3] PG_MASK            - Mask PG toggle event
// [2] AC2_PRESENT_MASK   - Mask VAC2 present change event
// [1] AC1_PRESENT_MASK   - Mask VAC1 present change event
// [0] VBUS_PRESENT_MASK  - Mask VBUS present change event

bool BQ25672::getMask0Raw(uint8_t &out)
{
    return read8(REG::MASK0, out);
}

bool BQ25672::setMask0Raw(uint8_t value)
{
    return write8(REG::MASK0, value);
}

bool BQ25672::getMask0(Mask0 &m)
{
    uint8_t v;
    if (!getMask0Raw(v))
        return false;

    m.iindpm_masked = (v & (1u << 7)) != 0;
    m.vindpm_masked = (v & (1u << 6)) != 0;
    m.wd_masked = (v & (1u << 5)) != 0;
    m.poor_source_masked = (v & (1u << 4)) != 0;
    m.pg_masked = (v & (1u << 3)) != 0;
    m.vac2_present_masked = (v & (1u << 2)) != 0;
    m.vac1_present_masked = (v & (1u << 1)) != 0;
    m.vbus_present_masked = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::setMask0(const Mask0 &m)
{
    uint8_t v = 0;

    if (m.iindpm_masked)
        v |= (1u << 7);
    if (m.vindpm_masked)
        v |= (1u << 6);
    if (m.wd_masked)
        v |= (1u << 5);
    if (m.poor_source_masked)
        v |= (1u << 4);
    if (m.pg_masked)
        v |= (1u << 3);
    if (m.vac2_present_masked)
        v |= (1u << 2);
    if (m.vac1_present_masked)
        v |= (1u << 1);
    if (m.vbus_present_masked)
        v |= (1u << 0);

    return setMask0Raw(v);
}

// MASK0 setters (1 = masked, 0 = unmasked)
bool BQ25672::setIindpmMask(bool masked)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVindpmMask(bool masked)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setWatchdogMask(bool masked)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setPoorSourceMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setPowerGoodMask(bool masked)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVac2PresentMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVac1PresentMask(bool masked)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVbusPresentMask(bool masked)
{
    const uint8_t mask = (1u << 0);
    return updateBits(REG::MASK0, mask, masked ? mask : 0);
}

// --- MASK1 (REG 0x29) ---
// Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
//
// [7] CHG_MASK           - Charge status change mask
// [6] ICO_MASK           - ICO status change mask
// [5] reserved
// [4] VBUS_MASK          - VBUS status change mask
// [3] reserved
// [2] TREG_MASK          - Thermal regulation entry mask
// [1] VBAT_PRESENT_MASK  - VBAT present status change mask
// [0] BC1_2_DONE_MASK    - BC1.2 detection status change mask

bool BQ25672::getMask1Raw(uint8_t &out)
{
    return read8(REG::MASK1, out);
}

bool BQ25672::getMask1(Mask1 &m)
{
    uint8_t v;
    if (!getMask1Raw(v))
        return false;

    m.chg_masked = (v & (1u << 7)) != 0;
    m.ico_masked = (v & (1u << 6)) != 0;
    m.vbus_masked = (v & (1u << 4)) != 0;
    m.treg_masked = (v & (1u << 2)) != 0;
    m.vbat_present_masked = (v & (1u << 1)) != 0;
    m.bc12_done_masked = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::setMask1(const Mask1 &m)
{
    uint8_t v = 0;

    if (m.chg_masked)
        v |= (1u << 7);
    if (m.ico_masked)
        v |= (1u << 6);
    if (m.vbus_masked)
        v |= (1u << 4);
    if (m.treg_masked)
        v |= (1u << 2);
    if (m.vbat_present_masked)
        v |= (1u << 1);
    if (m.bc12_done_masked)
        v |= (1u << 0);

    return write8(REG::MASK1, v);
}

// Per-bit setters (1 = masked, 0 = unmasked)

bool BQ25672::setChgMask(bool masked)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setIcoMask(bool masked)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setVbusMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setTregMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setVbatPresentMask(bool masked)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setBc12DoneMask(bool masked)
{
    const uint8_t mask = (1u << 0);
    return updateBits(REG::MASK1, mask, masked ? mask : 0);
}

// --- MASK2 (REG 0x2A) ---
// Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
//
// [7] reserved
// [6] DPDM_DONE_MASK     - D+/D- detection done mask
// [5] ADC_DONE_MASK      - ADC conversion done mask (one-shot mode)
// [4] VSYS_MASK          - VSYSMIN regulation enter/exit mask
// [3] CHG_TMR_MASK       - Fast charge timer expire mask
// [2] TRICHG_TMR_MASK    - Trickle charge timer expire mask
// [1] PRECHG_TMR_MASK    - Pre-charge timer expire mask
// [0] TOPOFF_TMR_MASK    - Top-off timer expire mask

bool BQ25672::getMask2Raw(uint8_t &out)
{
    return read8(REG::MASK2, out);
}

bool BQ25672::getMask2(Mask2 &m)
{
    uint8_t v;
    if (!getMask2Raw(v))
        return false;

    m.dpdm_done_masked = (v & (1u << 6)) != 0;
    m.adc_done_masked = (v & (1u << 5)) != 0;
    m.vsysmin_masked = (v & (1u << 4)) != 0;
    m.fast_charge_timer_masked = (v & (1u << 3)) != 0;
    m.trickle_timer_masked = (v & (1u << 2)) != 0;
    m.precharge_timer_masked = (v & (1u << 1)) != 0;
    m.topoff_timer_masked = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::setMask2(const Mask2 &m)
{
    uint8_t v = 0;

    if (m.dpdm_done_masked)
        v |= (1u << 6);
    if (m.adc_done_masked)
        v |= (1u << 5);
    if (m.vsysmin_masked)
        v |= (1u << 4);
    if (m.fast_charge_timer_masked)
        v |= (1u << 3);
    if (m.trickle_timer_masked)
        v |= (1u << 2);
    if (m.precharge_timer_masked)
        v |= (1u << 1);
    if (m.topoff_timer_masked)
        v |= (1u << 0);

    return write8(REG::MASK2, v);
}

bool BQ25672::setDpdmDoneMask(bool masked)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setAdcDoneMask(bool masked)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setVsysMinMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setFastChargeTimerMask(bool masked)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setTrickleTimerMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setPrechargeTimerMask(bool masked)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

bool BQ25672::setTopoffTimerMask(bool masked)
{
    const uint8_t mask = (1u << 0);
    return updateBits(REG::MASK2, mask, masked ? mask : 0);
}

// --- MASK3 (REG 0x2B) ---
// Mask bits: 0 = event produces INT pulse, 1 = event does NOT produce INT pulse (masked)
//
// [7:5] reserved
// [4] VBATOTG_LOW_MASK  - VBAT too low for OTG event mask
// [3] TS_COLD_MASK      - TS crossed cold threshold (T1) mask
// [2] TS_COOL_MASK      - TS crossed cool threshold (T2) mask
// [1] TS_WARM_MASK      - TS crossed warm threshold (T3) mask
// [0] TS_HOT_MASK       - TS crossed hot threshold (T5) mask

bool BQ25672::getMask3Raw(uint8_t &out)
{
    return read8(REG::MASK3, out);
}

bool BQ25672::getMask3(Mask3 &m)
{
    uint8_t v;
    if (!getMask3Raw(v))
        return false;

    m.vbat_otg_low_masked = (v & (1u << 4)) != 0;
    m.ts_cold_masked = (v & (1u << 3)) != 0;
    m.ts_cool_masked = (v & (1u << 2)) != 0;
    m.ts_warm_masked = (v & (1u << 1)) != 0;
    m.ts_hot_masked = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::setMask3(const Mask3 &m)
{
    uint8_t v = 0;

    if (m.vbat_otg_low_masked)
        v |= (1u << 4);
    if (m.ts_cold_masked)
        v |= (1u << 3);
    if (m.ts_cool_masked)
        v |= (1u << 2);
    if (m.ts_warm_masked)
        v |= (1u << 1);
    if (m.ts_hot_masked)
        v |= (1u << 0);

    return write8(REG::MASK3, v);
}

bool BQ25672::setVbatOtgLowMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::MASK3, mask, masked ? mask : 0);
}

bool BQ25672::setTsColdMask(bool masked)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::MASK3, mask, masked ? mask : 0);
}

bool BQ25672::setTsCoolMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::MASK3, mask, masked ? mask : 0);
}

bool BQ25672::setTsWarmMask(bool masked)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::MASK3, mask, masked ? mask : 0);
}

bool BQ25672::setTsHotMask(bool masked)
{
    const uint8_t mask = (1u << 0);
    return updateBits(REG::MASK3, mask, masked ? mask : 0);
}

// --- FAULT_MASK0 (REG 0x2C) ---
// Mask bits: 0 = fault event produces INT pulse, 1 = fault event does NOT produce INT pulse (masked)
//
// [7] IBAT_REG_MASK   - IBAT regulation enter/exit mask
// [6] VBUS_OVP_MASK   - VBUS OVP entry mask
// [5] VBAT_OVP_MASK   - VBAT OVP entry mask
// [4] IBUS_OCP_MASK   - IBUS OCP fault mask
// [3] IBAT_OCP_MASK   - IBAT OCP fault mask
// [2] CONV_OCP_MASK   - Converter OCP fault mask
// [1] VAC2_OVP_MASK   - VAC2 OVP entry mask
// [0] VAC1_OVP_MASK   - VAC1 OVP entry mask

bool BQ25672::getFaultMask0Raw(uint8_t &out)
{
    return read8(REG::FAULT_MASK0, out);
}

bool BQ25672::getFaultMask0(FaultMask0 &m)
{
    uint8_t v;
    if (!getFaultMask0Raw(v))
        return false;

    m.ibat_reg_masked = (v & (1u << 7)) != 0;
    m.vbus_ovp_masked = (v & (1u << 6)) != 0;
    m.vbat_ovp_masked = (v & (1u << 5)) != 0;
    m.ibus_ocp_masked = (v & (1u << 4)) != 0;
    m.ibat_ocp_masked = (v & (1u << 3)) != 0;
    m.conv_ocp_masked = (v & (1u << 2)) != 0;
    m.vac2_ovp_masked = (v & (1u << 1)) != 0;
    m.vac1_ovp_masked = (v & (1u << 0)) != 0;

    return true;
}

bool BQ25672::setFaultMask0(const FaultMask0 &m)
{
    uint8_t v = 0;

    if (m.ibat_reg_masked)
        v |= (1u << 7);
    if (m.vbus_ovp_masked)
        v |= (1u << 6);
    if (m.vbat_ovp_masked)
        v |= (1u << 5);
    if (m.ibus_ocp_masked)
        v |= (1u << 4);
    if (m.ibat_ocp_masked)
        v |= (1u << 3);
    if (m.conv_ocp_masked)
        v |= (1u << 2);
    if (m.vac2_ovp_masked)
        v |= (1u << 1);
    if (m.vac1_ovp_masked)
        v |= (1u << 0);

    return write8(REG::FAULT_MASK0, v);
}

bool BQ25672::setIbatRegFaultMask(bool masked)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVbusOvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVbatOvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setIbusOcpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setIbatOcpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setConvOcpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVac2OvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

bool BQ25672::setVac1OvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 0);
    return updateBits(REG::FAULT_MASK0, mask, masked ? mask : 0);
}

// --- FAULT_MASK1 (REG 0x2D) ---
// Mask bits: 0 = fault event produces INT pulse, 1 = fault event does NOT produce INT pulse (masked)
//
// [7] VSYS_SHORT_MASK  - VSYS short circuit fault mask
// [6] VSYS_OVP_MASK    - VSYS over-voltage fault mask
// [5] OTG_OVP_MASK     - OTG VBUS over-voltage fault mask
// [4] OTG_UVP_MASK     - OTG VBUS under-voltage fault mask
// [3] reserved
// [2] TSHUT_MASK       - Thermal shutdown fault mask
// [1:0] reserved

bool BQ25672::getFaultMask1Raw(uint8_t &out)
{
    return read8(REG::FAULT_MASK1, out);
}

bool BQ25672::getFaultMask1(FaultMask1 &m)
{
    uint8_t v;
    if (!getFaultMask1Raw(v))
        return false;

    m.vsys_short_masked = (v & (1u << 7)) != 0;
    m.vsys_ovp_masked = (v & (1u << 6)) != 0;
    m.otg_ovp_masked = (v & (1u << 5)) != 0;
    m.otg_uvp_masked = (v & (1u << 4)) != 0;
    m.tshut_masked = (v & (1u << 2)) != 0;

    return true;
}

bool BQ25672::setFaultMask1(const FaultMask1 &m)
{
    // Preserve reserved bits (3,1,0) by doing RMW on the known mask.
    const uint8_t knownMask = (1u << 7) | (1u << 6) | (1u << 5) | (1u << 4) | (1u << 2);

    uint8_t value = 0;
    if (m.vsys_short_masked)
        value |= (1u << 7);
    if (m.vsys_ovp_masked)
        value |= (1u << 6);
    if (m.otg_ovp_masked)
        value |= (1u << 5);
    if (m.otg_uvp_masked)
        value |= (1u << 4);
    if (m.tshut_masked)
        value |= (1u << 2);

    return updateBits(REG::FAULT_MASK1, knownMask, value);
}

bool BQ25672::setVsysShortFaultMask(bool masked)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::FAULT_MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setVsysOvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::FAULT_MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setOtgOvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::FAULT_MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setOtgUvpFaultMask(bool masked)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::FAULT_MASK1, mask, masked ? mask : 0);
}

bool BQ25672::setTshutFaultMask(bool masked)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::FAULT_MASK1, mask, masked ? mask : 0);
}

// --- ADC configuration ---

// --- ADC_CTRL (REG 0x2E) ---
// [7]   ADC_EN          - ADC enable
// [6]   ADC_RATE        - 0=continuous, 1=one-shot
// [5:4] ADC_SAMPLE_1:0  - sample speed / effective resolution
// [3]   ADC_AVG         - 0=single, 1=running average
// [2]   ADC_AVG_INIT    - 0=use existing value, 1=start with new conversion
// [1:0] reserved

bool BQ25672::getAdcCtrlRaw(uint8_t &out)
{
    return read8(REG::ADC_CTRL, out);
}

bool BQ25672::getAdcCtrl(AdcCtrl &a)
{
    uint8_t v;
    if (!getAdcCtrlRaw(v))
        return false;

    a.adc_enabled = (v & (1u << 7)) != 0;
    a.rate = static_cast<AdcRate>((v >> 6) & 0x01);
    a.sample = static_cast<AdcSample>((v >> 4) & 0x03);
    a.running_average = (v & (1u << 3)) != 0;
    a.avg_init_new_conversion = (v & (1u << 2)) != 0;

    return true;
}

bool BQ25672::setAdcCtrl(const AdcCtrl &a)
{
    const uint8_t knownMask =
        (1u << 7) | (1u << 6) | (3u << 4) | (1u << 3) | (1u << 2);

    uint8_t value = 0;
    if (a.adc_enabled)
        value |= (1u << 7);
    if (static_cast<uint8_t>(a.rate) & 0x01)
        value |= (1u << 6);
    value |= (static_cast<uint8_t>(a.sample) & 0x03) << 4;
    if (a.running_average)
        value |= (1u << 3);
    if (a.avg_init_new_conversion)
        value |= (1u << 2);

    return updateBits(REG::ADC_CTRL, knownMask, value);
}

bool BQ25672::setAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::ADC_CTRL, mask, enable ? mask : 0);
}

bool BQ25672::setAdcRate(AdcRate rate)
{
    const uint8_t mask = (1u << 6);
    const uint8_t val = (static_cast<uint8_t>(rate) & 0x01) ? mask : 0;
    return updateBits(REG::ADC_CTRL, mask, val);
}

bool BQ25672::setAdcSample(AdcSample sample)
{
    const uint8_t mask = (3u << 4);
    const uint8_t val = (static_cast<uint8_t>(sample) & 0x03) << 4;
    return updateBits(REG::ADC_CTRL, mask, val);
}

bool BQ25672::setAdcRunningAverage(bool enable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::ADC_CTRL, mask, enable ? mask : 0);
}

bool BQ25672::setAdcAvgInitNewConversion(bool enable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::ADC_CTRL, mask, enable ? mask : 0);
}

// --- ADC_DIS0 (REG 0x2F) ---
// Note: 0 = enabled, 1 = disabled
//
// [7] IBUS_ADC_DIS  - IBUS ADC disable
// [6] IBAT_ADC_DIS  - IBAT ADC disable
// [5] VBUS_ADC_DIS  - VBUS ADC disable
// [4] VBAT_ADC_DIS  - VBAT ADC disable
// [3] VSYS_ADC_DIS  - VSYS ADC disable
// [2] TS_ADC_DIS    - TS ADC disable
// [1] TDIE_ADC_DIS  - TDIE ADC disable
// [0] reserved

bool BQ25672::getAdcDis0Raw(uint8_t &out)
{
    return read8(REG::ADC_DIS0, out);
}

bool BQ25672::getAdcDis0(AdcDis0 &a)
{
    uint8_t v;
    if (!getAdcDis0Raw(v))
        return false;

    a.ibus_disabled = (v & (1u << 7)) != 0;
    a.ibat_disabled = (v & (1u << 6)) != 0;
    a.vbus_disabled = (v & (1u << 5)) != 0;
    a.vbat_disabled = (v & (1u << 4)) != 0;
    a.vsys_disabled = (v & (1u << 3)) != 0;
    a.ts_disabled = (v & (1u << 2)) != 0;
    a.tdie_disabled = (v & (1u << 1)) != 0;

    return true;
}

bool BQ25672::setAdcDis0(const AdcDis0 &a)
{
    const uint8_t knownMask =
        (1u << 7) | (1u << 6) | (1u << 5) | (1u << 4) |
        (1u << 3) | (1u << 2) | (1u << 1);

    uint8_t value = 0;

    if (a.ibus_disabled)
        value |= (1u << 7);
    if (a.ibat_disabled)
        value |= (1u << 6);
    if (a.vbus_disabled)
        value |= (1u << 5);
    if (a.vbat_disabled)
        value |= (1u << 4);
    if (a.vsys_disabled)
        value |= (1u << 3);
    if (a.ts_disabled)
        value |= (1u << 2);
    if (a.tdie_disabled)
        value |= (1u << 1);

    return updateBits(REG::ADC_DIS0, knownMask, value);
}

bool BQ25672::setIbusAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setIbatAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setVbusAdcEnabled(bool enable)
{ // 0=disabled, 1=enabled
    const uint8_t mask = (1u << 5);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setVbatAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setVsysAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 3);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setTsAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 2);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

bool BQ25672::setTdieAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 1);
    return updateBits(REG::ADC_DIS0, mask, enable ? 0 : mask);
}

// --- ADC_DIS1 (REG 0x30) ---
// Note: 0 = enabled, 1 = disabled
//
// [7] DP_ADC_DIS    - D+ ADC disable
// [6] DM_ADC_DIS    - D- ADC disable
// [5] VAC2_ADC_DIS  - VAC2 ADC disable
// [4] VAC1_ADC_DIS  - VAC1 ADC disable
// [3:0] reserved

bool BQ25672::getAdcDis1Raw(uint8_t &out)
{
    return read8(REG::ADC_DIS1, out);
}

bool BQ25672::getAdcDis1(AdcDis1 &a)
{
    uint8_t v;
    if (!getAdcDis1Raw(v))
        return false;

    a.dp_disabled = (v & (1u << 7)) != 0;
    a.dm_disabled = (v & (1u << 6)) != 0;
    a.vac2_disabled = (v & (1u << 5)) != 0;
    a.vac1_disabled = (v & (1u << 4)) != 0;

    return true;
}

bool BQ25672::setAdcDis1(const AdcDis1 &a)
{
    const uint8_t knownMask = (1u << 7) | (1u << 6) | (1u << 5) | (1u << 4);

    uint8_t value = 0;
    if (a.dp_disabled)
        value |= (1u << 7);
    if (a.dm_disabled)
        value |= (1u << 6);
    if (a.vac2_disabled)
        value |= (1u << 5);
    if (a.vac1_disabled)
        value |= (1u << 4);

    return updateBits(REG::ADC_DIS1, knownMask, value);
}

bool BQ25672::setDpAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 7);
    return updateBits(REG::ADC_DIS1, mask, enable ? 0 : mask);
}

bool BQ25672::setDmAdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 6);
    return updateBits(REG::ADC_DIS1, mask, enable ? 0 : mask);
}

bool BQ25672::setVac2AdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 5);
    return updateBits(REG::ADC_DIS1, mask, enable ? 0 : mask);
}

bool BQ25672::setVac1AdcEnabled(bool enable)
{
    const uint8_t mask = (1u << 4);
    return updateBits(REG::ADC_DIS1, mask, enable ? 0 : mask);
}

// --- DPDM ---

// --- DPDM_DRIVER (REG 0x47) ---
// [7:5] DPLUS_DAC_2:0   - D+ output driver level
// [4:2] DMINUS_DAC_2:0  - D- output driver level
// [1:0] reserved

bool BQ25672::getDpdmDriverRaw(uint8_t &out)
{
    return read8(REG::DPDM_DRIVER, out);
}

bool BQ25672::getDpdmDriver(DpdmDriver &d)
{
    uint8_t v;
    if (!getDpdmDriverRaw(v))
        return false;

    d.dp = static_cast<DpDac>((v >> 5) & 0x07);
    d.dm = static_cast<DmDac>((v >> 2) & 0x07);

    return true;
}

bool BQ25672::setDpdmDriver(const DpdmDriver &d)
{
    const uint8_t knownMask = (7u << 5) | (7u << 2);

    uint8_t value = 0;
    value |= (static_cast<uint8_t>(d.dp) & 0x07) << 5;
    value |= (static_cast<uint8_t>(d.dm) & 0x07) << 2;

    return updateBits(REG::DPDM_DRIVER, knownMask, value);
}

bool BQ25672::setDplusDac(DpDac dp)
{
    const uint8_t mask = (7u << 5);
    const uint8_t val = (static_cast<uint8_t>(dp) & 0x07) << 5;
    return updateBits(REG::DPDM_DRIVER, mask, val);
}

bool BQ25672::setDminusDac(DmDac dm)
{
    const uint8_t mask = (7u << 2);
    const uint8_t val = (static_cast<uint8_t>(dm) & 0x07) << 2;
    return updateBits(REG::DPDM_DRIVER, mask, val);
}

// --- Part information ---

bool BQ25672::getPartInfo(uint8_t &out)
{
    return read8(REG::PART_INFO, out);
}

// --- Measurements ---

// IBUS ADC (1 LSB = 1 mA)
bool BQ25672::readIbus_mA(uint16_t &ma)
{
    return read16(REG::IBUS_ADC, ma);
}

// IBAT ADC (1 LSB = 1 mA)
bool BQ25672::readIbat_mA(uint16_t &ma)
{
    return read16(REG::IBAT_ADC, ma);
}

// VBUS ADC (1 LSB = 1 mV)
bool BQ25672::readVbus_mV(uint16_t &mv)
{
    return read16(REG::VBUS_ADC, mv);
}

// VAC1 ADC (1 LSB = 1 mV)
bool BQ25672::readVac1_mV(uint16_t &mv)
{
    return read16(REG::VAC1_ADC, mv);
}

// VAC2 ADC (1 LSB = 1 mV)
bool BQ25672::readVac2_mV(uint16_t &mv)
{
    return read16(REG::VAC2_ADC, mv);
}

// VBAT ADC (1 LSB = 1 mV)
bool BQ25672::readVbat_mV(uint16_t &mv)
{
    return read16(REG::VBAT_ADC, mv);
}

// VSYS ADC (1 LSB = 1 mV)
bool BQ25672::readVsys_mV(uint16_t &mv)
{
    return read16(REG::VSYS_ADC, mv);
}

// TS ADC (raw value, scaling TBD)
uint16_t BQ25672::readTsRaw()
{
    uint16_t v;
    if (!read16(REG::TS_ADC, v))
        return 0xFFFF;
    return v;
}

// TDIE ADC (raw value, scaling TBD)
uint16_t BQ25672::readTdieRaw()
{
    uint16_t v;
    if (!read16(REG::TDIE_ADC, v))
        return 0xFFFF;
    return v;
}

// D+ ADC (raw value)
uint16_t BQ25672::readDpRaw()
{
    uint16_t v;
    if (!read16(REG::DP_ADC, v))
        return 0xFFFF;
    return v;
}

// D- ADC (raw value)
uint16_t BQ25672::readDmRaw()
{
    uint16_t v;
    if (!read16(REG::DM_ADC, v))
        return 0xFFFF;
    return v;
}

bool BQ25672::readBytes(REG reg, uint8_t *buf, uint8_t len)
{
    m_wire->beginTransmission(m_addr);
    m_wire->write(static_cast<uint8_t>(reg));
    if (m_wire->endTransmission(false) != 0)
        return false;

    uint8_t got = m_wire->requestFrom(m_addr, len);
    if (got != len)
        return false;

    for (uint8_t i = 0; i < len; i++)
    {
        if (!m_wire->available())
            return false;
        buf[i] = static_cast<uint8_t>(m_wire->read());
    }
    return true;
}

bool BQ25672::read8(REG reg, uint8_t &out)
{
    return readBytes(reg, &out, 1);
}

bool BQ25672::write8(REG reg, uint8_t val)
{
    m_wire->beginTransmission(m_addr);
    m_wire->write(static_cast<uint8_t>(reg));
    m_wire->write(val);
    return (m_wire->endTransmission(true) == 0);
}

bool BQ25672::updateBits(REG reg, uint8_t mask, uint8_t value)
{
    uint8_t cur;
    if (!read8(reg, cur))
        return false;

    uint8_t next = (cur & ~mask) | (value & mask);
    if (next == cur)
        return true;

    return write8(reg, next);
}

bool BQ25672::read16(REG reg_msb, uint16_t &out)
{
    uint8_t b[2];
    if (!readBytes(reg_msb, b, 2))
        return false;
    out = (uint16_t(b[0]) << 8) | b[1];
    return true;
}

bool BQ25672::write16(REG reg_msb, uint16_t val)
{
    const uint8_t msb = static_cast<uint8_t>((val >> 8) & 0xFF);
    const uint8_t lsb = static_cast<uint8_t>(val & 0xFF);

    m_wire->beginTransmission(m_addr);
    m_wire->write(static_cast<uint8_t>(reg_msb));
    m_wire->write(msb);
    m_wire->write(lsb);
    return (m_wire->endTransmission(true) == 0);
}

bool BQ25672::isBQ25672()
{
    uint8_t v;
    if (!getPartInfo(v))
        return false;

    const uint8_t pn = (v >> 3) & 0x07; // PN_2:0 (bits 5..3)
    const uint8_t rev = v & 0x07;       // DEV_REV_2:0 (bits 2..0)

    return (pn == 0b100) && (rev == 0b001);
}

#if defined(BQ25672_ENABLE_STRINGS) && BQ25672_ENABLE_STRINGS
    const char *BQ25672::errorToString(Error e)
    {
        switch (e)
        {
        case Error::OK:
            return "OK";
        case Error::I2C_READ_FAIL:
            return "I2C read failed";
        case Error::I2C_WRITE_FAIL:
            return "I2C write failed";
        case Error::PART_INFO_READ_FAIL:
            return "PART_INFO read failed";
        case Error::WRONG_PART_NUMBER:
            return "Wrong part number";
        case Error::WRONG_DEVICE_REV:
            return "Wrong device revision";
        case Error::ADC_ENABLE_FAIL:
            return "Enable ADC failed";
        case Error::VBUS_ADC_ENABLE_FAIL:
            return "Enable VBUS ADC failed";
        default:
            return "Unknown error";
        }
    }

    const char *BQ25672::prechargeThresholdToString(PrechargeVbatThreshold t)
    {
        switch (t)
        {
        case PrechargeVbatThreshold::VREG_15_0pct:
            return "15.0% * VREG";
        case PrechargeVbatThreshold::VREG_62_2pct:
            return "62.2% * VREG";
        case PrechargeVbatThreshold::VREG_66_7pct:
            return "66.7% * VREG";
        case PrechargeVbatThreshold::VREG_71_4pct:
            return "71.4% * VREG";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::cellToString(CellCount c)
    {
        switch (c)
        {
        case CellCount::S1:
            return "1S";
        case CellCount::S2:
            return "2S";
        case CellCount::S3:
            return "3S";
        case CellCount::S4:
            return "4S";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::rechgDeglitchToString(RechargeDeglitch t)
    {
        switch (t)
        {
        case BQ25672::RechargeDeglitch::MS_64:
            return "64 ms";
        case BQ25672::RechargeDeglitch::MS_256:
            return "256 ms";
        case BQ25672::RechargeDeglitch::MS_1024:
            return "1024 ms";
        case BQ25672::RechargeDeglitch::MS_2048:
            return "2048 ms";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::topOffToString(BQ25672::TopOffTimer t)
    {
        switch (t)
        {
        case BQ25672::TopOffTimer::Disabled:
            return "Disabled";
        case BQ25672::TopOffTimer::Min15:
            return "15 min";
        case BQ25672::TopOffTimer::Min30:
            return "30 min";
        case BQ25672::TopOffTimer::Min45:
            return "45 min";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::fastChgToString(BQ25672::FastChargeTimer t)
    {
        switch (t)
        {
        case BQ25672::FastChargeTimer::Hrs5:
            return "5 hrs";
        case BQ25672::FastChargeTimer::Hrs8:
            return "8 hrs";
        case BQ25672::FastChargeTimer::Hrs12:
            return "12 hrs";
        case BQ25672::FastChargeTimer::Hrs24:
            return "24 hrs";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::vacOvpToString(BQ25672::VacOvpThreshold t)
    {
        switch (t)
        {
        case BQ25672::VacOvpThreshold::V26:
            return "26V";
        case BQ25672::VacOvpThreshold::V22:
            return "22V";
        case BQ25672::VacOvpThreshold::V12:
            return "12V";
        case BQ25672::VacOvpThreshold::V7:
            return "7V";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::watchdogToString(BQ25672::WatchdogTimer t)
    {
        switch (t)
        {
        case BQ25672::WatchdogTimer::Disabled:
            return "Disabled";
        case BQ25672::WatchdogTimer::S0_5:
            return "0.5s";
        case BQ25672::WatchdogTimer::S1:
            return "1s";
        case BQ25672::WatchdogTimer::S2:
            return "2s";
        case BQ25672::WatchdogTimer::S20:
            return "20s";
        case BQ25672::WatchdogTimer::S40:
            return "40s";
        case BQ25672::WatchdogTimer::S80:
            return "80s";
        case BQ25672::WatchdogTimer::S160:
            return "160s";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::ibatRegToString(BQ25672::IbatRegOtg r)
    {
        switch (r)
        {
        case BQ25672::IbatRegOtg::A3:
            return "3A";
        case BQ25672::IbatRegOtg::A4:
            return "4A";
        case BQ25672::IbatRegOtg::A5:
            return "5A";
        case BQ25672::IbatRegOtg::Disabled:
            return "Disabled";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::vbusStatusToString(BQ25672::VbusStatus s)
    {
        switch (s)
        {
        case BQ25672::VbusStatus::NoInputOrBhotOrBcold:
            return "No input / BHOT / BCOLD (OTG)";
        case BQ25672::VbusStatus::UsbSdp:
            return "USB SDP (500mA)";
        case BQ25672::VbusStatus::UsbCdp:
            return "USB CDP (1.5A)";
        case BQ25672::VbusStatus::UsbDcp:
            return "USB DCP (3.25A)";
        case BQ25672::VbusStatus::Hvdcp_1_5A:
            return "HVDCP (1.5A)";
        case BQ25672::VbusStatus::UnknownAdapter_3A:
            return "Unknown adapter (3A)";
        case BQ25672::VbusStatus::NonStandardAdapter:
            return "Non-standard adapter";
        case BQ25672::VbusStatus::OtgMode:
            return "OTG mode";
        case BQ25672::VbusStatus::NotQualifiedAdapter:
            return "Not qualified adapter";
        case BQ25672::VbusStatus::VbusPowered:
            return "Powered from VBUS";
        case BQ25672::VbusStatus::BackupMode:
            return "Backup mode";
        default:
            return "Reserved/Unknown";
        }
    }

    const char *BQ25672::chargeStatusToString(BQ25672::ChargeStatus s)
    {
        switch (s)
        {
        case BQ25672::ChargeStatus::NotCharging:
            return "Not charging";
        case BQ25672::ChargeStatus::TrickleCharge:
            return "Trickle charge";
        case BQ25672::ChargeStatus::PreCharge:
            return "Pre-charge";
        case BQ25672::ChargeStatus::FastCharge_CC:
            return "Fast charge (CC)";
        case BQ25672::ChargeStatus::TaperCharge_CV:
            return "Taper charge (CV)";
        case BQ25672::ChargeStatus::TopOffActive:
            return "Top-off active";
        case BQ25672::ChargeStatus::TerminationDone:
            return "Charge terminated";
        case BQ25672::ChargeStatus::Reserved5:
            return "Reserved";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::icoStatusToString(BQ25672::IcoStatus s)
    {
        switch (s)
        {
        case BQ25672::IcoStatus::Disabled:
            return "Disabled";
        case BQ25672::IcoStatus::OptimizationRunning:
            return "Optimization running";
        case BQ25672::IcoStatus::MaxInputDetected:
            return "Max input detected";
        case BQ25672::IcoStatus::Reserved:
            return "Reserved";
        default:
            return "Unknown";
        }
    }

    const char *BQ25672::tsRangeToString(TsRange r)
    {
        switch (r)
        {
        case TsRange::Normal:
            return "NORMAL";
        case TsRange::Cold:
            return "COLD";
        case TsRange::Cool:
            return "COOL";
        case TsRange::Warm:
            return "WARM";
        case TsRange::Hot:
            return "HOT";
        case TsRange::Invalid:
            return "INVALID";
        default:
            return "UNKNOWN";
        }
    }
#endif

#if defined(BQ25672_ENABLE_FLOAT_API) && BQ25672_ENABLE_FLOAT_API
    bool BQ25672::getMinSystemVoltage_V(float &v)
    {
        uint16_t mv;
        if (!getMinSystemVoltage_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::setMinSystemVoltage_V(float v)
    {
        // simple sanity check, bez math.h
        if (!(v == v))
            return false; // NaN check
        if (v < 0.0f)
            return false;

        uint16_t mv = static_cast<uint16_t>(v * 1000.0f + 0.5f);
        return setMinSystemVoltage_mV(mv);
    }

    bool BQ25672::getChargeVoltageLimit_V(float &v)
    {
        uint16_t mv;
        if (!getChargeVoltageLimit_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::setChargeVoltageLimit_V(float v)
    {
        if (!(v == v))
            return false; // NaN
        if (v < 0.0f)
            return false;

        const uint16_t mv = static_cast<uint16_t>(v * 1000.0f + 0.5f);
        return setChargeVoltageLimit_mV(mv);
    }

    bool BQ25672::getChargeCurrentLimit_A(float &a)
    {
        uint16_t ma;
        if (!getChargeCurrentLimit_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::setChargeCurrentLimit_A(float a)
    {
        if (!(a == a))
            return false; // NaN
        if (a < 0.0f)
            return false;

        const uint16_t ma = static_cast<uint16_t>(a * 1000.0f + 0.5f);
        return setChargeCurrentLimit_mA(ma);
    }

    bool BQ25672::getInputVoltageLimit_V(float &v)
    {
        uint16_t mv;
        if (!getInputVoltageLimit_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::setInputVoltageLimit_V(float v)
    {
        if (!(v == v))
            return false; // NaN
        if (v < 0.0f)
            return false;

        const uint16_t mv = static_cast<uint16_t>(v * 1000.0f + 0.5f);
        return setInputVoltageLimit_mV(mv);
    }

    bool BQ25672::getInputCurrentLimit_A(float &a)
    {
        uint16_t ma;
        if (!getInputCurrentLimit_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::setInputCurrentLimit_A(float a)
    {
        if (!(a == a))
            return false; // NaN
        if (a < 0.0f)
            return false;

        const uint16_t ma = static_cast<uint16_t>(a * 1000.0f + 0.5f);
        return setInputCurrentLimit_mA(ma);
    }

    bool BQ25672::getPrechargeCurrentLimit_A(float &a)
    {
        uint16_t ma;
        if (!getPrechargeCurrentLimit_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::getTerminationCurrent_A(float &a)
    {
        uint16_t ma;
        if (!getTerminationCurrent_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::getOtgVoltage_V(float &v)
    {
        uint16_t mv;
        if (!getOtgVoltage_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::setOtgVoltage_V(float v)
    {
        if (!(v == v))
            return false; // NaN
        if (v < 0.0f)
            return false;

        const uint16_t mv = static_cast<uint16_t>(v * 1000.0f + 0.5f);
        return setOtgVoltage_mV(mv);
    }

    bool BQ25672::getOtgCurrentLimit_A(float &a)
    {
        uint16_t ma;
        if (!getOtgCurrentLimit_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::getVocPctFactor(float &factor)
    {
        uint16_t permille;
        if (!getVocPct_permille(permille))
            return false;
        factor = static_cast<float>(permille) * 0.001f;
        return true;
    }

    bool BQ25672::getIcoInputCurrentLimit_A(float &a)
    {
        uint16_t ma;
        if (!getIcoInputCurrentLimit_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::readIbus_A(float &a)
    {
        uint16_t ma;
        if (!readIbus_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::readIbat_A(float &a)
    {
        uint16_t ma;
        if (!readIbat_mA(ma))
            return false;
        a = static_cast<float>(ma) * 0.001f;
        return true;
    }

    bool BQ25672::readVbus_V(float &v)
    {
        uint16_t mv;
        if (!readVbus_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::readVac1_V(float &v)
    {
        uint16_t mv;
        if (!readVac1_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::readVac2_V(float &v)
    {
        uint16_t mv;
        if (!readVac2_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::readVbat_V(float &v)
    {
        uint16_t mv;
        if (!readVbat_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

    bool BQ25672::readVsys_V(float &v)
    {
        uint16_t mv;
        if (!readVsys_mV(mv))
            return false;
        v = static_cast<float>(mv) * 0.001f;
        return true;
    }

#endif