/******************************************************************************
basic_read.ino
BQ25672 Arduino Library – Basic Read Example

Author: Piotr Malek
Repository: https://github.com/piotrmalek/BQ25672

This example demonstrates basic communication with the TI BQ25672 battery
charger and power-path management IC.

It shows how to:
- Initialize the BQ25672 device
- Check for initialization errors
- Read basic voltage and current measurements using the integer (mV / mA) API

Compile-time configuration:
Library features can be enabled or disabled using preprocessor macros.
These macros may be defined:
- directly in this sketch before including BQ25672.h, or
- in an optional "BQ25672_config.h" file (recommended for larger projects).

Available options:
- BQ25672_ENABLE_FLOAT_API   (0/1) – enable floating-point helpers
- BQ25672_ENABLE_STRINGS     (0/1) – enable enum-to-string helpers

The example uses the default I2C interface (Wire).

License:
This example is distributed under the MIT License.
See the LICENSE file for details.

Distributed as-is; no warranty is given.
******************************************************************************/


#include <Wire.h>
#include "BQ25672.h"

BQ25672 chg(Wire);

void setup() {
  Serial.begin(115200);
  Serial.println("Starting basic read BQ25672 app...");

  Wire.begin();

  auto err = chg.begin();
  if (err != BQ25672::Error::OK) {
      Serial.print("BQ25672 init failed: ");
      Serial.println(BQ25672::errorToString(err));  // Requires BQ25672_ENABLE_STRINGS = 1
      while (1) delay(1000);
  }

  Serial.println("BQ25672 OK");
}

void loop() {
    uint16_t v;

    if (chg.readVbat_mV(v)) {                       // Read VBAT
        Serial.print("VBAT: ");
        Serial.print(v);
        Serial.println(" mV");
    } else {
        Serial.println("VBAT: read error");
    }

    if (chg.readVsys_mV(v)) {                       // Read VSYS
        Serial.print("VSYS: ");
        Serial.print(v);
        Serial.println(" mV");
    } else {
        Serial.println("VSYS: read error");
    }

    if (chg.readVac1_mV(v)) {                       // Read VAC1
        Serial.print("VAC1: ");
        Serial.print(v);
        Serial.println(" mV");
    } else {
        Serial.println("VAC1: read error");
    }

    if (chg.readVac2_mV(v)) {                       // Read VAC2
        Serial.print("VAC2: ");
        Serial.print(v);
        Serial.println(" mV");
    } else {
        Serial.println("VAC2: read error");
    }

    if (chg.readIbus_mA(v)) {                       // Read IBUS
        Serial.print("IBUS: ");
        Serial.print(v);
        Serial.println(" mA");
    } else {
        Serial.println("IBUS: read error");
    }

    if (chg.readIbat_mA(v)) {                       // Read IBAT
        Serial.print("IBAT: ");
        Serial.print(v);
        Serial.println(" mA");
    } else {
        Serial.println("IBAT: read error");
    }

    delay(1000);
}