# BQ25672 Arduino Library

Arduino / PlatformIO library for the **Texas Instruments BQ25672** battery charger
and power-path management IC.

The library provides **register-level access**, structured helpers, and optional
high-level APIs for measurements and status decoding.

---

## Features

- Full register coverage of the BQ25672
- Typed enums for all configuration fields
- Structured status, fault, flag, and mask registers
- Integer measurement API (mV / mA)
- Optional floating-point helpers (V / A)
- Optional enum-to-string helpers for debugging
- Compatible with Arduino IDE and PlatformIO

---

## Supported devices

- Texas Instruments **BQ25672**

---

## Installation

### Arduino IDE

1. Clone or download this repository
2. Copy the `BQ25672` folder into your Arduino `libraries` directory
3. Restart the Arduino IDE

### PlatformIO

Add the library as a Git dependency:

```ini
lib_deps =
    https://github.com/piotrmalek/BQ25672
```

---

## Basic usage

```cpp
#include <Wire.h>
#include <BQ25672.h>

BQ25672 chg(Wire);

void setup() {
    Serial.begin(115200);
    Wire.begin();

    auto err = chg.begin();
    if (err != BQ25672::Error::OK) {
        Serial.println("BQ25672 init failed");
        while (1);
    }

    Serial.println("BQ25672 ready");
}

void loop() {
    uint16_t vbus;
    if (chg.readVbus_mV(vbus)) {
        Serial.print("VBUS: ");
        Serial.print(vbus);
        Serial.println(" mV");
    }

    delay(1000);
}
```

See `examples` for a complete working examples.

---

## Compile-time configuration

Some features can be enabled or disabled at **compile time** using preprocessor
macros.

These macros can be defined in an optional `BQ25672_config.h` file (recommended for larger projects)

### Available options

| Macro                         | Default | Description |
|------------------------------|---------|-------------|
| `BQ25672_ENABLE_FLOAT_API`   | `0`     | Enable floating-point helpers (V / A) |
| `BQ25672_ENABLE_STRINGS`     | `0`     | Enable enum-to-string helper functions |

### Example (`BQ25672_config.h`)

```cpp
#pragma once

#define BQ25672_ENABLE_FLOAT_API   0
#define BQ25672_ENABLE_STRINGS     1
```

---

## APIs overview

The library exposes a register-level API grouped by functionality.
All functions return `bool` (success/failure) unless stated otherwise.

### Measurement API (integer)
Read ADC measurements using integer units (mV / mA):
- `readVbus_mV(uint16_t&)`
- `readVbat_mV(uint16_t&)`
- `readVsys_mV(uint16_t&)`
- `readVac1_mV(uint16_t&)`
- `readVac2_mV(uint16_t&)`
- `readIbus_mA(uint16_t&)`
- `readIbat_mA(uint16_t&)`

All functions return `true` on success.

### Configuration and control API

Register-level accessors are provided for:

- Charge voltage and current limits
- Input voltage and current limits (VINDPM / IINDPM)
- OTG voltage and current control
- Safety timers (precharge, fast charge, top-off)
- Charger enable / disable and HIZ mode
- MPPT (VOC-based) configuration
- JEITA temperature control
- ADC configuration and channel enable/disable

Each register has:
- a raw accessor (`getXxxRaw`)
- decoded helpers (`getXxx`, `setXxx`)
- optional unit-converted helpers (`_mV`, `_mA`)

### Status and fault registers

Structured access is provided for:

- Charger status (`Stat0`–`Stat4`)
- Input source and charge state
- ICO / MPPT state
- Thermal regulation
- Faults and fault flags
- Interrupt flags and interrupt masks

Instead of bit masking, decoded structures are returned, e.g.:

- `Stat0`, `Stat1`, `Stat2`, `Stat3`, `Stat4`
- `Fault0`, `Fault1`
- `Flag0`–`Flag3`
- `Mask0`–`Mask3`

### Optional APIs

#### Float API

If enabled in `BQ25672_config.h`:

- Unit-converted helpers returning `float` (V / A)
- Convenience wrappers around integer API

```c
#define BQ25672_ENABLE_FLOAT_API 1

```

#### String API

If enabled in `BQ25672_config.h`:

- Enum-to-string helpers for debugging and logging
- Functions like errorToString(), chargeStatusToString()

```c
#define BQ25672_ENABLE_STRINGS 1
```

---

## Design notes

- The library favors **explicit register access** over hidden logic
- No dynamic memory allocation
- Minimal overhead unless optional features are enabled
- Suitable for embedded and low-resource systems

---

## License

This library is licensed under the **MIT License**.

See the `LICENSE` file for details.

---

## Author

**Piotr Małek**  
GitHub: https://github.com/piotrmalek
