#ifndef HS300x_H
#define HS300x_H

#include <Arduino.h>
#include <Wire.h>

/**
 * @mainpage HS300x Arduino Library
 *
 * @section intro Introduction
 * This library provides a high-level API for reading temperature and humidity
 * from the Renesas HS300x series of sensors over I2C.
 *
 * It supports both blocking and non-blocking measurement modes and allows
 * configuration of temperature units.
 *
 * @section usage Example
 * @code
 * #include <Wire.h>
 * #include <HS300x.h>
 *
 * HS300x sensor;
 *
 * void setup() {
 *   Serial.begin(9600);
 *   Wire.begin();
 *   if (!sensor.begin()) {
 *     Serial.println("Sensor not found!");
 *     while (1);
 *   }
 * }
 *
 * void loop() {
 *   auto result = sensor.readMeasurement();
 *   Serial.print("Temperature: ");
 *   Serial.print(result.temperature);
 *   Serial.print(" °C, Humidity: ");
 *   Serial.print(result.humidity);
 *   Serial.println(" %");
 *   delay(1000);
 * }
 * @endcode
 */
class HS300x {
public:
  /** @brief Temperature output unit. */
  enum class TempUnit {
    CELSIUS,    ///< Output in degrees Celsius
    FAHRENHEIT, ///< Output in degrees Fahrenheit
    KELVIN      ///< Output in Kelvin
  };

  /**
   * @brief Sensor resolution modes.
   *
   * Lower resolution allows faster measurements.
   * The default is `RES_14_BIT`.
   * Note that changing resolution does not change the internal resolution of
   * the sensor; it only affects the measurement time. To change the actual
   * resolution, the sensor must be configured in programming mode.
   */
  enum class Resolution {
    RES_14_BIT, // takes 35 ms
    RES_12_BIT, // takes 10 ms
    RES_10_BIT, // takes 2.9 ms
    RES_8_BIT   // takes 1.4 ms
  };

  /**
   * @brief Raw sensor data (ADC values).
   */
  struct RawMeasurementResult {
    uint16_t rawTemperature; ///< 14-bit temperature ADC count
    uint16_t rawHumidity;    ///< 14-bit humidity ADC count
  };

  /**
   * @brief Processed sensor data (engineering units).
   */
  struct MeasurementResult {
    float temperature; ///< Temperature in selected unit
    float humidity;    ///< Relative humidity in %RH
  };

  // --- Default configuration constants ---
  static constexpr TempUnit DEFAULT_UNIT = TempUnit::CELSIUS;
  static constexpr Resolution DEFAULT_RESOLUTION = Resolution::RES_14_BIT;
  static constexpr uint8_t DEFAULT_ADDRESS = 0x44;

  // --- Timing and protocol constants ---
  static constexpr uint8_t POLLING_INTERVAL_COARSE_MS = 1;
  static constexpr uint8_t POLLING_INTERVAL_FINE_US = 100;
  static constexpr uint8_t FRAME_SIZE = 4;
  static constexpr uint8_t STATUS_BITS_SHIFT = 6;
  static constexpr uint8_t STATUS_BITS_MASK = 0x3F;
  static constexpr uint8_t TEMP_BITS_SHIFT = 2;
  static constexpr unsigned long SENSOR_TIMEOUT_MS = 100;

  // --- Data conversion constants ---
  static constexpr uint16_t INVALID_RAW_VALUE = 0xFFFF;
  // --- Conversion factors from datasheet ---
  static constexpr float HUMIDITY_FACTOR = 100.0f / 16383.0f;
  static constexpr float TEMPERATURE_FACTOR = 165.0f / 16383.0f;
  static constexpr float TEMPERATURE_OFFSET = -40.0f;
  // --- Temperature conversion constants ---
  static constexpr float KELVIN_OFFSET = 273.15f;
  static constexpr float FAHRENHEIT_SCALE = 9.0f / 5.0f;
  static constexpr float FAHRENHEIT_OFFSET = 32.0f;

  /**
   * @brief Construct a new HS300x sensor object.
   * @param address I2C address of the sensor (default 0x44)
   * @param unit Output temperature unit (default CELSIUS)
   * @param resolution Desired measurement resolution (default RES_14_BIT)
   */
  HS300x(uint8_t address = DEFAULT_ADDRESS, TempUnit unit = DEFAULT_UNIT,
         Resolution resolution = DEFAULT_RESOLUTION);

  /**
   * @brief Initialize the sensor interface.
   *
   * Must be called after `Wire.begin()`. Returns `true` if the sensor
   * acknowledges on the I2C bus.
   *
   * @param wirePort Reference to the I2C interface (default: `Wire`)
   * @return `true` if sensor is detected
   */
  bool begin(TwoWire &wirePort = Wire);

  /**
   * @brief Get the I2C address of the sensor.
   * @return I2C address (7-bit)
   */
  uint8_t getAddress() const;
  
  // --------------------------------------------------------
  // Blocking measurement API
  // --------------------------------------------------------

  /**
   * @brief Read raw temperature value (blocking).
   * @return 14-bit raw temperature, or `INVALID_RAW_VALUE` on failure.
   */
  uint16_t readRawTemperature();

  /**
   * @brief Read raw humidity value (blocking).
   * @return 14-bit raw humidity, or `INVALID_RAW_VALUE` on failure.
   */
  uint16_t readRawHumidity();

  /**
   * @brief Read both raw temperature and humidity values (blocking).
   * @return Struct with raw ADC values, or invalid values on failure.
   */
  RawMeasurementResult readRawMeasurement();

  /**
   * @brief Read processed temperature (blocking).
   * @return Temperature in selected unit, or `NAN` on failure.
   */
  float readTemperature();

  /**
   * @brief Read processed humidity (blocking).
   * @return Relative humidity in %RH, or `NAN` on failure.
   */
  float readHumidity();

  /**
   * @brief Read both temperature and humidity (blocking).
   * @return Struct with temperature and humidity, or `NAN` fields on failure.
   */
  MeasurementResult readMeasurement();

  // --------------------------------------------------------
  // Non-blocking measurement API
  // --------------------------------------------------------

  /**
   * @brief Start a measurement (non-blocking).
   *
   * Initiates a new measurement by sending an I2C command.
   * Use `isMeasurementReady()` to check completion.
   */
  void startMeasurement();

  /**
   * @brief Check if the sensor has completed a measurement.
   *
   * If data is ready, it is read and cached internally.
   * @return `true` if valid data was received.
   */
  bool isMeasurementReady();

  /**
   * @brief Get last raw temperature (non-blocking result).
   * @return Raw temperature ADC value.
   */
  uint16_t getRawTemperature() const;

  /**
   * @brief Get last raw humidity (non-blocking result).
   * @return Raw humidity ADC value.
   */
  uint16_t getRawHumidity() const;

  /**
   * @brief Get last raw temperature and humidity together.
   * @return Struct containing raw ADC values.
   */
  RawMeasurementResult getRawMeasurement() const;

  /**
   * @brief Get last measured temperature in selected units.
   * @return Temperature or `NAN` if no valid data.
   */
  float getTemperature() const;

  /**
   * @brief Get last measured relative humidity.
   * @return Humidity (%RH) or `NAN` if no valid data.
   */
  float getHumidity() const;

  /**
   * @brief Get last temperature and humidity values together.
   * @return MeasurementResult struct with converted values.
   */
  MeasurementResult getMeasurement() const;

  /**
   * @brief Change output temperature unit.
   * @param unit Desired unit (Celsius, Fahrenheit, Kelvin)
   */
  void setOutputUnit(TempUnit unit);

  /**
   * @brief Get current output temperature unit.
   * @return Configured temperature unit.
   */
  TempUnit getOutputUnit() const;

private:
  uint8_t _address;
  TwoWire *_wire;
  TempUnit _unit;
  Resolution _resolution;
  RawMeasurementResult _rawData;
  bool _waitForMeasurement();
};

#endif