#include "HS300x.h"

HS300x::HS300x(uint8_t address, TempUnit unit, Resolution resolution)
    : _address(address), _wire(nullptr), _unit(unit), _resolution(resolution),
      _rawData{INVALID_RAW_VALUE, INVALID_RAW_VALUE} {}

bool HS300x::begin(TwoWire &wirePort) {
  // we assume that Wire.begin() has already been called in the main program
  _wire = &wirePort;
  // check if sensor is connected
  _wire->beginTransmission(_address);
  return _wire->endTransmission() == 0;
}

uint8_t HS300x::getAddress() const { return _address; }

uint16_t HS300x::readRawTemperature() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getRawTemperature();
  }
  return INVALID_RAW_VALUE;
}

uint16_t HS300x::readRawHumidity() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getRawHumidity();
  }
  return INVALID_RAW_VALUE;
}

HS300x::RawMeasurementResult HS300x::readRawMeasurement() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getRawMeasurement();
  }
  return {INVALID_RAW_VALUE, INVALID_RAW_VALUE};
}

float HS300x::readTemperature() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getTemperature();
  }
  return NAN;
}

float HS300x::readHumidity() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getHumidity();
  }
  return NAN;
}

HS300x::MeasurementResult HS300x::readMeasurement() {
  startMeasurement();
  if (_waitForMeasurement()) {
    return getMeasurement();
  }
  return {NAN, NAN};
}

void HS300x::startMeasurement() {
  // start measurement by simply sending the I2C address with write flag
  _wire->beginTransmission(_address);
  _wire->endTransmission();
}

bool HS300x::isMeasurementReady() {
  // we check the flag and if new data is available we store it directly to
  // avoid having to read it multiple times
  uint8_t humMSB, humLSB, tempMSB, tempLSB;
  _wire->requestFrom(_address, FRAME_SIZE);
  if (_wire->available() == FRAME_SIZE) {
    humMSB = _wire->read();
    humLSB = _wire->read();
    tempMSB = _wire->read();
    tempLSB = _wire->read();
    // check if data is valid
    if ((humMSB >> STATUS_BITS_SHIFT) == 0) {
      // parse valid data
      _rawData.rawHumidity =
          (static_cast<uint16_t>(humMSB & STATUS_BITS_MASK) << 8) | humLSB;
      _rawData.rawTemperature =
          ((static_cast<uint16_t>(tempMSB) << 8) | tempLSB) >> TEMP_BITS_SHIFT;
      return true;
    } else {
      return false;
    }
  } else {
    return false;
  }
}

uint16_t HS300x::getRawTemperature() const { return _rawData.rawTemperature; }

uint16_t HS300x::getRawHumidity() const { return _rawData.rawHumidity; }

HS300x::RawMeasurementResult HS300x::getRawMeasurement() const {
  return _rawData;
}

float HS300x::getTemperature() const {
  if (_rawData.rawTemperature == INVALID_RAW_VALUE) {
    return NAN;
  }
  float tempC =
      (_rawData.rawTemperature * TEMPERATURE_FACTOR) + TEMPERATURE_OFFSET;
  switch (_unit) {
  case TempUnit::CELSIUS:
    return tempC;
  case TempUnit::FAHRENHEIT:
    return (tempC * FAHRENHEIT_SCALE) + FAHRENHEIT_OFFSET;
  case TempUnit::KELVIN:
    return tempC + KELVIN_OFFSET;
  default:
    // should not reach here
    return NAN;
  }
}

float HS300x::getHumidity() const {
  if (_rawData.rawHumidity == INVALID_RAW_VALUE) {
    return NAN;
  }
  return _rawData.rawHumidity * HUMIDITY_FACTOR;
}

HS300x::MeasurementResult HS300x::getMeasurement() const {
  return HS300x::MeasurementResult{getTemperature(), getHumidity()};
}

void HS300x::setOutputUnit(TempUnit unit) { _unit = unit; }

HS300x::TempUnit HS300x::getOutputUnit() const { return _unit; }

bool HS300x::_waitForMeasurement() {
  unsigned long startTime = millis();
  while (!isMeasurementReady()) {
    if (millis() - startTime > SENSOR_TIMEOUT_MS) {
      return false;
    }
    if (_resolution == Resolution::RES_14_BIT ||
        _resolution == Resolution::RES_12_BIT) {
      delay(POLLING_INTERVAL_COARSE_MS);
    } else {
      delayMicroseconds(POLLING_INTERVAL_FINE_US);
    }
  }
  return true;
}
