#include "Arduino.h"
#include "MCP4461.h"
#include <stdio.h>
#include <Wire.h>

/*
Library to control the MCP4461 Digital Potentiometer over I2C.
http://ww1.microchip.com/downloads/en/DeviceDoc/22265a.pdf
This library does not fully implement the functionality of
the MCP4461 just the basics of changing the wiper values.
The master joins the bus with the default address of 0

No warranty given or implied, use at your own risk.

Tony@think3dprint3d.com
GPL v3
*/

//ensure you call begin() before any other functions but note
//begin can only be called once for all MCP* objects as it initialises
//the local master through the Wire library
//if the MCP4461 does not have a default address, call set address before
//trying to communicate
MCP4461::MCP4461(uint8_t i2c_address) {
  _mcp4461_address = i2c_address;
}

//set the MCP4461 address
void MCP4461::setMCP4461Address(uint8_t mcp4461_addr) {
	_mcp4461_address = mcp4461_addr;
}

uint16_t MCP4461::readAddress(uint8_t address) {
  uint16_t ret = 0;
  uint16_t cmdByte = 0;
  cmdByte |= address;
  cmdByte |= MCP4461_READ;
  Wire.beginTransmission(_mcp4461_address);
  Wire.write(cmdByte);
  Wire.endTransmission(false);
  Wire.requestFrom((uint8_t)_mcp4461_address,(uint8_t)2);
  int i = 0;
  while(Wire.available()) 
  { 
    ret |= Wire.read();
    if (i==0) ret = ret<<8;
    i++;
  }
  return ret;
}

void MCP4461::writeValue(uint8_t addressByte, uint16_t data) {
	uint8_t commandByte;
	uint8_t dataByte = (uint8_t)data;
	if (data > 0xFF) commandByte = 0x1;
	else commandByte = 0;
	commandByte |= addressByte;
	commandByte |= MCP4461_WRITE;
	Wire.beginTransmission(_mcp4461_address);
	Wire.write(commandByte);
	Wire.write(dataByte);
	Wire.endTransmission();
	while(getEEPRomWriteActive()) {
	  delayMicroseconds(1);
	}
}

uint16_t MCP4461::getWiper(uint8_t wiper, bool nonvolatile = false) {
	uint8_t addressByte;
	switch (wiper) {
		case 0:
			addressByte = MCP4461_VW0;
			break;
		case 1:
			addressByte = MCP4461_VW1;
			break;
		case 2:
			addressByte = MCP4461_VW2;
			break;
		case 3:
			addressByte = MCP4461_VW3;
			break;
		default: 
			return;
	}
	if(nonvolatile) addressByte = addressByte + 0x20;
	return readAddress(addressByte);
}

void MCP4461::setWiper(uint8_t wiper, uint16_t wiper_value, bool nonvolatile = false){
  if (wiper_value > 257) return; //max 257 taps allowed
  uint8_t dataByte = (uint8_t)wiper_value;
  uint8_t addressByte;
  switch (wiper) {
      case 0:
		addressByte = MCP4461_VW0;
        break;
      case 1:
        addressByte = MCP4461_VW1;
        break;
      case 2:
        addressByte = MCP4461_VW2;
        break;
      case 3:
        addressByte = MCP4461_VW3;
        break;
      default: 
        return;
  }
  if(nonvolatile) addressByte = addressByte + 0x20;
  writeValue(addressByte, dataByte);
}

void MCP4461::setWipers(uint16_t wiper_value, bool nonvolatile = false ){
  uint16_t value = wiper_value;
  uint8_t i = 0;
  while (i < 4) {
	  setWiper(i, wiper_value, nonvolatile);
	  i++;
  }
}

uint8_t MCP4461::getStatus() {
  uint16_t ret;
  ret = readAddress(MCP4461_STATUS);
  return (ret & 0x00ff);
}

bool MCP4461::getEEPRomWriteActive() {
  return bitRead(getStatus(), 4);
}

uint8_t MCP4461::getTerminalRegister(uint8_t reg) {
  if (reg == 0) {
    return (readAddress(MCP4461_TCON0) & 0x00ff);
  }
  else {
    return (readAddress(MCP4461_TCON1) & 0x00ff);
  }
}

void MCP4461::setTerminalRegister(uint8_t reg, uint8_t value) {
  uint8_t addressByte = 0;
  if (reg == 0) {
    addressByte |= MCP4461_TCON0;
  }
  else {
	addressByte |= MCP4461_TCON1;
  }
  addressByte |= MCP4461_WRITE;
  writeValue(addressByte, value);
}

uint8_t MCP4461::getTerminalState(uint8_t wiper, char terminal) {
  uint8_t reg = 0;
  if (wiper >= 2) {
    reg++;
  }
  uint8_t bitNum;
  bitNum = getTerminalStateBitNumber(wiper, terminal);
  uint8_t tcon;
  tcon = getTerminalRegister(reg);
  return bitRead(tcon, bitNum);
  
}

void MCP4461::setTerminalState(uint8_t wiper, char terminal, uint8_t state) {
	uint8_t reg = 0;
	uint8_t tcon = 0;
	uint8_t bitNum;
	if (wiper >= 2) reg++;
	bitNum = getTerminalStateBitNumber(wiper, terminal);
	tcon = getTerminalRegister(reg);
	if(state == 0) bitClear(tcon, bitNum);
	else bitSet(tcon, bitNum);
	setTerminalRegister(reg, tcon);
}

uint8_t MCP4461::getTerminalStateBitNumber(uint8_t wiper, char terminal) {
	uint8_t bitNum;
	switch(terminal) {
		case 'B':
			bitNum = 0;
			break;
		case 'W':
			bitNum = 1;
			break;
		case 'A':
			bitNum = 2;
			break;
		case 'H':
			bitNum = 3;
			break;
	}
	if (wiper == 1 || wiper == 3) bitNum = bitNum + 4;
	return bitNum;
}

void MCP4461::connectWiper(uint8_t wiper) {
	setTerminalState(wiper, 'H', 1);
}

void MCP4461::disconnectWiper(uint8_t wiper) {
	setTerminalState(wiper, 'H', 0);
}

void MCP4461::toggleWiper(uint8_t wiper) {
	bool state;
    state = getTerminalState(wiper, 'H');
	if(state) {
		disconnectWiper(wiper);
	}
	else {
		connectWiper(wiper);
	}
}

uint16_t MCP4461::getEEPRomGeneralPurposeData(uint8_t location){
  if (location > 4) return 0;
  uint16_t addressByte = 1;
  addressByte |= (MCP4461_EEPROM_DATA_ADDRESS_1 + (location * 0x10));
  uint16_t ret;
  ret = readAddress(addressByte);
  byte retVal[2];
  retVal[0] = (ret >> 8);
  retVal[1] = (ret & 0x00ff);
  uint16_t returnInt;
  returnInt = (retVal[0]*256 + retVal[1]);
  return returnInt;
}

void MCP4461::setEEPRomGeneralPurposeData(uint8_t location, uint16_t value){
	uint8_t addressByte = 0;
	if (value > 511)return;
	addressByte |= (MCP4461_EEPROM_DATA_ADDRESS_1 + (location * 0x10));
	Serial.println(addressByte, HEX);
	writeValue(addressByte, value);
}
