/** \copyright
 * Copyright (c) 2015, Balazs Racz
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * \file PIPClient.hxx
 *
 * A client library for talking to an arbitrary openlcb Node and ask it for the
 * supported protocol data.
 *
 * @author Balazs Racz
 * @date 31 Oct 2015
 */

#ifndef _OPENLCB_PIPCLIENT_HXX_
#define _OPENLCB_PIPCLIENT_HXX_

#include "executor/StateFlow.hxx"
#include "openlcb/Defs.hxx"
#include "openlcb/If.hxx"
#include "os/os.h"

namespace openlcb
{

/** Specifies how long to wait for a PIP request to get a response. Writable
 * for unittesting purposes. Defaults to 4 seconds. */
extern long long PIP_CLIENT_TIMEOUT_NSEC;

/// State flow to request PIP information from a remote node on the OpenLCB
/// network.
///
/// Usage:
/// 
/// Create a global or local instance of this flow. Call the @ref request()
/// function with the arguments, supplying as notifiable the calling flow or a
/// sync notifiable for blocking operation on a thread. Wait for the
/// notification. Check that @ref error_code() == OPEATION_SUCCESS, then access
/// the returned protocol bitmask via the @ref response() accessor.
class PIPClient : public StateFlowBase
{
public:
    PIPClient(If *iface)
        : StateFlowBase(iface)
    {
    }

    /** Sends a PIP request to the specified node.
     *
     * The current
     *
     * @param dst is the target node to query
     * @param src is the source node from which to send query
     * @param done will be notified if the request succeeds or fails or
     * timeouts)
     */
    void request(NodeHandle dst, Node *src, Notifiable *done)
    {
        src_ = src;
        dst_ = dst;
        done_ = done;
        errorCode_ = OPERATION_PENDING;
        pipResponse_ = 0;
        start_flow(STATE(request_buffer));
    }

    /** @return the error code of the last request, or one of the internal
     * error codes from \ref PIPClient::ResultCodes */
    uint32_t error_code()
    {
        return errorCode_;
    }

    /** Returns the response of the last request out, or unspecified if the
     * last request has not succeeded. */
    uint64_t response()
    {
        return pipResponse_;
    }

    enum ResultCodes
    {
        // Internal error codes generated by the send flow
        OPERATION_SUCCESS = 0x10000, //< set when the Datagram OK arrives
        OPERATION_PENDING = 0x20000, //< cleared when done is called.
        TIMEOUT = 0x80000,           //< Timeout waiting for ack/nack.

        IDLE = 0xFFFF0000, //< The current flow is not in use.

    };

private:
    enum
    {
        MTI_1a = Defs::MTI_TERMINATE_DUE_TO_ERROR,
        MTI_1b = Defs::MTI_OPTIONAL_INTERACTION_REJECTED,
        MASK_1 = ~(MTI_1a ^ MTI_1b),
        MTI_1 = MTI_1a,

        MTI_2 = Defs::MTI_PROTOCOL_SUPPORT_REPLY,
        MASK_2 = Defs::MTI_EXACT,
    };

    Action request_buffer()
    {
        return allocate_and_call(
            iface()->addressed_message_write_flow(), STATE(write_request));
    }

    Action write_request()
    {
        auto *b =
            get_allocation_result(iface()->addressed_message_write_flow());
        b->data()->reset(Defs::MTI_PROTOCOL_SUPPORT_INQUIRY, src_->node_id(),
            dst_, EMPTY_PAYLOAD);

        iface()->dispatcher()->register_handler(
            &responseHandler_, MTI_1, MASK_1);
        iface()->dispatcher()->register_handler(
            &responseHandler_, MTI_2, MASK_2);

        iface()->addressed_message_write_flow()->send(b);

        return sleep_and_call(
            &timer_, PIP_CLIENT_TIMEOUT_NSEC, STATE(response_came));
    }

    // Callback from the response handler.
    void handle_response(Buffer<GenMessage> *message)
    {
        AutoReleaseBuffer<GenMessage> rb(message);
        if (src_ != message->data()->dstNode ||
            !iface()->matching_node(dst_, message->data()->src))
        {
            // Not from the right place.
            return;
        }
        if (message->data()->mti == Defs::MTI_OPTIONAL_INTERACTION_REJECTED ||
            message->data()->mti == Defs::MTI_TERMINATE_DUE_TO_ERROR)
        {
            uint16_t mti, error_code;
            buffer_to_error(
                message->data()->payload, &error_code, &mti, nullptr);
            if (mti && mti != Defs::MTI_PROTOCOL_SUPPORT_INQUIRY)
            {
                // Got error response for a different interaction. Ignore.
                return;
            }
            errorCode_ = error_code;
        }
        else if (message->data()->mti == Defs::MTI_PROTOCOL_SUPPORT_REPLY)
        {
            pipResponse_ = buffer_to_node_id(message->data()->payload);
            errorCode_ = OPERATION_SUCCESS;
        }
        else
        {
            // Dunno what this MTI is. Ignore.
            LOG(INFO, "Unexpected MTI for PIP response handler: %04x",
                message->data()->mti);
            return;
        }

        // Wakes up parent flow.
        errorCode_ &= ~OPERATION_PENDING;
        timer_.trigger();
    }

    Action response_came()
    {
        if (errorCode_ & OPERATION_PENDING)
        {
            errorCode_ = TIMEOUT;
        }
        iface()->dispatcher()->unregister_handler_all(&responseHandler_);
        done_->notify();
        return exit();
    }

    /// Message handler for incoming PIP responses. Gets registered in the
    /// input inteface's dispatcher and proxies an incoming PIP response to
    /// wake up the parent flow.
    class PIPResponseHandler : public MessageHandler
    {
    public:
        PIPResponseHandler(PIPClient *parent)
            : parent_(parent)
        {
        }

        void send(Buffer<GenMessage> *message, unsigned priority) OVERRIDE
        {
            parent_->handle_response(message);
        }

    private:
        PIPClient *parent_;
    };

    If *iface()
    {
        return static_cast<If *>(service());
    }

    StateFlowTimer timer_{this};
    Node *src_;
    Notifiable *done_;
    NodeHandle dst_;
    uint64_t pipResponse_{0};
    uint32_t errorCode_{IDLE};
    PIPResponseHandler responseHandler_{this};
};

} // namespace openlcb

#endif // _OPENLCB_PIPCLIENT_HXX_
