/** \copyright
 * Copyright (c) 2017, Balazs Racz
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are  permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * \file DccDebug.cxx
 *
 * Defines helper functions for debugging DCC packets
 *
 * @author Balazs Racz
 * @date 28 Dec 2017
 */

#include "dcc/DccDebug.hxx"
#include "dcc/Defs.hxx"
#include "utils/StringPrintf.hxx"
#include "utils/logging.h"

namespace dcc
{

string packet_to_string(const DCCPacket &pkt, bool bin_payload)
{
    if (pkt.packet_header.is_pkt)
    {
        return StringPrintf("[cmd:%u]", pkt.command_header.cmd);
    }
    // First render the option bits
    string options;
    if (pkt.packet_header.is_marklin)
    {
        options += "[marklin]";
    }
    else
    {
        options += "[dcc]";
    }
    if (pkt.packet_header.send_long_preamble)
    {
        options += "[long_preamble]";
    }
    if (pkt.packet_header.sense_ack)
    {
        options += "[sense_ack]";
    }
    if (pkt.packet_header.rept_count)
    {
        options +=
            StringPrintf("[repeat %u times]", pkt.packet_header.rept_count + 1);
    }
    if (!pkt.dlc)
    {
        return options + " no payload";
    }
    if (bin_payload || pkt.packet_header.is_marklin)
    {
        options += "[";
        for (unsigned i = 0; i < pkt.dlc; ++i)
        {
            options += StringPrintf("%02x ", pkt.payload[i]);
        }
        options.pop_back();
        options += "]";
    }
    if (pkt.packet_header.is_marklin)
    {
        return options;
    }
    unsigned ofs = 0;
    bool is_idle_packet = false;
    bool is_basic_accy_packet = false;
    bool is_unknown_packet = false;
    bool is_svc_packet = false;
    unsigned accy_address = 0;
    if (pkt.packet_header.send_long_preamble)
    {
        // Checks for service mode packets.

        // WARNING: This code is only guaranteed to correctly decide between
        // service mode packets and basic decoder packets if the packets were
        // generated by OpenMRN. Otherwise it is not guaranteed that
        // long_preamble is equivalent to service mode.
        using namespace dcc::Defs;
        is_svc_packet = true;
        uint8_t cmd = pkt.payload[ofs];
        int cv = (((pkt.payload[ofs] & 0x3) << 8) | pkt.payload[ofs + 1]) + 1;
        int reg = (pkt.payload[ofs] & 0x7) + 1;
        // How many bytes does the command have (cmd and arguments), usually 2
        // for paged mode and 3 for direct mode commands.
        int num_bytes =
            pkt.packet_header.skip_ec ? pkt.dlc - ofs - 1 : pkt.dlc - ofs;
        if (num_bytes == 3 && (cmd & DCC_SVC_MASK) == DCC_SVC_WRITE)
        {
            options += StringPrintf(
                "[svc] Direct Write Byte CV %d = %d", cv, pkt.payload[ofs + 2]);
            ofs += 2;
        }
        else if (num_bytes == 3 && (cmd & DCC_SVC_MASK) == DCC_SVC_VERIFY)
        {
            options += StringPrintf("[svc] Direct Verify Byte CV %d =?= %d", cv,
                pkt.payload[ofs + 2]);
            ofs += 2;
        }
        else if (num_bytes == 3 &&
            ((cmd & DCC_SVC_MASK) == DCC_SVC_BIT_MANIPULATE) &&
            ((pkt.payload[ofs + 2] & DCC_SVC_BITVAL_MASK) ==
                DCC_SVC_BITVAL_WRITE))
        {
            int bitnum = pkt.payload[ofs + 2] & 7;
            int bitval = pkt.payload[ofs + 2] & DCC_SVC_BITVAL_VALUE ? 1 : 0;
            options += StringPrintf(
                "[svc] Direct Write Bit CV %d bit %d = %d", cv, bitnum, bitval);
            ofs += 2;
        }
        else if (num_bytes == 3 &&
            ((cmd & DCC_SVC_MASK) == DCC_SVC_BIT_MANIPULATE) &&
            ((pkt.payload[ofs + 2] & DCC_SVC_BITVAL_MASK) ==
                DCC_SVC_BITVAL_VERIFY))
        {
            int bitnum = pkt.payload[ofs + 2] & 7;
            int bitval = pkt.payload[ofs + 2] & DCC_SVC_BITVAL_VALUE ? 1 : 0;
            options +=
                StringPrintf("[svc] Direct Verify Bit CV %d bit %d =?= %d", cv,
                    bitnum, bitval);
            ofs += 2;
        }
        else if (num_bytes == 2 &&
            (cmd & DCC_SVC_PAGED_MASK) == DCC_SVC_PAGED_WRITE)
        {
            options += StringPrintf("[svc] Paged Write Byte Reg %d = %d", reg,
                pkt.payload[ofs + 1]);
            ofs++;
        }
        else if (num_bytes == 2 &&
            (cmd & DCC_SVC_PAGED_MASK) == DCC_SVC_PAGED_VERIFY)
        {
            options += StringPrintf("[svc] Paged Verify Byte Reg %d =?= %d",
                reg, pkt.payload[ofs + 1]);
            ofs++;
        }
        else
        {
            // Not recognized.
            is_svc_packet = false;
        }
    }

    if (is_svc_packet)
    {
        // Do not use the regular address partition logic here.
    }
    else if (pkt.payload[ofs] == 0xff)
    {
        options += " Idle packet";
        ofs++;
        if (pkt.payload[ofs] != 0)
        {
            options += StringPrintf(" unexpected[0x%02x]", pkt.payload[ofs]);
        }
        is_idle_packet = true;
    }
    else if (pkt.payload[ofs] == 0)
    {
        options += " Broadcast";
        ofs++;
        if (pkt.payload[ofs] == 0)
        {
            options += " reset";
        }
    }
    else if ((pkt.payload[ofs] & 0x80) == 0)
    {
        options += StringPrintf(" Short Address %u", pkt.payload[ofs]);
        ofs++;
    }
    else if ((pkt.payload[ofs] & 0xC0) == 0x80)
    {
        // accessory decoder
        is_basic_accy_packet = true;
        accy_address = (pkt.payload[ofs] & 0b111111) << 3;
        ofs++;
    }
    else if (pkt.payload[ofs] >= 192 && pkt.payload[ofs] <= 231)
    {
        // long address
        unsigned addr = pkt.payload[ofs] & 0x3F;
        addr <<= 8;
        ofs++;
        addr |= pkt.payload[ofs];
        ofs++;
        options += StringPrintf(" Long Address %u", addr);
    } else if (pkt.payload[ofs] == 254) {
        options += " Logon packet";
        is_unknown_packet = true;
        while (ofs < pkt.dlc)
        {
            options += StringPrintf(" 0x%02x", pkt.payload[ofs++]);
        }
    } else if (pkt.payload[ofs] == 253) {
        options += " Advanced extended packet";
        is_unknown_packet = true;
        while (ofs < pkt.dlc)
        {
            options += StringPrintf(" 0x%02x", pkt.payload[ofs++]);
        }
    }
    uint8_t cmd = pkt.payload[ofs];
    if (!is_unknown_packet)
    {
        ofs++;
    }
    if (is_basic_accy_packet && ((cmd & 0x80) == 0x80))
    {
        accy_address |= cmd & 0b111;
        cmd >>= 3;
        bool is_activate = cmd & 1;
        cmd >>= 1;
        accy_address |= ((~cmd) & 0b111) << 9;
        unsigned user_address =
            Defs::accy_address_binary_to_user(accy_address >> 1);
        const char *n_r =
            accy_address & 1 ? "closed/normal/on" : "thrown/reverse/off";
        const char* a_d = is_activate ? "activate" : "deactivate";
        options += StringPrintf(
            " Accy %u (user %u %s) %s", accy_address, user_address, n_r, a_d);
    }
    else if (is_unknown_packet || is_svc_packet)
    {
    }
    else if ((cmd & 0xC0) == 0x40)
    {
        // Speed and direction
        bool is_forward = (cmd & 0x20) != 0;
        options += " SPD ";
        options += is_forward ? 'F' : 'R';
        uint8_t speed = ((cmd & 0xF) << 1) | ((cmd & 0x10) >> 4);
        switch (speed)
        {
            case 0:
                options += " 0";
                break;
            case 1:
                options += " 0'";
                break;
            case 2:
                options += " E-STOP";
                break;
            case 3:
                options += " E-STOP'";
                break;
            default:
                options += StringPrintf(" %u", speed - 3);
        }
    }
    else if (cmd == 0x3F) {
        // 128-speed step
        uint8_t val = pkt.payload[ofs];
        ofs++;
        bool is_forward = (val & 0x80) != 0;
        uint8_t speed = val & 0x7F;
        options += " SPD128 ";
        options += is_forward ? 'F' : 'R';
        switch (speed)
        {
            case 0:
                options += " 0";
                break;
            case 1:
                options += " E-STOP";
                break;
            default:
                options += StringPrintf(" %u", speed - 1);
        }        
    }
    else if ((cmd >> 5) == 0b100)
    {
        // function group 0
        options += StringPrintf(" F[0-4]=%d%d%d%d%d", (cmd >> 4) & 1,
            (cmd >> 0) & 1, (cmd >> 1) & 1, (cmd >> 2) & 1, (cmd >> 3) & 1);
    }
    else if ((cmd >> 5) == 0b101)
    {
        // function group 1 or 2
        if (cmd & 0x10)
        {
            options += " F[5-8]=";
        }
        else
        {
            options += " F[9-12]=";
        }
        options += StringPrintf("%d%d%d%d", (cmd >> 0) & 1, (cmd >> 1) & 1,
            (cmd >> 2) & 1, (cmd >> 3) & 1);
    }
    else if ((cmd >> 5) == 0b110)
    {
        // expansion
        uint8_t c = cmd & 0x1F;
        if ((c & ~1) == 0b11110)
        {
            if (c & 1)
            {
                options += " F[21-28]=";
            }
            else
            {
                options += " F[13-20]=";
            }
            c = pkt.payload[ofs];
            ofs++;
            for (int i = 0; i < 8; ++i, c >>= 1)
                options += '0' + (c & 1);
        }
        else
        {
            /// @todo
        }
    }
    else if (cmd == 0 && is_idle_packet)
    {
    }
    else if ((cmd >> 4) == 0b1110)
    {
        // POM command
        options += " POM CV";
        unsigned kk = (cmd >> 2) & 3;
        unsigned cv = (cmd & 3) << 8;
        cv |= pkt.payload[ofs];
        ofs++;
        options += StringPrintf("%d", cv + 1);
        uint8_t d = pkt.payload[ofs++];

        switch (kk)
        {
            case 0b00:
            {
                options += StringPrintf(" resvd %02x", d);
                break;
            }
            case 0b01:
            {
                options += StringPrintf(" read/verify %d", d);
                break;
            }
            case 0b11:
            {
                options += StringPrintf(" write = %d", d);
                break;
            }
            case 0b10:
            {
                unsigned bit = d & 7;
                unsigned value = (d >> 3) & 1;
                if ((d & 0xE0) != 0xE0)
                {
                    options += StringPrintf(" bit manipulate unknown (%02x)", d);
                    break;
                }
                if ((d & 0x10) == 0x10)
                {
                    options += StringPrintf(" bit %d write = %d", bit, value);
                }
                else
                {
                    options += StringPrintf(" bit %d verify ?= %d", bit, value);
                }
                break;
            }
        }
    }

    // checksum of packet
    if (ofs == pkt.dlc && (pkt.packet_header.skip_ec == 0 || is_unknown_packet))
    {
        // EC skipped.
    }
    else if (((ofs + 1) == pkt.dlc) && pkt.packet_header.skip_ec == 1)
    {
        uint8_t x = 0;
        for (unsigned i = 0; i + 1 < pkt.dlc; ++i)
        {
            x ^= pkt.payload[i];
        }
        if (x != pkt.payload[pkt.dlc - 1])
        {
            options += StringPrintf(" [bad EC expected 0x%02x actual 0x%02x]",
                x, pkt.payload[pkt.dlc - 1]);
        }
    }
    else
    {
        options += StringPrintf(" [bad dlc, exp %u, actual %u]", ofs+1, pkt.dlc);
        while (ofs < pkt.dlc)
        {
            options += StringPrintf(" 0x%02x", pkt.payload[ofs++]);
        }
    }
    if (pkt.packet_header.csum_error)
    {
        options += " [csum err]";
    }
    return options;
}

} // namespace dcc
