/**
 * examples/ButtonControl/ButtonControl.ino
 * Example for controlling LEDs with buttons using BCC_ESP32S3 library
 * Created by Nonpawit Ekburanawat, 2025
 * Released under MIT License
 */

#include <BCC_ESP32S3.h>

/* LED array for easy indexing. */

const uint8_t leds[] = {RED_LED, YELLOW_LED, GREEN_LED};
const int NUM_LEDS = 3;

/* Button pins. */

const uint8_t BTN_UP   = BT1;  // Button to increase LED index
const uint8_t BTN_DOWN = BT2;  // Button to decrease LED index

/* Variables for button state tracking. */

int currentLedIndex   = 0;           // Current LED index (0=RED, 1=YELLOW, 2=GREEN)
bool btnUpState       = HIGH;        // Current state of UP button
bool btnDownState     = HIGH;        // Current state of DOWN button
bool lastBtnUpState   = HIGH;        // Previous state of UP button
bool lastBtnDownState = HIGH;        // Previous state of DOWN button

unsigned long lastDebounceTime = 0;  // Last time a button was pressed
unsigned long debounceDelay = 50;    // Debounce time in milliseconds

void setup()
{
  /* Initialize the on-board LEDs as outputs. */

  pinMode(RED_LED, OUTPUT);
  pinMode(YELLOW_LED, OUTPUT);
  pinMode(GREEN_LED, OUTPUT);

  /* Initialize the buttons as inputs with pull-up resistors. */

  pinMode(BTN_UP, INPUT_PULLUP);
  pinMode(BTN_DOWN, INPUT_PULLUP);

  /* Initially turn on the first LED (RED). */

  updateLeds();
}

void loop()
{
  /* Read the current state of both buttons. */

  bool reading_up   = digitalRead(BTN_UP);
  bool reading_down = digitalRead(BTN_DOWN);

  /* Check if either button state has changed. */

  if (reading_up != lastBtnUpState || reading_down != lastBtnDownState) 
  {
    /* Reset the debounce timer. */

    lastDebounceTime = millis();
  }

  /* If enough time has passed since the last button state change. */

  if ((millis() - lastDebounceTime) > debounceDelay) 
  {
    /* Check if the UP button state has changed. */
    if (reading_up != btnUpState)
    {
      btnUpState = reading_up;

      /* If the button is pressed (LOW when using INPUT_PULLUP). */

      if (btnUpState == LOW)
      {
        /* Increase the LED index and wrap around if needed. */
        currentLedIndex = (currentLedIndex + 1) % NUM_LEDS;
        updateLeds();
      }
    }

    /* Check if the DOWN button state has changed. */

    if (reading_down != btnDownState)
    {
      btnDownState = reading_down;

      /* If the button is pressed (LOW when using INPUT_PULLUP). */

      if (btnDownState == LOW) 
      {
        /* Decrease the LED index and wrap around if needed. */

        currentLedIndex = (currentLedIndex - 1 + NUM_LEDS) % NUM_LEDS;
        updateLeds();
      }
    }
  }

  /* Save the current button states for the next loop. */

  lastBtnUpState   = reading_up;
  lastBtnDownState = reading_down;
}

void updateLeds()
{
  /* Turn off all LEDs first. */

  digitalWrite(RED_LED, LOW);
  digitalWrite(YELLOW_LED, LOW);
  digitalWrite(GREEN_LED, LOW);

  /* Turn on only the current LED. */

  digitalWrite(leds[currentLedIndex], HIGH);
}
