#include "ZenRTC.h"

// DS3231 registers/bits
static constexpr uint8_t REG_SECONDS = 0x00;
static constexpr uint8_t REG_MINUTES = 0x01;
static constexpr uint8_t REG_HOURS   = 0x02;
static constexpr uint8_t REG_DAY     = 0x03;
static constexpr uint8_t REG_DATE    = 0x04;
static constexpr uint8_t REG_MONTH   = 0x05; // bit7 = Century
static constexpr uint8_t REG_YEAR    = 0x06;
static constexpr uint8_t REG_CONTROL = 0x0E;
static constexpr uint8_t REG_STATUS  = 0x0F;

// Status bits
static constexpr uint8_t STS_A1F     = 0;
static constexpr uint8_t STS_A2F     = 1;
static constexpr uint8_t STS_OSF     = 7;

// Control bits
static constexpr uint8_t CTRL_EOSC   = 7; // 1=stop oscillator
static constexpr uint8_t CTRL_INTCN  = 2;

static inline uint8_t dec2bcd(uint8_t v){ return ((v/10)<<4) | (v%10); }
static inline uint8_t bcd2dec(uint8_t v){ return ((v>>4)*10) + (v & 0x0F); }

ZenRTC::ZenRTC(uint8_t sda, uint8_t scl, uint32_t freq): _sda(sda), _scl(scl), _freq(freq) {}

bool ZenRTC::begin(bool setBuildIfInvalid) {
  Wire.begin(_sda, _scl, _freq);
  if (!_prefsBegun) { _prefsBegun = _prefs.begin(NAMESPACE, false); }

  // Ensure oscillator enabled, interrupts mode selected
  uint8_t ctrl=0; 
  if (readControl(ctrl)) { 
    ctrl &= ~(1<<CTRL_EOSC); 
    ctrl |= (1<<CTRL_INTCN); 
    writeControl(ctrl); 
  }

  // Clear alarm flags; clear OSF so we can detect new stops later
  uint8_t status=0; 
  if (readStatus(status)) { 
    status &= ~((1<<STS_A1F)|(1<<STS_A2F)); 
    writeStatus(status & ~(1<<STS_OSF)); 
  }

  DateTime rt{}; 
  bool ok = readRTC(rt);
  time_t lastGood = _prefsBegun ? _prefs.getLong64(KEY_LASTGOOD, 0) : 0;

  if (!ok || !isReasonable(rt)) {
    if (setBuildIfInvalid) {
      setToBuildTime();
      return true;
    }
    return false;
  }

  time_t nowE = toEpoch(rt);
  if (lastGood == 0) {
    if (_prefsBegun) _prefs.putLong64(KEY_LASTGOOD, nowE);
    return true;
  }

  time_t delta = llabs(nowE - lastGood);
  if ((uint32_t)delta > _maxJump) {
    // Clamp back to last-good
    DateTime clampT = fromEpoch(lastGood);
    writeRTC(clampT);
    if (_prefsBegun) _prefs.putLong64(KEY_LASTGOOD, lastGood);
  } else {
    if (_prefsBegun) _prefs.putLong64(KEY_LASTGOOD, nowE);
  }
  return true;
}

bool ZenRTC::now(DateTime &out) { 
  return readRTC(out) && isReasonable(out); 
}

bool ZenRTC::set(const DateTime &dt) {
  if (!isReasonable(dt)) return false;
  if (!writeRTC(dt)) return false;
  if (_prefsBegun) _prefs.putLong64(KEY_LASTGOOD, toEpoch(dt));
  return true;
}

void ZenRTC::setMaxAcceptableJumpSeconds(uint32_t s){ _maxJump = s; }

time_t ZenRTC::lastGoodEpoch(){ 
  return _prefsBegun ? _prefs.getLong64(KEY_LASTGOOD, 0) : 0; 
}

bool ZenRTC::setToBuildTime(){
  // Parse __DATE__ and __TIME__ (UTC assumed)
  const char months[][4] = {"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
  char mstr[4]; int d,y,hh,mm,ss; 
  sscanf(__DATE__, "%3s %d %d", mstr, &d, &y); 
  sscanf(__TIME__, "%d:%d:%d", &hh, &mm, &ss);
  int m=1; for (int i=0;i<12;i++) if (!strncmp(mstr, months[i],3)) { m=i+1; break; }
  DateTime build{ (uint16_t)y, (uint8_t)m, (uint8_t)d, (uint8_t)hh, (uint8_t)mm, (uint8_t)ss, 1 };
  bool ok = writeRTC(build);
  if (ok && _prefsBegun) _prefs.putLong64(KEY_LASTGOOD, toEpoch(build));
  return ok;
}

// ---- low-level ----
bool ZenRTC::readStatus(uint8_t &status){ 
  uint8_t v; 
  if(!readBurst(REG_STATUS,&v,1)) return false; 
  status=v; 
  return true; 
}
bool ZenRTC::writeStatus(uint8_t status){ return writeReg(REG_STATUS, status); }
bool ZenRTC::readControl(uint8_t &ctrl){ 
  uint8_t v; 
  if(!readBurst(REG_CONTROL,&v,1)) return false; 
  ctrl=v; 
  return true; 
}
bool ZenRTC::writeControl(uint8_t ctrl){ return writeReg(REG_CONTROL, ctrl); }

bool ZenRTC::readBurst(uint8_t startReg, uint8_t *buf, size_t n){
  Wire.beginTransmission(I2C_ADDR); 
  Wire.write(startReg); 
  if (Wire.endTransmission(false) != 0) return false;
  if (Wire.requestFrom(I2C_ADDR, (uint8_t)n) != (int)n) return false; 
  for (size_t i=0;i<n;i++) buf[i]=Wire.read(); 
  return true;
}

bool ZenRTC::writeBurst(uint8_t startReg, const uint8_t *buf, size_t n){
  Wire.beginTransmission(I2C_ADDR); 
  Wire.write(startReg); 
  for(size_t i=0;i<n;i++) Wire.write(buf[i]); 
  return Wire.endTransmission()==0;
}

bool ZenRTC::writeReg(uint8_t reg, uint8_t val){ 
  Wire.beginTransmission(I2C_ADDR); 
  Wire.write(reg); 
  Wire.write(val); 
  return Wire.endTransmission()==0; 
}

bool ZenRTC::readRTC(DateTime &t){
  uint8_t buf[7]; 
  if(!readBurst(REG_SECONDS, buf, 7)) return false;
  t.second = bcd2dec(buf[0] & 0x7F);
  t.minute = bcd2dec(buf[1] & 0x7F);
  // Hours: handle 12/24h
  if (buf[2] & 0x40){ // 12h mode
    uint8_t hr = bcd2dec(buf[2] & 0x1F); 
    bool pm = buf[2] & 0x20;
    if (pm && hr != 12) hr += 12; 
    if (!pm && hr == 12) hr = 0; 
    t.hour = hr;
  } else { 
    t.hour = bcd2dec(buf[2] & 0x3F); 
  }
  t.weekday = bcd2dec(buf[3] & 0x07);
  t.day     = bcd2dec(buf[4] & 0x3F);
  uint8_t monthReg = buf[5]; 
  t.month = bcd2dec(monthReg & 0x1F);
  uint16_t year = 2000 + bcd2dec(buf[6]); 
  if (monthReg & 0x80) year += 100; 
  t.year = year;
  return true;
}

bool ZenRTC::writeRTC(const DateTime &t){
  uint8_t monthReg = dec2bcd(t.month); 
  if (t.year >= 2100) monthReg |= 0x80; // century bit
  uint8_t year2 = (t.year >= 2000) ? (t.year - 2000) : (t.year % 100);
  uint8_t buf[7];
  buf[0]=dec2bcd(t.second); 
  buf[1]=dec2bcd(t.minute); 
  buf[2]=dec2bcd(t.hour);
  buf[3]=dec2bcd(t.weekday ? t.weekday : 1); 
  buf[4]=dec2bcd(t.day); 
  buf[5]=monthReg; 
  buf[6]=dec2bcd(year2);
  return writeBurst(REG_SECONDS, buf, 7);
}

bool ZenRTC::isReasonable(const DateTime &t) const{
  if (t.year < 2020 || t.year > 2099) return false;
  if (t.month < 1 || t.month > 12) return false;
  if (t.day < 1 || t.day > 31) return false;
  if (t.hour > 23 || t.minute > 59 || t.second > 59) return false;
  return true;
}

void ZenRTC::print(const char* tag, const DateTime &t) const{
  Serial.printf("%s %04u-%02u-%02u %02u:%02u:%02u\n", tag, t.year, t.month, t.day, t.hour, t.minute, t.second);
}

// ---- epoch conversions (UTC) ----
static tm makeTmUTC(const ZenRTC::DateTime &d){ 
  tm tmv{}; 
  tmv.tm_year=d.year-1900; 
  tmv.tm_mon=d.month-1; 
  tmv.tm_mday=d.day; 
  tmv.tm_hour=d.hour; 
  tmv.tm_min=d.minute; 
  tmv.tm_sec=d.second; 
  tmv.tm_isdst=-1; 
  return tmv; 
}

time_t ZenRTC::toEpoch(const DateTime &dt){ 
  tm tmv = makeTmUTC(dt); 
  return mktime(&tmv); 
}

ZenRTC::DateTime ZenRTC::fromEpoch(time_t e){ 
  tm *tmv = gmtime(&e); 
  DateTime d{}; 
  d.year=tmv->tm_year+1900; 
  d.month=tmv->tm_mon+1; 
  d.day=tmv->tm_mday; 
  d.hour=tmv->tm_hour; 
  d.minute=tmv->tm_min; 
  d.second=tmv->tm_sec; 
  d.weekday = tmv->tm_wday==0?7:tmv->tm_wday; 
  return d; 
}
