// Plays a RAW (16-bit signed) PCM audio file at slower or faster rate
// this example plays a sample stored in an array
#include <Arduino.h>
#include <Audio.h>
#include "playsdresmp.h"
#include "output_soundio.h"
#include <soundio/soundio.h>
#include <SD.h>
#include <iostream>
#include <execinfo.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <string>
#include <cassert>

// GUItool: begin automatically generated code
AudioPlaySdResmp         playSd1;        //xy=306,225
AudioRecordQueue         queue1;         //xy=609,267
AudioOutputSoundIO       sio_out1;       //xy=612,224
AudioConnection          patchCord1(playSd1, 0, sio_out1, 0);
AudioConnection          patchCord2(playSd1, 1, sio_out1, 1);
AudioConnection          patchCord3(playSd1, 0, queue1, 0);
// GUItool: end automatically generated code

char** _filenames  = nullptr;
uint16_t _fileIndex = 0;
uint16_t _numWaveFiles = 0;

int16_t buffer[512] = {0};
File frec;

unsigned long lastSamplePlayed = 0;
void my_handler(sig_atomic_t i);
static char stack_body[64*1024];
static stack_t sigseg_stack;
static struct sigaction sigseg_handler;

void crash_handler(sig_atomic_t i);



uint16_t getNumWavFilesInDirectory(char *directory);
void populateFilenames(char *directory, char **filenames);


void setup() {
    Serial.begin(9600);       

    playSd1.setPlaybackRate(1.0f);
    playSd1.enableInterpolation(true);
    //rraw_a1.play((int16_t*)kick_raw, kick_raw_len/2);    
    Serial.println("setup done");

    if (SD.exists("RECORD_SD.RAW")) {
        // The SD library writes new data to the end of the
        // file, so to start a new recording, the old file
        // must be deleted before new data is written.
        SD.remove("RECORD_SD.RAW");
    }
    frec = SD.open("RECORD_SD.RAW", O_WRITE);

    _numWaveFiles = getNumWavFilesInDirectory("/");
    Serial.printf("Num wave files: %d\n", _numWaveFiles);
    _filenames = new char*[_numWaveFiles];
    populateFilenames("/", _filenames);

    Serial.println("Populated...");

    AudioMemory(120);
    
    if (frec) {
        queue1.begin();
        Serial.println("startRecording");
    } else {
        Serial.println("recording failed...");
    }
}

void loop() {
    unsigned currentMillis = millis();
    if (currentMillis > lastSamplePlayed + 1000) {
        if (!playSd1.isPlaying()) {
            if (playSd1.playWav(_filenames[_fileIndex])) {
              lastSamplePlayed = currentMillis;
              Serial.printf("playing %s\n", _filenames[_fileIndex]);
            } else 
              Serial.printf("failed to play%s\n", _filenames[_fileIndex]);
            _fileIndex++;
            _fileIndex %= _numWaveFiles;

            Serial.print("Memory: ");
            Serial.print(AudioMemoryUsage());
            Serial.print(",");
            Serial.print(AudioMemoryUsageMax());
            Serial.println();
        }
    }

    if (queue1.available() >= 1) {   
        int16_t* incomming = queue1.readBuffer();
        //Serial.printf("sizeof(incomming)=%i\n", sizeof(incomming));
        //if (incomming != NULL && sizeof(incomming) >= 256) {
        if (incomming != NULL) {
            memcpy(buffer, incomming, 256);
            queue1.freeBuffer();
            frec.write((unsigned char *)buffer, 256);
            frec.flush();
        } 
        //else {
        //    arduino_should_exit = true;        
            //Serial.printf("sizeof(incomming)=%i\n", sizeof(incomming));
        //}
    } 
    delay(1);
}

int main(int numArgs, char **args) {

    if (numArgs < 2)
    {
        std::cout << "usage: " << args[0] << " <path-to-SDCard>";
        exit(0);
    }
    std::cout << args[1] << std::endl;
    signal (SIGINT,my_handler);
    signal (SIGSEGV,crash_handler);    

    sigseg_stack.ss_sp = stack_body;
    sigseg_stack.ss_flags = SS_ONSTACK;
    sigseg_stack.ss_size = sizeof(stack_body);
 //   assert(!sigaltstack(&sigseg_stack, nullptr));
    sigseg_handler.sa_flags = SA_ONSTACK;
    sigseg_handler.sa_handler = &crash_handler;
 //   assert(!sigaction(SIGSEGV, &sigseg_handler, nullptr));

    initialize_mock_arduino();
    //SD.setSDCardFileData((char *) kick_raw, kick_raw_len);
    SD.setSDCardFolderPath(args[1]);
    setup();
    while(!arduino_should_exit){
        loop();
        yield();
    }
    delay(1000);
    frec.close();
}


uint16_t getNumWavFilesInDirectory(char *directory) {
  File dir = SD.open(directory);
  uint16_t numWaveFiles = 0;

  while (true) { 

    File files =  dir.openNextFile();
    if (!files) {
      //If no more files, break out.
      break;
    }

    String curfile = files.name(); //put file in string
    
    int m = curfile.lastIndexOf(".WAV");
    int a = curfile.lastIndexOf(".wav");
    int underscore = curfile.indexOf("_");
    int underscore2 = curfile.indexOf("._");
    // if returned results is more then 0 add them to the list.
    if ((m > 0 || a > 0) && (underscore != 0) && (underscore2 != 0)) {  
        numWaveFiles++;
    }
    
    files.close();
  }
  // close 
  dir.close();
  return numWaveFiles;
}

void populateFilenames(char *directory, char **filenames) {
  File dir = SD.open(directory);
  uint16_t index = 0;

  while (true) { 

    File files =  dir.openNextFile();
    if (!files) {
      //If no more files, break out.
      break;
    }

    String curfile = files.name(); //put file in string
    
    int m = curfile.lastIndexOf(".WAV");
    int a = curfile.lastIndexOf(".wav");
    int underscore = curfile.indexOf("_");
    int underscore2 = curfile.indexOf("._");
    // if returned results is more then 0 add them to the list.
    if ((m > 0 || a > 0) && (underscore != 0) && (underscore2 != 0) ) {  
    
        Serial.printf(" ---- %s\n", curfile.c_str());
        filenames[index] = new char[curfile.length()+1] {0};
        memcpy(filenames[index], curfile.c_str(), curfile.length()+1);
        Serial.printf(" ------ %s\n", filenames[index]);
        index++;
    } 
    files.close();
  }
  // close 
  dir.close();
}


void my_handler(sig_atomic_t i){
    if ( i== SIGINT) {
        arduino_should_exit = true;
        printf("Caught signal %d\n",i);
    } else 
    {
        std::cerr << "sig seg fault handler" << std::endl;
        const int asize = 10;
        void *array[asize];
        size_t size;

        // get void*'s for all entries on the stack
        size = backtrace(array, asize);

        // print out all the frames to stderr
        std::cerr << "stack trace: " << std::endl;
        backtrace_symbols_fd(array, size, STDERR_FILENO);
        std::cerr << "resend SIGSEGV to get core dump" << std::endl;
        signal(i, SIG_DFL);
        kill(getpid(), i);
    }
}
void crash_handler(sig_atomic_t i){
    std::cerr << "sig seg fault handler" << std::endl;
    const int asize = 10;
    void *array[asize];
    size_t size;

    // get void*'s for all entries on the stack
    size = backtrace(array, asize);

    // print out all the frames to stderr
    std::cerr << "stack trace: " << std::endl;
    backtrace_symbols_fd(array, size, STDERR_FILENO);
    std::cerr << "resend SIGSEGV to get core dump" << std::endl;
    signal(i, SIG_DFL);
    kill(getpid(), i);
}

unsigned char kick_raw[] = {
  0x99, 0x02, 0xd7, 0x02, 0xfa, 0x02, 0x5f, 0x03, 0xc1, 0x03, 0x2a, 0x04,
  0xad, 0x04, 0xa5, 0x05, 0x76, 0x06, 0x2f, 0x07, 0x9e, 0x07, 0xe2, 0x07,
  0x43, 0x08, 0x92, 0x08, 0xb2, 0x08, 0xe8, 0x08, 0x16, 0x09, 0xda, 0x08,
  0x51, 0x08, 0x01, 0x08, 0x25, 0x08, 0x70, 0x08, 0xc3, 0x08, 0x23, 0x09,
  0x95, 0x09, 0x19, 0x0a, 0x83, 0x0a, 0x7e, 0x0a, 0xd0, 0x0a, 0x65, 0x0b,
  0xf6, 0x0b, 0x89, 0x0c, 0xd1, 0x0c, 0xcf, 0x0c, 0x1a, 0x0d, 0xe5, 0x0d,
  0x5e, 0x0e, 0xbb, 0x0e, 0xec, 0x0e, 0xd9, 0x0e, 0x07, 0x0f, 0xc8, 0x0f,
  0x2a, 0x10, 0x04, 0x10, 0x28, 0x10, 0x54, 0x11, 0x8e, 0x13, 0x4b, 0x16,
  0x09, 0x19, 0x91, 0x1b, 0xf7, 0x1d, 0x55, 0x20, 0xd1, 0x22, 0xcb, 0x25,
  0x4d, 0x29, 0xa8, 0x2c, 0x7f, 0x2f, 0xda, 0x31, 0xac, 0x34, 0x0a, 0x3a,
  0x24, 0x47, 0x9d, 0x5b, 0xe9, 0x67, 0x29, 0x67, 0x24, 0x66, 0x26, 0x66,
  0xd2, 0x65, 0x9c, 0x65, 0x38, 0x65, 0x05, 0x65, 0x9f, 0x64, 0x64, 0x64,
  0x12, 0x64, 0xce, 0x63, 0x7c, 0x63, 0x32, 0x63, 0xe6, 0x62, 0x97, 0x62,
  0x49, 0x62, 0x01, 0x62, 0xb3, 0x61, 0x63, 0x61, 0x15, 0x61, 0xc4, 0x60,
  0x75, 0x60, 0x20, 0x60, 0xce, 0x5f, 0x7a, 0x5f, 0x28, 0x5f, 0xd5, 0x5e,
  0x81, 0x5e, 0x2d, 0x5e, 0xd3, 0x5d, 0x80, 0x5d, 0x2e, 0x5d, 0xe6, 0x5c,
  0x1a, 0x5c, 0x16, 0x5a, 0x01, 0x58, 0xb9, 0x56, 0x6d, 0x55, 0xf4, 0x53,
  0x49, 0x52, 0x83, 0x50, 0x87, 0x4e, 0x5f, 0x4c, 0x68, 0x4a, 0x5c, 0x48,
  0x62, 0x46, 0x5a, 0x44, 0xe2, 0x41, 0x08, 0x3f, 0x1c, 0x3c, 0x44, 0x39,
  0x35, 0x36, 0xcb, 0x32, 0xaf, 0x2f, 0xc8, 0x2c, 0xf8, 0x29, 0x55, 0x27,
  0x6a, 0x24, 0x0f, 0x21, 0x5e, 0x1d, 0xc3, 0x19, 0x7b, 0x16, 0x71, 0x13,
  0x6c, 0x10, 0x00, 0x0d, 0xd2, 0x08, 0x7f, 0x04, 0x7a, 0x01, 0x43, 0xff,
  0xb9, 0xfc, 0xfa, 0xf9, 0x3b, 0xf7, 0xcb, 0xf4, 0x2b, 0xf2, 0x02, 0xef,
  0x0c, 0xec, 0x3d, 0xe9, 0x21, 0xe6, 0xa6, 0xe2, 0x8a, 0xdf, 0x00, 0xdd,
  0xbc, 0xda, 0x9e, 0xd8, 0xc1, 0xd6, 0xd6, 0xd4, 0xd6, 0xd2, 0xad, 0xd0,
  0x5f, 0xce, 0xf0, 0xcb, 0xe9, 0xc9, 0x61, 0xc8, 0x75, 0xc7, 0x97, 0xc6,
  0x3e, 0xc5, 0x07, 0xc4, 0x8e, 0xc3, 0x18, 0xc3, 0x3a, 0xc2, 0x15, 0xc1,
  0x0e, 0xc0, 0xb3, 0xbf, 0xcf, 0xbf, 0xf8, 0xbf, 0xcc, 0xbf, 0x72, 0xbf,
  0x41, 0xbf, 0x2b, 0xbf, 0xe2, 0xbe, 0x99, 0xbe, 0x4e, 0xbe, 0x0e, 0xbe,
  0xcd, 0xbd, 0x7c, 0xbd, 0x8a, 0xbd, 0x88, 0xbd, 0x04, 0xbd, 0x0c, 0xbc,
  0xb3, 0xbb, 0xf6, 0xbb, 0xf1, 0xbb, 0x12, 0xbc, 0x6f, 0xbc, 0xcb, 0xbc,
  0xe4, 0xbc, 0x33, 0xbd, 0x1b, 0xbe, 0xac, 0xbe, 0x1e, 0xbf, 0x91, 0xbf,
  0x50, 0xc0, 0x40, 0xc1, 0x3d, 0xc2, 0x32, 0xc3, 0xdf, 0xc3, 0xad, 0xc4,
  0x77, 0xc5, 0xbe, 0xc6, 0xc7, 0xc8, 0x1d, 0xcb, 0x0e, 0xcd, 0x83, 0xce,
  0xf1, 0xcf, 0xb4, 0xd1, 0x7d, 0xd3, 0x86, 0xd5, 0x89, 0xd7, 0xd2, 0xd9,
  0x34, 0xdc, 0x28, 0xde, 0x23, 0xe0, 0x33, 0xe2, 0x0a, 0xe4, 0x59, 0xe5,
  0xfc, 0xe6, 0x98, 0xe9, 0x30, 0xec, 0x91, 0xee, 0xc2, 0xf0, 0x0d, 0xf3,
  0x35, 0xf5, 0xf3, 0xf6, 0xc4, 0xf8, 0xcb, 0xfa, 0xef, 0xfc, 0x65, 0xff,
  0x05, 0x02, 0x7c, 0x04, 0xde, 0x06, 0x75, 0x09, 0x2b, 0x0c, 0x9b, 0x0e,
  0xf3, 0x10, 0xb3, 0x13, 0x3a, 0x16, 0xeb, 0x18, 0x55, 0x1c, 0xad, 0x1f,
  0xa8, 0x22, 0x54, 0x25, 0xae, 0x27, 0x33, 0x2a, 0x16, 0x2d, 0x36, 0x30,
  0x84, 0x33, 0x94, 0x36, 0xbd, 0x38, 0xa2, 0x3a, 0x7d, 0x3c, 0x06, 0x3e,
  0x24, 0x3f, 0x27, 0x40, 0x7c, 0x41, 0xef, 0x42, 0x14, 0x44, 0xeb, 0x44,
  0x06, 0x46, 0x53, 0x47, 0x47, 0x48, 0x9b, 0x48, 0xaf, 0x48, 0xd7, 0x48,
  0x4c, 0x49, 0xa0, 0x49, 0xbe, 0x49, 0xd4, 0x49, 0xfa, 0x49, 0x5e, 0x4a,
  0xcc, 0x4a, 0x14, 0x4b, 0xfe, 0x4a, 0x22, 0x4b, 0x10, 0x4c, 0x0c, 0x4d,
  0xb2, 0x4d, 0x4c, 0x4e, 0x3e, 0x4e, 0x77, 0x4d, 0x98, 0x4c, 0xf6, 0x4b,
  0x67, 0x4b, 0xf0, 0x4a, 0x2a, 0x4a, 0xea, 0x48, 0x06, 0x48, 0x47, 0x47,
  0xb2, 0x46, 0xda, 0x45, 0xad, 0x44, 0x5c, 0x43, 0x43, 0x42, 0x9e, 0x41,
  0x0a, 0x41, 0x49, 0x40, 0xa6, 0x3f, 0x9d, 0x3e, 0x3c, 0x3d, 0xc6, 0x3b,
  0xf6, 0x39, 0x87, 0x37, 0xf6, 0x34, 0x87, 0x32, 0x2b, 0x30, 0x6f, 0x2d,
  0xfa, 0x2a, 0x3d, 0x29, 0x48, 0x27, 0xc2, 0x24, 0x49, 0x22, 0xca, 0x1f,
  0xa0, 0x1c, 0x7c, 0x19, 0x06, 0x17, 0xbf, 0x14, 0x9f, 0x12, 0x96, 0x10,
  0xf9, 0x0d, 0x3e, 0x0b, 0xe8, 0x08, 0x5c, 0x06, 0xe7, 0x02, 0x6e, 0xff,
  0xca, 0xfc, 0x5b, 0xfa, 0xa0, 0xf7, 0xe9, 0xf4, 0x9c, 0xf2, 0x66, 0xf0,
  0xaf, 0xed, 0xfd, 0xea, 0xcc, 0xe8, 0x6e, 0xe6, 0x82, 0xe3, 0x97, 0xe0,
  0xed, 0xdd, 0x62, 0xdb, 0x7b, 0xd8, 0xd3, 0xd5, 0x5f, 0xd3, 0x1a, 0xd1,
  0x44, 0xcf, 0xeb, 0xcd, 0x89, 0xcc, 0xca, 0xca, 0x4d, 0xc9, 0x35, 0xc8,
  0x53, 0xc7, 0x0c, 0xc6, 0x06, 0xc4, 0xca, 0xc1, 0x09, 0xc0, 0x9c, 0xbe,
  0xa8, 0xbd, 0xfd, 0xbc, 0xf2, 0xbb, 0x9b, 0xba, 0x20, 0xb9, 0xe4, 0xb7,
  0xc1, 0xb6, 0xcd, 0xb5, 0x12, 0xb5, 0x55, 0xb4, 0xd1, 0xb3, 0x86, 0xb3,
  0x19, 0xb3, 0xe8, 0xb2, 0xd7, 0xb2, 0x72, 0xb2, 0x27, 0xb2, 0xb7, 0xb1,
  0x67, 0xb1, 0x65, 0xb1, 0xae, 0xb1, 0x6b, 0xb1, 0xf2, 0xb0, 0xeb, 0xb0,
  0x0f, 0xb1, 0xfe, 0xb0, 0xeb, 0xb0, 0xcf, 0xb0, 0x94, 0xb0, 0x3e, 0xb0,
  0x29, 0xb0, 0x56, 0xb0, 0x0c, 0xb0, 0xb7, 0xaf, 0xfb, 0xaf, 0x37, 0xb0,
  0x96, 0xb0, 0x42, 0xb1, 0xe8, 0xb1, 0xb5, 0xb2, 0xc5, 0xb3, 0x93, 0xb4,
  0x93, 0xb4, 0xee, 0xb4, 0x59, 0xb6, 0xca, 0xb7, 0x87, 0xb8, 0x6f, 0xb8,
  0x33, 0xb8, 0xaf, 0xb8, 0x4a, 0xb9, 0x9d, 0xb9, 0xf2, 0xb9, 0x48, 0xba,
  0xd0, 0xba, 0xe5, 0xbb, 0x4e, 0xbd, 0xaf, 0xbe, 0xe9, 0xbf, 0xba, 0xc1,
  0xc2, 0xc3, 0x73, 0xc5, 0xa6, 0xc6, 0x6a, 0xc7, 0x83, 0xc8, 0x42, 0xca,
  0xc8, 0xcb, 0x34, 0xcd, 0x94, 0xce, 0xcc, 0xcf, 0x31, 0xd1, 0x27, 0xd3,
  0x8c, 0xd5, 0x61, 0xd7, 0x78, 0xd9, 0x3b, 0xdc, 0x40, 0xdf, 0xdd, 0xe1,
  0x0c, 0xe4, 0xe4, 0xe5, 0xd0, 0xe7, 0x65, 0xea, 0xc9, 0xec, 0xd7, 0xee,
  0xfc, 0xf0, 0x7c, 0xf3, 0xf6, 0xf5, 0x09, 0xf8, 0xde, 0xf9, 0xca, 0xfb,
  0xac, 0xfd, 0xc3, 0xff, 0x33, 0x02, 0xb1, 0x04, 0x24, 0x07, 0x57, 0x09,
  0x5f, 0x0b, 0xe6, 0x0d, 0xd1, 0x10, 0x6d, 0x13, 0x8f, 0x15, 0xfb, 0x17,
  0x43, 0x1a, 0x8e, 0x1c, 0x1a, 0x1f, 0x69, 0x21, 0x80, 0x23, 0x74, 0x25,
  0x62, 0x27, 0x07, 0x29, 0xa1, 0x2a, 0xa5, 0x2c, 0xdf, 0x2e, 0x57, 0x31,
  0xff, 0x33, 0xd1, 0x36, 0x6e, 0x39, 0x8a, 0x3b, 0x58, 0x3d, 0x32, 0x3f,
  0xc8, 0x40, 0x1b, 0x42, 0x22, 0x43, 0x1a, 0x44, 0x25, 0x45, 0xe5, 0x45,
  0x43, 0x46, 0x9b, 0x46, 0x6a, 0x47, 0x6f, 0x48, 0x69, 0x49, 0x6f, 0x4a,
  0xc7, 0x4b, 0x0e, 0x4d, 0x03, 0x4e, 0x78, 0x4e, 0xdf, 0x4e, 0x0b, 0x4f,
  0xea, 0x4e, 0xcb, 0x4e, 0x1b, 0x4f, 0x6e, 0x4f, 0xc3, 0x4f, 0xdc, 0x4f,
  0xcb, 0x4f, 0xd2, 0x4f, 0x16, 0x50, 0x24, 0x50, 0xf2, 0x4f, 0x00, 0x50,
  0x37, 0x50, 0x4e, 0x50, 0x5e, 0x50, 0x7c, 0x50, 0xab, 0x50, 0x69, 0x50,
  0xad, 0x4f, 0xa3, 0x4e, 0xe6, 0x4d, 0x42, 0x4d, 0xdc, 0x4c, 0x7c, 0x4c,
  0xbe, 0x4b, 0x08, 0x4b, 0x7b, 0x4a, 0xe4, 0x49, 0x14, 0x49, 0x07, 0x48,
  0x98, 0x46, 0x2f, 0x45, 0x16, 0x44, 0x23, 0x43, 0x55, 0x42, 0xac, 0x41,
  0x06, 0x41, 0x4b, 0x40, 0x8f, 0x3f, 0xde, 0x3e, 0xe1, 0x3d, 0x75, 0x3c,
  0xc0, 0x3a, 0xe4, 0x38, 0x83, 0x37, 0x9a, 0x36, 0xe5, 0x35, 0xc0, 0x34,
  0xf9, 0x32, 0xe1, 0x30, 0xfa, 0x2e, 0xd4, 0x2c, 0x4d, 0x2a, 0xed, 0x27,
  0xb9, 0x25, 0xb2, 0x23, 0x7d, 0x21, 0xfd, 0x1e, 0x89, 0x1c, 0x38, 0x1a,
  0xe2, 0x17, 0x67, 0x15, 0xf3, 0x12, 0xb3, 0x10, 0x9e, 0x0e, 0x79, 0x0c,
  0xea, 0x09, 0x54, 0x07, 0x26, 0x05, 0x0a, 0x03, 0xd7, 0x00, 0x98, 0xfe,
  0x41, 0xfc, 0x8d, 0xf9, 0x3c, 0xf7, 0x4f, 0xf5, 0x73, 0xf3, 0x7b, 0xf1,
  0x68, 0xef, 0x6f, 0xed, 0x7a, 0xeb, 0x87, 0xe9, 0x48, 0xe7, 0xc6, 0xe4,
  0x65, 0xe2, 0xf6, 0xdf, 0x86, 0xdd, 0x9f, 0xdb, 0x1b, 0xda, 0xa6, 0xd8,
  0xce, 0xd6, 0xe4, 0xd4, 0xe3, 0xd2, 0x84, 0xd0, 0xec, 0xcd, 0x08, 0xcc,
  0x89, 0xca, 0x1b, 0xc9, 0xe2, 0xc7, 0x9d, 0xc6, 0xe5, 0xc4, 0x79, 0xc3,
  0x6d, 0xc2, 0xe3, 0xc0, 0x3c, 0xbf, 0xd3, 0xbd, 0x41, 0xbc, 0xd2, 0xba,
  0x6a, 0xb9, 0xa1, 0xb7, 0xa9, 0xb5, 0x47, 0xb4, 0x91, 0xb3, 0xd5, 0xb2,
  0xb7, 0xb1, 0x51, 0xb0, 0x14, 0xaf, 0xf5, 0xad, 0x95, 0xac, 0x34, 0xab,
  0x05, 0xaa, 0xe1, 0xa8, 0xb3, 0xa7, 0xd9, 0xa6, 0x25, 0xa6, 0x6e, 0xa5,
  0x2b, 0xa5, 0x7b, 0xa5, 0x9a, 0xa5, 0x88, 0xa5, 0xc3, 0xa5, 0xe7, 0xa5,
  0xaf, 0xa5, 0x8b, 0xa5, 0x80, 0xa5, 0x65, 0xa5, 0x8c, 0xa5, 0x7e, 0xa5,
  0x22, 0xa5, 0x40, 0xa5, 0xed, 0xa5, 0x27, 0xa6, 0x2b, 0xa6, 0x1c, 0xa6,
  0xe5, 0xa5, 0x7b, 0xa5, 0x45, 0xa5, 0x37, 0xa5, 0x04, 0xa5, 0x91, 0xa4,
  0x8d, 0xa4, 0x2d, 0xa5, 0x9f, 0xa5, 0xf6, 0xa5, 0x7e, 0xa6, 0x34, 0xa7,
  0x14, 0xa8, 0x7e, 0xa8, 0x87, 0xa8, 0xc4, 0xa8, 0x51, 0xa9, 0xec, 0xa9,
  0x74, 0xaa, 0xf7, 0xaa, 0x40, 0xab, 0xd9, 0xab, 0xca, 0xac, 0x6b, 0xad,
  0xb3, 0xad, 0x08, 0xae, 0x10, 0xaf, 0x2c, 0xb0, 0xcb, 0xb0, 0x23, 0xb1,
  0xac, 0xb1, 0x0e, 0xb2, 0x42, 0xb2, 0xd7, 0xb2, 0xef, 0xb3, 0x21, 0xb5,
  0x30, 0xb6, 0xe9, 0xb6, 0x54, 0xb7, 0xf0, 0xb7, 0x9e, 0xb8, 0x2e, 0xb9,
  0x03, 0xba, 0x55, 0xbb, 0xdf, 0xbc, 0x5f, 0xbe, 0xb8, 0xbf, 0x01, 0xc1,
  0x83, 0xc2, 0x0c, 0xc4, 0x65, 0xc5, 0x7e, 0xc6, 0x86, 0xc7, 0xba, 0xc8,
  0x11, 0xca, 0x66, 0xcb, 0x28, 0xcd, 0x4d, 0xcf, 0x60, 0xd1, 0x69, 0xd3,
  0x7b, 0xd5, 0x5f, 0xd7, 0xe3, 0xd8, 0xca, 0xda, 0xda, 0xdc, 0xda, 0xde,
  0xab, 0xe0, 0x59, 0xe2, 0x3b, 0xe4, 0x21, 0xe6, 0xfc, 0xe7, 0xcb, 0xe9,
  0xbe, 0xeb, 0x9c, 0xed, 0x47, 0xef, 0x0b, 0xf1, 0xd3, 0xf2, 0xbc, 0xf4,
  0x9c, 0xf6, 0x67, 0xf8, 0x2a, 0xfa, 0xc1, 0xfb, 0x7f, 0xfd, 0x41, 0xff,
  0x12, 0x01, 0xd2, 0x02, 0xfc, 0x04, 0xe8, 0x06, 0x9a, 0x08, 0x59, 0x0a,
  0x48, 0x0c, 0x36, 0x0e, 0x37, 0x10, 0x36, 0x12, 0x59, 0x14, 0x62, 0x16,
  0x86, 0x18, 0xb9, 0x1a, 0xc2, 0x1c, 0xcd, 0x1e, 0xf1, 0x20, 0x27, 0x23,
  0x7a, 0x25, 0xf9, 0x27, 0x2a, 0x2a, 0x1f, 0x2c, 0xf8, 0x2d, 0xa3, 0x2f,
  0x23, 0x31, 0xf4, 0x32, 0x2c, 0x35, 0x40, 0x37, 0x23, 0x39, 0xfe, 0x3a,
  0x11, 0x3d, 0x2c, 0x3f, 0xe8, 0x40, 0x8c, 0x42, 0x55, 0x44, 0x37, 0x46,
  0x99, 0x47, 0xcb, 0x48, 0x12, 0x4a, 0x60, 0x4b, 0x86, 0x4c, 0x9b, 0x4d,
  0xc8, 0x4e, 0xec, 0x4f, 0xe3, 0x50, 0x8a, 0x51, 0x23, 0x52, 0xd8, 0x52,
  0x68, 0x53, 0x9b, 0x53, 0xb1, 0x53, 0x11, 0x54, 0x94, 0x54, 0xf7, 0x54,
  0x4f, 0x55, 0xa4, 0x55, 0x03, 0x56, 0x51, 0x56, 0x92, 0x56, 0xfa, 0x56,
  0x59, 0x57, 0xad, 0x57, 0xcd, 0x57, 0xc5, 0x57, 0xa8, 0x57, 0x64, 0x57,
  0x49, 0x57, 0x63, 0x57, 0x64, 0x57, 0x40, 0x57, 0xf6, 0x56, 0xfc, 0x56,
  0x36, 0x57, 0x3b, 0x57, 0x1e, 0x57, 0x1c, 0x57, 0x03, 0x57, 0xee, 0x56,
  0xa5, 0x56, 0x80, 0x56, 0xd4, 0x56, 0xe4, 0x56, 0x92, 0x56, 0xf0, 0x55,
  0x02, 0x55, 0xab, 0x53, 0xb5, 0x52, 0x51, 0x52, 0x08, 0x52, 0x80, 0x51,
  0xb4, 0x50, 0xde, 0x4f, 0x27, 0x4f, 0x63, 0x4e, 0x58, 0x4d, 0x72, 0x4c,
  0x82, 0x4b, 0x81, 0x4a, 0x87, 0x49, 0xb4, 0x48, 0xb1, 0x47, 0x99, 0x46,
  0xb4, 0x45, 0x34, 0x45, 0xb8, 0x44, 0x2f, 0x44, 0x7f, 0x43, 0xa0, 0x42,
  0xcb, 0x41, 0xd1, 0x40, 0xeb, 0x3f, 0x28, 0x3f, 0x3d, 0x3e, 0x09, 0x3d,
  0x9d, 0x3b, 0x40, 0x3a, 0x1c, 0x39, 0xeb, 0x37, 0xd1, 0x36, 0xb3, 0x35,
  0x8b, 0x34, 0x0b, 0x33, 0x51, 0x31, 0xfb, 0x2f, 0xb9, 0x2e, 0x54, 0x2d,
  0xaf, 0x2b, 0xdf, 0x29, 0xf1, 0x27, 0x3a, 0x26, 0x6f, 0x24, 0x56, 0x22,
  0x20, 0x20, 0x0a, 0x1e, 0x3b, 0x1c, 0x55, 0x1a, 0x6c, 0x18, 0xaa, 0x16,
  0xef, 0x14, 0x0a, 0x13, 0x17, 0x11, 0x1c, 0x0f, 0x22, 0x0d, 0x46, 0x0b,
  0x53, 0x09, 0x46, 0x07, 0x46, 0x05, 0x4d, 0x03, 0x1a, 0x01, 0xd3, 0xfe,
  0x94, 0xfc, 0x8c, 0xfa, 0x8f, 0xf8, 0xc1, 0xf6, 0x27, 0xf5, 0x8f, 0xf3,
  0xf2, 0xf1, 0x74, 0xf0, 0xfa, 0xee, 0x4d, 0xed, 0x9e, 0xeb, 0xa2, 0xe9,
  0x73, 0xe7, 0xa9, 0xe5, 0x2d, 0xe4, 0xa3, 0xe2, 0xf9, 0xe0, 0x50, 0xdf,
  0xb9, 0xdd, 0x34, 0xdc, 0xad, 0xda, 0x0b, 0xd9, 0x77, 0xd7, 0xb9, 0xd5,
  0x37, 0xd4, 0xff, 0xd2, 0x8a, 0xd1, 0x14, 0xd0, 0xba, 0xce, 0x7d, 0xcd,
  0x2c, 0xcc, 0xa1, 0xca, 0x22, 0xc9, 0xa9, 0xc7, 0x0e, 0xc6, 0x66, 0xc4,
  0xfc, 0xc2, 0xa5, 0xc1, 0x4a, 0xc0, 0xff, 0xbe, 0xf7, 0xbd, 0x0e, 0xbd,
  0x1d, 0xbc, 0x16, 0xbb, 0xad, 0xb9, 0x44, 0xb8, 0xd7, 0xb6, 0x9e, 0xb5,
  0x9f, 0xb4, 0x97, 0xb3, 0x72, 0xb2, 0x5f, 0xb1, 0x67, 0xb0, 0x89, 0xaf,
  0x9d, 0xae, 0xbe, 0xad, 0x08, 0xad, 0x5d, 0xac, 0x8b, 0xab, 0xad, 0xaa,
  0xe4, 0xa9, 0x59, 0xa9, 0xc4, 0xa8, 0x02, 0xa8, 0x4d, 0xa7, 0xc2, 0xa6,
  0x4a, 0xa6, 0x09, 0xa6, 0xd2, 0xa5, 0x50, 0xa5, 0xec, 0xa4, 0xc2, 0xa4,
  0xd9, 0xa4, 0xbd, 0xa4, 0x97, 0xa4, 0xa5, 0xa4, 0xbf, 0xa4, 0xb4, 0xa4,
  0x8f, 0xa4, 0x31, 0xa4, 0x39, 0xa4, 0x7d, 0xa4, 0xab, 0xa4, 0xc7, 0xa4,
  0xb3, 0xa4, 0xab, 0xa4, 0xca, 0xa4, 0x05, 0xa5, 0x1e, 0xa5, 0x2a, 0xa5,
  0x2c, 0xa5, 0x18, 0xa5, 0x01, 0xa5, 0x33, 0xa5, 0x8c, 0xa5, 0xb2, 0xa5,
  0x81, 0xa5, 0x5c, 0xa5, 0x6e, 0xa5, 0x79, 0xa5, 0x4e, 0xa5, 0x17, 0xa5,
  0xff, 0xa4, 0x1c, 0xa5, 0x45, 0xa5, 0x8a, 0xa5, 0xbf, 0xa5, 0xdb, 0xa5,
  0x41, 0xa6, 0xfb, 0xa6, 0xc6, 0xa7, 0x86, 0xa8, 0x29, 0xa9, 0x97, 0xa9,
  0x27, 0xaa, 0xd7, 0xaa, 0x6d, 0xab, 0xf4, 0xab, 0x34, 0xac, 0x69, 0xac,
  0xc6, 0xac, 0x37, 0xad, 0xaa, 0xad, 0x34, 0xae, 0xd3, 0xae, 0x81, 0xaf,
  0x16, 0xb0, 0x72, 0xb0, 0xc8, 0xb0, 0x36, 0xb1, 0x91, 0xb1, 0xe0, 0xb1,
  0x3b, 0xb2, 0x8e, 0xb2, 0xe0, 0xb2, 0x2a, 0xb3, 0xab, 0xb3, 0x3a, 0xb4,
  0xf1, 0xb4, 0xc0, 0xb5, 0xae, 0xb6, 0x91, 0xb7, 0x82, 0xb8, 0x98, 0xb9,
  0x8f, 0xba, 0x88, 0xbb, 0x65, 0xbc, 0x20, 0xbd, 0xcb, 0xbd, 0xaf, 0xbe,
  0x94, 0xbf, 0x84, 0xc0, 0x85, 0xc1, 0x7b, 0xc2, 0x67, 0xc3, 0x66, 0xc4,
  0x7f, 0xc5, 0x89, 0xc6, 0x6b, 0xc7, 0x38, 0xc8, 0x29, 0xc9, 0x4a, 0xca,
  0x82, 0xcb, 0xc1, 0xcc, 0x12, 0xce, 0x5e, 0xcf, 0xb0, 0xd0, 0x25, 0xd2,
  0x66, 0xd3, 0xde, 0xd4, 0x6c, 0xd6, 0x28, 0xd8, 0xd7, 0xd9, 0x80, 0xdb,
  0xf7, 0xdc, 0x6f, 0xde, 0xd0, 0xdf, 0x22, 0xe1, 0x7a, 0xe2, 0xe5, 0xe3,
  0x7a, 0xe5, 0xea, 0xe6, 0x51, 0xe8, 0xeb, 0xe9, 0x70, 0xeb, 0xb3, 0xec,
  0x01, 0xee, 0x60, 0xef, 0xcb, 0xf0, 0x1d, 0xf2, 0x7b, 0xf3, 0xcf, 0xf4,
  0x39, 0xf6, 0xc8, 0xf7, 0x82, 0xf9, 0x4a, 0xfb, 0xf2, 0xfc, 0x8c, 0xfe,
  0x01, 0x00, 0x68, 0x01, 0x15, 0x03, 0xe6, 0x04, 0xbc, 0x06, 0x42, 0x08,
  0xbb, 0x09, 0x2b, 0x0b, 0xc7, 0x0c, 0x35, 0x0e, 0x94, 0x0f, 0xf2, 0x10,
  0x59, 0x12, 0xbc, 0x13, 0x34, 0x15, 0xac, 0x16, 0x1c, 0x18, 0x79, 0x19,
  0xdc, 0x1a, 0x6d, 0x1c, 0x21, 0x1e, 0xca, 0x1f, 0x70, 0x21, 0x02, 0x23,
  0x6c, 0x24, 0x9e, 0x25, 0x45, 0x27, 0x02, 0x29, 0x95, 0x2a, 0xff, 0x2b,
  0x72, 0x2d, 0xe8, 0x2e, 0x48, 0x30, 0x7c, 0x31, 0xd2, 0x32, 0x3d, 0x34,
  0xac, 0x35, 0x26, 0x37, 0xbc, 0x38, 0x4c, 0x3a, 0x90, 0x3b, 0xef, 0x3c,
  0x61, 0x3e, 0xe8, 0x3f, 0x59, 0x41, 0xab, 0x42, 0xf7, 0x43, 0x2d, 0x45,
  0x6c, 0x46, 0x78, 0x47, 0xb4, 0x48, 0x2e, 0x4a, 0x8e, 0x4b, 0xd7, 0x4c,
  0xf4, 0x4d, 0xee, 0x4e, 0xcb, 0x4f, 0xc3, 0x50, 0xc2, 0x51, 0xac, 0x52,
  0x61, 0x53, 0xf3, 0x53, 0xac, 0x54, 0x5f, 0x55, 0xf4, 0x55, 0x7f, 0x56,
  0x04, 0x57, 0x9c, 0x57, 0x1f, 0x58, 0x7c, 0x58, 0xbc, 0x58, 0x0b, 0x59,
  0x71, 0x59, 0xc1, 0x59, 0x0b, 0x5a, 0x3e, 0x5a, 0x8b, 0x5a, 0xac, 0x5a,
  0xa9, 0x5a, 0x8a, 0x5a, 0x6c, 0x5a, 0x4c, 0x5a, 0x39, 0x5a, 0x32, 0x5a,
  0x40, 0x5a, 0x33, 0x5a, 0x44, 0x5a, 0x58, 0x5a, 0x79, 0x5a, 0x7a, 0x5a,
  0x69, 0x5a, 0x49, 0x5a, 0x54, 0x5a, 0x78, 0x5a, 0x72, 0x5a, 0x5f, 0x5a,
  0x31, 0x5a, 0x11, 0x5a, 0xf9, 0x59, 0xea, 0x59, 0xd1, 0x59, 0xa1, 0x59,
  0x48, 0x59, 0xf0, 0x58, 0xa2, 0x58, 0x6e, 0x58, 0x60, 0x58, 0x46, 0x58,
  0x3f, 0x58, 0x47, 0x58, 0x40, 0x58, 0x08, 0x58, 0x9c, 0x57, 0x11, 0x57,
  0x83, 0x56, 0xd7, 0x55, 0x42, 0x55, 0xab, 0x54, 0x06, 0x54, 0x63, 0x53,
  0xa6, 0x52, 0xf0, 0x51, 0x55, 0x51, 0xbc, 0x50, 0x1e, 0x50, 0x8e, 0x4f,
  0xf3, 0x4e, 0x34, 0x4e, 0x6b, 0x4d, 0xc9, 0x4c, 0x30, 0x4c, 0x7d, 0x4b,
  0xd8, 0x4a, 0x49, 0x4a, 0xad, 0x49, 0x11, 0x49, 0x6c, 0x48, 0xa4, 0x47,
  0xec, 0x46, 0x6c, 0x46, 0xde, 0x45, 0x5d, 0x45, 0xbc, 0x44, 0xf1, 0x43,
  0x34, 0x43, 0x8c, 0x42, 0xf8, 0x41, 0x47, 0x41, 0x6a, 0x40, 0x72, 0x3f,
  0x8b, 0x3e, 0xb3, 0x3d, 0xf2, 0x3c, 0x2f, 0x3c, 0x5c, 0x3b, 0x96, 0x3a,
  0xbc, 0x39, 0xb2, 0x38, 0x91, 0x37, 0x8c, 0x36, 0x75, 0x35, 0x63, 0x34,
  0x5e, 0x33, 0x63, 0x32, 0x4b, 0x31, 0x3c, 0x30, 0x1e, 0x2f, 0x07, 0x2e,
  0xd0, 0x2c, 0xb2, 0x2b, 0x84, 0x2a, 0x4d, 0x29, 0xfc, 0x27, 0xa5, 0x26,
  0x2f, 0x25, 0x95, 0x23, 0x1f, 0x22, 0xad, 0x20, 0x3a, 0x1f, 0x9e, 0x1d,
  0x10, 0x1c, 0x6e, 0x1a, 0xe3, 0x18, 0x56, 0x17, 0xdb, 0x15, 0x58, 0x14,
  0xba, 0x12, 0x2a, 0x11, 0x8b, 0x0f, 0xf4, 0x0d, 0x5b, 0x0c, 0xd3, 0x0a,
  0x5f, 0x09, 0xe5, 0x07, 0x4a, 0x06, 0xc0, 0x04, 0x3e, 0x03, 0xb6, 0x01,
  0x37, 0x00, 0xca, 0xfe, 0x74, 0xfd, 0x00, 0xfc, 0xa5, 0xfa, 0x4a, 0xf9,
  0xd1, 0xf7, 0x7e, 0xf6, 0x1c, 0xf5, 0x9d, 0xf3, 0x4f, 0xf2, 0x0e, 0xf1,
  0xc0, 0xef, 0x3a, 0xee, 0xd8, 0xec, 0x69, 0xeb, 0x0c, 0xea, 0xad, 0xe8,
  0x56, 0xe7, 0xf8, 0xe5, 0xbe, 0xe4, 0x98, 0xe3, 0x61, 0xe2, 0x03, 0xe1,
  0x99, 0xdf, 0x4d, 0xde, 0x0e, 0xdd, 0xde, 0xdb, 0xad, 0xda, 0x84, 0xd9,
  0x62, 0xd8, 0x32, 0xd7, 0x01, 0xd6, 0xb7, 0xd4, 0x87, 0xd3, 0x97, 0xd2,
  0x89, 0xd1, 0x89, 0xd0, 0x6e, 0xcf, 0x65, 0xce, 0x5d, 0xcd, 0x44, 0xcc,
  0x13, 0xcb, 0xf2, 0xc9, 0xd3, 0xc8, 0xad, 0xc7, 0x9b, 0xc6, 0x94, 0xc5,
  0x8e, 0xc4, 0x7b, 0xc3, 0x92, 0xc2, 0xad, 0xc1, 0xb2, 0xc0, 0xb9, 0xbf,
  0xb3, 0xbe, 0xc0, 0xbd, 0xd5, 0xbc, 0xf2, 0xbb, 0xe1, 0xba, 0xc7, 0xb9,
  0xc7, 0xb8, 0xe8, 0xb7, 0x09, 0xb7, 0x2c, 0xb6, 0x66, 0xb5, 0xb1, 0xb4,
  0xf6, 0xb3, 0x32, 0xb3, 0x53, 0xb2, 0x92, 0xb1, 0xdb, 0xb0, 0x0d, 0xb0,
  0x46, 0xaf, 0x62, 0xae, 0x98, 0xad, 0xd1, 0xac, 0x3f, 0xac, 0xb0, 0xab,
  0x34, 0xab, 0xb1, 0xaa, 0x45, 0xaa, 0xda, 0xa9, 0x6a, 0xa9, 0xf1, 0xa8,
  0x7d, 0xa8, 0x04, 0xa8, 0xbb, 0xa7, 0x80, 0xa7, 0x41, 0xa7, 0x02, 0xa7,
  0xb6, 0xa6, 0x68, 0xa6, 0x3b, 0xa6, 0x1c, 0xa6, 0x0d, 0xa6, 0xfc, 0xa5,
  0xd4, 0xa5, 0xa6, 0xa5, 0x81, 0xa5, 0x76, 0xa5, 0x61, 0xa5, 0x54, 0xa5,
  0x48, 0xa5, 0x47, 0xa5, 0x4b, 0xa5, 0x49, 0xa5, 0x38, 0xa5, 0x4b, 0xa5,
  0x6c, 0xa5, 0xa3, 0xa5, 0xda, 0xa5, 0xfe, 0xa5, 0x21, 0xa6, 0x4a, 0xa6,
  0x8a, 0xa6, 0xb4, 0xa6, 0xe0, 0xa6, 0xf3, 0xa6, 0x1b, 0xa7, 0x17, 0xa7,
  0x40, 0xa7, 0x8e, 0xa7, 0xcd, 0xa7, 0x06, 0xa8, 0x3e, 0xa8, 0x86, 0xa8,
  0xbd, 0xa8, 0xde, 0xa8, 0xf6, 0xa8, 0x1c, 0xa9, 0x49, 0xa9, 0x5c, 0xa9,
  0x79, 0xa9, 0x93, 0xa9, 0xab, 0xa9, 0xe6, 0xa9, 0x2c, 0xaa, 0x82, 0xaa,
  0xbc, 0xaa, 0xec, 0xaa, 0x32, 0xab, 0x69, 0xab, 0xa7, 0xab, 0xde, 0xab,
  0x19, 0xac, 0x53, 0xac, 0xbb, 0xac, 0x14, 0xad, 0x8a, 0xad, 0xeb, 0xad,
  0x6c, 0xae, 0xe1, 0xae, 0x62, 0xaf, 0xd9, 0xaf, 0x58, 0xb0, 0xc0, 0xb0,
  0x36, 0xb1, 0x99, 0xb1, 0x11, 0xb2, 0x71, 0xb2, 0xdc, 0xb2, 0x58, 0xb3,
  0xbb, 0xb3, 0x37, 0xb4, 0xb6, 0xb4, 0x46, 0xb5, 0xca, 0xb5, 0x50, 0xb6,
  0xd9, 0xb6, 0x6c, 0xb7, 0x03, 0xb8, 0x7e, 0xb8, 0xf0, 0xb8, 0x64, 0xb9,
  0xcf, 0xb9, 0x47, 0xba, 0xd3, 0xba, 0x60, 0xbb, 0xe8, 0xbb, 0x70, 0xbc,
  0xf3, 0xbc, 0x82, 0xbd, 0x05, 0xbe, 0xa1, 0xbe, 0x2a, 0xbf, 0xae, 0xbf,
  0x25, 0xc0, 0xac, 0xc0, 0x52, 0xc1, 0xde, 0xc1, 0x75, 0xc2, 0x11, 0xc3,
  0xb5, 0xc3, 0x42, 0xc4, 0xc5, 0xc4, 0x6d, 0xc5, 0x0b, 0xc6, 0xb2, 0xc6,
  0x4b, 0xc7, 0xf6, 0xc7, 0x9b, 0xc8, 0x4f, 0xc9, 0x17, 0xca, 0xf3, 0xca,
  0xc6, 0xcb, 0x9a, 0xcc, 0x7b, 0xcd, 0x54, 0xce, 0x32, 0xcf, 0x09, 0xd0,
  0xd7, 0xd0, 0xc9, 0xd1, 0xc1, 0xd2, 0xb8, 0xd3, 0xb6, 0xd4, 0xae, 0xd5,
  0xbc, 0xd6, 0xca, 0xd7, 0xd2, 0xd8, 0xca, 0xd9, 0xbd, 0xda, 0xc8, 0xdb,
  0xd8, 0xdc, 0xf9, 0xdd, 0x08, 0xdf, 0x1e, 0xe0, 0x39, 0xe1, 0x4e, 0xe2,
  0x60, 0xe3, 0x7a, 0xe4, 0x96, 0xe5, 0xc1, 0xe6, 0xde, 0xe7, 0x0e, 0xe9,
  0x2a, 0xea, 0x42, 0xeb, 0x7a, 0xec, 0x9f, 0xed, 0xbb, 0xee, 0xd6, 0xef,
  0xef, 0xf0, 0x0e, 0xf2, 0x30, 0xf3, 0x55, 0xf4, 0x68, 0xf5, 0x7e, 0xf6,
  0xa4, 0xf7, 0xc0, 0xf8, 0xea, 0xf9, 0x10, 0xfb, 0x41, 0xfc, 0x7d, 0xfd,
  0xad, 0xfe, 0xe3, 0xff, 0x10, 0x01, 0x27, 0x02, 0x58, 0x03, 0x74, 0x04,
  0xa4, 0x05, 0xd5, 0x06, 0x0b, 0x08, 0x36, 0x09, 0x5c, 0x0a, 0x76, 0x0b,
  0x9a, 0x0c, 0xc1, 0x0d, 0xd8, 0x0e, 0xf5, 0x0f, 0x15, 0x11, 0x3f, 0x12,
  0x70, 0x13, 0x8e, 0x14, 0xcc, 0x15, 0xf2, 0x16, 0x31, 0x18, 0x4b, 0x19,
  0x82, 0x1a, 0xa1, 0x1b, 0xbc, 0x1c, 0xde, 0x1d, 0xe9, 0x1e, 0x13, 0x20,
  0x33, 0x21, 0x6c, 0x22, 0x80, 0x23, 0xa7, 0x24, 0xbc, 0x25, 0xde, 0x26,
  0xe7, 0x27, 0x07, 0x29, 0x28, 0x2a, 0x36, 0x2b, 0x4d, 0x2c, 0x71, 0x2d,
  0x91, 0x2e, 0xbe, 0x2f, 0xd6, 0x30, 0xfb, 0x31, 0x20, 0x33, 0x46, 0x34,
  0x59, 0x35, 0x7a, 0x36, 0xa1, 0x37, 0xc2, 0x38, 0xdf, 0x39, 0x01, 0x3b,
  0x0f, 0x3c, 0x27, 0x3d, 0x1f, 0x3e, 0x2e, 0x3f, 0x43, 0x40, 0x5b, 0x41,
  0x73, 0x42, 0x8e, 0x43, 0xaf, 0x44, 0xbc, 0x45, 0xcc, 0x46, 0xd8, 0x47,
  0xec, 0x48, 0xe1, 0x49, 0xd1, 0x4a, 0xc4, 0x4b, 0xb6, 0x4c, 0xa9, 0x4d,
  0x9d, 0x4e, 0x88, 0x4f, 0x75, 0x50, 0x4c, 0x51, 0x3b, 0x52, 0x05, 0x53,
  0xd9, 0x53, 0xa1, 0x54, 0x6f, 0x55, 0x33, 0x56, 0xe5, 0x56, 0x7d, 0x57,
  0x15, 0x58, 0xb3, 0x58, 0x4e, 0x59, 0xdb, 0x59, 0x55, 0x5a, 0xc8, 0x5a,
  0x36, 0x5b, 0xa9, 0x5b, 0x09, 0x5c, 0x5a, 0x5c, 0xab, 0x5c, 0xf1, 0x5c,
  0x34, 0x5d, 0x6a, 0x5d, 0x9e, 0x5d, 0xc1, 0x5d, 0xea, 0x5d, 0x0b, 0x5e,
  0x28, 0x5e, 0x3d, 0x5e, 0x41, 0x5e, 0x51, 0x5e, 0x5f, 0x5e, 0x69, 0x5e,
  0x79, 0x5e, 0x6a, 0x5e, 0x78, 0x5e, 0x76, 0x5e, 0x7f, 0x5e, 0x72, 0x5e,
  0x70, 0x5e, 0x61, 0x5e, 0x61, 0x5e, 0x42, 0x5e, 0x36, 0x5e, 0x2f, 0x5e,
  0x2c, 0x5e, 0x1a, 0x5e, 0xec, 0x5d, 0xbf, 0x5d, 0x94, 0x5d, 0x71, 0x5d,
  0x4c, 0x5d, 0x24, 0x5d, 0xfa, 0x5c, 0xc3, 0x5c, 0x8c, 0x5c, 0x59, 0x5c,
  0x3e, 0x5c, 0x0c, 0x5c, 0xd3, 0x5b, 0xb1, 0x5b, 0x78, 0x5b, 0x45, 0x5b,
  0x14, 0x5b, 0xed, 0x5a, 0xc0, 0x5a, 0x89, 0x5a, 0x4f, 0x5a, 0x16, 0x5a,
  0xe5, 0x59, 0xb5, 0x59, 0x84, 0x59, 0x4e, 0x59, 0x0d, 0x59, 0xcf, 0x58,
  0xa6, 0x58, 0x79, 0x58, 0x3e, 0x58, 0xf8, 0x57, 0xc0, 0x57, 0x83, 0x57,
  0x38, 0x57, 0x03, 0x57, 0xc1, 0x56, 0x65, 0x56, 0xe0, 0x55, 0x52, 0x55,
  0xd0, 0x54, 0x4c, 0x54, 0xbe, 0x53, 0x32, 0x53, 0xa7, 0x52, 0x2a, 0x52,
  0x95, 0x51, 0x09, 0x51, 0x76, 0x50, 0xe9, 0x4f, 0x69, 0x4f, 0xe2, 0x4e,
  0x49, 0x4e, 0xc0, 0x4d, 0x38, 0x4d, 0xbd, 0x4c, 0x2d, 0x4c, 0x98, 0x4b,
  0x07, 0x4b, 0x76, 0x4a, 0xdc, 0x49, 0x4a, 0x49, 0xb9, 0x48, 0x2d, 0x48,
  0x9c, 0x47, 0x19, 0x47, 0x80, 0x46, 0xf9, 0x45, 0x66, 0x45, 0xd6, 0x44,
  0x47, 0x44, 0xb5, 0x43, 0x14, 0x43, 0x74, 0x42, 0xd4, 0x41, 0x42, 0x41,
  0xa4, 0x40, 0x18, 0x40, 0x85, 0x3f, 0xeb, 0x3e, 0x52, 0x3e, 0xb2, 0x3d,
  0x0c, 0x3d, 0x7a, 0x3c, 0xdb, 0x3b, 0x3a, 0x3b, 0x87, 0x3a, 0xe0, 0x39,
  0x29, 0x39, 0x88, 0x38, 0xd1, 0x37, 0x16, 0x37, 0x4d, 0x36, 0x9e, 0x35,
  0xdc, 0x34, 0x19, 0x34, 0x4a, 0x33, 0x83, 0x32, 0xb7, 0x31, 0xec, 0x30,
  0x1c, 0x30, 0x47, 0x2f, 0x71, 0x2e, 0x88, 0x2d, 0xa6, 0x2c, 0xbb, 0x2b,
  0xd0, 0x2a, 0xd8, 0x29, 0xdb, 0x28, 0xe3, 0x27, 0xf5, 0x26, 0xf1, 0x25,
  0xf8, 0x24, 0xdc, 0x23, 0xdb, 0x22, 0xca, 0x21, 0xa7, 0x20, 0x85, 0x1f,
  0x5a, 0x1e, 0x30, 0x1d, 0x0a, 0x1c, 0xd5, 0x1a, 0xa6, 0x19, 0x6d, 0x18,
  0x3d, 0x17, 0xff, 0x15, 0xbf, 0x14, 0x81, 0x13, 0x57, 0x12, 0x15, 0x11,
  0xe0, 0x0f, 0xa2, 0x0e, 0x65, 0x0d, 0x27, 0x0c, 0xf1, 0x0a, 0xb2, 0x09,
  0x77, 0x08, 0x4b, 0x07, 0x0a, 0x06, 0xc9, 0x04, 0x97, 0x03, 0x65, 0x02,
  0x3b, 0x01, 0x03, 0x00, 0xd2, 0xfe, 0x95, 0xfd, 0x6f, 0xfc, 0x50, 0xfb,
  0x19, 0xfa, 0xf0, 0xf8, 0xb8, 0xf7, 0x97, 0xf6, 0x65, 0xf5, 0x48, 0xf4,
  0x24, 0xf3, 0xfe, 0xf1, 0xec, 0xf0, 0xd2, 0xef, 0xa9, 0xee, 0x95, 0xed,
  0x76, 0xec, 0x64, 0xeb, 0x61, 0xea, 0x43, 0xe9, 0x3a, 0xe8, 0x21, 0xe7,
  0x14, 0xe6, 0xff, 0xe4, 0x06, 0xe4, 0x04, 0xe3, 0x04, 0xe2, 0x01, 0xe1,
  0x02, 0xe0, 0x04, 0xdf, 0x16, 0xde, 0x1f, 0xdd, 0x29, 0xdc, 0x2d, 0xdb,
  0x3d, 0xda, 0x45, 0xd9, 0x5a, 0xd8, 0x6f, 0xd7, 0x74, 0xd6, 0x8a, 0xd5,
  0x91, 0xd4, 0xa5, 0xd3, 0xbb, 0xd2, 0xdb, 0xd1, 0x01, 0xd1, 0x17, 0xd0,
  0x42, 0xcf, 0x5e, 0xce, 0x94, 0xcd, 0xb1, 0xcc, 0xe8, 0xcb, 0x04, 0xcb,
  0x29, 0xca, 0x4e, 0xc9, 0x78, 0xc8, 0x99, 0xc7, 0xc4, 0xc6, 0xff, 0xc5,
  0x3d, 0xc5, 0x76, 0xc4, 0xa6, 0xc3, 0xe4, 0xc2, 0x17, 0xc2, 0x4e, 0xc1,
  0x8a, 0xc0, 0xc5, 0xbf, 0x0b, 0xbf, 0x42, 0xbe, 0x8e, 0xbd, 0xd8, 0xbc,
  0x20, 0xbc, 0x60, 0xbb, 0xbb, 0xba, 0x08, 0xba, 0x5a, 0xb9, 0xba, 0xb8,
  0x08, 0xb8, 0x64, 0xb7, 0xb1, 0xb6, 0x0c, 0xb6, 0x5d, 0xb5, 0xbf, 0xb4,
  0x10, 0xb4, 0x7b, 0xb3, 0xd0, 0xb2, 0x3d, 0xb2, 0xa1, 0xb1, 0x0f, 0xb1,
  0x84, 0xb0, 0xf2, 0xaf, 0x63, 0xaf, 0xd3, 0xae, 0x50, 0xae, 0xbd, 0xad,
  0x49, 0xad, 0xc1, 0xac, 0x4d, 0xac, 0xc6, 0xab, 0x56, 0xab, 0xe4, 0xaa,
  0x83, 0xaa, 0x0d, 0xaa, 0xac, 0xa9, 0x49, 0xa9, 0xf3, 0xa8, 0x9e, 0xa8,
  0x4e, 0xa8, 0x07, 0xa8, 0xc5, 0xa7, 0x8d, 0xa7, 0x56, 0xa7, 0x1d, 0xa7,
  0xf6, 0xa6, 0xc7, 0xa6, 0xad, 0xa6, 0x8c, 0xa6, 0x74, 0xa6, 0x58, 0xa6,
  0x47, 0xa6, 0x2d, 0xa6, 0x24, 0xa6, 0x23, 0xa6, 0x21, 0xa6, 0x19, 0xa6,
  0x20, 0xa6, 0x1f, 0xa6, 0x28, 0xa6, 0x3c, 0xa6, 0x4e, 0xa6, 0x59, 0xa6,
  0x6e, 0xa6, 0x84, 0xa6, 0x9a, 0xa6, 0xb7, 0xa6, 0xd4, 0xa6, 0xef, 0xa6,
  0x0a, 0xa7, 0x2f, 0xa7, 0x53, 0xa7, 0x77, 0xa7, 0x9c, 0xa7, 0xbc, 0xa7,
  0xe5, 0xa7, 0x10, 0xa8, 0x35, 0xa8, 0x6d, 0xa8, 0x9b, 0xa8, 0xcb, 0xa8,
  0xf5, 0xa8, 0x23, 0xa9, 0x5a, 0xa9, 0x80, 0xa9, 0xb7, 0xa9, 0xee, 0xa9,
  0x1c, 0xaa, 0x46, 0xaa, 0x78, 0xaa, 0xb2, 0xaa, 0xeb, 0xaa, 0x1a, 0xab,
  0x57, 0xab, 0x84, 0xab, 0xba, 0xab, 0xef, 0xab, 0x22, 0xac, 0x55, 0xac,
  0x88, 0xac, 0xc1, 0xac, 0xf5, 0xac, 0x2a, 0xad, 0x67, 0xad, 0x9f, 0xad,
  0xe0, 0xad, 0x14, 0xae, 0x4c, 0xae, 0x82, 0xae, 0xb8, 0xae, 0xf8, 0xae,
  0x2c, 0xaf, 0x6a, 0xaf, 0x9d, 0xaf, 0xd4, 0xaf, 0x05, 0xb0, 0x38, 0xb0,
  0x77, 0xb0, 0xb2, 0xb0, 0xf8, 0xb0, 0x3c, 0xb1, 0x97, 0xb1, 0x05, 0xb2,
  0x6f, 0xb2, 0xdf, 0xb2, 0x48, 0xb3, 0xb2, 0xb3, 0x16, 0xb4, 0x7b, 0xb4,
  0xe5, 0xb4, 0x54, 0xb5, 0xc3, 0xb5, 0x26, 0xb6, 0x97, 0xb6, 0x03, 0xb7,
  0x68, 0xb7, 0xce, 0xb7, 0x44, 0xb8, 0xae, 0xb8, 0x20, 0xb9, 0x87, 0xb9,
  0xf1, 0xb9, 0x5d, 0xba, 0xc6, 0xba, 0x24, 0xbb, 0xa1, 0xbb, 0x05, 0xbc,
  0x79, 0xbc, 0xdc, 0xbc, 0x4b, 0xbd, 0xba, 0xbd, 0x24, 0xbe, 0x9b, 0xbe,
  0x07, 0xbf, 0x72, 0xbf, 0xe5, 0xbf, 0x46, 0xc0, 0xc3, 0xc0, 0x1f, 0xc1,
  0x91, 0xc1, 0x01, 0xc2, 0x79, 0xc2, 0xf1, 0xc2, 0x66, 0xc3, 0xe5, 0xc3,
  0x55, 0xc4, 0xc8, 0xc4, 0x3e, 0xc5, 0xb3, 0xc5, 0x2d, 0xc6, 0xa7, 0xc6,
  0x27, 0xc7, 0x97, 0xc7, 0x1f, 0xc8, 0xa0, 0xc8, 0x29, 0xc9, 0xad, 0xc9,
  0x36, 0xca, 0xad, 0xca, 0x3d, 0xcb, 0xc0, 0xcb, 0x47, 0xcc, 0xe9, 0xcc,
  0x6f, 0xcd, 0x06, 0xce, 0x84, 0xce, 0x1a, 0xcf, 0xbb, 0xcf, 0x55, 0xd0,
  0xf6, 0xd0, 0x8b, 0xd1, 0x30, 0xd2, 0xd2, 0xd2, 0x79, 0xd3, 0x1c, 0xd4,
  0xcf, 0xd4, 0x83, 0xd5, 0x36, 0xd6, 0xe1, 0xd6, 0x96, 0xd7, 0x5f, 0xd8,
  0x1c, 0xd9, 0xe2, 0xd9, 0xa8, 0xda, 0x6c, 0xdb, 0x43, 0xdc, 0x04, 0xdd,
  0xd8, 0xdd, 0xac, 0xde, 0x8a, 0xdf, 0x5b, 0xe0, 0x39, 0xe1, 0x0b, 0xe2,
  0xf5, 0xe2, 0xde, 0xe3, 0xbc, 0xe4, 0x9e, 0xe5, 0x83, 0xe6, 0x62, 0xe7,
  0x51, 0xe8, 0x34, 0xe9, 0x1e, 0xea, 0x12, 0xeb, 0xf7, 0xeb, 0xe5, 0xec,
  0xc5, 0xed, 0xc2, 0xee, 0xb1, 0xef, 0xa6, 0xf0, 0x90, 0xf1, 0x7e, 0xf2,
  0x6f, 0xf3, 0x56, 0xf4, 0x48, 0xf5, 0x35, 0xf6, 0x29, 0xf7, 0x17, 0xf8,
  0x0a, 0xf9, 0x00, 0xfa, 0xe7, 0xfa, 0xde, 0xfb, 0xc9, 0xfc, 0xc1, 0xfd,
  0xac, 0xfe, 0xae, 0xff, 0xa2, 0x00, 0x8d, 0x01, 0x70, 0x02, 0x62, 0x03,
  0x4e, 0x04, 0x3a, 0x05, 0x26, 0x06, 0x16, 0x07, 0xff, 0x07, 0xf3, 0x08,
  0xda, 0x09, 0xbe, 0x0a, 0xb9, 0x0b, 0xa1, 0x0c, 0x9b, 0x0d, 0x7d, 0x0e,
  0x66, 0x0f, 0x54, 0x10, 0x36, 0x11, 0x20, 0x12, 0x11, 0x13, 0xf3, 0x13,
  0xea, 0x14, 0xd1, 0x15, 0xb6, 0x16, 0x9b, 0x17, 0x83, 0x18, 0x7f, 0x19,
  0x58, 0x1a, 0x53, 0x1b, 0x37, 0x1c, 0x22, 0x1d, 0x0a, 0x1e, 0xf3, 0x1e,
  0xda, 0x1f, 0xca, 0x20, 0xb1, 0x21, 0x96, 0x22, 0x70, 0x23, 0x55, 0x24,
  0x39, 0x25, 0x39, 0x26, 0x1a, 0x27, 0x07, 0x28, 0xe1, 0x28, 0xc5, 0x29,
  0xa7, 0x2a, 0x85, 0x2b, 0x70, 0x2c, 0x4e, 0x2d, 0x3e, 0x2e, 0x19, 0x2f,
  0xf3, 0x2f, 0xdf, 0x30, 0xc0, 0x31, 0xaf, 0x32, 0x93, 0x33, 0x77, 0x34,
  0x48, 0x35, 0x2a, 0x36, 0x0f, 0x37, 0xf8, 0x37, 0xd9, 0x38, 0xbe, 0x39,
  0x8b, 0x3a, 0x74, 0x3b, 0x48, 0x3c, 0x2d, 0x3d, 0xfa, 0x3d, 0xe7, 0x3e,
  0xb0, 0x3f, 0x98, 0x40, 0x67, 0x41, 0x3a, 0x42, 0x16, 0x43, 0xf4, 0x43,
  0xd8, 0x44, 0xa4, 0x45, 0x7c, 0x46, 0x4c, 0x47, 0x2a, 0x48, 0x06, 0x49,
  0xce, 0x49, 0xa5, 0x4a, 0x77, 0x4b, 0x4e, 0x4c, 0x13, 0x4d, 0xdf, 0x4d,
  0xaa, 0x4e, 0x74, 0x4f, 0x3d, 0x50, 0xfc, 0x50, 0xbd, 0x51, 0x8c, 0x52,
  0x3e, 0x53, 0xfe, 0x53, 0x99, 0x54, 0x57, 0x55, 0xfc, 0x55, 0xab, 0x56,
  0x45, 0x57, 0xd6, 0x57, 0x87, 0x58, 0x2f, 0x59, 0xbf, 0x59, 0x48, 0x5a,
  0xc7, 0x5a, 0x54, 0x5b, 0xc8, 0x5b, 0x45, 0x5c, 0xb4, 0x5c, 0x25, 0x5d,
  0x7e, 0x5d, 0xda, 0x5d, 0x36, 0x5e, 0xa5, 0x5e, 0xe9, 0x5e, 0x37, 0x5f,
  0x72, 0x5f, 0xab, 0x5f, 0xe5, 0x5f, 0x1d, 0x60, 0x4f, 0x60, 0x81, 0x60,
  0x9c, 0x60, 0xc3, 0x60, 0xdc, 0x60, 0xf7, 0x60, 0x04, 0x61, 0x16, 0x61,
  0x1b, 0x61, 0x21, 0x61, 0x20, 0x61, 0x0d, 0x61, 0x0b, 0x61, 0xf0, 0x60,
  0xe5, 0x60, 0xc7, 0x60, 0xb3, 0x60, 0x98, 0x60, 0x7c, 0x60, 0x59, 0x60,
  0x4c, 0x60, 0x24, 0x60, 0x0d, 0x60, 0xdf, 0x5f, 0xb7, 0x5f, 0x87, 0x5f,
  0x6b, 0x5f, 0x44, 0x5f, 0x28, 0x5f, 0xf5, 0x5e, 0xbd, 0x5e, 0x8f, 0x5e,
  0x5c, 0x5e, 0x2a, 0x5e, 0xfe, 0x5d, 0xcd, 0x5d, 0xa4, 0x5d, 0x75, 0x5d,
  0x4d, 0x5d, 0x11, 0x5d, 0xdf, 0x5c, 0xa9, 0x5c, 0x78, 0x5c, 0x42, 0x5c,
  0x13, 0x5c, 0xde, 0x5b, 0xaf, 0x5b, 0x75, 0x5b, 0x3e, 0x5b, 0xfd, 0x5a,
  0xbc, 0x5a, 0x90, 0x5a, 0x54, 0x5a, 0x1f, 0x5a, 0xe5, 0x59, 0xc2, 0x59,
  0x80, 0x59, 0x39, 0x59, 0x0a, 0x59, 0xd6, 0x58, 0x96, 0x58, 0x55, 0x58,
  0x1e, 0x58, 0xf1, 0x57, 0xc8, 0x57, 0x7e, 0x57, 0x44, 0x57, 0x11, 0x57,
  0xd4, 0x56, 0x93, 0x56, 0x4c, 0x56, 0x11, 0x56, 0xd7, 0x55, 0xae, 0x55,
  0x72, 0x55, 0x3c, 0x55, 0x0c, 0x55, 0xd1, 0x54, 0x98, 0x54, 0x63, 0x54,
  0x2b, 0x54, 0xf8, 0x53, 0xc6, 0x53, 0x81, 0x53, 0x4b, 0x53, 0x19, 0x53,
  0xea, 0x52, 0xb2, 0x52, 0x7e, 0x52, 0x30, 0x52, 0xc5, 0x51, 0x46, 0x51,
  0xca, 0x50, 0x55, 0x50, 0xd7, 0x4f, 0x5d, 0x4f, 0xe4, 0x4e, 0x79, 0x4e,
  0xfb, 0x4d, 0x83, 0x4d, 0x02, 0x4d, 0x8a, 0x4c, 0x1f, 0x4c, 0x98, 0x4b,
  0x20, 0x4b, 0xa7, 0x4a, 0x36, 0x4a, 0xc0, 0x49, 0x44, 0x49, 0xc3, 0x48,
  0x44, 0x48, 0xd5, 0x47, 0x59, 0x47, 0xe6, 0x46, 0x63, 0x46, 0xf8, 0x45,
  0x79, 0x45, 0xfa, 0x44, 0x80, 0x44, 0x01, 0x44, 0x7a, 0x43, 0x04, 0x43,
  0x85, 0x42, 0x07, 0x42, 0x92, 0x41, 0x17, 0x41, 0x94, 0x40, 0x16, 0x40,
  0x9f, 0x3f, 0x20, 0x3f, 0x9c, 0x3e, 0x29, 0x3e, 0x9c, 0x3d, 0x20, 0x3d,
  0x88, 0x3c, 0x0c, 0x3c, 0x87, 0x3b, 0x07, 0x3b, 0x89, 0x3a, 0xfb, 0x39,
  0x6f, 0x39, 0xe7, 0x38, 0x55, 0x38, 0xcf, 0x37, 0x43, 0x37, 0xb5, 0x36,
  0x2a, 0x36, 0x9c, 0x35, 0x0a, 0x35, 0x83, 0x34, 0xe6, 0x33, 0x53, 0x33,
  0xba, 0x32, 0x1d, 0x32, 0x78, 0x31, 0xcb, 0x30, 0x34, 0x30, 0x8e, 0x2f,
  0xf6, 0x2e, 0x56, 0x2e, 0xab, 0x2d, 0xfe, 0x2c, 0x4c, 0x2c, 0x9f, 0x2b,
  0xed, 0x2a, 0x36, 0x2a, 0x7b, 0x29, 0xb9, 0x28, 0x02, 0x28, 0x41, 0x27,
  0x81, 0x26, 0xb9, 0x25, 0xfb, 0x24, 0x2d, 0x24, 0x66, 0x23, 0x98, 0x22,
  0xc6, 0x21, 0xe9, 0x20, 0x0d, 0x20, 0x28, 0x1f, 0x4c, 0x1e, 0x64, 0x1d,
  0x66, 0x1c, 0x8c, 0x1b, 0x96, 0x1a, 0xac, 0x19, 0xb2, 0x18, 0xbb, 0x17,
  0xc2, 0x16, 0xc7, 0x15, 0xcb, 0x14, 0xcd, 0x13, 0xc4, 0x12, 0xca, 0x11,
  0xb6, 0x10, 0xb8, 0x0f, 0xa3, 0x0e, 0x9e, 0x0d, 0x88, 0x0c, 0x86, 0x0b,
  0x7b, 0x0a, 0x7c, 0x09, 0x6d, 0x08, 0x5b, 0x07, 0x4a, 0x06, 0x38, 0x05,
  0x2f, 0x04, 0x1c, 0x03, 0x10, 0x02, 0xf6, 0x00, 0xe9, 0xff, 0xe5, 0xfe,
  0xcf, 0xfd, 0xc8, 0xfc, 0xb1, 0xfb, 0xa7, 0xfa, 0x8d, 0xf9, 0x7b, 0xf8,
  0x7e, 0xf7, 0x77, 0xf6, 0x68, 0xf5, 0x62, 0xf4, 0x49, 0xf3, 0x42, 0xf2,
  0x41, 0xf1, 0x3e, 0xf0, 0x49, 0xef, 0x47, 0xee, 0x56, 0xed, 0x46, 0xec,
  0x56, 0xeb, 0x5c, 0xea, 0x6c, 0xe9, 0x7d, 0xe8, 0x9c, 0xe7, 0x99, 0xe6,
  0xbf, 0xe5, 0xc9, 0xe4, 0xe5, 0xe3, 0xfd, 0xe2, 0x17, 0xe2, 0x26, 0xe1,
  0x44, 0xe0, 0x6d, 0xdf, 0x9f, 0xde, 0xc5, 0xdd, 0xec, 0xdc, 0x26, 0xdc,
  0x58, 0xdb, 0x91, 0xda, 0xb7, 0xd9, 0xf5, 0xd8, 0x27, 0xd8, 0x63, 0xd7,
  0x96, 0xd6, 0xd5, 0xd5, 0x1a, 0xd5, 0x52, 0xd4, 0x9d, 0xd3, 0xde, 0xd2,
  0x27, 0xd2, 0x6c, 0xd1, 0xb3, 0xd0, 0xf6, 0xcf, 0x51, 0xcf, 0x98, 0xce,
  0xe9, 0xcd, 0x2f, 0xcd, 0x88, 0xcc, 0xdc, 0xcb, 0x41, 0xcb, 0x9f, 0xca,
  0x01, 0xca, 0x60, 0xc9, 0xc4, 0xc8, 0x13, 0xc8, 0x7e, 0xc7, 0xe2, 0xc6,
  0x51, 0xc6, 0xb6, 0xc5, 0x1a, 0xc5, 0x8e, 0xc4, 0xf6, 0xc3, 0x74, 0xc3,
  0xdf, 0xc2, 0x59, 0xc2, 0xd2, 0xc1, 0x47, 0xc1, 0xbe, 0xc0, 0x3b, 0xc0,
  0xbe, 0xbf, 0x3e, 0xbf, 0xc4, 0xbe, 0x39, 0xbe, 0xbf, 0xbd, 0x3e, 0xbd,
  0xd7, 0xbc, 0x5d, 0xbc, 0xec, 0xbb, 0x73, 0xbb, 0x07, 0xbb, 0x96, 0xba,
  0x24, 0xba, 0xb5, 0xb9, 0x49, 0xb9, 0xdd, 0xb8, 0x76, 0xb8, 0x02, 0xb8,
  0xa8, 0xb7, 0x3c, 0xb7, 0xee, 0xb6, 0x82, 0xb6, 0x2b, 0xb6, 0xc3, 0xb5,
  0x6e, 0xb5, 0x14, 0xb5, 0xc4, 0xb4, 0x6b, 0xb4, 0x20, 0xb4, 0xd2, 0xb3,
  0x78, 0xb3, 0x2c, 0xb3, 0xe6, 0xb2, 0x9e, 0xb2, 0x58, 0xb2, 0x1a, 0xb2,
  0xea, 0xb1, 0xa6, 0xb1, 0x63, 0xb1, 0x2b, 0xb1, 0xf3, 0xb0, 0xb8, 0xb0,
  0x7b, 0xb0, 0x4e, 0xb0, 0x20, 0xb0, 0xf9, 0xaf, 0xc6, 0xaf, 0xa2, 0xaf,
  0x84, 0xaf, 0x5e, 0xaf, 0x44, 0xaf, 0x21, 0xaf, 0x06, 0xaf, 0xe9, 0xae,
  0xe2, 0xae, 0xd0, 0xae, 0xc3, 0xae, 0xba, 0xae, 0xc1, 0xae, 0xaf, 0xae,
  0xbc, 0xae, 0xb5, 0xae, 0xc0, 0xae, 0xc3, 0xae, 0xd7, 0xae, 0xdd, 0xae,
  0xfb, 0xae, 0x0a, 0xaf, 0x16, 0xaf, 0x37, 0xaf, 0x4a, 0xaf, 0x70, 0xaf,
  0x86, 0xaf, 0xa9, 0xaf, 0xc7, 0xaf, 0xec, 0xaf, 0x12, 0xb0, 0x42, 0xb0,
  0x66, 0xb0, 0x91, 0xb0, 0xc0, 0xb0, 0xee, 0xb0, 0x1a, 0xb1, 0x4c, 0xb1,
  0x77, 0xb1, 0xb3, 0xb1, 0xdd, 0xb1, 0x10, 0xb2, 0x47, 0xb2, 0x70, 0xb2,
  0xb8, 0xb2, 0xef, 0xb2, 0x26, 0xb3, 0x64, 0xb3, 0x90, 0xb3, 0xd8, 0xb3,
  0x07, 0xb4, 0x4b, 0xb4, 0x83, 0xb4, 0xc7, 0xb4, 0xf4, 0xb4, 0x35, 0xb5,
  0x70, 0xb5, 0xbc, 0xb5, 0xf0, 0xb5, 0x39, 0xb6, 0x7d, 0xb6, 0xb7, 0xb6,
  0xfa, 0xb6, 0x2e, 0xb7, 0x7d, 0xb7, 0xbb, 0xb7, 0x01, 0xb8, 0x39, 0xb8,
  0x80, 0xb8, 0xbd, 0xb8, 0x05, 0xb9, 0x46, 0xb9, 0x94, 0xb9, 0xcd, 0xb9,
  0x17, 0xba, 0x5b, 0xba, 0x9e, 0xba, 0xdf, 0xba, 0x27, 0xbb, 0x6a, 0xbb,
  0xbc, 0xbb, 0xf6, 0xbb, 0x3a, 0xbc, 0x83, 0xbc, 0xc3, 0xbc, 0x16, 0xbd,
  0x58, 0xbd, 0xa2, 0xbd, 0xe0, 0xbd, 0x24, 0xbe, 0x70, 0xbe, 0xa1, 0xbe,
  0xfa, 0xbe, 0x37, 0xbf, 0x8d, 0xbf, 0xc9, 0xbf, 0x1d, 0xc0, 0x5d, 0xc0,
  0xa6, 0xc0, 0xe6, 0xc0, 0x2f, 0xc1, 0x7a, 0xc1, 0xcb, 0xc1, 0x2f, 0xc2,
  0x95, 0xc2, 0xfa, 0xc2, 0x55, 0xc3, 0xbb, 0xc3, 0x1e, 0xc4, 0x86, 0xc4,
  0xec, 0xc4, 0x4b, 0xc5, 0xab, 0xc5, 0x04, 0xc6, 0x72, 0xc6, 0xc6, 0xc6,
  0x3a, 0xc7, 0x8e, 0xc7, 0xfd, 0xc7, 0x5a, 0xc8, 0xc0, 0xc8, 0x17, 0xc9,
  0x79, 0xc9, 0xd5, 0xc9, 0x43, 0xca, 0x98, 0xca, 0x05, 0xcb, 0x64, 0xcb,
  0xc3, 0xcb, 0x1e, 0xcc, 0x81, 0xcc, 0xe3, 0xcc, 0x40, 0xcd, 0xa6, 0xcd,
  0x0a, 0xce, 0x6a, 0xce, 0xc0, 0xce, 0x26, 0xcf, 0x81, 0xcf, 0xf0, 0xcf,
  0x48, 0xd0, 0xad, 0xd0, 0x07, 0xd1, 0x71, 0xd1, 0xca, 0xd1, 0x2b, 0xd2,
  0x8a, 0xd2, 0xed, 0xd2, 0x50, 0xd3, 0xa1, 0xd3, 0x0f, 0xd4, 0x6f, 0xd4,
  0xd6, 0xd4, 0x30, 0xd5, 0x93, 0xd5, 0xf6, 0xd5, 0x55, 0xd6, 0xb5, 0xd6,
  0x12, 0xd7, 0x7b, 0xd7, 0xde, 0xd7, 0x34, 0xd8, 0x93, 0xd8, 0xe9, 0xd8,
  0x61, 0xd9, 0xbb, 0xd9, 0x2b, 0xda, 0x88, 0xda, 0xeb, 0xda, 0x4f, 0xdb,
  0xb9, 0xdb, 0x21, 0xdc, 0x80, 0xdc, 0xf7, 0xdc, 0x53, 0xdd, 0xbb, 0xdd,
  0x2b, 0xde, 0xa4, 0xde, 0x05, 0xdf, 0x74, 0xdf, 0xd8, 0xdf, 0x44, 0xe0,
  0xb4, 0xe0, 0x26, 0xe1, 0x91, 0xe1, 0x01, 0xe2, 0x77, 0xe2, 0xeb, 0xe2,
  0x5e, 0xe3, 0xc7, 0xe3, 0x44, 0xe4, 0xb2, 0xe4, 0x25, 0xe5, 0xa3, 0xe5,
  0x15, 0xe6, 0x97, 0xe6, 0x0b, 0xe7, 0x8b, 0xe7, 0x10, 0xe8, 0x7d, 0xe8,
  0x06, 0xe9, 0x74, 0xe9, 0xff, 0xe9, 0x82, 0xea, 0x01, 0xeb, 0x85, 0xeb,
  0xfa, 0xeb, 0x91, 0xec, 0x07, 0xed, 0x8f, 0xed, 0x15, 0xee, 0x9a, 0xee,
  0x26, 0xef, 0xa5, 0xef, 0x26, 0xf0, 0xb2, 0xf0, 0x3e, 0xf1, 0xca, 0xf1,
  0x45, 0xf2, 0xcc, 0xf2, 0x51, 0xf3, 0xdc, 0xf3, 0x5c, 0xf4, 0xe8, 0xf4,
  0x6e, 0xf5, 0xf9, 0xf5, 0x79, 0xf6, 0xf9, 0xf6, 0x7d, 0xf7, 0x08, 0xf8,
  0x91, 0xf8, 0x18, 0xf9, 0x95, 0xf9, 0x18, 0xfa, 0xa5, 0xfa, 0x21, 0xfb,
  0xa9, 0xfb, 0x2a, 0xfc, 0xa7, 0xfc, 0x23, 0xfd, 0xa3, 0xfd, 0x2b, 0xfe,
  0x9c, 0xfe, 0x2b, 0xff, 0xa0, 0xff, 0x1c, 0x00, 0x9a, 0x00, 0x11, 0x01,
  0x9a, 0x01, 0x14, 0x02, 0x8c, 0x02, 0x07, 0x03, 0x84, 0x03, 0xf7, 0x03,
  0x75, 0x04, 0xf3, 0x04, 0x65, 0x05, 0xe8, 0x05, 0x54, 0x06, 0xd3, 0x06,
  0x48, 0x07, 0xc2, 0x07, 0x38, 0x08, 0xad, 0x08, 0x28, 0x09, 0x99, 0x09,
  0x11, 0x0a, 0x79, 0x0a, 0xf5, 0x0a, 0x5c, 0x0b, 0xd1, 0x0b, 0x3f, 0x0c,
  0xb5, 0x0c, 0x2c, 0x0d, 0x9a, 0x0d, 0x0b, 0x0e, 0x74, 0x0e, 0xe6, 0x0e,
  0x45, 0x0f, 0xbd, 0x0f, 0x2f, 0x10, 0x9a, 0x10, 0x05, 0x11, 0x6c, 0x11,
  0xdb, 0x11, 0x40, 0x12, 0xaa, 0x12, 0x14, 0x13, 0x81, 0x13, 0xe9, 0x13,
  0x4d, 0x14, 0xb9, 0x14, 0x18, 0x15, 0x88, 0x15, 0xe6, 0x15, 0x58, 0x16,
  0xb9, 0x16, 0x1c, 0x17, 0x7e, 0x17, 0xe3, 0x17, 0x43, 0x18, 0xac, 0x18,
  0x08, 0x19, 0x6d, 0x19, 0xce, 0x19, 0x34, 0x1a, 0x90, 0x1a, 0xf5, 0x1a,
  0x53, 0x1b, 0xbb, 0x1b, 0x07, 0x1c, 0x6f, 0x1c, 0xc2, 0x1c, 0x28, 0x1d,
  0x84, 0x1d, 0xe8, 0x1d, 0x35, 0x1e, 0xa2, 0x1e, 0xf8, 0x1e, 0x59, 0x1f,
  0xb8, 0x1f, 0x0c, 0x20, 0x69, 0x20, 0xbe, 0x20, 0x19, 0x21, 0x66, 0x21,
  0xc6, 0x21, 0x1f, 0x22, 0x70, 0x22, 0xbe, 0x22, 0x18, 0x23, 0x72, 0x23,
  0xbf, 0x23, 0x0f, 0x24, 0x67, 0x24, 0xc0, 0x24, 0x08, 0x25, 0x51, 0x25,
  0xa6, 0x25, 0xf4, 0x25, 0x42, 0x26, 0x93, 0x26, 0xd6, 0x26, 0x25, 0x27,
  0x6c, 0x27, 0xb7, 0x27, 0x07, 0x28, 0x49, 0x28, 0x96, 0x28, 0xda, 0x28,
  0x22, 0x29, 0x66, 0x29, 0xa4, 0x29, 0xe8, 0x29, 0x20, 0x2a, 0x62, 0x2a,
  0xa9, 0x2a, 0xe5, 0x2a, 0x21, 0x2b, 0x5a, 0x2b, 0x97, 0x2b, 0xc1, 0x2b,
  0xfe, 0x2b, 0x28, 0x2c, 0x57, 0x2c, 0x88, 0x2c, 0xae, 0x2c, 0xd3, 0x2c,
  0xfa, 0x2c, 0x17, 0x2d, 0x33, 0x2d, 0x4b, 0x2d, 0x5d, 0x2d, 0x68, 0x2d,
  0x81, 0x2d, 0x92, 0x2d, 0x9e, 0x2d, 0xa9, 0x2d, 0xbb, 0x2d, 0xc3, 0x2d,
  0xcd, 0x2d, 0xd5, 0x2d, 0xcd, 0x2d, 0xdc, 0x2d, 0xe3, 0x2d, 0xe1, 0x2d,
  0xdd, 0x2d, 0xd9, 0x2d, 0xd7, 0x2d, 0xcf, 0x2d, 0xe0, 0x2d, 0xd2, 0x2d,
  0xd2, 0x2d, 0xcb, 0x2d, 0xb7, 0x2d, 0xb6, 0x2d, 0xa1, 0x2d, 0x9e, 0x2d,
  0x8b, 0x2d, 0x84, 0x2d, 0x68, 0x2d, 0x6a, 0x2d, 0x4e, 0x2d, 0x45, 0x2d,
  0x2f, 0x2d, 0x18, 0x2d, 0x15, 0x2d, 0xef, 0x2c, 0xe8, 0x2c, 0xc9, 0x2c,
  0xc7, 0x2c, 0xaa, 0x2c, 0x9b, 0x2c, 0x7c, 0x2c, 0x64, 0x2c, 0x3c, 0x2c,
  0x2e, 0x2c, 0x16, 0x2c, 0x05, 0x2c, 0xe1, 0x2b, 0xc8, 0x2b, 0xb5, 0x2b,
  0x9a, 0x2b, 0x7c, 0x2b, 0x61, 0x2b, 0x3f, 0x2b, 0x21, 0x2b, 0x01, 0x2b,
  0xee, 0x2a, 0xd0, 0x2a, 0xb3, 0x2a, 0x95, 0x2a, 0x6d, 0x2a, 0x50, 0x2a,
  0x2b, 0x2a, 0x0b, 0x2a, 0xe9, 0x29, 0xd1, 0x29, 0xa8, 0x29, 0x97, 0x29,
  0x6c, 0x29, 0x51, 0x29, 0x24, 0x29, 0x03, 0x29, 0xe6, 0x28, 0xbe, 0x28,
  0xa1, 0x28, 0x80, 0x28, 0x64, 0x28, 0x38, 0x28, 0x12, 0x28, 0xf0, 0x27,
  0xd3, 0x27, 0xac, 0x27, 0x94, 0x27, 0x67, 0x27, 0x4a, 0x27, 0x1c, 0x27,
  0xf7, 0x26, 0xd1, 0x26, 0xb6, 0x26, 0x8d, 0x26, 0x6e, 0x26, 0x4b, 0x26,
  0x25, 0x26, 0x02, 0x26, 0xd8, 0x25, 0xbe, 0x25, 0x93, 0x25, 0x7c, 0x25,
  0x4a, 0x25, 0x27, 0x25, 0x07, 0x25, 0xdc, 0x24, 0xbb, 0x24, 0x9a, 0x24,
  0x78, 0x24, 0x4c, 0x24, 0x26, 0x24, 0x00, 0x24, 0xd7, 0x23, 0xc0, 0x23,
  0x94, 0x23, 0x71, 0x23, 0x45, 0x23, 0x1c, 0x23, 0x00, 0x23, 0xd4, 0x22,
  0xb5, 0x22, 0x8e, 0x22, 0x67, 0x22, 0x39, 0x22, 0x00, 0x22, 0xbd, 0x21,
  0x87, 0x21, 0x46, 0x21, 0x01, 0x21, 0xca, 0x20, 0x89, 0x20, 0x4a, 0x20,
  0x17, 0x20, 0xd6, 0x1f, 0x9a, 0x1f, 0x5e, 0x1f, 0x20, 0x1f, 0xe0, 0x1e,
  0xa3, 0x1e, 0x5f, 0x1e, 0x29, 0x1e, 0xf0, 0x1d, 0xb8, 0x1d, 0x78, 0x1d,
  0x36, 0x1d, 0x04, 0x1d, 0xc6, 0x1c, 0x8b, 0x1c, 0x4d, 0x1c, 0x18, 0x1c,
  0xd9, 0x1b, 0x97, 0x1b, 0x5c, 0x1b, 0x2b, 0x1b, 0xf4, 0x1a, 0xb1, 0x1a,
  0x7c, 0x1a, 0x32, 0x1a, 0x02, 0x1a, 0xc6, 0x19, 0x8b, 0x19, 0x59, 0x19,
  0x10, 0x19, 0xd7, 0x18, 0x9c, 0x18, 0x62, 0x18, 0x22, 0x18, 0xe8, 0x17,
  0xb1, 0x17, 0x7c, 0x17, 0x34, 0x17, 0xf9, 0x16, 0xc0, 0x16, 0x8b, 0x16,
  0x55, 0x16, 0x14, 0x16, 0xdd, 0x15, 0x9f, 0x15, 0x65, 0x15, 0x2b, 0x15,
  0xea, 0x14, 0xb9, 0x14, 0x70, 0x14, 0x3d, 0x14, 0x05, 0x14, 0xc1, 0x13,
  0x86, 0x13, 0x45, 0x13, 0x09, 0x13, 0xca, 0x12, 0x90, 0x12, 0x57, 0x12,
  0x1a, 0x12, 0xe1, 0x11, 0x9b, 0x11, 0x61, 0x11, 0x1d, 0x11, 0xe1, 0x10,
  0x9d, 0x10, 0x64, 0x10, 0x17, 0x10, 0xde, 0x0f, 0xa1, 0x0f, 0x60, 0x0f,
  0x1b, 0x0f, 0xd8, 0x0e, 0x90, 0x0e, 0x5b, 0x0e, 0x0e, 0x0e, 0xd3, 0x0d,
  0x86, 0x0d, 0x4d, 0x0d, 0xf5, 0x0c, 0xba, 0x0c, 0x68, 0x0c, 0x2a, 0x0c,
  0xde, 0x0b, 0x94, 0x0b, 0x4e, 0x0b, 0x07, 0x0b, 0xbb, 0x0a, 0x6c, 0x0a,
  0x25, 0x0a, 0xd4, 0x09, 0x89, 0x09, 0x39, 0x09, 0xe6, 0x08, 0xa3, 0x08,
  0x4f, 0x08, 0x02, 0x08, 0xa7, 0x07, 0x5f, 0x07, 0x08, 0x07, 0xba, 0x06,
  0x68, 0x06, 0x1f, 0x06, 0xc5, 0x05, 0x7a, 0x05, 0x1f, 0x05, 0xd7, 0x04,
  0x85, 0x04, 0x31, 0x04, 0xde, 0x03, 0x89, 0x03, 0x3a, 0x03, 0xe9, 0x02,
  0x96, 0x02, 0x47, 0x02, 0xf7, 0x01, 0xa4, 0x01, 0x5d, 0x01, 0x06, 0x01,
  0xb9, 0x00, 0x00, 0x00
};
unsigned int kick_raw_len = 6352;
