/**
 * SPDX-FileCopyrightText: 2025 Suwatchai K. <suwatchai@outlook.com>
 *
 * SPDX-License-Identifier: MIT
 */

#ifndef NUSOCK_TYPES_H
#define NUSOCK_TYPES_H

#include "NuSockConfig.h"

// Forward declarations
class NuSockServer;
class NuSockServerSecure;

/**
 * @brief Event types generated by the NuSock Server.
 */
enum NuServerEvent
{
    SERVER_EVENT_UBDEFINED,
    SERVER_EVENT_CONNECT,             // Server Started / Network Connected
    SERVER_EVENT_DISCONNECTED,        // Server Stopped / Network Disconnected
    SERVER_EVENT_CLIENT_HANDSHAKE,    // Handshake Request Received
    SERVER_EVENT_CLIENT_CONNECTED,    // Handshake Successful (WebSocket Open)
    SERVER_EVENT_CLIENT_DISCONNECTED, // Client Closed
    SERVER_EVENT_MESSAGE_TEXT,        // Text Frame
    SERVER_EVENT_MESSAGE_BINARY,      // Binary Frame
    SERVER_EVENT_FRAGMENT_START,      // First chunk of a fragmented message (FIN=0, Opcode > 0)
    SERVER_EVENT_FRAGMENT_CONT,       // Middle chunk (FIN=0, Opcode=0)
    SERVER_EVENT_FRAGMENT_FIN,        // Last chunk (FIN=1, Opcode=0)
    SERVER_EVENT_ERROR                // Error
};

/**
 * @brief Event types generated by the NuSock Client.
 */
enum NuClientEvent
{
    CLIENT_EVENT_HANDSHAKE,
    CLIENT_EVENT_CONNECTED,
    CLIENT_EVENT_DISCONNECTED,
    CLIENT_EVENT_MESSAGE_TEXT,
    CLIENT_EVENT_MESSAGE_BINARY,
    CLIENT_EVENT_FRAGMENT_START,
    CLIENT_EVENT_FRAGMENT_CONT,
    CLIENT_EVENT_FRAGMENT_FIN,
    CLIENT_EVENT_ERROR
};

/**
 * @brief Internal Client Wrapper Structure.
 * Holds state, buffers, and the underlying connection handle for a WebSocket client.
 */
struct NuClient
{
    void *server;
    bool isSecure;

#ifdef NUSOCK_USE_LWIP
    struct tcp_pcb *pcb;
#else
    Client *client;
    bool isConnected;
    bool ownsClient;

    // Stored locally to allow duplicate detection without accessing the potentially invalid client object.
    IPAddress remoteIP;
    uint16_t remotePort = 0;
#endif

    char id[32];
    uint8_t *rxBuffer;
    size_t rxLen;
    uint8_t *txBuffer;
    size_t txLen;
    size_t txCap;

    // Stores the opcode of the FIRST fragment (1=Text, 2=Binary)
    // 0 = No active fragmentation
    uint8_t fragmentOpcode = 0;

    // UTF-8 Validation State (0 = Accept)
    uint32_t utf8State = 0; // 0 = NuUTF8::UTF8_ACCEPT

    enum State
    {
        STATE_SSL_HANDSHAKE,
        STATE_HANDSHAKE,
        STATE_CONNECTED,
        STATE_CLOSING
    };
    State state;

    int8_t index = -1;
    NuServerEvent last_event = SERVER_EVENT_UBDEFINED;

#ifdef NUSOCK_USE_LWIP
    template <typename Server>
    NuClient(Server *s, struct tcp_pcb *p)
        : server((void *)s), isSecure(false), pcb(p), rxLen(0), txLen(0), txCap(0), state(STATE_HANDSHAKE)
    {
        rxBuffer = (uint8_t *)malloc(MAX_WS_BUFFER);
        txBuffer = nullptr;
        id[0] = 0;
    }
#else
    template <typename Server>
    NuClient(Server *s, Client *c, bool owns = true)
        : server((void *)s), isSecure(false), client(c), isConnected(true), ownsClient(owns), rxLen(0), txLen(0), txCap(0), state(STATE_HANDSHAKE)
    {
        rxBuffer = (uint8_t *)malloc(MAX_WS_BUFFER);
        txBuffer = nullptr;
        id[0] = 0;
    }
#endif

    ~NuClient()
    {
        if (rxBuffer)
            free(rxBuffer);
        rxBuffer = nullptr;
        if (txBuffer)
            free(txBuffer);
        txBuffer = nullptr;
#ifndef NUSOCK_USE_LWIP
        if (client)
        {
            client->stop();
            if (ownsClient)
            {
#if defined(__GNUC__)
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdelete-non-virtual-dtor"
#endif
                delete client;
#if defined(__GNUC__)
#pragma GCC diagnostic pop
#endif
            }
        }
#endif
    }

    void appendTx(uint8_t b)
    {
        if (txLen >= txCap)
        {
            size_t newCap = (txCap == 0) ? 64 : txCap * 2;
            uint8_t *newBuf = (uint8_t *)realloc(txBuffer, newCap);
            if (!newBuf)
                return;
            txBuffer = newBuf;
            txCap = newCap;
        }
        txBuffer[txLen++] = b;
    }

    void clearTx()
    {
        txLen = 0;
    }
};

#endif